#include "woundDetection.hpp"

#include <girl/girlToger.hpp>
#include <girl/girlHashFunc.hpp>
#include <girl/girlUnorderedSet.hpp>
#include <girl/girlHashMap.hpp>
#include <girl/euclideanPathSet.hpp>

#include <girlTools/girlToolsRegionInfos.hpp>
#include <girlTools/girlToolsPainter.hpp>

#include <ise/iseRegionUnaryPredicateThreshold.hpp>
#include <ise/iseRegionBinaryPredicateFromUnaryPredicate.hpp>
#include <ise/iseRegionBinaryPredicateBoolCombine.hpp>
#include <ise/iseRegionBinaryFunctionWithRegSetEuclideanDistance.hpp>
#include <ise/iseMergeAlgorithmUntil.hpp>
#include <ise/iseMergeDriverDefault.hpp>
#include <ise/iseMergeDriverMeta.hpp>
#include <ise/iseSegmentationContext.hpp>
#include <ise/iseSplitAlgorithmDefault.hpp>
#include <ise/iseSplitDriverDefault.hpp>
#include <ise/iseRegionBinaryFunctionConstantNull.hpp>
#include <ise/iseQuantificationNbColors.hpp>
#include <ise/iseTogerUtils.hpp> //mergeSmallRegions
#include <ise/iseSegmentationClusteringGonogo1.hpp>

#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/ml/ml.hpp>
#include <opencv2/highgui/highgui.hpp>

#include "defineUrgoEye.hpp"
#include "utilsSeg.hpp"
#include "utilsClassifier.hpp"
#include "pointsUtils.hpp"


#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
typedef CvSVM *Classifier;
#else
typedef cv::Ptr<cv::ml::SVM> Classifier;
#endif


typedef girl::unorderedSet<girl::region> USet;

#ifdef SAVE_PROBA
extern std::string g_outputProbaFilename;
extern std::string g_outputProbaFilename1;
extern std::string g_outputProbaFilename2;
extern std::string g_outputProbaFilename3;
extern std::string g_outputProbaFilename4;
extern std::string g_outputCarteProbaFilename;
extern std::string g_outputGrowMarkerFilename;
extern std::string g_outputInsideMaskFilename;
extern std::string g_outputKeptImageFilename;
extern std::string g_outputBurnedPixelFilename;
extern std::string g_outputSkinWoundMarkerFilename;
extern std::string g_outputOutsideMaskFilename;
extern std::string g_outputSkinWithProbaFilename;
extern std::string g_outputProbaFinalFilename;
extern std::string g_outputBeforeFilename;
extern std::string g_outputAfterFilename;
extern std::string g_outputProbaExtendConvexHull;
#endif //SAVE_PROBA

#ifdef SAVE_SEG
extern std::string g_outputSegImageFilename;
extern std::string g_outputSegImageWithPtsFilename;
#endif //SAVE_SEG


float thresholdWoundMerge = 0.70;
float thresholdWoundMergeFinal = 0.667;


#ifdef SAVE_PROBA
static
int 
P_boundedRand(int min, int max)
{
  return static_cast<int>(min + (static_cast<float>(rand()) / RAND_MAX * (max - min + 1)));
}
#endif //SAVE_PROBA


struct Moments
{
  float m0;
  float m1;
  float m2;

  Moments(float _m0=0.f, float _m1=0.f, float _m2=0.f)
    : m0(_m0), m1(_m1), m2(_m2)
  {}

  float getMean() const
  {
    if (m0 != 0)
      return m1 / m0;
    return 0.f;
  }
  
  float getVariance() const
  {
    if (m0 != 0) {
      const float inv_m0 = 1.f / m0;
      return (m2 - m1*m1*inv_m0)*inv_m0;
    }
    return 0.f;
  }

};



namespace {

  class MeanComputer : public girl::toger::pointIterator
  {
  public:
  
    explicit MeanComputer(const cv::Mat &img)
      : m_img(img),
	m_m2(0.0f),
	m_m1(0.0f),
	m_m0(0.0f)
    {
    }

    void clear()
    {
      m_m2 = 0.0f;
      m_m1 = 0.0f;
      m_m0 = 0.0f;
    }

    void processPoint(const girl::point &pt)
    {
      const float v = m_img.at<float>(pt.y(), pt.x());
      m_m2 += v*v;
      m_m1 += v;
      m_m0 += 1;
    }

    void getMoments(float &m0, float &m1) const
    {
      m1 = m_m1;
      m0 = m_m0;
    }
   
    void getMoments(float &m0, float &m1, float &m2) const
    {
      m2 = m_m2;
      m1 = m_m1;
      m0 = m_m0;
    }
   
    float getMean() const
    {
      if (m_m0 != 0)
	return m_m1 / m_m0;
      return 0.f;
    }

    float getVariance() const
    {
      if (m_m0 != 0) {
	const float inv_m0 = 1.f / m_m0;
	return (m_m2 - m_m1*m_m1*inv_m0)*inv_m0;
      }
      return 0.f;
    }

  private:

    MeanComputer(const MeanComputer&);
    MeanComputer &operator=(const MeanComputer&);

  public:
    const cv::Mat &m_img;
    float m_m2;
    float m_m1;
    float m_m0;
  };


  class ValueSetter : public girl::toger::pointIterator
  {
  public:
  
    ValueSetter(cv::Mat &img, float v=0.f)
      : m_img(img),
	m_v(v)
    {
    }

    void reset(float v)
    {
      m_v = v;
    }

    void processPoint(const girl::point &pt)
    {
      m_img.at<float>(pt.y(), pt.x()) = m_v;
    }

  private:

    ValueSetter(const ValueSetter&);
    ValueSetter &operator=(const ValueSetter&);

  public:
    cv::Mat &m_img;
    float m_v;
  };


  class FPainter : public girl::toger::pointIterator {
  
  public:
    explicit FPainter(cv::Mat &mat) 
      : m_mat(mat), m_v(0.f)
    {
      assert(mat.type() == CV_32F);
    }

    void setValue(float v)
    {
      m_v = v;
    }
  
    virtual void processPoint(const girl::point &pt)
    {
      m_mat.at<float>(pt.y(), pt.x()) = m_v;
    }
  
  private:
    cv::Mat &m_mat;
    float m_v;
  };

} //end anonymous namespace


typedef std::vector<cv::Point> PointVec;
typedef std::vector<std::vector<cv::Point> > PointVecVec;
typedef girl::hashMap<girl::region, Moments > RegionMeanProbaH;

/*
  Compute moments and fill region with mean value.

  @param[out] probas  filled probas (moments) for each region in @a regions and toger @a tog.
  @param[in, out] res  once probas are computed, filled with mean value.

*/
template <typename T>
void
meanWithToger(cv::Mat &res, const girl::toger &tog, const T &regions, RegionMeanProbaH &probas)
{
  assert(res.type() == CV_32F);
  assert(res.cols == tog.width() && res.rows == tog.height());

  MeanComputer mc(res);
  ValueSetter vs(res);

  for (typename T::const_iterator it = regions.begin();
       it != regions.end();
       ++it) {

    const girl::region r = *it;

    assert(tog.isRegionValid(r));

    mc.clear();
    tog.iterateOnRegionDomain(r, mc);
    
    float m0, m1, m2;
    mc.getMoments(m0, m1, m2);
    assert(m0 > 0);
    const float mean = m1/m0;

    vs.reset(mean);
    tog.iterateOnRegionDomain(r, vs);

    probas[*it] = Moments(m0, m1, m2);
  }
}

//Fill woundRegionsNew and update @a res for regions which mean value is superior to threshold.
void
meanWithThreshold(cv::Mat &res, USet &woundRegionsNew, RegionMeanProbaH &probas, const girl::toger &tog, float threshold)
{
  assert(res.type() == CV_32F);
  assert(res.cols == tog.width() && res.rows == tog.height());

  woundRegionsNew.clear();
  
  MeanComputer mc(res);
  ValueSetter vs(res);

  girl::regionDeque regions;
  tog.allRegions(regions);
  
  for (girl::regionDeque::const_iterator it = regions.begin();
       it != regions.end();
       ++it) {

    const girl::region r = *it;

    assert(tog.isRegionValid(r));

    mc.clear();
    tog.iterateOnRegionDomain(r, mc);
    
    float m0, m1, m2;
    mc.getMoments(m0, m1, m2);
    assert(m0 > 0);
    const float mean = m1/m0;

    if (mean > threshold) {

      //std::cerr<<"r="<<*it<<" m0="<<m0<<" m1="<<m1<<" m2="<<m2<<" mean="<<mean<<" var="<<(m2 - m1*m1/m0)/m0<<"  kept?="<<(mean > threshold)<<"\n";
    
      woundRegionsNew.insert(r);
      
      vs.reset(mean);
      tog.iterateOnRegionDomain(r, vs);

      probas[*it] = Moments(m0, m1, m2);
    }
  }

  //std::cerr<<"woundRegionsNew="<<woundRegionsNew.size()<<"/"<<regions.size()<<"\n";
}



namespace {

  class MergeDriverLabel1 : public ise::MergeDriverDefault
  {
  public:

    typedef girl::hashMap<girl::region, int> Region2label;

    MergeDriverLabel1(ise::SegmentationContext *segContext, 
		      const Region2label &labels)
      : MergeDriverDefault(segContext), m_labels(labels)
    {
    }

    virtual bool allowed(girl::region r1, girl::region r2, int /*level*/)
    {
      assert(m_labels.find(r1) != m_labels.end());
      assert(m_labels.find(r2) != m_labels.end());
    
      return m_labels.find(r1)->second == m_labels.find(r2)->second; 
    }

  protected:
  
    const Region2label &m_labels;  
  };


  class PainterMask : public girl::toger::pointIterator
  {
  public:

    PainterMask(cv::Mat &imgMask, unsigned char value)
      : m_imgMask(imgMask), m_value(value)
    {
      assert(imgMask.type() == CV_8UC1);
    }

    virtual void processPoint(const girl::point &pt)
    {
      m_imgMask.at<unsigned char>(pt.y(), pt.x()) = m_value;
    }

    virtual void setParameter(unsigned char value)
    {
      m_value = value;
    }

  protected:
    cv::Mat &m_imgMask;
    unsigned char m_value;
  };

} //end anonymous namespace


#ifndef NDEBUG
static
bool
checkNoIntersection(const girl::regionDeque &regions1, 
		    const girl::regionDeque &regions2)
{
  for (girl::regionDeque::const_iterator it1 = regions1.begin();
       it1 != regions1.end(); ++it1) {
  
    for (girl::regionDeque::const_iterator it2 = regions2.begin();
	 it2 != regions2.end(); ++it2) {
      
      if (*it1 == *it2) {
	return false;
      }
    }
  }
  return true;
}

static
bool
checkNoIntersection(const USet &regions1, 
		    const USet &regions2)
{
  for (USet::const_iterator it1 = regions1.begin();
       it1 != regions1.end(); ++it1) {
    if (regions2.find(*it1) != regions2.end()) {
      return false;
    }
  }
  return true;
}

static
bool
checkNoIntersection(const girl::regionDeque &regions1, 
		    const USet &regions2)
{
  for (girl::regionDeque::const_iterator it1 = regions1.begin();
       it1 != regions1.end(); ++it1) {
    if (regions2.find(*it1) != regions2.end()) {
      std::cerr<<"region1 "<<*it1<<" found in regions2\n";
      return false;
    }
  }
  return true;
}

#endif //NDEBUG



#include "SplitAgain.cpp"

#include "SkinExtend.cpp"

//#include "MeanVarianceComputer.cpp"

#include "Hull.cpp"


#if 1 //#ifdef SAVE_SEG
static
void
drawPtsB(cv::Mat &img,
	 const std::vector<cv::Point> &pts,
	 const cv::Scalar &color = cv::Scalar(0, 255, 0))
{
  assert(img.type() == CV_8UC3);
  
  const int radius = 5;

  const size_t numPts = pts.size();
  for (size_t i=0; i<numPts; ++i) {
    cv::circle(img, pts[i], radius, color, -1);
  }
}

static
void
drawPtsB(cv::Mat &img,
	 const std::vector< std::vector<cv::Point> > &ptsV,
	 const cv::Scalar &color = cv::Scalar(0, 255, 0))
{
  const size_t numV = ptsV.size();
  for (size_t i=0; i<numV; ++i) {
    drawPtsB(img, ptsV[i], color);
  }
}
#endif //SAVE_SEG

#ifndef NDEBUG
static
bool
checkPtsNotInRegions(const std::vector< std::vector<cv::Point> > &skinPts, 
		     const girl::toger &tog,
		     const USet &woundRegions)
{
  for (PointVecVec::const_iterator itVV = skinPts.begin();
       itVV != skinPts.end(); 
       ++itVV) {
    const PointVec &pts = *itVV;       
    for (PointVec::const_iterator itV = pts.begin();
	 itV != pts.end(); 
	 ++itV) {
      const girl::region r = tog.findRegion(girl::point(itV->x, itV->y));
      if (woundRegions.find(r) != woundRegions.end())
	return false;
    }
  }
  return true;
}

static
bool
checkPtsNotInRegions(const std::vector< std::vector<cv::Point> > &skinPts, 
		     const girl::toger &tog,
		     const girl::regionDeque &woundRegions)
{
  for (PointVecVec::const_iterator itVV = skinPts.begin();
       itVV != skinPts.end(); 
       ++itVV) {
    const PointVec &pts = *itVV;       
    for (PointVec::const_iterator itV = pts.begin();
	 itV != pts.end(); 
	 ++itV) {
      const girl::region r = tog.findRegion(girl::point(itV->x, itV->y));
      for (girl::regionDeque::const_iterator it = woundRegions.begin(); it!=woundRegions.end(); ++it) {
	if (*it == r) 
	  return false;
      }
    }
  }
  return true;
}

static
bool
checkNotTwoPtsInSameRegion(const std::vector< std::vector<cv::Point> > &woundPts, 
			   const std::vector< std::vector<cv::Point> > &skinPts, 
			   const girl::toger &tog)
{
  USet s1, s2;

  for (PointVecVec::const_iterator itVV = skinPts.begin();
       itVV != skinPts.end(); 
       ++itVV) {
    const PointVec &pts = *itVV;
    for (PointVec::const_iterator itV = pts.begin();
	 itV != pts.end(); 
	 ++itV) {
      const girl::region r1 = tog.findRegion(girl::point(itV->x, itV->y));
      s1.insert(r1);
    }
  }

  for (PointVecVec::const_iterator itVV2 = woundPts.begin();
       itVV2 != woundPts.end(); 
       ++itVV2) {
    const PointVec &pts2 = *itVV2;
    for (PointVec::const_iterator itV2 = pts2.begin();
	 itV2 != pts2.end(); 
	 ++itV2) {

      const girl::region r2 = tog.findRegion(girl::point(itV2->x, itV2->y));
      s2.insert(r2);
    }
  }

  for (USet::const_iterator it1=s1.begin(); it1!=s1.end(); ++it1) {
    for (USet::const_iterator it2=s2.begin(); it2!=s2.end(); ++it2) {
      if (*it1 == *it2) {
	std::cerr<<"two points are in region "<<*it1<<"\n";
	return false;
      }
    }
  }

  return true;
}

#endif //NDEBUG


static
void
updateRegionMeanProbas(const cv::Mat &imgProba,
		       RegionMeanProbaH &regionProbas4,
		       const USet &woundRegionsNew,
		       const ise::SegmentationContext &segContext)
{
  const girl::toger &tog = *(segContext.getToger());

  MeanComputer mc(imgProba);
  
  for (USet::const_iterator it = woundRegionsNew.begin(); it!=woundRegionsNew.end(); ++it) {
    const girl::region r = *it;

    assert(tog.isRegionValid(r));

    mc.clear();
    tog.iterateOnRegionDomain(r, mc);
    
    float m0, m1, m2;
    mc.getMoments(m0, m1, m2);
    assert(m0 > 0);
    const float mean = m1/m0;

    regionProbas4[*it] = Moments(m0, m1, m2);
  }
   
}
		       
// update @a imgProba
static
void
extendWoundWithNeighboursInConvexHull(cv::Mat &imgProba,
				      RegionMeanProbaH &regionProbas4,
				      const USet &woundRegionsNew,
				      ise::SegmentationContext &segContext,
				      const USet &skinRegions)
{

  const girl::toger &tog = *(segContext.getToger());
  
  meanWithToger(imgProba, tog, woundRegionsNew, regionProbas4); //update imgProba & regionProbas4
  
  USet processedRegions;
  
#ifdef SAVE_PROBA
  std::cerr<<"extendWoundWithNeighboursInConvexHull tog.numRegions()="<<tog.numRegions()<<" woundRegionsNew.size()="<<woundRegionsNew.size()<<" skinRegions.size()="<<skinRegions.size()<<"\n";
  size_t cpt = 0;
#endif //SAVE_PROBA
  //size_t totalArea = 0;

  for (USet::const_iterator it = woundRegionsNew.begin(); it!=woundRegionsNew.end(); ++it) 
    {

      const girl::region r = *it;
      ConvexHull hull(tog, r);


      girl::regionDeque holes;
      tog.regionNeighbourhood(r, girl::toger::NEIGHBOUR_INNER, girl::toger::MARK_NONE, holes);
      
      //std::cerr<<"   holes.size()="<<holes.size()<<"\n";

      girl::regionDeque tmp;
      tmp.push_back(r);
      while (! tmp.empty())
	{
	  girl::region rcur = tmp.front();
	  tmp.pop_front();
	  
	  if ((processedRegions.find(rcur) == processedRegions.end()) && (skinRegions.find(rcur) == skinRegions.end()))
	    {
	      processedRegions.insert(rcur);

	      girl::regionDeque neighbors;
	      tog.regionNeighbourhood(rcur, girl::toger::NEIGHBOUR_ANY , girl::toger::MARK_NONE, neighbors);
	      
	      for (girl::regionDeque::const_iterator itN = neighbors.begin();
		   itN != neighbors.end();
		   ++itN)
		{
		  if (std::find(holes.begin(), holes.end(), *itN) == holes.end()) //not an original hole
		    {
		      const girl::region r2a = *itN;
		      //B: add this to also traverse holes of neighbours regions that would not pass the test on meanReg !
		      
		      girl::regionDeque neighbors2;
		      tog.regionNeighbourhood(r2a, girl::toger::NEIGHBOUR_INNER , girl::toger::MARK_NONE, neighbors2);
		      neighbors2.push_back(r2a);
		      for (girl::regionDeque::const_iterator itN2 = neighbors2.begin();
			   itN2 != neighbors2.end();
			   ++itN2) 
			{
			  const girl::region r2 = *itN2;
			  RegionMeanProbaH::const_iterator itH_ALL = regionProbas4.find(r2);
			  assert(itH_ALL != regionProbas4.end());
			  const float meanReg = itH_ALL->second.getMean();
			  if (meanReg > thresholdWoundMergeFinal1) 
			    {		    
			      const float ratioAreaInside = hull.ratioAreaInside(tog, r2);
			      if (ratioAreaInside > thresholdRatioAreaIncluded) 
				tmp.push_back(r2);
			    }
			}
		    }
		}
	    }
	}
    }
  
  girl::hashMap<girl::region, int> region2label;
  {
    girl::regionDeque allRegions;
    tog.allRegions(allRegions);
    //std::cerr<<"allRegions.size()="<<allRegions.size()<<"\n";
    for (girl::regionDeque::const_iterator it = allRegions.begin(); it!=allRegions.end(); ++it) {
      const girl::region r = *it;
      region2label[r] = 0;
    }
    for (USet::const_iterator it = processedRegions.begin(); it!=processedRegions.end(); ++it) {
      const girl::region r = *it;
      region2label[r] = 1;
    }
    
    MergeDriverLabel1 md(&segContext, region2label);
    ise::MergeAlgorithmUntil mergeAlgo(&segContext);
    
    girl::regionDeque mergedRegions;
    mergeAlgo.apply(allRegions, md, 1, mergedRegions);
  }

  //std::cerr<<"After merge :  tog.numRegions()="<<tog.numRegions()<<"\n";
}


// all region for which mean proba is superior to threshold are considered as wound regions
// They are then merged and regionProbas4 is updated.
// update @a imgProba
static
void
extendWoundWithThreshold(cv::Mat &imgProba,
			 RegionMeanProbaH &regionProbas4,
			 USet &woundRegionsNew,
			 ise::SegmentationContext &segContext,
			 float threshold)
{
  const girl::toger &tog = *(segContext.getToger());

  meanWithThreshold(imgProba, woundRegionsNew, regionProbas4, tog, threshold); //update imgProba & regionProbas4, and fill woundRegionsNew

  girl::hashMap<girl::region, int> region2label;
  {
    girl::regionDeque allRegions;
    tog.allRegions(allRegions);
    //std::cerr<<"allRegions.size()="<<allRegions.size()<<"\n";
    for (girl::regionDeque::const_iterator it = allRegions.begin(); it!=allRegions.end(); ++it) {
      const girl::region r = *it;
      region2label[r] = 0;
    }
    for (USet::const_iterator it = woundRegionsNew.begin(); it!=woundRegionsNew.end(); ++it) {
      const girl::region r = *it;
      region2label[r] = 1;
    }
    
    MergeDriverLabel1 md(&segContext, region2label);
    ise::MergeAlgorithmUntil mergeAlgo(&segContext);
    
    girl::regionDeque mergedRegions;
    mergeAlgo.apply(allRegions, md, 1, mergedRegions);
  }

  //B:TODO: merge too small (wound) regions with background and remove them from woundRegionsNew.
  
  {
    USet woundRegionsNew2;
    girl::regionDeque allRegions;
    tog.allRegions(allRegions);
    for (girl::regionDeque::const_iterator it = allRegions.begin(); it!=allRegions.end(); ++it) {
      if (woundRegionsNew.find(*it) != woundRegionsNew.end())
	woundRegionsNew2.insert(*it);
    }
    std::swap(woundRegionsNew2, woundRegionsNew);
  }

  
  updateRegionMeanProbas(imgProba, regionProbas4, woundRegionsNew, segContext);
}




static
void
segmentation(const cv::Mat &img, 
	     cv::Mat &imgProba, 
	     cv::Mat &outImgC,			    
	     const std::vector< std::vector<cv::Point> > &skinPts,
	     const std::vector< std::vector<cv::Point> > &woundPts,
	     const cv::Mat &markerMask)
{
  assert(imgProba.type() == CV_32FC1);
  assert(outImgC.type() == CV_8UC1);
  
  girl::toger tog(img.cols, img.rows);
  girl::regionDeque maskRegions, nonMaskRegions;

  cv::Mat imgProbaInit = imgProba; 

  splitWithMask(markerMask, tog, maskRegions, nonMaskRegions);

  girl::tools::Image gtImg;
  BGR2girlToolsImage(img, gtImg);
  ise::SegmentationContext segContext(&tog, &gtImg);
  
  
  ise::SplitAlgorithmDefault splitAlgo(&segContext);
  ise::SplitDriverDefault    splitDriver(&segContext);

#if 1
  ise::SegmentationClusteringGonogo1 g(&segContext);
  const int nbClasses = 20; 
  const int regionArea = 25;
  const int flashLuminanceThreshold = thBurnedPixel;
  g.setParameters(ISE_RGB_to_Lab, nbClasses, regionArea, flashLuminanceThreshold);
#else
  const int nbColors = 12; //26; //12;
  ise::QuantificationNbColors g(&segContext, ISE_RGB_to_Lab, nbColors);
#endif

  ise::SegmentationClustering &segClust = g;

  girl::regionDeque createdRegionsOutsideMask; 
  splitAlgo.apply(nonMaskRegions, splitDriver, segClust, 0, createdRegionsOutsideMask);

  girl::regionDeque createdRegionsInsideMask; 
  splitAlgo.apply(maskRegions, splitDriver, segClust, 0, createdRegionsInsideMask);
  
  assert(checkNoIntersection(createdRegionsOutsideMask, createdRegionsInsideMask));

  //assert(checkPtsNotInRegions(woundPts, tog, createdRegionsInsideMask));
  //assert(checkPtsNotInRegions(skinPts, tog, createdRegionsOutsideMask));


#ifdef SAVE_PROBA
  { 
    cv::Mat skinImg = img.clone();
    girl::tools::Painter<cv::Vec3b> p(skinImg, cv::Scalar(0, 255, 0));
    for (girl::regionDeque::const_iterator it = createdRegionsOutsideMask.begin(); it!=createdRegionsOutsideMask.end(); ++it) {
      p.setColor(cv::Scalar(0, P_boundedRand(55, 255), 0));
      tog.iterateOnRegionDomain(*it, p);
    }
    const std::string filename = g_outputOutsideMaskFilename;
    std::cerr<<"save proba file: "<<filename<<"\n";
    cv::imwrite(filename, skinImg);
  }
  { 
    cv::Mat skinImg = img.clone();
    girl::tools::Painter<cv::Vec3b> p(skinImg, cv::Scalar(0, 0, 255));
    for (girl::regionDeque::const_iterator it = createdRegionsInsideMask.begin(); it!=createdRegionsInsideMask.end(); ++it) {
      p.setColor(cv::Scalar(0, 0, P_boundedRand(55, 255)));
      tog.iterateOnRegionDomain(*it, p);
    }
    const std::string filename = g_outputInsideMaskFilename;
    std::cerr<<"save proba file: "<<filename<<"\n";
    cv::imwrite(filename, skinImg);
  }
#endif


  RegionMeanProbaH regionProbas4;

  //update createdRegionsOutsideMask & createdRegionsInsideMask
  splitAgain(imgProba, regionProbas4, splitAlgo, splitDriver, createdRegionsOutsideMask, segContext);
  splitAgain(imgProba, regionProbas4, splitAlgo, splitDriver, createdRegionsInsideMask, segContext);
  
  //assert(checkPtsNotInRegions(woundPts, tog, createdRegionsInsideMask));
  assert(checkNoIntersection(createdRegionsOutsideMask, createdRegionsInsideMask));

  splitAgainIfPtsInWrongRegions(imgProba, regionProbas4, splitAlgo, splitDriver, createdRegionsOutsideMask, createdRegionsInsideMask, segContext, skinPts, woundPts);
  splitAgainIfPtsInWrongRegions(imgProba, regionProbas4, splitAlgo, splitDriver, createdRegionsInsideMask, createdRegionsOutsideMask, segContext, woundPts, skinPts);


  assert(checkNoIntersection(createdRegionsOutsideMask, createdRegionsInsideMask));
  assert(checkPtsNotInRegions(skinPts, tog, createdRegionsOutsideMask));

  assert(checkNotTwoPtsInSameRegion(woundPts, skinPts, tog));


#ifdef SAVE_PROBA
  { 
    cv::Mat skinImg = img.clone();
    girl::tools::Painter<cv::Vec3b> p(skinImg, cv::Scalar(0, 255, 0));
    for (girl::regionDeque::const_iterator it = createdRegionsOutsideMask.begin(); it!=createdRegionsOutsideMask.end(); ++it) {
      p.setColor(cv::Scalar(0, P_boundedRand(55, 255), 0));
      tog.iterateOnRegionDomain(*it, p);
    }
    const std::string filename = g_outputOutsideMaskFilename + "_2.png";
    std::cerr<<"save proba file: "<<filename<<"\n";
    cv::imwrite(filename, skinImg);
  }
  { 
    cv::Mat skinImg = img.clone();
    girl::tools::Painter<cv::Vec3b> p(skinImg, cv::Scalar(0, 0, 255));
    for (girl::regionDeque::const_iterator it = createdRegionsInsideMask.begin(); it!=createdRegionsInsideMask.end(); ++it) {
      p.setColor(cv::Scalar(0, 0, P_boundedRand(55, 255)));
      tog.iterateOnRegionDomain(*it, p);
    }
    const std::string filename = g_outputInsideMaskFilename + "_2.png";
    std::cerr<<"save proba file: "<<filename<<"\n";
    cv::imwrite(filename, skinImg);
  }
#endif



  girl::regionDeque allRegions1;

  tog.allRegions(allRegions1);
  meanWithToger(imgProba, tog, allRegions1, regionProbas4); //update imgProba & regionProbas4

  USet skinRegions;

  for (girl::regionDeque::const_iterator it = createdRegionsInsideMask.begin(); 
       it!=createdRegionsInsideMask.end();
       ++it) 
    skinRegions.insert(*it);

  {
    for (PointVecVec::const_iterator itVV = skinPts.begin();
	 itVV != skinPts.end(); 
	 ++itVV) {
      const PointVec &pts = *itVV;       
      for (PointVec::const_iterator itV = pts.begin();
	   itV != pts.end(); 
	   ++itV) {
	const girl::region r = tog.findRegion(girl::point(itV->x, itV->y));
	skinRegions.insert(r);
      }
    }
  }
    
  assert(checkPtsNotInRegions(woundPts, tog, skinRegions));

  girl::regionDeque woundRegions;
  selectRegionsWound(regionProbas4, woundRegions, woundPts, tog, skinRegions); //update woundRegions


  assert(checkPtsNotInRegions(skinPts, tog, woundRegions));
  assert(checkPtsNotInRegions(woundPts, tog, skinRegions));





#ifdef SAVE_PROBA
  { 
    cv::Mat imgProbaIMG = imgProba.clone();
    makeBGRImageFromImageProba(imgProba, imgProba.rows, imgProba.cols, imgProbaIMG, false);
    const std::string filename1 = g_outputProbaFinalFilename;
    cv::imwrite(filename1, imgProbaIMG);
    
    girl::tools::Painter<cv::Vec3b> p(imgProbaIMG, cv::Scalar(255, 0, 0));
    for (USet::const_iterator it = skinRegions.begin(); it!=skinRegions.end(); ++it) {
      p.setColor(cv::Scalar(P_boundedRand(210, 255), 0, 0));
      tog.iterateOnRegionDomain(*it, p);
    }
    
    girl::tools::Painter<cv::Vec3b> p1(imgProbaIMG, cv::Scalar(255, 255, 0));
    for (girl::regionDeque::const_iterator it = woundRegions.begin(); it!=woundRegions.end(); ++it) {
      p1.setColor(cv::Scalar(255, 255, P_boundedRand(0, 55)));
      tog.iterateOnRegionDomain(*it, p1);
    }
    
    const std::string filename = g_outputSkinWithProbaFilename;
    std::cerr<<"write "<<filename<<"\n";
    cv::imwrite(filename, imgProbaIMG);
  }
  {
    cv::Mat skinImg = img.clone();
    girl::tools::Painter<cv::Vec3b> p1(skinImg, cv::Scalar(255, 255, 0));
    for (girl::regionDeque::const_iterator it = woundRegions.begin(); it!=woundRegions.end(); ++it) {
      tog.iterateOnRegionDomain(*it, p1);
    }
    const std::string filename = g_outputSkinWoundMarkerFilename;
    std::cerr<<"save proba file: "<<filename<<"\n";
    cv::imwrite(filename, skinImg);
  }
#endif //SAVE_PROBA
  
  girl::regionDeque allRegions;
  
  girl::hashMap<girl::region, int> region2label1;
  USet woundRegionsNew;
  
  {
    tog.allRegions(allRegions);
    for (girl::regionDeque::const_iterator it = allRegions.begin(); it!=allRegions.end(); ++it) {
      const girl::region r = *it;
      region2label1[r] = 0;
    }
    for (girl::regionDeque::const_iterator it = woundRegions.begin(); it!=woundRegions.end(); ++it) {
      const girl::region r = *it;
      region2label1[r] = 1;
    }

#ifdef SAVE_PROBA //DEBUG
    std::cerr<<"BEFORE MERGE: tog.numRegions()="<<tog.numRegions()<<" allREgions.size()="<<allRegions.size()<<" woundRegions.size()="<<woundRegions.size()<<" skinRegions=tog.numRegions()-woundRegions.size()="<<tog.numRegions()-woundRegions.size()<<"\n";
#endif //SAVE_PROBA


    assert(region2label1.size() == allRegions.size());

    //merge all wound regions with the same label           
    MergeDriverLabel1 md(&segContext, region2label1);
    ise::MergeAlgorithmUntil mergeAlgo(&segContext);
    girl::regionDeque mergedRegions;
    mergeAlgo.apply(woundRegions, md, 1, mergedRegions);
    //mergedRegions is not valid here !!!
    
    //update wound regions
    for (PointVecVec::const_iterator itVV = woundPts.begin();
	 itVV != woundPts.end(); 
	 ++itVV) {
      const PointVec &pts = *itVV;       
      for (PointVec::const_iterator itV = pts.begin();
	   itV != pts.end(); 
	   ++itV) {
	const girl::region r = tog.findRegion(girl::point(itV->x, itV->y));
	assert(r != 0);
	woundRegionsNew.insert(r);
      }
    }

    //update skin regions
    skinRegions.clear();
    for (PointVecVec::const_iterator itVV = skinPts.begin();
	 itVV != skinPts.end(); 
	 ++itVV) {
      const PointVec &pts = *itVV;       
      for (PointVec::const_iterator itV = pts.begin();
	   itV != pts.end(); 
	   ++itV) {
	const girl::region r = tog.findRegion(girl::point(itV->x, itV->y));
	assert(r != 0);
	skinRegions.insert(r);
      }
    }    

    assert(checkPtsNotInRegions(skinPts, tog, woundRegionsNew));
    assert(checkPtsNotInRegions(woundPts, tog, skinRegions));
    //B: REM: WE DO NOT CHECK WETHER A REGION APPEARS BOTH IN woundRegionsNew & skinRegions !

  }


#ifdef SAVE_PROBA

  std::cerr<<"AFTER MERGE: tog.numRegions()="<<tog.numRegions()<<" woundRegionsNew.size()="<<woundRegionsNew.size()<<" skinRegions.size()="<<skinRegions.size()<<"\n";
  { 
    cv::Mat imgProbaIMG = imgProba.clone();
    makeBGRImageFromImageProba(imgProba, imgProba.rows, imgProba.cols, imgProbaIMG, false);
    
    girl::tools::Painter<cv::Vec3b> p(imgProbaIMG, cv::Scalar(255, 0, 0));
    for (USet::const_iterator it = skinRegions.begin(); it!=skinRegions.end(); ++it) {
      p.setColor(cv::Scalar(P_boundedRand(210, 255), 0, 0));
      tog.iterateOnRegionDomain(*it, p);
    }
    
    girl::tools::Painter<cv::Vec3b> p1(imgProbaIMG, cv::Scalar(255, 255, 0));
    for (USet::const_iterator it = woundRegionsNew.begin(); it!=woundRegionsNew.end(); ++it) {
      p1.setColor(cv::Scalar(255, 255, P_boundedRand(0, 55)));
      tog.iterateOnRegionDomain(*it, p1);
    }

    drawPtsB(imgProbaIMG, woundPts, cv::Scalar(255, 100, 20));
    drawPtsB(imgProbaIMG, skinPts, cv::Scalar(240, 240, 210));

    
    const std::string filename = g_outputSkinWithProbaFilename + "_2.png";
    std::cerr<<"write "<<filename<<"\n";
    cv::imwrite(filename, imgProbaIMG);
  }
#endif //SAVE_PROBA

  //20/03/2018 : to handle "no wound point" case
  if (! woundRegionsNew.empty()) {

    extendWoundWithNeighboursInConvexHull(imgProba,
					  regionProbas4,
					  woundRegionsNew,
					  segContext,
					  skinRegions);

    woundRegionsNew.clear();
    for (PointVecVec::const_iterator itVV = woundPts.begin();
	 itVV != woundPts.end(); 
	 ++itVV) {
      
      const PointVec &pts = *itVV;      

      for (PointVec::const_iterator itV = pts.begin();
	   itV != pts.end(); 
	   ++itV) {
	const girl::region r = tog.findRegion(girl::point(itV->x, itV->y));
	assert(r != 0);
	woundRegionsNew.insert(r);
      }
    }

    updateRegionMeanProbas(imgProba, regionProbas4, woundRegionsNew, segContext);
  }
  else {
    const float THRESHOLD_WOUND = 0.62;
    extendWoundWithThreshold(imgProba, regionProbas4, woundRegionsNew, segContext, THRESHOLD_WOUND);
  }


#ifdef SAVE_PROBA
    std::cerr<<"After extendWoundWithNeighboursInConvexHull tog.numRegions()="<<tog.numRegions()<<" woundRegionsNew.size()="<<woundRegionsNew.size()<<"\n";
    {

      cv::Mat imgProbaIMG = imgProba.clone();
      makeBGRImageFromImageProba(imgProba, imgProba.rows, imgProba.cols, imgProbaIMG, false);

      const cv::Scalar color(255, 0, 0, 0);
      girl::tools::Painter<cv::Vec3b> p1(imgProbaIMG, cv::Scalar(255, 255, 0));
      for (USet::const_iterator it = woundRegionsNew.begin(); it!=woundRegionsNew.end(); ++it) 
	{
	  ConvexHull hull(tog, *it);
	  tog.iterateOnRegionDomain(*it, p1);
	  drawHull(imgProbaIMG, hull, color);
	}
      
      const std::string filename2 = g_outputKeptImageFilename + "_tmp.png";
      std::cerr<<"save proba file: "<<filename2<<"\n";
      cv::imwrite(filename2, imgProbaIMG);
    }
#endif //SAVE_PROBA

  girl::hashMap<girl::region, int> region2label;
  {
    tog.allRegions(allRegions);
    for (girl::regionDeque::const_iterator it = allRegions.begin(); it!=allRegions.end(); ++it) {
      const girl::region r = *it;
      region2label[r] = 0;
    }
    for (USet::const_iterator  it = woundRegionsNew.begin(); it!=woundRegionsNew.end(); ++it) {
      const girl::region r = *it;
      region2label[r] = 1;
    }
  }


#ifdef SAVE_PROBA
  std::cerr<<"after merge: tog.numRegions()="<<tog.numRegions()<<"\n";
#endif //SAVE_PROBA

  //-merge small holes in wound regions
  //
  // - merge very small regions without checking their mean proba
  // - merge not so small regions checking their mean proba

  const double THRESHOLD_AREAG_PERCENT1 = 0.1; //0.05; //0.0068;
  const double THRESHOLD_AREAG_PERCENT2 = 0.2; //0.05; //0.0068;
  const float THRESHOLD_PROBA = 0.2;
  
  assert(THRESHOLD_AREAG_PERCENT1 < THRESHOLD_AREAG_PERCENT2);

  tog.allRegions(allRegions);

  for (girl::regionDeque::const_iterator it = allRegions.begin(); it!=allRegions.end(); ++it) 
    {
      girl::region r = *it;
      if (tog.isRegionValid(r)) 
	{
	  
	  if(region2label[r] == 1) //this is a valid wound region
	    {
	      const size_t areaReg        = girl::tools::computeArea(tog, r);

	      const double thresholdAreaG1 = THRESHOLD_AREAG_PERCENT1*areaReg;
	      const double thresholdAreaG2 = THRESHOLD_AREAG_PERCENT2*areaReg;
	      
	      girl::regionDeque holes;
	      tog.regionNeighbourhood(r, girl::toger::NEIGHBOUR_INNER, girl::toger::MARK_NONE, holes);
	      if (! holes.empty()) {
		//region has at least one hole, it will be merged with its mother if its area is small enough
		for (girl::regionDeque::const_iterator it1 = holes.begin();
		     it1 != holes.end();
		     ++it1) 
		  {
		    const girl::region r2 = *it1;

		    const size_t area = girl::tools::computeArea(tog, r2);

		    if (area < thresholdAreaG1) {
		      r = tog.mergeRegions(r, *it1);
		    }
		    else {
		      RegionMeanProbaH::const_iterator itH = regionProbas4.find(r2);
		      assert(itH != regionProbas4.end());
		      const float meanP = itH->second.getMean();
		      if (area < thresholdAreaG2 && meanP > THRESHOLD_PROBA) {
			r = tog.mergeRegions(r, r2);
		      }
		    }
		    
		  }
	      }
	    }
	}
    }


  PainterMask pr(outImgC, 255);
  for (USet::const_iterator  it = woundRegionsNew.begin(); it!=woundRegionsNew.end(); ++it) {
    const girl::region r = *it;
    if (tog.isRegionValid(r)) {
      tog.iterateOnRegionDomain(r, pr);
    }
  }

  
  filterKeepOnlyRegionsWithWoundPts(outImgC, woundPts);

}


#ifdef SAVE_SEG

namespace {

  class Splitter8 : public girl::toger::sameRegion
  {
    const cv::Mat &m_imgP;

  public:
    explicit Splitter8(const cv::Mat &imgP)
      : m_imgP(imgP)
    {
      assert(imgP.type() == CV_8UC1);
    }

    virtual bool same(const girl::point &p, girl::direction dir)
    {
      girl::point p2 = p;
      p2.move(dir);

      const bool b1 = ( m_imgP.at<unsigned char>(p.y(), p.x()) < 128 );
      const bool b2 = ( m_imgP.at<unsigned char>(p2.y(), p2.x()) < 128 );
    
      return b1 == b2;
    }
		
  };

} //end anonymous namespace


cv::Mat
makeImageWithBorderAndPts(const cv::Mat &img, 
			  const cv::Mat &woundMask, 
			  const std::vector< std::vector<cv::Point> > &skinPts,
			  const std::vector< std::vector<cv::Point> > &woundPts)
{
  assert(img.rows == woundMask.rows && img.cols == woundMask.cols);
  assert(img.type() == CV_8UC3);
  assert(woundMask.type() == CV_8UC1);

  girl::toger tog(img.cols, img.rows);
  
  girl::regionDeque createdRegions;
  Splitter8 s(woundMask);
  tog.splitRegion(1, s, createdRegions);


  girl::euclideanPathSet eps(tog.width(), tog.height());
  eps.initFromAllRegions(tog);

  cv::Mat labelImg = img.clone();
  
  drawPtsB(labelImg, woundPts, cv::Scalar(0, 0, 255));
  drawPtsB(labelImg, skinPts, cv::Scalar(255, 255, 0));

  girl::tools::CvPainter p(labelImg, cv::Scalar(0, 255, 0), 2);
  girl::tools::drawToger(p, tog, eps);
  
  return labelImg;
}

#endif //SAVE_SEG



static 
void 
makeProbaCumulation(const cv::Mat &res1, const cv::Mat &res2, const cv::Mat &/*res3*/, cv::Mat &res4)
{
  
  //size_t nbSkinPixels = 0;
  size_t nbWoundPixels = 0;
  float sumWoundPixels = 0;

  res4 = res1.clone();

  const float *pf1 = res1.ptr<float>(0);
  const float *pf2 = res2.ptr<float>(0);
  
  float *d = res4.ptr<float>(0);
  
  for (int j=0; j<res1.rows*res1.cols; ++j) {
    float value = *pf2;

    if(*pf1 > value) 
      value = *pf1;
   

    if(value < 0.5) {
      value = 0.0f;
      //++nbSkinPixels;      
    }
    else {
      ++nbWoundPixels;
      sumWoundPixels += value;
    }

    *d =  value;

    ++pf1;
    ++pf2;
    //++pf3;
    ++d;
  }

  float mean = sumWoundPixels/(float)nbWoundPixels;

  mean = mean < 0.60 ? 0.65:mean;
  mean = mean > 0.75 ? 0.75:mean; 

  thresholdWoundMerge = mean + 0.05;
  thresholdWoundMergeFinal = mean  - 0.05;

}


/*
  res is of type CV_32FC1 and has 1 column.
  mask is of type CV_8UC1.

  we resize res to be of the same size than mask,
  and put 0 where mask is 0. 
*/
static
cv::Mat reshapeResultWithMask(const cv::Mat &res, const cv::Mat &mask)
{
  assert(res.type() == CV_32FC1);
  assert(mask.type() == CV_8UC1);
  assert(res.cols == 1);
  assert(res.rows <= mask.rows*mask.cols);

  cv::Mat res2(mask.rows, mask.cols, CV_32FC1);
  int rows = mask.rows;
  int cols = mask.cols;
  if (mask.isContinuous() && res.isContinuous() && res2.isContinuous()) {
    cols *= rows;
    rows = 1;
  }

  for (int i=0; i<rows; ++i) {
    float *d = res2.ptr<float>(i);
    const float *s = res.ptr<float>(i);
    const unsigned char *m = mask.ptr<unsigned char>(i);
    for (int j=0; j<cols; ++j) {
      if (*m) {
	*d = *s;
	++s;
      }
      else {
	*d = 0;
      }
      ++d;
      ++m;
    }
  }
  
  return res2;
}

 
#define DONT_USE_MASK 0


static 
void
computeImageProba(Classifier &classifier1, 
		  Classifier &classifier2, 
		  Classifier &classifier3, 
		  const cv::Mat &img, 
		  const cv::Mat &markerMask,
		  const cv::Mat &imgLAB, 
		  ColorSpace cs,
		  int patchSize, PatchType patchType, 
		  bool useL, 
		  const Normalizer &n1, const Normalizer &n2, const Normalizer &n3, 
		  const MinMaxP &m1, const MinMaxP &m2, const MinMaxP &m3,
		  bool useClassifier3,
		  cv::Mat &res4)
{

  cv::Mat res1, res2, res3;
  
  assert(img.channels() == 3);
  
#if DONT_USE_MASK
  cv::Mat mask2 = cv::Mat::ones(img.rows, img.cols, CV_8UC1); //TODO:OPTIM: remove !!
#else
  cv::Mat mask2 = 255-markerMask;
#endif //USE_MASK
  assert(mask2.rows == img.rows && mask2.cols == img.cols);
  
  cv::Mat imgCS = imgLAB;
  if (cs != eLAB) 
    convert(img, cs, imgCS);
  
  cv::Mat testData(img.rows*img.cols, getLABsSize(patchSize, patchType, useL), CV_32F);
  
  const bool removeSaturated = false;
  
#ifdef TIMING
  double t = (double)cv::getTickCount();
#endif
  
  accumulateImagePatch(imgCS, mask2, testData, patchSize, patchType, useL, removeSaturated);
  
#ifdef TIMING
  t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
  std::cout<<"[classification] Time for data accumulation : "<<t<<"s   (rows="<<testData.rows<<" cols="<<testData.cols<<")"<<std::endl;
  //t = (double)cv::getTickCount();
#endif
  
  //std::cerr<<"testData: rows="<<testData.rows<<" cols="<<testData.cols<<" type="<<testData.type()<<"\n";
  
  
  cv::Mat testData1 = testData.clone();
  cv::Mat testData2 = testData.clone();
  cv::Mat testData3;
  
  n1.normalizeMat(testData1); //do nothing if normalizeFilename was empty
  n2.normalizeMat(testData2); //do nothing if normalizeFilename was empty
  if (useClassifier3) {
    testData3 = testData.clone();
    n3.normalizeMat(testData3); //do nothing if normalizeFilename was empty
  }


  const bool returnDFVal = true;
  const bool normalize_r = true;

  //we use minP/maxP obtained during training
  classifyData(classifier1, testData1, res1, returnDFVal, normalize_r, 0, 0, m1.minP, m1.maxP);
  classifyData(classifier2, testData2, res2, returnDFVal, normalize_r, 0, 0, m2.minP, m2.maxP);
  if (useClassifier3)
    classifyData(classifier3, testData3, res3, returnDFVal, normalize_r, 0, 0, m2.minP, m2.maxP);


#if DONT_USE_MASK
  res1 = res1.reshape(1, img.rows);
  res2 = res2.reshape(1, img.rows);
  if (useClassifier3)
    res3 = res3.reshape(1, img.rows);
  else 
    res3 = cv::Mat::zeros(img.rows, img.cols, res1.type());

#else
  res1 = reshapeResultWithMask(res1, mask2);
  res2 = reshapeResultWithMask(res2, mask2);
  if (useClassifier3)
    res3= reshapeResultWithMask(res3, mask2);
  else
    res3 = cv::Mat::zeros(mask2.rows, mask2.cols, res1.type());
#endif



  
  makeProbaCumulation (res1, res2, res3, res4);
   

#if SAVE_PROBA
  {
    cv::Mat imgRes1, imgRes2, imgRes3, imgRes4;
    bool normalize = true;
    makeBGRImageFromImageProba(res1, img.rows, img.cols, imgRes1, normalize);
    std::cerr<<"save proba1 file: "<<g_outputProbaFilename1<<"\n";
    cv::imwrite(g_outputProbaFilename1, imgRes1);
    makeBGRImageFromImageProba(res2, img.rows, img.cols, imgRes2, normalize);
    std::cerr<<"save proba2 file: "<<g_outputProbaFilename2<<"\n";
    cv::imwrite(g_outputProbaFilename2, imgRes2);
    if (useClassifier3) {
      makeBGRImageFromImageProba(res3, img.rows, img.cols, imgRes3, normalize);
      std::cerr<<"save proba3 file: "<<g_outputProbaFilename3<<"\n";
      cv::imwrite(g_outputProbaFilename3, imgRes3);
    }
     
    makeBGRImageFromImageProba(res4, img.rows, img.cols, imgRes4, false);
    std::cerr<<"save proba4 file: "<<g_outputProbaFilename4<<"\n";
    cv::imwrite(g_outputProbaFilename4, imgRes4);
     
  }
#endif //SAVE_PROBA
  
}




#include "save_proba_facto.cpp"


#include "detectWoundMask.cpp"


static
void
readMinMax(const std::string &inputFilename, MinMaxP &m)
{
  std::ifstream in(inputFilename.c_str());
  if (! in) {
    std::cerr<<"ERROR: unable to open input file: "<<inputFilename<<"\n";
    exit(10);
  }
  in >> m.minP;
  if (in.bad()) {
    std::cerr<<"ERROR: reading: "<<inputFilename<<"\n";
    exit(10);
  }
  in >> m.maxP;
} 


cv::Mat
computeWoundMask(const cv::Mat &img, 
		   const cv::Mat &markerMask,
		   const std::vector<std::vector<cv::Point> > &inputOtherPts,
		   const std::vector<std::vector<cv::Point> > &inputWoundPts,
		   const std::string &modelFilename1, const std::string &normalizeFilename1,
		   const std::string &modelFilename2, const std::string &normalizeFilename2,
		   const std::string &modelFilename3, const std::string &normalizeFilename3,
		   int smallSize,
		   int nbClasses,
		   ColorSpace cs,
		   int patchSize,
		   PatchType patchType, 
		   bool useL, 
		   bool saveInColors,
		   const std::string &minMaxDistFilename1, 
		   const std::string &minMaxDistFilename2, 
		   const std::string &minMaxDistFilename3)
{
#ifdef TIMING
  double t = (double)cv::getTickCount();
#endif
  
  Normalizer n1(normalizeFilename1);
  Normalizer n2(normalizeFilename2);
  Normalizer n3(normalizeFilename3);


  MinMaxP m1, m2, m3;
  readMinMax(minMaxDistFilename1, m1);
  readMinMax(minMaxDistFilename2, m2);
  const bool useClassifier3 = (! modelFilename3.empty());
  if (useClassifier3)
    readMinMax(minMaxDistFilename3, m3);

  cv::Mat out = detectWoundMask(modelFilename1, modelFilename2, modelFilename3, 
				img, 
				markerMask,
				inputOtherPts, inputWoundPts,
				cs, patchSize, patchType, smallSize, nbClasses, useL, 
				saveInColors, 
				n1, n2, n3,
				m1, m2, m3);


#ifdef TIMING
  t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
  std::cout<<"Total time: "<<t<<"s"<<std::endl;
#endif

  return out;
}
