#ifndef UTILSCONCEPTSTRAIN_HPP
#define UTILSCONCEPTSTRAIN_HPP


#include "utilsConcepts.hpp"

#if defined(_MSC_VER)  || ( defined(__clang__) && __clang_major__ >= 5 )
#include <unordered_map>
#include <unordered_set>
#else
#include <tr1/unordered_map>
#include <tr1/unordered_set>
#endif

#include <cassert>
#include <iostream>
#include <fstream>

#include <ise/iseConversion.hpp>


struct LConcept
{
  std::string name;
  cv::Vec3b color;
  float percent;

  explicit LConcept(cv::Vec3b color_)
    : name(), color(color_), percent(0.f)
  {}

  LConcept(const std::string &n, cv::Vec3b color_, float percent_=0)
    : name(n),
      color(color_),
      percent(percent_)
  {}

  bool operator==(const LConcept &lc) const { return color == lc.color; } //compare just color !
  bool operator!=(const LConcept &lc) const { return ! (*this == lc); }

  bool operator<(const LConcept &lc) const
  {
    return color[0] < lc.color[0] || (color[0] == lc.color[0] && color[1] < lc.color[1]) || (color[0] == lc.color[0] && color[1] == lc.color[1] && color[2] < lc.color[2]) ;
    //return color < lc.color;
  }

};



namespace std {

#if ( defined(__clang__) && __clang_major__ >= 5 )
  //nothing
#elif !defined(_MSC_VER) || (_MSC_VER < 1600)
  //From Visual C++ 2010, i.e., _MSC_VER >= 1600, hash is not in the tr1 namespace
  namespace tr1 {
#endif

    template<>
    struct hash<LConcept>
    //      : public std::unary_function<girl::dart, std::size_t>
    {
      std::size_t
      operator()(const LConcept &d) const
      { 
	return (size_t)(((int)d.color[0])<<16 | ((int)d.color[1])<<8 | ((int)d.color[2]));
      }
    };
#if ( defined(__clang__) && __clang_major__ >= 5 )
    //nothing
#elif !defined(_MSC_VER) || (_MSC_VER < 1600)
  }
#endif

}


static inline
bool
isNotSaturatedPixel(const cv::Vec3b &color)
{
  //color is in BGR !

  const int threshold = 253 + 254 + 253;
  const int sum = color[0] + color[1] + color[2];
  return sum < threshold;
}
static inline
bool
isSaturatedPixel(const cv::Vec3b &color)
{
  return ! isNotSaturatedPixel(color);
}


#if defined(_MSC_VER)  || ( defined(__clang__) && __clang_major__ >= 5 )
typedef std::unordered_map<LConcept, cv::Mat> MapType;
#else
typedef std::tr1::unordered_map<LConcept, cv::Mat> MapType;
#endif

static inline
size_t
myHash(cv::Vec3b v)
{
  return (size_t)(((int)v[0])<<16 | ((int)v[1])<<8 | ((int)v[2]));
}


static
void accumulateImagePatch(const cv::Mat &origImg, 
			  const cv::Mat &labelImg, 
			  MapType &conceptsPixels,
			  ColorSpace cs = eLAB,
			  int patchSize = 5,
			  PatchType type = eLAB_VALUES,
			  float lSaturatedThreshold = L_SATURATED_THRESHOLD,
			  bool useL = true,
			  bool removeSaturated = true)
{
  assert(origImg.cols == labelImg.cols);
  assert(origImg.rows == labelImg.rows);
  assert(origImg.type() == CV_8UC3);
  assert(labelImg.type() == CV_8UC3);

  //We want to resize the matrices faster than one row at a time.

  //const int nbConcepts = conceptsPixels.size();
  
  cv::Mat imgCS;
  convert(origImg, cs, imgCS);

  cv::Mat mask(origImg.rows, origImg.cols, CV_8UC1);

  for (MapType::iterator it = conceptsPixels.begin(); 
       it != conceptsPixels.end();
       ++it) {

    cv::Vec3b color = it->first.color; //concept color in RGB
    std::swap(color[0], color[2]); //we pass it in BGR

    //build mask for this concept
    
    //TODO:OPTIM: we could be less general, but traverse the origImg only once !?!

    assert(labelImg.isContinuous());
    assert(mask.isContinuous());
    int nbPix = 0;
    //const cv::Vec3b *pi = imgLAB.ptr<cv::Vec3b>(0);
    //int nbPixL = 0;
    const cv::Vec3b *ll = labelImg.ptr<cv::Vec3b>(0);
    unsigned char *ml = mask.ptr<unsigned char>(0);
    for (int y=0; y<labelImg.rows; ++y) {
      for (int x=0; x<labelImg.cols; ++x) {
	if (*ll == color) {
	  ++nbPix;
	  //if (removeSaturated && (*pi)[0] < lSaturatedThreshold)
	  //++nbPixL;

	  //TODO:OPTIM: do not set 255 in mask if (removeSaturated && (*pi)[0] < lSaturatedThreshold) !?!

	  *ml = 255;

	}
	else {

	  *ml = 0;

	}

	++ll;
	++ml;
	//++pi;
      }
    }
    
    cv::Mat data;
    if (nbPix > 0) {
      accumulateImagePatch(imgCS, mask, data, patchSize, type, useL, removeSaturated, lSaturatedThreshold);
   
      cv::Mat &mat = it->second;
      mat.push_back(data);

    }
  }

}




typedef std::vector<int> IndiceVector; //int because cv::RNG::uniform takes 'int'

/**
   Get @a nb indices in [@a minI; @a maxI[ in @a indices.

   returned indices are unique and in increasing order.
*/
static inline
void
getIndices(IndiceVector &indices, 
	   size_t nb, 
	   int minI, int maxI)
{
  assert(minI <= maxI);
  assert(nb <= (size_t)(maxI-minI+1));

  cv::RNG rng(time(NULL));

#if defined(_MSC_VER)  || ( defined(__clang__) && __clang_major__ >= 5 )
  std::unordered_set<size_t> indicesUSet;
#else
  std::tr1::unordered_set<size_t> indicesUSet;
#endif
  while (indicesUSet.size() < nb) {
    indicesUSet.insert(rng.uniform(minI, maxI)); //warning: cv::RNG::uniform takes 'int'
  }
  indices.resize(nb);
  std::copy(indicesUSet.begin(), indicesUSet.end(), indices.begin());

  std::sort(indices.begin(), indices.end());
}

static
void
subsampleRows(const cv::Mat &mIn, int maxRows, cv::Mat &mOut)
{
  assert(&mIn != &mOut);

  if (mIn.rows <= maxRows) {
    mOut = mIn;
  }
  else {
    IndiceVector indices;
    getIndices(indices, maxRows, 0, mIn.rows);
    assert(indices.size() == (size_t)maxRows);
    mOut = cv::Mat(maxRows, mIn.cols, mIn.type());
    for (int i=0; i<maxRows; ++i) {
      cv::Mat r = mOut.row(i);
      mIn.row(indices[i]).copyTo(r);
    }
  }
  
  assert(mOut.rows <= maxRows);
}



static
void
saveLibSVMFormat(const cv::Mat &trainDataP, const cv::Mat &responses, const std::string &outputFilename)
{
  assert(trainDataP.rows == responses.rows);
  assert(responses.cols == 1);
  assert(responses.type() == CV_32S);
  assert(trainDataP.type() == CV_32F);

  std::ofstream out(outputFilename.c_str());
  if (! out) {
    std::cerr<<"ERROR: unable to open output file: "<<outputFilename<<"\n";
    exit(10);
  }
  
  const int *resp = responses.ptr<int>(0);
  const float *d = trainDataP.ptr<float>(0);
  for (int i=0; i<trainDataP.rows; ++i) {
    out<<*resp<<" ";
    for (int j=0; j<trainDataP.cols; ++j) {
      out<<j+1<<":"<<*d<<" ";
      ++d;
    }
    out<<"\n";
    ++resp;
  }

}

static
void
saveCSVFormat(const cv::Mat &trainDataP, const cv::Mat &responses, const std::string &outputFilename)
{
  assert(trainDataP.rows == responses.rows);
  assert(responses.cols == 1);
  assert(responses.type() == CV_32S);
  assert(trainDataP.type() == CV_32F);

  std::ofstream out(outputFilename.c_str());
  if (! out) {
    std::cerr<<"ERROR: unable to open output file: "<<outputFilename<<"\n";
    exit(10);
  }
  
  const int *resp = responses.ptr<int>(0);
  const float *d = trainDataP.ptr<float>(0);
  for (int i=0; i<trainDataP.rows; ++i) {
    for (int j=0; j<trainDataP.cols; ++j) {
      out<<*d<<", ";
      ++d;
    }
    out<<*resp<<"\n";
    ++resp;
  }

}



#endif //* ! UTILSCONCEPTSTRAIN_HPP */
