#ifndef UTILSCONCEPTS_HPP
#define UTILSCONCEPTS_HPP

#include <iostream>
#include <fstream>
#include "lab.hpp"


typedef enum {eLAB=0, eHSV, eCMY, eHSL, eSMY, eRGB, eLUV, eXYZ} ColorSpace;
static const char *colorSpaceStr[] = {"LAB", "HSV", "CMY", "HSL", "SMY", "RGB", "LUV", "XYZ"};
typedef enum {eMTS=0, eMEAN_VAR, eLAB_VALUES, eMEAN_STDDEV, eLAB_VALUES_UNSORTED, eMEAN, eLA_SORTED} PatchType;
static const char *patchTypeStr[] = {"Moments", "Mean+Variance", "LAB sorted values", "Mean+StdDev", "LAB unsorted values", "Mean", "LA sorted"};


static const float L_SATURATED_THRESHOLD = 80;


static
int 
getLABsSizeWithDim(int patchSize, PatchType type, int dim)
{
  int dataSize = 0;
  switch (type) {
  case eMTS:
    dataSize = dim*2+1;
    break;
  case eMEAN:
    dataSize = dim;
    break;
  case eMEAN_VAR:
  case eMEAN_STDDEV:
    dataSize = dim*2;
    break;
  case eLAB_VALUES:
  case eLAB_VALUES_UNSORTED:
    dataSize = patchSize*patchSize*dim;
    break;
  case eLA_SORTED:
    dataSize = patchSize*patchSize*(dim-1);
    break;    
  default:
    std::cerr<<"ERROR: unknown patch type "<<(int)type<<"\n";
    exit(10);
  }
  return dataSize;
}

struct ColorSpaceInfo
{
  ColorSpace m_cs;
  bool m_use[3];

  ColorSpaceInfo(ColorSpace cs_=eLAB, bool use0=true, bool use1=true, bool use2=true)
    : m_cs(cs_)
  {
    m_use[0] = use0;
    m_use[1] = use1;
    m_use[2] = use2;
  }

  ColorSpaceInfo(ColorSpace cs_, bool use[3])
    : m_cs(cs_)
  {
    m_use[0] = use[0];
    m_use[1] = use[1];
    m_use[2] = use[2];
  }

  bool use(int i) const 
  {
    assert(i>=0 && i<3);
    return m_use[i];
  }

  int dim() const
  {
    int d = 0;
    d += (m_use[0] ? 1 : 0);
    d += (m_use[1] ? 1 : 0);
    d += (m_use[2] ? 1 : 0);
    return d;
  }

};

typedef std::vector<ColorSpaceInfo> ColorSpaceInfos;


#ifndef NDEBUG
static
int 
getLABsSizeTill(int patchSize, PatchType type, const ColorSpaceInfos &csi, int n)
{
  assert((size_t)n <= csi.size());
  int dataSize = 0;
  for (int i=0; i<n; ++i) {
    const ColorSpaceInfo &cs = csi[i];
    const int dim = cs.dim();
    dataSize += getLABsSizeWithDim(patchSize, type, dim);
  }
  return dataSize;
}

static
int 
getLABsSize(int patchSize, PatchType type, const ColorSpaceInfos &csi)
{
  const int nbCS = csi.size();
  return getLABsSizeTill(patchSize, type, csi, nbCS);
}
#endif //NDEBUG

static
int 
getLABsSize(int patchSize, PatchType type, bool useL)
{
  const int dim = (useL? 3 : 2);
  return getLABsSizeWithDim(patchSize, type, dim);
}



struct NormSorter
{
  const std::vector<float> &m_norms;

  explicit NormSorter(const std::vector<float> &norms)
    : m_norms(norms)
  {}

  bool operator()(size_t i, size_t j) const
  {
    assert(i<m_norms.size());
    assert(j<m_norms.size());

    return m_norms[i] < m_norms[j];
  }

};


static
void
sortLABs(std::vector<float> &labs, int dim, cv::Mat &labsM)
{
  assert(labsM.rows == 1 && labsM.cols >= dim);
  assert(labsM.type() == CV_32F);
  assert(labs.size() == (size_t)labsM.cols);

  //We sort LABs according to their norm

  std::vector<float> norms;
  norms.resize(labs.size()/dim);
  assert(norms.size()*dim == labs.size());

  //compute L2 norms
  for (int i=0; i<(int)norms.size(); ++i) {
    float norm = 0;
    for (int j=i*dim; j<(i+1)*dim; ++j) {
      norm += labs[j]*labs[j];
    }
    norms[i] = norm;
  }
  
  std::vector<size_t> indices(norms.size());
  for (size_t i=0; i<norms.size(); ++i)
    indices[i] = i;
  
  std::sort(indices.begin(), indices.end(), NormSorter(norms));

  float *l = labsM.ptr<float>(0);
  for (size_t i=0; i<indices.size(); ++i) {
    assert(indices[i]*dim+dim-1 < labs.size());
    const size_t ind = indices[i];
    const size_t indSrc = ind*dim;
    const size_t indDst = i*dim;
    assert(indSrc+dim-1 < labs.size());
    assert(indDst+dim-1 < (size_t)labsM.cols);
    for (int j=0; j<dim; ++j) {
      l[indDst+j] = labs[indSrc+j];
    }
  }
  
}

struct laSorter
{
  bool operator()(const std::pair<float, float> &p1, const std::pair<float, float> &p2)
  {
    return (p1.first*p1.first + p1.second*p1.second) < (p2.first*p2.first + p2.second*p2.second);
  }

};

static
bool
getLABMts(int x, int y, 
	  const cv::Mat &imgLAB, 
	  //const cv::Mat &mask, 
	  int patchSize, 
	  PatchType type,
	  float lSaturatedThreshold, 
	  bool useL,
	  cv::Mat &data)
{
  assert(imgLAB.type() == CV_32FC3);
  //assert(mask.type() == CV_8UC1);
  //assert(img.rows == mask.rows && img.cols == mask.cols);
  assert(data.type() == CV_32F && data.rows == 1);
  
  const int dataSize = data.cols;
  assert(data.cols == getLABsSize(patchSize, type, useL));
  
  const int halfPatchSize = patchSize/2;

  const int x0 = std::max(x-halfPatchSize, 0);
  const int y0 = std::max(y-halfPatchSize, 0);
  const int x1 = std::min(x+halfPatchSize, imgLAB.cols-1);
  const int y1 = std::min(y+halfPatchSize, imgLAB.rows-1);

  const int shift = y0*imgLAB.cols + x0;
  
  const cv::Vec3f *ol = imgLAB.ptr<cv::Vec3f>(0) + shift;
  //const unsigned char *ml = mask.ptr<unsigned char>(0) + shift;
  
  float *d = data.ptr<float>(0);
  std::vector<float> labs;
  size_t m0 = 0;
  if (type == eMTS || type == eMEAN_VAR || type == eMEAN_STDDEV || type == eMEAN)
    memset(d, 0, dataSize*sizeof(float));
  else
    labs.resize(dataSize);

  std::vector<std::pair<float, float> > laValues(patchSize*patchSize);
	   
  const int dim = (useL ? 3 : 2);
  //const int start = 3-dim;
  const int shiftW = imgLAB.cols - (x1-x0+1);
  assert(shiftW>=0);

  int i = 0;
  bool hasOneSaturated = false;
  for (int yy=y0; yy<=y1; ++yy) {
    for (int xx=x0; xx<=x1; ++xx) {
      const cv::Vec3f &lab = *ol;

      if (lab[0] > lSaturatedThreshold)
	hasOneSaturated = true;
	
      if (type == eMTS) {
	if (useL) {
	  d[0] += 1;
	  d[1] += lab[0]; d[2] += lab[1]; d[3] += lab[2]; 
	  d[4] += lab[0]*lab[0]; d[5] += lab[1]*lab[1]; d[6] += lab[2]*lab[2]; 
	}
	else {
	  d[0] += 1;
	  d[1] += lab[1]; d[2] += lab[2]; 
	  d[3] += lab[1]*lab[1]; d[4] += lab[2]*lab[2]; 
	}
      }
      else if (type == eMEAN_VAR || type ==eMEAN_STDDEV) {
	++m0;
	if (useL) {
	  d[0] += lab[0]; d[1] += lab[1]; d[2] += lab[2]; 
	  d[3] += lab[0]*lab[0]; d[4] += lab[1]*lab[1]; d[5] += lab[2]*lab[2]; 
	}
	else {
	  d[0] += lab[1]; d[1] += lab[2]; 
	  d[2] += lab[1]*lab[1]; d[3] += lab[2]*lab[2]; 
	}	  
      }
      else if (type == eMEAN) {
	++m0;
	if (useL) {
	  d[0] += lab[0]; d[1] += lab[1]; d[2] += lab[2]; 
	}
	else {
	  d[0] += lab[1]; d[1] += lab[2]; 
	}	  
      }
      else if (type == eLAB_VALUES || type == eLAB_VALUES_UNSORTED) {

	if (useL) {
	  labs[i] = lab[0]; labs[i+1] = lab[1]; labs[i+2] = lab[2];
	}
	else {
	  labs[i] = lab[1]; labs[i+1] = lab[2];
	}
	i += dim;
      }	  
      else if (type == eLA_SORTED) {
	
	laValues[i] = std::pair<float, float>(lab[0], lab[1]);
	
	++i;
      }

      
      ++ol;
    }
    ol += shiftW;
  }      
  
  if (type == eMEAN_VAR || type == eMEAN_STDDEV) {
    if (m0 > 0) {
      float inv_m0 = 1.f/(float)m0;
      if (useL) {
	d[3] = (d[3] - d[0]*d[0]*inv_m0)*inv_m0;
	d[4] = (d[4] - d[1]*d[1]*inv_m0)*inv_m0;
	d[5] = (d[5] - d[2]*d[2]*inv_m0)*inv_m0;

	if (d[3] < 0)
	  d[3] = 0;
	if (d[4] < 0)
	  d[4] = 0;
	if (d[5] < 0)
	  d[5] = 0;

	d[0] *= inv_m0; d[1] *= inv_m0; d[2] *= inv_m0;

	if (type == eMEAN_STDDEV) {
	  d[3] = sqrt(d[3]);
	  d[4] = sqrt(d[4]);
	  d[5] = sqrt(d[5]);
	}

      }
      else {
	d[2] = (d[2] - d[0]*d[0]*inv_m0)*inv_m0;
	d[3] = (d[3] - d[1]*d[1]*inv_m0)*inv_m0;
	
	if (d[2] < 0)
	  d[2] = 0;
	if (d[3] < 0)
	  d[3] = 0;

	d[0] *= inv_m0; d[1] *= inv_m0;

	if (type == eMEAN_STDDEV) {
	  d[2] = sqrt(d[2]);
	  d[3] = sqrt(d[3]);
	}
      }
    }
  }
  if (type == eMEAN) {
    if (m0 > 0) {
      float inv_m0 = 1.f/(float)m0;
      if (useL) {
	d[0] *= inv_m0; d[1] *= inv_m0; d[2] *= inv_m0;
      }
      else {
	d[0] *= inv_m0; d[1] *= inv_m0;
      }
    }
  }
  else if (type == eLAB_VALUES) {
    assert(labs.size() == (size_t)data.cols);
    sortLABs(labs, dim, data);
  }
  else if (type == eLAB_VALUES_UNSORTED) {
    assert(labs.size() == (size_t)data.cols);
    for (size_t i=0; i<labs.size(); ++i)
      d[i] = labs[i];
  }
  else if (type == eLA_SORTED) {
    std::sort(laValues.begin(), laValues.end(), laSorter());
    
    for (int i=0; i<data.cols; i+=2) {
      d[i] = laValues[i/2].first;
      d[i+1] = laValues[i/2].second;
    }

    
  }
  
  return hasOneSaturated;
}




/*
  accumulate data only where mask is not 0.
*/
static
void accumulateImagePatch(const cv::Mat &imgLAB, 
			  const cv::Mat &mask,
			  cv::Mat &data,
			  int patchSize = 5,
			  PatchType type = eLAB_VALUES,
			  bool useL = true,
			  bool removeSaturated = true,
			  float lSaturatedThreshold = L_SATURATED_THRESHOLD)
{
  assert(imgLAB.cols == mask.cols);
  assert(imgLAB.rows == mask.rows);
  assert(imgLAB.type() == CV_32FC3);
  assert(mask.type() == CV_8UC1);

  const int dataSize = getLABsSize(patchSize, type, useL);
  data.create(imgLAB.rows*imgLAB.cols, dataSize, CV_32F);

  assert(imgLAB.isContinuous());
  assert(mask.isContinuous());

  //cv::Vec3f *p = imgLAB.ptr<float>(0);
  const unsigned char *m = mask.ptr<unsigned char>(0);
  int k = 0;
  for (int i=0; i<imgLAB.rows; ++i) {
    for (int j=0; j<imgLAB.cols; ++j) {
      if (*m) {
	cv::Mat dR(1, data.cols, data.type(), data.ptr<float>(k));
	bool hasSaturated = getLABMts(j, i, imgLAB, patchSize, type, lSaturatedThreshold, useL, dR);
	if (! removeSaturated || ! hasSaturated)
	  ++k;
      }
      ++m;
    }
  }
  data.resize(k);

}




struct ConverterLAB
{
  ConverterLAB() {}

  template <typename T> void operator()(T r, T g, T b, float &x, float &y, float &z) const
  {
    RGB_to_LAB(r, g, b, x, y, z);

  }

  template <typename T> void operator()(T r, T g, T b, float &y, float &z) const
  {
    //TODO:OPTIM: no need to compute L
    // but we will not gain much...
    float x;
    RGB_to_LAB(r, g, b, x, y, z);
  }
};


struct ConverterRGB
{
  ConverterRGB() {}

  template <typename T> void operator()(T r, T g, T b, float &x, float &y, float &z) const
  {
    x = r;
    y = g;
    z = b;
  }

  template <typename T> void operator()(T /*r*/, T g, T b, float &y, float &z) const
  {
    y = g;
    z = b;
  }
};

struct ConverterLUV
{
  ConverterLUV() {}

  template <typename T> void operator()(T r, T g, T b, float &x, float &y, float &z) const
  {
    RGB_to_LUV(r, g, b, x, y, z);

  }

  template <typename T> void operator()(T r, T g, T b, float &y, float &z) const
  {
    //TODO:OPTIM: no need to compute L
    // but we will not gain much...
    float x;
    RGB_to_LUV(r, g, b, x, y, z);
  }
};

struct ConverterXYZ
{
  ConverterXYZ() {}

  template <typename T> void operator()(T r, T g, T b, float &x, float &y, float &z) const
  {
    RGB_to_XYZ(r, g, b, x, y, z);

  }

  template <typename T> void operator()(T r, T g, T b, float &y, float &z) const
  {
    //TODO:OPTIM: no need to compute L
    // but we will not gain much...
    float x;
    RGB_to_XYZ(r, g, b, x, y, z);
  }
};


//TODO:OPTIM: no need to compute L of LAB if it will not be used !!!!!
//TODO:OPTIM: parallelize conversion !?!?
template<typename Converter>
void
convert(const cv::Mat &img, cv::Mat &imgOut)
{
  assert(img.type() == CV_8UC3);
  const Converter c;

  imgOut.create(img.rows, img.cols, CV_32FC3);

  assert(img.isContinuous());
  assert(imgOut.isContinuous());
  const cv::Vec3b *src = img.ptr<cv::Vec3b>(0);
  cv::Vec3f *dst = imgOut.ptr<cv::Vec3f>(0);

  for (int i=0; i<img.rows*img.cols; ++i) {
    const cv::Vec3b &color = *src;
    cv::Vec3f &out = *dst;
    //BGR
    c(color[2], color[1], color[0], out[0], out[1], out[2]);
    ++src;
    ++dst;
  }
  
}

static
void
convert(const cv::Mat &origImg, ColorSpace cs, cv::Mat &imgCS)
{
  switch (cs) {
  case eLAB:
    convert<ConverterLAB>(origImg, imgCS);
    break;
  case eRGB:
    convert<ConverterRGB>(origImg, imgCS);
    break;    
  case eLUV:
    convert<ConverterLUV>(origImg, imgCS);
    break;
  case eXYZ:
    convert<ConverterXYZ>(origImg, imgCS);
    break;
  default:
    std::cerr<<"ERROR: utilsConcept.hpp: convert(): unknown color space "<<cs<<"\n";
    exit(EXIT_FAILURE);
  }
  
}



static
void 
P_normalizeMat(const std::vector<float> &mins,
	       const std::vector<float> &maxs, 
	       cv::Mat &mat)
{
  assert(mins.size() == maxs.size());
  assert(mins.size() == (size_t)mat.cols);
  
  for (int i=0; i<mat.rows; ++i) {
    float *p = mat.ptr<float>(i);
    for (int j=0; j<mat.cols; ++j) {
      if (p[j] >= maxs[j])
	p[j] = 1.0f;
      else if (p[j] <= mins[j])
	p[j] = 0.f;
      else 
	p[j] = (p[j]-mins[j])/(maxs[j]-mins[j]);
    }
  }

}


static
void
readMinMaxs(const std::string &inputFilename, 
	    std::vector<float> &mins, 
	    std::vector<float> &maxs)
{
  std::ifstream in(inputFilename.c_str());
  if (! in) {
    std::cerr<<"ERROR: unable to open input file: "<<inputFilename<<"\n";
    exit(10);
  }
  int size = 0;
  in >> size;
  mins.resize(size);
  maxs.resize(size);
  for (int i=0; i<size; ++i)
    in >> mins[i];
  for (int i=0; i<size; ++i)
    in >> maxs[i];

  //std::cerr<<"FileName"<< inputFilename<< "Taille" << size <<"\n";

}

struct Normalizer
{
  std::vector<float> m_mins;
  std::vector<float> m_maxs;

  Normalizer()
    : m_mins(), m_maxs()
  {}

  explicit Normalizer(const std::string &inputFilename)
    : m_mins(), m_maxs()
  {
    if (! inputFilename.empty())
      load(inputFilename);
  }

  void load(const std::string &inputFilename)
  {
    readMinMaxs(inputFilename, m_mins, m_maxs);
  }

  bool active() const
  {
    return ! m_mins.empty();
  }

  void normalizeMat(cv::Mat &m) const
  {
    if (! active())
      return ;

    if ((size_t)m.cols != m_mins.size()) {
      std::cerr<<"ERROR: Normalizer: matrix cols="<<m.cols<<" mins.size()="<<m_mins.size()<<"\n";
      exit(100);
    }

    P_normalizeMat(m_mins, m_maxs, m);
  }

};


#endif //* ! UTILSCONCEPTS_HPP */
