#ifndef UTILS_CLASSIFIER_HPP
#define UTILS_CLASSIFIER_HPP


#include <fstream>
#include <limits>
#include <opencv2/ml/ml.hpp>


static
double sigmoid_predict(double decision_value, double A, double B)
{
  double fApB = decision_value*A+B;
  // 1-p used later; avoid catastrophic cancellation
  if (fApB >= 0)
    return exp(-fApB)/(1.0+exp(-fApB));
  else
    return 1.0/(1+exp(fApB)) ;
}


/**
   @a out is of type CV_32FC1 with values in [0; 1]

   @warning with returnDFVal=true, we have a distance and not a probability
*/
static
void
classifyData(
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
	     CvSVM *classifier, 
#else
	     cv::Ptr<cv::ml::SVM> classifier, 
#endif
	     const cv::Mat &testData,
	     cv::Mat &out,
	     bool returnDFVal = true,
	     bool normalize = false,
	     double sigmoid_A=0, double sigmoid_B=0,
	     float minV = std::numeric_limits<float>::max(),
	     float maxV = std::numeric_limits<float>::max())
{
  out.create(testData.rows, 1, CV_32FC1);

  float max =  -std::numeric_limits<float>::max(); 
  float min =  std::numeric_limits<float>::max(); ;

#if ! (CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300)
  int flags = returnDFVal ? cv::ml::StatModel::RAW_OUTPUT : 0;
#endif


  classifier->predict(testData, out, 
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
		      returnDFVal //parallelized version, taking returnDFVal, is only available in OpenCV 2.4.x patched version...
#else
		     flags
#endif
		     );
		        
  float *d1 = out.ptr<float>(0);
  for (int i=0; i<out.rows; ++i) {
    const float response = *d1;
    if(response > max)
      max = response;
    if (response < min)
      min = response;
    ++d1;
  }


  if (returnDFVal == true && (sigmoid_A != 0 && sigmoid_B != 0)) {

    //TODO:OPTIM: should be parallelized ???
    float *d = out.ptr<float>(0);
    for (int i=0; i<out.rows; ++i) {
      *d = sigmoid_predict(*d, sigmoid_A, sigmoid_B);
      ++d;
    }
    
  }
  else if (normalize) {

    if (maxV != std::numeric_limits<float>::max()) {
      min = minV;
      max = maxV;
    }
    assert(min < max);
    float inv = 1.f / (max - min);

    float *d = out.ptr<float>(0);
    for (int i=0; i<out.rows*out.cols; ++i) {
      float f = *d;      
      float val = (f - min) * inv;
      if (val < 0)
	val = 0;
      else if (val > 1.f)
	val = 1.f;

      *d = (returnDFVal ? 1.f - val : val);
      ++d;
    }
    
  }



}



#ifdef SAVE_PROBA

/*
  @a imgProba is of type CV_32FC1 and values are in [0; 1]

*/
static
void
makeBGRImageFromImageProba(const cv::Mat &imgProba, cv::Mat &outImg)
{
  assert(imgProba.type() == CV_32F);

  outImg.create(imgProba.rows, imgProba.cols, CV_8UC3);
  
  int rows = imgProba.rows;
  int cols = imgProba.cols;
  if (imgProba.isContinuous() && outImg.isContinuous()) {
    cols *= rows;
    rows = 1;
  }

  for (int i=0; i<rows; ++i) {
    cv::Vec3b *d = outImg.ptr<cv::Vec3b>(i);  
    const float *pf = imgProba.ptr<float>(i);
    for (int j=0; j<cols; ++j) {
      const float f = *pf;
      const int v = (int)(f * 255);
      assert(v>=0 && v<=255);
      unsigned char r = v;
      cv::Vec3b &dst = *d;
      dst[0] = dst[1] = dst[2] = r;
      ++pf;
      ++d;
    }
  }
 
}


/*
  @a m is a of type CV_32FC1
*/
static
cv::Mat
P_normalize(const cv::Mat &m, float minP, float maxP)
{
  assert(m.type() == CV_32F);
  assert(m.isContinuous());

  cv::Mat mr = m.clone();

  if (minP != maxP) {
    const float inv = 1.f / (maxP-minP);
    const float *pf = m.ptr<float>(0);
    float *df = mr.ptr<float>(0);
    for (int i=0; i<m.rows*m.cols; ++i) {
      const float p = *pf;
      *df = (p - minP) * inv;
      ++pf;
      ++df;
    }
  }

  return mr;
}


/*
  @a m is a of type CV_32FC1
*/
static
cv::Mat
P_normalize(const cv::Mat &m)
{
  assert(m.type() == CV_32F);
  assert(m.isContinuous());
  
  float maxP =  -std::numeric_limits<float>::max(); 
  float minP =  std::numeric_limits<float>::max();
  {
    const float *pf = m.ptr<float>(0);
    for (int i=0; i<m.rows*m.cols; ++i) {
      const float p = *pf;
      if (p < minP)
	minP = p;
      if (p > maxP)
	maxP = p;
      ++pf;
    }
  }
  return P_normalize(m, minP, maxP);
}


/*
  @a imgProba is a of type CV_32FC1
*/
static
void
makeBGRImageFromImageProba(const cv::Mat &imgProba, int rows, int cols, cv::Mat &outImg, bool normalize = true)
{
  assert(imgProba.rows*imgProba.cols == rows*cols);
  assert(imgProba.type() == CV_32F);
  assert(imgProba.isContinuous());

  cv::Mat imgProbaN = imgProba.reshape(1, rows);
  if (normalize) {
    imgProbaN = P_normalize(imgProbaN);
  }
  assert(imgProbaN.rows == rows && imgProbaN.cols == cols);

  makeBGRImageFromImageProba(imgProbaN, outImg);
   
  assert(outImg.rows == rows && outImg.cols == cols && outImg.channels() == 3);
}
#endif //SAVE_PROBA

static
void
readProbaSigmoidFile(const std::string &probaSigmoidFilename,
		     double &sigmoid_A,
		     double &sigmoid_B)
{
  
  std::ifstream in(probaSigmoidFilename.c_str());
  if (! in) {
    std::cerr<<"ERROR: unable to open input proba file: "<<probaSigmoidFilename<<"\n";
    exit(EXIT_FAILURE);
  }
  in >> sigmoid_A;
  in >> sigmoid_B;
}

static
void
readFileLibSVM(const char *filename,
	       cv::Mat &data,
	       cv::Mat &responses)
{
  std::ifstream in(filename);
  if (! in) {
    std::cerr<<"ERROR: unable to open input file: "<<filename<<"\n";
    exit(10);
  }

  //We suppose we have a dense matrix !!!

  //We also suppose that responses are "int" (and not "float");

  responses = cv::Mat(0, 1, CV_32S);
  data = cv::Mat();

  std::string s;
  std::vector<float> v;
  v.reserve(10);
  while (! in.eof() && ! in.bad()) {

    v.clear();
    std::getline(in, s);

    if (in.bad() || in.eof()) 
      break;

    std::stringstream ss(s);
    int label;
    ss >> label;
    
    while (ss.good()) {
      int index=-1;
      ss >> index;
      if (ss.bad() || ss.eof())
	break;
      assert(index > 0);
      index -= 1; //index starts from 1 in libSVM format

      //std::cerr<<"ss.good()="<<ss.good()<<" ss.bad()="<<ss.bad()<<" ss.eof()="<<ss.eof()<<"\n";
      char sep;
      ss >> sep;
      if (ss.bad() || ss.eof())
	break;
      float val;
      ss >> val;
      
      if (index >= (int)v.size())
	v.resize(index+1);

      v[index] = val;
    }
    
    if (data.empty()) {
      data = cv::Mat(0, v.size(), CV_32F);
    }
    if (! v.empty() && (int)v.size() != data.cols) {
      std::cerr<<"ERROR: we only handle dense matrix as input ! v.size()="<<v.size()<<" != data.cols="<<data.cols<<"\n";
      exit(10);
    }

    if (in.bad())
      break;
    if (! v.empty()) {
      cv::Mat r(1, 1, CV_32S);
      *(r.ptr<int>(0)) = label;
      responses.push_back(r);
      
      cv::Mat d(1, data.cols, CV_32F);
      float *p = d.ptr<float>(0);
      assert(data.cols == (int)v.size());
      for (int i=0; i<data.cols; ++i) 
	p[i] = v[i];
      data.push_back(d);
    }
  }

}

class Ellipse
{
public:
  Ellipse(float a, float b, float muX, float muY)
    : m_a(a),
      m_b(b),
      m_muX(muX),
      m_muY(muY)
  {
    m_fX = 1.f/(a*a);
    m_fY = 1.f/(b*b);
  }

  float f(float x, float y) const
  {
    return (x-m_muX)*(x-m_muX)*m_fX + (y-m_muY)*(y-m_muY)*m_fY;
  }

  bool outside(float x, float y) const
  {
    return f(x, y) > 1;
  }

protected:
  float m_a;
  float m_b;
  float m_muX;
  float m_muY;
  float m_fX;
  float m_fY;
};



template <typename T, typename TMask>
void
applyMask(cv::Mat &m, const cv::Mat &mask)
{
  assert(m.type() == CV_MAKETYPE(cv::DataDepth<T>::value, 1));
  assert(mask.type() == CV_MAKETYPE(cv::DataDepth<TMask>::value, 1));
  assert(m.isContinuous());
  assert(mask.isContinuous());
  assert(m.cols == mask.cols && m.rows == mask.rows);

  const TMask *ms = mask.ptr<TMask>(0);
  T *md = m.ptr<T>(0);
  for (int i=0; i<mask.rows*mask.cols; ++i) {
    *md = (T)(*md * *ms);  //Warning: static_cast ! Should we use cv::saturate_cast ???
    ++md;
    ++ms;
  }
}


#endif /* ! UTILS_CLASSIFIER_HPP */
