#ifndef POINTS_UTILS_HPP
#define POINTS_UTILS_HPP

#include <cassert>
#include <vector>
#include <sstream>
#include <opencv2/core/core.hpp>

static
bool
parsePoints(const std::string &s,
	    std::vector<cv::Point> &pts)
{
  //we consider points are specified this way :
  // "x1,y1:x2,y2:...:xn,yn"

  pts.clear();

  if (s.empty())
    return true;

  const char sep2=':';
  const char sep1=',';

  std::stringstream ss;

  std::string::size_type pos = 0;
  std::string::size_type start = 0;
  do {

    pos = s.find_first_of(sep1, start);
    
    ss.str(std::string());
    ss.clear();

    ss << s.substr(start, pos-start);

    int x;
    ss >> x;
    if (ss.fail())
      return false;

    //start = pos+1;
    const std::string::size_type newStart = pos+1;
    pos = s.find_first_of(sep2, start); //from start & not newStart ! [1]
    
    if (pos < newStart)
      return false;
    //[1]: allows to avoid : ",3:10,20"

    start = newStart;

    ss.str(std::string());
    ss.clear();

    ss << s.substr(start, pos-start);

    start = pos+1;
    int y;
    ss >> y;
    if (ss.fail())
      return false;
    
    pts.push_back(cv::Point(x, y));

#if __GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ > 4)
    //does not work on CentOS 6.5 with gcc 4.4.7
    if (ss.tellg() != -1)
      return false;
    //allows to avoid garbage at the end. Ex: "1,3," or '1,3:4,5abcd"
#endif 

  }
  while (pos != std::string::npos);

  return true;
}

static
bool
parsePoints(const std::string &s,
	    std::vector<std::vector<cv::Point> > &ptsGroups)
{

  //we consider points are specified this way :
  // "x1_1,y1_1:x1_2,y1_2:...:x1_n,y1_n#x1_1,y1_1:x1_2,y1_2:...:x1_m,y1_m# ... #x1_1,y1_1:x1_2,y1_2:...:x1_l,y1_l"

  ptsGroups.clear();

  if (s.empty())
    return true;

  const char sep='#';


  std::string::size_type pos = 0;
  std::string::size_type start = 0;
  do {

    pos = s.find_first_of(sep, start);
    
    std::vector<cv::Point> pts;
    std::string subS = s.substr(start, pos-start);
    const bool parseOk = parsePoints(subS, pts);
    if (! parseOk)
      return false;
    ptsGroups.push_back(pts);

    start = pos+1;

  }
  while (pos != std::string::npos);

  return true;

}


static
bool
checkPoints(const std::vector<cv::Point> &pts, const cv::Mat &mask)
{
  assert(mask.type() == CV_8UC1);

  const int width = mask.cols;
  const int height = mask.rows;

  const size_t nbPts = pts.size();
  for (size_t i=0; i<nbPts; ++i) {
    if (pts[i].x <0 || pts[i].x >= width) {
      std::cerr<<"pts["<<i<<"].x="<<pts[i].x<<" not in [0; "<<width<<"[\n";
      return false;
    }
    if (pts[i].y <0 || pts[i].y >= height) {
      std::cerr<<"pts["<<i<<"].y="<<pts[i].y<<" not in [0; "<<height<<"[\n";
      return false;
    }
  }
  return true;
}

static
bool
checkPoints(const std::vector<std::vector<cv::Point> >&ptsGroup, const cv::Mat &mask)
{
  for (std::vector<std::vector<cv::Point> >::const_iterator itG = ptsGroup.begin();
       itG != ptsGroup.end(); ++itG) {

    if (! checkPoints(*itG, mask))
      return false;
  }
  return true;
}


static
bool
checkPointsNotInMask(const std::vector<cv::Point> &pts, const cv::Mat &mask)
{
  assert(mask.type() == CV_8UC1);

  const int width = mask.cols;
  const int height = mask.rows;

  const size_t nbPts = pts.size();
  for (size_t i=0; i<nbPts; ++i) {
    if (pts[i].x <0 || pts[i].x >= width) {
      std::cerr<<"pts["<<i<<"].x="<<pts[i].x<<" not in [0; "<<width<<"[\n";
      return false;
    }
    if (pts[i].y <0 || pts[i].y >= height) {
      std::cerr<<"pts["<<i<<"].y="<<pts[i].y<<" not in [0; "<<height<<"[\n";
      return false;
    }
    if (mask.at<unsigned char>(pts[i].y, pts[i].x)) {
      std::cerr<<"pts["<<i<<"].x="<<pts[i].x<<" y="<<pts[i].y<<" in mask \n";
      return false;
    }
  }
  return true;
}


static
bool
checkPointsNotInMask(const std::vector<std::vector<cv::Point> >&ptsGroup, const cv::Mat &mask)
{
  for (std::vector<std::vector<cv::Point> >::const_iterator itG = ptsGroup.begin();
       itG != ptsGroup.end(); ++itG) {

    if (! checkPointsNotInMask(*itG, mask))
      return false;
  }
  return true;
}


static
void
zoomPoints(const std::vector<cv::Point> &inPts, 
	   float xRatio, float yRatio,
	   std::vector<cv::Point> &outPts)
{
  const size_t nbPts = inPts.size();
  outPts.resize(nbPts);
  
  for (size_t i=0; i<nbPts; ++i) {
    int newX = static_cast<int>(inPts[i].x * xRatio);
    int newY = static_cast<int>(inPts[i].y * yRatio);
    outPts[i].x = newX;
    outPts[i].y = newY;
  }
  
}

/*
static
void
zoomPoints(const std::vector<cv::Point> &inPts, 
	   int width, int height,
	   int newWidth, int newHeight, 
	   std::vector<cv::Point> &outPts)
{
  assert(width>0);
  assert(height>0);

  float xRatio = newWidth/(float)width;
  float yRatio = newHeight/(float)height;

  zoomPoints(inPts, xRatio, yRatio, outPts);
}
*/

static
void
zoomPoints(const std::vector<std::vector<cv::Point> > &inPts, 
	   int width, int height,
	   int newWidth, int newHeight, 
	   std::vector<std::vector<cv::Point> >&outPts)
{
  assert(width>0);
  assert(height>0);

  float xRatio = newWidth/(float)width;
  float yRatio = newHeight/(float)height;

  const size_t s = inPts.size();
  outPts.resize(s);

  for (size_t i=0; i<s; ++i) {
    zoomPoints(inPts[i], xRatio, yRatio, outPts[i]);
  }

}

namespace {

  struct PtSorter
  {
    bool operator()(cv::Point p1, cv::Point p2) const
    {
      return p1.x < p2.x || (p1.x == p2.x && p1.y < p2.y);
    }

  };

} //end annonymous namespace


/*
  Keep only points inside mask

 */
static
std::vector< std::vector<cv::Point> >
keepOnlyOutsideMask(const std::vector< std::vector<cv::Point> >&inPtsV,
		    const cv::Mat &mask)
{
  
  assert(mask.type() == CV_8UC1);
  const int width = mask.cols;
  const int height = mask.rows;

  const size_t nbPtsV = inPtsV.size();
  std::vector< std::vector<cv::Point> > outPtsV;
  outPtsV.reserve(nbPtsV);
  for (size_t i=0; i<nbPtsV; ++i) {

    const std::vector<cv::Point> &inPts = inPtsV[i];
    const size_t nbInPts = inPts.size();
    std::vector<cv::Point> outPts;
    outPts.reserve(nbInPts);
    for (size_t j=0; j<nbInPts; ++j) {
      if (mask.at<unsigned char>(inPts[j].y, inPts[j].x) == 0) {
	outPts.push_back(inPts[j]);
      }
      else {
	//pt in mask
	if (! outPts.empty()) {
	  outPtsV.push_back(outPts);
	  outPts.resize(0);
	}
      }
    }
    if (! outPts.empty()) {
      outPtsV.push_back(outPts);
    }
    assert(outPtsV.size() >= nbPtsV);
  }

  return outPtsV;
}


/*
  expand points but not on the mask

  //REM: we do check for duplicates but only inside a sub-vector.
  //[This check is necessary because otherwise we would have lots of points when we link them !].
 */
static
void
expandPointsNotInMask(const std::vector< std::vector<cv::Point> >&inPtsV, 
		      const cv::Mat &mask,
		      int radius,
		      std::vector< std::vector<cv::Point> > &outPts)
{
  assert(mask.type() == CV_8UC1);
  const int width = mask.cols;
  const int height = mask.rows;

  const size_t nbPts = inPtsV.size();
  outPts.resize(nbPts);
  for (size_t i=0; i<nbPts; ++i) {

    const std::vector<cv::Point> &inPts = inPtsV[i];
    const size_t nbInPts = inPts.size();
    std::vector<cv::Point> &outPts_i = outPts[i];
    outPts_i.reserve(nbInPts * radius*radius);

    for (size_t j=0; j<nbInPts; ++j) {

      int startX = inPts[j].x-radius;
      if (startX < 0)
	startX=0;
      int endX = inPts[j].x+radius;
      if (endX > width)
	endX=width;
      int startY = inPts[j].y-radius;
      if (startY < 0)
	startY=0;
      int endY = inPts[j].y+radius;
      if (endY > height)
	endY=height;
      
      if (radius > 0) {
	for (int y=startY; y<endY; ++y) {
	  for (int x=startX; x<endX; ++x) {
	    if (mask.at<unsigned char>(y, x) == 0)
	      outPts_i.push_back(cv::Point(x, y));
	  }
	}
      }
      else {
	//if (mask.at<unsigned char>(inPts[j].y, inPts[j].x) == 0)
	outPts_i.push_back(inPts[j]);
      }

      std::sort(outPts_i.begin(), outPts_i.end(), PtSorter());
      outPts_i.erase(std::unique(outPts_i.begin(), outPts_i.end()), outPts_i.end());

    }    

  }
   
}




//http://www.edepot.com/linec.html
static
void
doBresenham(cv::Point p1, cv::Point p2, int imgWidth, int imgHeight, std::vector<cv::Point> &pts)
{
  int x = p1.x;
  int y = p1.y;
  int x2 = p2.x;
  int y2 = p2.y;
  cv::Rect bbox(0, 0, imgWidth, imgHeight);

  bool yLonger=false;
  int incrementVal, endVal;
  
  int shortLen=y2-y;
  int longLen=x2-x;
  if (abs(shortLen)>abs(longLen)) {
    int swap=shortLen;
    shortLen=longLen;
    longLen=swap;
    yLonger=true;
  }
	
  endVal=longLen;
  if (longLen<0) {
    incrementVal=-1;
    longLen=-longLen;
  } 
  else 
    incrementVal=1;
  
  double decInc;
  if (longLen==0) 
    decInc=(double)shortLen;
  else 
    decInc=((double)shortLen/(double)longLen);
  double j=0.0;
  if (yLonger) {
    for (int i=0;i!=endVal;i+=incrementVal) {
      if (bbox.contains(cv::Point(x+(int)j, y+i)))
	pts.push_back(cv::Point(x+(int)j, y+i));
      j+=decInc;
    }
  } 
  else {
    for (int i=0;i!=endVal;i+=incrementVal) {
      if (bbox.contains(cv::Point(x+i, y+(int)j)))
	pts.push_back(cv::Point(x+i, y+(int)j));
      j+=decInc;
    }
  } 


}

static
size_t 
getNumPoints(const std::vector< std::vector<cv::Point> > &pts)
{
  size_t s = 0;

  std::vector<std::vector<cv::Point> >::const_iterator itO = pts.begin();
  for ( ; itO != pts.end(); ++itO) {
    const std::vector<cv::Point> &v = *itO;
    s += v.size();
  }

  return s;
}


static
std::vector< std::vector<cv::Point> > 
linkPoints(const std::vector< std::vector<cv::Point> > &inputPts, int imgWidth, int imgHeight)
{
  std::vector< std::vector<cv::Point> > outPts;
  outPts.resize(inputPts.size());

  std::vector<std::vector<cv::Point> >::const_iterator itI = inputPts.begin();
  std::vector<std::vector<cv::Point> >::iterator itO = outPts.begin();
  for ( ; itI != inputPts.end(); ++itI, ++itO) {

    const std::vector<cv::Point> &inV = *itI;
    std::vector<cv::Point> &outV = *itO;

    const size_t s = inV.size();
    if (s <= 1) {
      outV = inV;
    }
    else {
      outV.reserve(6*inV.size()); //arbitrary

      cv::Point prev = inV[0];
      for (size_t i=1; i<s; ++i) { //start from 1

	cv::Point curr = inV[i];

	doBresenham(prev, curr, imgWidth, imgHeight, outV);

	prev =curr;
      }

    }
    
  }

  return outPts;
}


/*
  Check if points of @a pts2 are present in @a pts1
  and if there are, they are removed.

 */
static
void
removeCommonPoints(const std::vector< std::vector<cv::Point> > &pts1,
		   std::vector< std::vector<cv::Point> > &pts2)
{
  
  std::vector<cv::Point> pts1F;
  std::vector<std::vector<cv::Point> >::const_iterator it = pts1.begin();
  for ( ; it != pts1.end(); ++it) {
    const std::vector<cv::Point> &ptsW = *it;
    for (std::vector<cv::Point>::const_iterator itW = ptsW.begin();
	 itW != ptsW.end(); 
	 ++itW) {
      pts1F.push_back(*itW);
    }
  }

  std::sort(pts1F.begin(), pts1F.end(), PtSorter());
  pts1F.erase(std::unique(pts1F.begin(), pts1F.end()), pts1F.end());

  std::vector<std::vector<cv::Point> >::iterator it2 = pts2.begin();
  for ( ; it2 != pts2.end(); ++it2) {
    std::vector<cv::Point> &ptsV = *it2;
    for (std::vector<cv::Point>::iterator itV = ptsV.begin();
	 itV != ptsV.end(); ) {
      if (std::find(pts1F.begin(), pts1F.end(), *itV) != pts1F.end()) {
	itV = ptsV.erase(itV);
      }
      else {
	++itV;
      }
    }
  }
  
}


#endif /* ! POINTS_UTILS_HPP */
