#ifndef PNG_UTILS_HPP
#define PNG_UTILS_HPP

/*
  These functions load/save binary (bitdepth=1) PNG images.

  With current OpenCV (2.4.8) imread/imwrite functions,
  it is not possible to save binary image, only 8-bit images.
  Thus theses functions load/save binary PNG images from 8-bit matrices.
 */

#include <png.h>

//inspired from 
// http://www.piko3d.net/tutorials/libpng-tutorial-loading-png-files-from-streams/
//http://www.libpng.org/pub/png/libpng-1.2.5-manual.html


static
void
userWriteData(png_structp pngPtr, png_bytep data, png_size_t length) {
  //Here we get our IO pointer back from the write struct.
  //This is the parameter we passed to the png_set_write_fn() function.
  //Our std::ostream pointer.
  png_voidp a = png_get_io_ptr(pngPtr);
  //Cast the pointer to std::ostream* and write 'length' bytes from 'data'
  ((std::ostream*)a)->write((const char*)data, length);
}

static
void
userFlushData(png_structp pngPtr)
{
  png_voidp a = png_get_io_ptr(pngPtr);
  ((std::ostream*)a)->flush();
}

static
bool
writeBinaryPNG(std::ofstream &out, const cv::Mat &m)
{
  assert(m.type() == CV_8UC1);

  png_structp pngPtr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!pngPtr) {
    std::cerr << "ERROR: Couldn't initialize png read struct" << std::endl;
    return false;
  }
  png_infop infoPtr = png_create_info_struct(pngPtr);
  if (!infoPtr) {
    std::cerr << "ERROR: Couldn't initialize png info struct" << std::endl;
    png_destroy_write_struct(&pngPtr, (png_infopp)0);
    return false;
  }

  png_bytep* rowPtrs = NULL;
  //char* data = NULL;
  if (setjmp(png_jmpbuf(pngPtr))) {

    //An error occured, so clean up what we have allocated so far...
    png_destroy_read_struct(&pngPtr, &infoPtr,(png_infopp)0);
    if (rowPtrs != NULL) delete [] rowPtrs;
    //if (data != NULL) delete [] data;
    std::cerr << "ERROR: An error occured while reading the PNG file"<<std::endl;

    //Make sure you return here. libPNG will jump to here if something
    //goes wrong, and if you continue with your normal code, you might
    //End up with an infinite loop.
    return false; // Do your own error handling here.
  }
  
  //to write to ostream
  png_set_write_fn(pngPtr,(png_voidp)&out, userWriteData, userFlushData);


  png_set_compression_level(pngPtr, 9); //[0; 9], default is 6 
  /*
  png_set_compression_mem_level(pngPtr, 8); //[0; 9], default is 8
  png_set_compression_strategy(pngPtr, 0);
  */

  png_uint_32 imgWidth =  m.cols;
  png_uint_32 imgHeight = m.rows;
  png_uint_32 bit_depth = 1;
  //png_uint_32 channels = 1;
  
  //png_uint_32 color_type = PNG_COLOR_TYPE_GRAY;
  /**/
  png_uint_32 color_type = PNG_COLOR_TYPE_PALETTE;
  png_color palette[2];
  //png_colorp palette = (png_colorp)png_malloc(pngPtr, 2*sizeof(png_color));
  palette[0].red = 0;
  palette[0].green = 0;
  palette[0].blue = 0;
  palette[1].red = 255;
  palette[1].green = 255;
  palette[1].blue = 255;
  png_set_PLTE(pngPtr, infoPtr, palette, 2);
  /**/

  png_set_packing(pngPtr);


  //infoPtr->valid |= PNG_INFO_sBIT;
  //infoPtr->sig_bit.gray = true_bit_depth;
  /*
  png_color_8 sig_bit;
  sig_bit.gray = 1;
  png_set_sBIT(pngPtr, infoPtr, &sig_bit);
  */


  png_set_packswap(pngPtr);  //B !!! because I do my own packing (on little endian)


     
  png_set_IHDR(pngPtr, infoPtr,
	       imgWidth, imgHeight, bit_depth, color_type, 
	       PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);



  int sz = m.cols/8;
  if (sz*8 < m.cols)
    sz+=1;
  //std::cerr<<"m.cols="<<m.cols<<" sz*8="<<sz*8<<"\n";
  assert(sz*8>= m.cols);
  cv::Mat v = cv::Mat::zeros(m.rows, sz, CV_8UC1); //init useless ???
  const unsigned char *c = m.ptr<unsigned char>(0);
  unsigned char *o = v.ptr<unsigned char>(0);
  for (int i=0; i<m.rows; ++i) {
    int s = 0;
    o = v.ptr<unsigned char>(i);
    for (int j=0; j<m.cols; ++j) {
      if (*c) {
	*o |= (1<<s);
      }
      ++s;
      if (s == 8) {
	s = 0;
	++o;
      }
      ++c;
    }
  }


  png_write_info(pngPtr, infoPtr);


  //Array of row pointers. One for every row.
  rowPtrs = new png_bytep[imgHeight];

  //This is the length in bytes, of one row.
  //const unsigned int stride = imgWidth;
  
  //A little for-loop here to set all the row pointers to the starting
  //Adresses for every row in the buffer
  for (size_t i = 0; i < imgHeight; i++) {
    //Set the pointer to the data pointer + i times the row stride.
    rowPtrs[i] = (unsigned char *)v.ptr<unsigned char>(i);
  }    

  png_write_image(pngPtr, rowPtrs);

  png_write_end(pngPtr, infoPtr);

  delete [] rowPtrs;

  png_destroy_write_struct(&pngPtr, &infoPtr);

  return true;

}


static
bool
writeBinaryPNG(const std::string &filename, const cv::Mat &m)
{
  assert(m.type() == CV_8UC1);

  std::ofstream out(filename.c_str(), std::ios::out|std::ios::binary);
  if (! out) {
    std::cerr<<"Error: unable to open output file: "<<filename<<std::endl;
    return false;
  }
  
  const bool writeOk = writeBinaryPNG(out, m);
  if (! writeOk) {
    std::cerr<<"ERROR: unable to write output file: "<<filename<<"\n";
    return false;
  }

  return true;
}





#if 0

static
void
userReadData(png_structp pngPtr, png_bytep data, png_size_t length) {
  //Here we get our IO pointer back from the read struct.
  //This is the parameter we passed to the png_set_read_fn() function.
  //Our std::istream pointer.
  png_voidp a = png_get_io_ptr(pngPtr);
  //Cast the pointer to std::istream* and read 'length' bytes into 'data'
  ((std::istream*)a)->read((char*)data, length);
}


static
bool
readBinaryPNG(std::ifstream &in, cv::Mat &m)
{
#define PNGSIGSIZE 8

  //Allocate a buffer of 8 bytes, where we can put the file signature.
  png_byte pngsig[PNGSIGSIZE];
  int is_png = 0;

  //Read the 8 bytes from the stream into the sig buffer.
  in.read((char*)pngsig, PNGSIGSIZE);

  //Check if the read worked...
  if (!in.good())
    return false;

  //Let LibPNG check the sig. If this function returns 0, everything is OK.
  is_png = png_sig_cmp(pngsig, 0, PNGSIGSIZE);
  if (is_png != 0) 
    return false;

  //Here we create the png read struct. The 3 NULL's at the end can be used
  //for your own custom error handling functions, but we'll just use the default.
  //if the function fails, NULL is returned. Always check the return values!
  png_structp pngPtr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!pngPtr) {
    std::cerr << "ERROR: Couldn't initialize png read struct" << std::endl;
    return false;
  }
  //Here we create the png info struct.
  //Note that this time, if this function fails, we have to clean up the read struct!
  png_infop infoPtr = png_create_info_struct(pngPtr);
  if (!infoPtr) {
    std::cerr << "ERROR: Couldn't initialize png info struct" << std::endl;
    png_destroy_read_struct(&pngPtr, (png_infopp)0, (png_infopp)0);
    return false;
  }

  png_bytep* rowPtrs = NULL;
  //char* data = NULL;
  if (setjmp(png_jmpbuf(pngPtr))) {

    //An error occured, so clean up what we have allocated so far...
    png_destroy_read_struct(&pngPtr, &infoPtr,(png_infopp)0);
    if (rowPtrs != NULL) delete [] rowPtrs;
    //if (data != NULL) delete [] data;
    std::cerr << "ERROR: An error occured while reading the PNG file"<<std::endl;

    //Make sure you return here. libPNG will jump to here if something
    //goes wrong, and if you continue with your normal code, you might
    //End up with an infinite loop.
    return false;
  }



  //to read from istream
  png_set_read_fn(pngPtr,(png_voidp)&in, userReadData);
    


  //Set the amount signature bytes we've already read:
  //We've defined PNGSIGSIZE as 8;
  png_set_sig_bytes(pngPtr, PNGSIGSIZE);

  //Now call png_read_info with our pngPtr as image handle, and infoPtr to receive the file info.
  png_read_info(pngPtr, infoPtr);

  png_uint_32 imgWidth =  png_get_image_width(pngPtr, infoPtr);
  png_uint_32 imgHeight = png_get_image_height(pngPtr, infoPtr);
  //bits per CHANNEL! note: not per pixel!
  png_uint_32 bit_depth   = png_get_bit_depth(pngPtr, infoPtr);
  //Number of channels
  //png_uint_32 channels   = png_get_channels(pngPtr, infoPtr);
  //Color type. (RGB, RGBA, Luminance, luminance alpha... palette... etc)
  png_uint_32 color_type = png_get_color_type(pngPtr, infoPtr);
    
  if (color_type == PNG_COLOR_TYPE_PALETTE) {

    png_colorp palette;
    int num_palette;
    png_get_PLTE(pngPtr, infoPtr, &palette, &num_palette);
    //std::cerr<<"color_type=PNG_COLOR_TYPE_PALETTE bitdepth="<<bit_depth<<" num_palette="<<num_palette<<"\n";
  }

  //B: we are interested only in 
  //color_type == PNG_COLOR_TYPE_GRAY && bit_depth==1
 
  //if (color_type == PNG_COLOR_TYPE_PALETTE)
  //png_set_palette_to_rgb(pngPtr);      //B ????
    
  /*
    if (color_type == PNG_COLOR_TYPE_GRAY ||
    color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
    png_set_gray_to_rgb(pngPtr);
  */

  //convert an RGB or RGBA image to grayscale or grayscale with alpha.
  if (color_type == PNG_COLOR_TYPE_RGB ||
      color_type == PNG_COLOR_TYPE_RGB_ALPHA) {
    png_set_rgb_to_gray_fixed(pngPtr, 1, -1, -1);
  }


  //if (color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8) 
  if (bit_depth < 8) {
    png_set_expand_gray_1_2_4_to_8(pngPtr);
    //B: convert to color_type=PNG_COLOR_TYPE_RGB !


    png_read_update_info(pngPtr, infoPtr);
    png_set_rgb_to_gray_fixed(pngPtr, 1, -1, -1);

    /*
    png_read_update_info(pngPtr, infoPtr);
    bit_depth   = png_get_bit_depth(pngPtr, infoPtr);
    //Color type. (RGB, RGBA, Luminance, luminance alpha... palette... etc)
    color_type = png_get_color_type(pngPtr, infoPtr);
    std::cerr<<" after update_info(): \n";
    if (color_type == PNG_COLOR_TYPE_GRAY)
      std::cerr<<"color_type=PNG_COLOR_TYPE_GRAY\n";
    else if (color_type == PNG_COLOR_TYPE_PALETTE)
      std::cerr<<"color_type=PNG_COLOR_TYPE_PALETTE\n";
    else if (color_type == PNG_COLOR_TYPE_RGB)
      std::cerr<<"color_type=PNG_COLOR_TYPE_RGB\n";
    else 
      std::cerr<<"color_type="<<color_type<<"\n";
    std::cerr<<"channels="<<channels<<"\n";
    std::cerr<<"bit_depth="<<bit_depth<<"\n";
    */

  }

  //if the image Has 16 bits per channel precision... round it down to 8.
  //if (bit_depth == 16)
  //png_set_strip_16(pngPtr);
    
  //strip alpha channel
  //if (color_type & PNG_COLOR_MASK_ALPHA)
  //png_set_strip_alpha(pngPtr);

  //B: useful ???
  //if (bit_depth < 8)
  //png_set_packing(pngPtr);

  png_read_update_info(pngPtr, infoPtr);
  //bit_depth = 8; 

  //assert(bit_depth == 8);
  //assert(channels == 1);

  
  //Array of row pointers. One for every row.
  rowPtrs = new png_bytep[imgHeight];

  
  //Alocate a buffer with enough space.
  m.create(imgHeight, imgWidth, CV_8UC1);
  assert(m.isContinuous());

  png_uint_32 rowb = png_get_rowbytes(pngPtr, infoPtr);
  //std::cerr<<"height="<<imgHeight<<" width="<<imgWidth<<" rowbytes="<<rowb<<"\n";
  if (imgWidth != rowb) {
    std::cerr<<"ERROR: wrong number of rows\n";
    return false;
  }

  //This is the length in bytes, of one row.
  //const unsigned int stride = imgWidth;
  
  //A little for-loop here to set all the row pointers to the starting
  //Adresses for every row in the buffer
  for (size_t i = 0; i < imgHeight; i++) {
    //Set the pointer to the data pointer + i times the row stride.
    rowPtrs[i] = m.ptr<unsigned char>(i);
  }

  //And here it is! The actuall reading of the image!
  //Read the imagedata and write it to the adresses pointed to
  //by rowptrs (in other words: our image databuffer)
  png_read_image(pngPtr, rowPtrs);

  
  //Delete the row pointers array....
  delete[] (png_bytep)rowPtrs;
  
  //And don't forget to clean up the read and info structs !
  png_destroy_read_struct(&pngPtr, &infoPtr,(png_infopp)0);

  return true;
}


static
bool
readBinaryPNG(const std::string &filename, cv::Mat &m)
{
  assert(m.type() == CV_8UC1);

  std::ifstream in(filename.c_str(), std::ios::in|std::ios::binary);
  if (! in) {
    std::cerr<<"Error: unable to open input file: "<<filename<<std::endl;
    return false;
  }
  
  const bool readOk = readBinaryPNG(in, m);
  if (! readOk) {
    std::cerr<<"ERROR: unable to read input file: "<<filename<<"\n";
    return false;
  }
  
  return true;
}

#endif //0




#endif /* ! PNG_UTILS_HPP */
