#ifndef MARKER_HPP
#define MARKER_HPP

#include <opencv2/core/core.hpp>

/**
   Detect white&blue marker on image @img.
   We consider that marker is square : a white square with a central blue square.

   @param[in] img  input image 
   @param[out] pixelSize  pixel size (according to marker size)
   @param[out] markerMask mask of marker (marker is white, background is black)
   @param[out] blue  mean color of blue part of marker.
   @param[out] white mean color of white part of marker.

   @return true if marker was correctly detected, false otherwise.
 */
extern bool detectMarker(const cv::Mat &img, float &pixelSize, cv::Mat &markerMask, cv::Vec3f &blue, cv::Vec3f &white);

/**
   Check if two masks intersect, i.e., have two non null pixels at the same coordinate.

   @return true if masks intersect, false otherwise.
 */
extern bool intersection(const cv::Mat &mask1, const cv::Mat &mask2);

/**
   Correct colors of image @a img according to white color @a white.

   @param[in] img  input image 
   @param[in] white  input reference white color in RGB format.

   @return corrected image
*/
extern cv::Mat correctWhite(const cv::Mat &img, const cv::Vec3f &white);

/**
   Compute Marker area according to pixel size.
 */
extern float computeMarkerSize(const cv::Mat &markerMask, float pixelSize);


/**
   Compute contours of markers in image @a img.

   @param[in] img  input image 
   @param[out] blueContour blue contour
   @param[out] whiteContour white contour (approximate)

   @return true if marker correctly found, false otherwise.
 */
extern bool findMarkerContours(const cv::Mat &img,
			       std::vector<cv::Point> &blueContour, 
			       std::vector<cv::Point> &whiteContour);

/**
   Compute stats on colours in contours of markers in image @a img.
*/
extern void computeMarkerColorStats(const cv::Mat &img,
				    const std::vector<cv::Point> &blueContour, 
				    const std::vector<cv::Point> &whiteContour,
				    float meanBlue[3], float &stdDevBlue,
				    float meanWhite[3], float &stdDevWhite);

#endif /* MARKER_HPP */
