
#include "marker.hpp"

#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgproc/imgproc.hpp>

//#include <girlTools/girlToolsPainter.hpp>

#include <limits>
#include <queue>
#include <iostream>

#include "hsv.hpp"
#include "lab.hpp"

static const float MARKER_SIZE = 1.f; //in centimeters.
//static const float MARKER_SIZE_OUTER = 3.f; //in centimeters.
static const float MARKER_CYAN = 1.0f;
static const float MARKER_MAGENTA=0.57f;
static const float MARKER_YELLOW=0.12f;
static const float MARKER_BLACK=0.66f;

static inline
void
RGB_to_CMYK(unsigned char R, unsigned char G, unsigned char B, float &c, float &m, float &y, float &k)
{
  const unsigned char M = std::max(R, std::max(G, B));
  if (M != 0) {
    const float rf = R/255.f;
    const float gf = G/255.f;
    const float bf = B/255.f;
    const float mf = M/255.f;
    float kf = 1.f - mf;
    const float inv = 1.f / (1.f - kf);
    c = (1.f - rf - kf) * inv;
    m = (1.f - gf - kf) * inv;
    y = (1.f - bf - kf) * inv;
    k = kf;
  }
  else {
    c = 0.f;
    m = 0.f;
    y = 0.f;
    k = 1.f;
  }
}

static const float THRESHOLD_COLOR_DIST_SQUARE = 0.85;

static inline
float
distanceMarker(float c, float m, float y, float k)
{
  assert(c>=0.f && c<=1.f);
  assert(m>=0.f && m<=1.f);
  assert(y>=0.f && y<=1.f);
  assert(k>=0.f && k<=1.f);

  const float distL2Square = (MARKER_CYAN-c)*(MARKER_CYAN-c)
    + (MARKER_MAGENTA-m)*(MARKER_MAGENTA-m)
    + (MARKER_YELLOW-y)*(MARKER_YELLOW-y)
    + (MARKER_BLACK-k)*(MARKER_BLACK-k);

  //std::cerr<<c<<" "<<m<<" "<<y<<" "<<k<<" "<<distL2Square<<"\n";

  return distL2Square;
}

static inline
float
distanceMarker(const cv::Vec3b &pix)
{
  float c, m, y, k;
  //warning: pix is in BGR
  RGB_to_CMYK(pix[2], pix[1], pix[0], c, m, y, k);

  return distanceMarker(c, m, y, k);
}

static size_t DBG_1 = 0;
static size_t DBG_2 = 0;

static inline
bool
checkIfMarker(const cv::Vec3b &pix, float thresholdDist)
{
  //return (distanceMarker(pix) <= thresholdDist);

  bool r = (distanceMarker(pix) <= thresholdDist);
  DBG_1 += r;
  return r;
}

// thresholdDist2 > thresholdDist1
static inline
bool
checkIfMarker2(const cv::Vec3b &pix, float thresholdDist1, float thresholdDist2)
{
  float c, m, y, k;
  //warning: pix is in BGR
  RGB_to_CMYK(pix[2], pix[1], pix[0], c, m, y, k);

  const float dist = distanceMarker(c, m, y, k);

  const float CYAN_THRES = MARKER_CYAN*0.23;
  const float BLACK_THRES = MARKER_BLACK*0.5;
  const float BLACK_THRES0 = 0.6; //0.76;

#if 0
  {
    if (dist > thresholdDist1 && dist< thresholdDist2) {

      if (! (k > BLACK_THRES0 || (c > CYAN_THRES && k > BLACK_THRES))) {
	std::cerr<<"NO : r="<<(int)(pix[2])<<" g="<<(int)(pix[1])<<" b="<<(int)(pix[0])<<" c="<<c<<" m="<<m<<" y="<<y<<" k="<<k<<"    distanceMarker(c, m, y, k)="<<distanceMarker(c, m, y, k)<<" distTh2="<<thresholdDist2<<"  CYAN_THRES="<<CYAN_THRES<<" BLACK_THRES="<<BLACK_THRES<<"\n";
      }
    }
  }
#endif

  return (dist <= thresholdDist1)
    || (dist <= thresholdDist2 && (k > BLACK_THRES0 || (c > CYAN_THRES && k > BLACK_THRES)));
  //k > BLACK_THRES0 means very dark 
}

static inline
bool
checkIfMarker(const cv::Vec3b &pix)
{
  static float thres1 = THRESHOLD_COLOR_DIST_SQUARE;
  static float thres2 = 1.15f*thres1;
  //return checkIfMarker2(pix, thres1, thres2);
  bool r = checkIfMarker2(pix, thres1, thres2);
  DBG_2 += r;
  return r;
}

static
void
binarizeForMarker(const cv::Mat &img,
		  cv::Mat &imgBin)
{
  assert(img.type() == CV_8UC3);

  imgBin.create(img.rows, img.cols, CV_8UC1);

  int rows = img.rows;
  int cols = img.cols;
  if (img.isContinuous() && imgBin.isContinuous()) {
    cols *= rows;
    rows = 1;
  }

  for (int i=0; i<rows; ++i) {
    const cv::Vec3b *s = img.ptr<cv::Vec3b>(i);
    unsigned char *d=imgBin.ptr<unsigned char>(i);
    
    for (int j=0; j<cols; ++j) {
      
#if 1
      const bool isMarker = checkIfMarker(*s);
      if (isMarker) {
	*d = 255;
      }
      else {
	*d = 0;
      }
#else
      //WARNING: does not work with gcc4.4.7 on CentOS 6.5
      *d = 255 * checkIfMarker(*s);
#endif

      ++d;
      ++s;
    }
  }
  
  assert(imgBin.type() == CV_8UC1);

}



//compute L2 square distance
static inline
float
distance(cv::Point2f p1, cv::Point2f p2)
{
  return (p1.x-p2.x)*(p1.x-p2.x) + (p1.y-p2.y)*(p1.y-p2.y);
}


struct SorterAreaOBB
{
public:

  explicit SorterAreaOBB(const std::vector<cv::RotatedRect> &rects)
  {
    const size_t nbRects = rects.size();
    m_areas.resize(nbRects);
    for (size_t i=0; i<nbRects; ++i) {
      m_areas[i] = rects[i].size.width * rects[i].size.height;
    }
  }

  bool operator()(size_t i, size_t j) const
  {
    return m_areas[i] > m_areas[j];
  }

protected:
  std::vector<float> m_areas;

};

/*
  We want the four points of the contour corresponding to the four corners of the marker.

  We want these corners always in the same order.
*/
static
bool
getCorners(const std::vector<cv::Point> &contour, std::vector<cv::Point2f> &corners)
{
  assert(contour.size() >= 4);

  corners.resize(4);

  const size_t nbPts = contour.size();

  if (nbPts == 4) {
    for (int i=0; i<4; ++i) 
      corners[i] = contour[i];
  }
  else {

    //first we store corners of OBB in @a corners
    //then we will store real points from @a contour

    cv::RotatedRect rr = cv::minAreaRect(contour);

    corners[0] = cv::Point2f(-rr.size.width/2, -rr.size.height/2);
    corners[1] = cv::Point2f(-rr.size.width/2, +rr.size.height/2);
    corners[2] = cv::Point2f(+rr.size.width/2, +rr.size.height/2);
    corners[3] = cv::Point2f(+rr.size.width/2, -rr.size.height/2);
    //rotation by rr.angle
    const float cos_angle = cosf(rr.angle);
    const float sin_angle = sinf(rr.angle);
    for (int i=0; i<4; ++i) {
      float x = rr.center.x + cos_angle*corners[i].x - sin_angle*corners[i].y;
      float y = rr.center.y + sin_angle*corners[i].x + cos_angle*corners[i].y;
      corners[i].x = x;
      corners[i].y = y;
    }
    
    std::vector<int> indices(4, nbPts);
    std::vector<float> minDists(4, std::numeric_limits<float>::max());
    for (int i=0; i<4; ++i) {
      for (size_t k=0; k<nbPts; ++k) {
	const float dist = distance(corners[i], contour[k]);
	if (dist < minDists[i]) {
	  indices[i] = k;
	  minDists[i] = dist;
	}
      }
    }
    //check that two points are not associated to the same corner
    for (int i=0; i<3; ++i) {
      for (int j=i+1; j<4; ++j) {
	if (indices[i] == indices[j]) {
	  std::cerr<<"corners associated to same point !\n";
	  return false;
	}
      }
    }
    
    for (int i=0; i<4; ++i) {
      corners[i] = contour[indices[i]];
    }

  }

  //We want them always in the same order
  std::vector<cv::Point2f> corners2(4);
  const cv::Point2f c0(0, 0);
  cv::Point2f p0 = corners[0];
  int ind0 = 0;
  float minDist = distance(p0, c0);
  for (int i=1; i<4; ++i) {
    const float dist = distance(corners[i], c0);
    if (dist < minDist) {
      minDist = dist;
      p0 = corners[i];
      ind0 = i;
    }
  }
  corners2[0] = corners[ind0];
  int ind1a = (ind0+1)%4;
  int ind1b = ind0 > 0 ? ind0-1 : 3;
  if (corners[ind1a].y > corners[ind1b].y) {
    for (int i=1; i<4; ++i) 
      corners2[i] = corners[(ind0+i)%4];
  }
  else {
    for (int i=1; i<4; ++i) 
      corners2[i] = corners[(ind0+4-i)%4];
  }

  corners = corners2;
  
  return true;
}


static const double axisRatioThreshold = 1.9;
static const double compactnessThreshold = 0.36; //0.5; //we don't want to filter too much !
static const double minAreaThresholdPercent = 0.002;
static const double maxAreaThresholdPercent = 0.85;

#if DEBUG_VISU
//@param[in, out] img
static
void
printContours(cv::Mat &img, std::vector<std::vector<cv::Point> > &contours)
{
  std::cerr<<contours.size()<<" contours\n";
  for (size_t i=0; i<contours.size(); ++i) {
    std::vector<cv::Point> &c = contours[i];

    const cv::RotatedRect r = cv::minAreaRect(c);
    const double areaOBB = r.size.area(); //r.area();
    int maxAxis = r.size.width;
    int minAxis = r.size.height;
    
    if (minAxis > maxAxis)
      std::swap(maxAxis, minAxis);
    const float ratioAxis = maxAxis / (float)minAxis;

    static double imgArea = img.rows*img.cols;;
    static const double minAreaThreshold = minAreaThresholdPercent * imgArea;
    static const double maxAreaThreshold = maxAreaThresholdPercent * imgArea;

    const double area = r.size.width*r.size.height; //cv::contourArea(contours0[i]); //contourArea seems to give wrong results sometimes
    const double perimeter = cv::arcLength(c, true);
    //const double y1 = area / (perimeter*perimeter);
    const double y3 = 4*sqrt(area) / perimeter;
    std::cerr<<"contour "<<i<<": size="<<c.size()<<"\n";
    std::cerr<<"areaOBB="<<areaOBB<<" >? minAreaThreshold="<<minAreaThreshold<<" => "<<(areaOBB > minAreaThreshold)<<"\n";
    std::cerr<<"areaOBB="<<areaOBB<<" <? maxAreaThreshold="<<maxAreaThreshold<<" => "<<(areaOBB < maxAreaThreshold)<<"\n";
    std::cerr<<"ratioAxis="<<ratioAxis<<" <? axisRatioThreshold="<<axisRatioThreshold<<" => "<<(ratioAxis < axisRatioThreshold)<<"\n";
    std::cerr<<"y3="<<y3<<" >? compactnessThreshold="<<compactnessThreshold<<" => "<<(y3 > compactnessThreshold)<<"\n";


    cv::Point2f vertices[4];
    r.points(vertices);
    for (int i = 0; i < 4; i++)
      cv::line(img, vertices[i], vertices[(i+1)%4], cv::Scalar(128,128,128));

  }

}
#endif //DEBUG_VISU


static
void
findSquareContours(const cv::Mat &img, std::vector<std::vector<cv::Point> > &contours)
{
  cv::Mat edges;
  const double th0 = 80; //40;//80;
  double th1 = 4*th0; //120; //4*th0;
  cv::Canny(img, edges, th0, th1);
  
  //close contour
  /*REM: we do not want a too large filter here 
    because it makes fat corners on the contour, 
    and then we have difficulties to find right corners later.
  */
  cv::Mat edges2;
  const int element2HalfSize = 3; //13
  const cv::Mat element = cv::getStructuringElement(cv::MORPH_RECT,
						    cv::Size(2*element2HalfSize+1, 2*element2HalfSize+1), 
						    cv::Point(element2HalfSize, element2HalfSize));
  cv::morphologyEx(edges, edges2, cv::MORPH_CLOSE, element);

  edges = edges2;

#if DEBUG_VISU
  {
    //cv::imshow("canny_morph", edges);
  }
#endif //DEBUG_VISU

  std::vector<cv::Vec4i> hierarchy0;
  std::vector<std::vector<cv::Point> > contours0;
  cv::findContours(edges, contours0, hierarchy0, cv::RETR_LIST, cv::CHAIN_APPROX_SIMPLE); //cv::RETR_TREE

#ifdef VERBOSE_MARKER
  std::cerr<<"contours0.size()="<<contours0.size()<<"\n";
#endif //VERBOSE_MARKER


  //-filter found contours: according to area & compactness

  //contours must have an area bigger than a certain percentage of image and lower than another
  // && an axis ratio superior to a threshold
  // && a compactness ratio superior to a threshold

  static double imgArea = img.rows*img.cols;;
  static const double minAreaThreshold = minAreaThresholdPercent * imgArea;
  static const double maxAreaThreshold = maxAreaThresholdPercent * imgArea;

#if DEBUG_VISU
#if 0
  {
    cv::Mat tmp = cv::Mat::zeros(img.rows, img.cols, CV_8UC3);
    for (int i=0; i<contours0.size(); ++i) {
      std::cerr<<"contours0["<<i<<"].size()="<<contours0[i].size()<<"\n";
      //printContours(tmp, contours0);  
      drawContours(tmp, contours0, i, cv::Scalar((40*(i+1))%255, 50*(contours0.size()-i), 32*i), 1); 
    }
    //cv::imshow("tmp", tmp);
    //cv::waitKey(0);
  }
#endif
#endif //DEBUG_VISU


  std::vector<std::vector<cv::Point> > contours1;
  for (size_t i=0; i<contours0.size(); ++i) {

    if (contours0[i].size() > 3) { //at least 4 points

      const cv::RotatedRect r = cv::minAreaRect(contours0[i]);
      const double areaOBB = r.size.area(); //r.area();
      int maxAxis = r.size.width;
      int minAxis = r.size.height;

      if (minAxis > maxAxis)
	std::swap(maxAxis, minAxis);
      const float ratioAxis = maxAxis / (float)minAxis;

      const double area = r.size.width*r.size.height; //cv::contourArea(contours0[i]); //contourArea seems to give wrong results sometimes
      const double perimeter = cv::arcLength(contours0[i], true);
      //const double y1 = area / (perimeter*perimeter);
      const double y3 = 4*sqrt(area) / perimeter;

#if DEBUG_VISU
#if 0
      std::cerr<<"countours0["<<i<<"]\n";
      std::cerr<<"areaOBB="<<areaOBB<<" > minAreaThreshold="<<minAreaThreshold<<" ? "<<(areaOBB > minAreaThreshold)<<"\n";
      std::cerr<<"areaOBB="<<areaOBB<<" < maxAreaThreshold="<<maxAreaThreshold<<" ? "<<(areaOBB < maxAreaThreshold)<<"\n";
      std::cerr<<"ratioAxis="<<ratioAxis<<" < axisRatioThreshold="<<axisRatioThreshold<<" ? "<<(ratioAxis < axisRatioThreshold)<<"\n";
      std::cerr<<"y3="<<y3<<" > compactnessThreshold="<<compactnessThreshold<<" ? "<<(y3 > compactnessThreshold)<<"\n";
      std::cerr<<"\n";
#endif
#endif //DEBUG_VISU


      if (areaOBB > minAreaThreshold && areaOBB < maxAreaThreshold && ratioAxis < axisRatioThreshold && y3 > compactnessThreshold) {
	contours1.push_back(contours0[i]);
      }
    }
  }

#ifdef VERBOSE_MARKER
  std::cerr<<"contours1.size()="<<contours1.size()<<"\n";
#endif //VERBOSE_MARKER
  
  std::vector<std::vector<cv::Point> > contours2(contours1.size());
  for( size_t k = 0; k < contours1.size(); k++ )
    approxPolyDP(cv::Mat(contours1[k]), contours2[k], 5, true);
  
#ifdef VERBOSE_MARKER
  std::cerr<<"contours2.size()="<<contours2.size()<<"\n";
#endif //VERBOSE_MARKER


  //- Remove duplicates among contours

  std::vector<std::vector<cv::Point> > contours3;

  std::vector<cv::RotatedRect> rects3;
  if (contours2.size() > 1) {

    std::vector<cv::RotatedRect> rects(contours2.size());
    for( size_t i = 0; i < contours2.size(); ++i ) {
      const cv::RotatedRect r = cv::minAreaRect(contours2[i]);
      rects[i] = r; 
    }
    //TODO: make thresholds dependant of image size ?
    static const float thresholdDistance = 17.f;
    static const float thresholdW = 10.f;
    static const float thresholdH = 10.f;
    
    std::vector<bool> duplicates(contours2.size(), false);
    size_t nbDuplicates = 0;
    for( size_t i = 0; i < contours2.size(); ++i ) {
      if (duplicates[i] == false) {
	const cv::RotatedRect &r_i = rects[i];
	const cv::Point2f &center_i = r_i.center;
	for( size_t j = i+1; j < contours2.size(); ++j ) {
	  const cv::RotatedRect &r_j = rects[j];
	  const cv::Point2f &center_j = r_j.center;
	  const float dist = distance(center_i, center_j);
	  //const float d_a = fabs(r_i.angle-r_j.angle);
	  const float d_w = fabs(r_i.size.width-r_j.size.width);
	  const float d_h = fabs(r_i.size.height-r_j.size.height);
	  //duplicate if almost same center, same width & same height
	  if (dist < thresholdDistance && d_w<thresholdW && d_h<thresholdH) {
	    //We want to keep the one with the smallest area of the twos
	    const float area_i = r_i.size.width*r_i.size.height;
	    const float area_j = r_j.size.width*r_j.size.height;
	    if (area_i <= area_j) {
	      duplicates[j] = true;
	    }
	    else {
	      duplicates[i] = true;
	      break;
	    }
	    ++nbDuplicates;
	  }
	}
      }
    }

    contours3.reserve(contours2.size()-nbDuplicates);
    rects3.reserve(contours2.size()-nbDuplicates);
    for (size_t i = 0; i < contours2.size(); ++i) {
      if (duplicates[i] == false) {
	contours3.push_back(contours2[i]);
	rects3.push_back(rects[i]);
      }
    }  

  }
  else {
    std::swap(contours2, contours3);
  }

#ifdef VERBOSE_MARKER
  std::cerr<<"contours3.size()="<<contours3.size()<<"\n";
#endif //VERBOSE_MARKER

  
  std::swap(contours, contours3);
  
}

/*
  Get mean blue and white color (and center point)
  from contour @a blueContour.

  We scale point according to center: inwards for blue, outwards for white.
*/
static
void
getMeanColorsOnContour(const cv::Mat &img, const std::vector<cv::Point> &blueContour, 
		       cv::Vec3b &meanBlue, cv::Vec3b &meanWhite, cv::Point &center)
{
#if DEBUG_VISU
  cv::Mat imgTmp = img.clone();
#endif

  const cv::RotatedRect r = cv::minAreaRect(blueContour);
  std::vector<cv::Point2f> pts(4);
  r.points(&pts[0]); 
  cv::Point2f c = r.center;
  cv::Vec3f meanBlueC(255, 0, 0);
  cv::Vec3f meanWhiteC(255, 255, 255);
  int nbPts = 0;
  const float scaleIn = 0.71f; //1.1f;
  const float scaleOut = 1.21f; //1.1f;
  for (int k=0; k<4; ++k) {
    cv::Point ptB = (pts[k]-c)*scaleIn + c;
    cv::Point ptW = (pts[k]-c)*scaleOut + c;
    if (ptW.x>=0 && ptW.y>=0 && ptW.x<img.cols && ptW.y<img.rows) {
      if (nbPts > 0) {
	meanBlueC += img.at<cv::Vec3b>(ptB.y, ptB.x);
	meanWhiteC += img.at<cv::Vec3b>(ptW.y, ptW.x);
      }
      else {
	meanBlueC = img.at<cv::Vec3b>(ptB.y, ptB.x);
	meanWhiteC = img.at<cv::Vec3b>(ptW.y, ptW.x);	
      }

#if DEBUG_VISU
      cv::circle(imgTmp, ptB, 3, cv::Scalar(128, 255, 255));
      cv::circle(imgTmp, ptW, 3, cv::Scalar(128, 255, 64));
#endif  //DEBUG_VISU


      ++nbPts;
    }
  }
  if (nbPts > 1) {//at least 2 points
    meanBlue = meanBlueC / static_cast<float>(nbPts);
    meanWhite = meanWhiteC / static_cast<float>(nbPts);
    center = c;
  }

#if DEBUG_VISU
  std::cerr<<"meanBlue: "<<(int)(meanBlue[0])<<" "<<(int)(meanBlue[1])<<" "<<(int)(meanBlue[2])<<"\n";
  std::cerr<<"meanWhite: "<<(int)(meanWhite[0])<<" "<<(int)(meanWhite[1])<<" "<<(int)(meanWhite[2])<<"\n";
  //cv::imshow("blueWhitePoints", imgTmp);
#endif  //DEBUG_VISU

}

//Compute euclidean distance between two colors @a a and @a b.
static
float
distL2S(const cv::Vec3b &a, const cv::Vec3b &b)
{
  return (a[0]-b[0])*(a[0]-b[0]) + (a[1]-b[1])*(a[1]-b[1]) + (a[2]-b[2])*(a[2]-b[2]);
}

//check if pixel @a Pix is closer to blue @a meanBlue than white @a meanWhite.
static
bool
isMoreBlueThanWhite(const cv::Vec3b &Pix, const cv::Vec3b &meanBlue, const cv::Vec3b &meanWhite)
{
  return distL2S(Pix, meanBlue) < 0.9f*distL2S(Pix, meanWhite);
}


static
cv::Vec3f
BGRtoHSV(cv::Vec3b bgr)
{
  cv::Vec3f r;
  RGB_to_HSV(bgr[2], bgr[1], bgr[0], r[0], r[1], r[2]);
  return r;
}

static
cv::Vec3f
BGRtoLAB(cv::Vec3b bgr)
{
  cv::Vec3f r;
  RGB_to_LAB(bgr[2], bgr[1], bgr[0], r[0], r[1], r[2]);
  return r;
}



/**
   Select among contours in @a contours1 the one with white outside.
   
   Pixels outside corners must have a similar color and be rather white (or gray).

   @param[in, out] contours1
*/
void
selectContourWithWhiteOutside(const cv::Mat &img,
			      std::vector<std::vector<cv::Point> > &contours1
#if DEBUG_VISU
			      , cv::Mat &imgTmp
#endif
			      )
{
  const float THRESHOLD_DISTANCE_WHITE = 87*87 + 87*87 + 87*87; //80*80 + 80*80 + 80*80;
  const unsigned char GRAY_DISTANCE = 18;
  const float THRESHOLD_DISTANCE_WHITE_2 = 38000;
  assert(THRESHOLD_DISTANCE_WHITE < THRESHOLD_DISTANCE_WHITE_2);
 
  const size_t nbContours1 = contours1.size();
 
  float minDist = std::numeric_limits<float>::max();
  size_t ind = nbContours1;
  for (size_t i=0; i<nbContours1; ++i) {
    const cv::RotatedRect r = cv::minAreaRect(contours1[i]);
    std::vector<cv::Point2f> pts(4);
    r.points(&pts[0]); 
    cv::Point2f c = r.center;
    int nbPts = 0;
    const float scale = 1.2f; //1.1f;
    std::vector<cv::Vec3b> ptsColor(4);
    for (int k=0; k<4; ++k) {
      pts[k] = (pts[k]-c)*scale + c;
      
#if DEBUG_VISU
      cv::circle(imgTmp, pts[k], 3, cv::Scalar(128, 255, 128));
#endif  //DEBUG_VISU
      
      cv::Point pt = pts[k];
      if (pt.x>=0 && pt.y>=0 && pt.x<img.cols && pt.y<img.rows) {
	ptsColor[nbPts] = (img.at<cv::Vec3b>(pt.y, pt.x));
	++nbPts;
      }
    }
    if (nbPts > 1) {//at least 2 points
      
#if DEBUG_VISU
      for (int k=0; k<nbPts; ++k)
	std::cerr<<"ptsColor["<<k<<"]="<<ptsColor[k]<<"\n";
#endif  //DEBUG_VISU
      
      
      cv::Vec3f meanColor = ptsColor[0];
      cv::Vec3f meanColorHSV = BGRtoHSV(ptsColor[0]);
      cv::Vec3f meanColorLAB = BGRtoLAB(ptsColor[0]);
      for (int j=1; j<nbPts; ++j) {//start from 1
	meanColor += ptsColor[j];
	meanColorHSV += BGRtoHSV(ptsColor[0]);
	meanColorLAB += BGRtoLAB(ptsColor[0]);
      }
      
      meanColor /= (float)nbPts;
      const float dist = (255-meanColor[0])*(255-meanColor[0]) + (255-meanColor[1])*(255-meanColor[1]) + (255-meanColor[2])*(255-meanColor[2]);
#if DEBUG_VISU
      std::cerr<<"i="<<i<<" meanColor="<<meanColor<<" nbPts="<<nbPts<<" dist="<<dist<<"<? THRESHOLD_DISTANCE_WHITE="<<THRESHOLD_DISTANCE_WHITE<<"\n";
      std::cerr<<"i="<<i<<" meanColorHSV="<<meanColorHSV<<"\n";
      std::cerr<<"i="<<i<<" meanColorLAB="<<meanColorLAB<<"\n";
#endif  //DEBUG_VISU
      if (dist < minDist && dist<THRESHOLD_DISTANCE_WHITE) {
	minDist = dist;
	ind = i;
      }
      else {
	
	bool isGray = true;
	for (int l=0; l<nbPts-1; ++l) {
	  cv::Vec3b color_i = ptsColor[l];
	  for (int j=1; j<nbPts; ++j) {
	    cv::Vec3b color_j = ptsColor[j];
	    for (int k=0; k<3; ++k) {
	      isGray &= (abs(color_i[k] - color_j[k]) < GRAY_DISTANCE);
	    }
	  }
	}
#if DEBUG_VISU
	std::cerr<<"i="<<i<<" meanColor="<<meanColor<<" nbPts="<<nbPts<<" dist="<<dist<<"<? THRESHOLD_DISTANCE_WHITE_2="<<THRESHOLD_DISTANCE_WHITE_2<<" isGray="<<isGray<<"\n";
#endif  //DEBUG_VISU
	if (isGray && dist < minDist && dist<THRESHOLD_DISTANCE_WHITE_2) {
	  minDist = dist;
	  ind = i;
	}
	
      }
    }
  }
    
#if DEBUG_VISU
  if (ind != nbContours1) {
    drawContours(imgTmp, contours1, ind, cv::Scalar(0, 255, 0)); 
    cv::imshow("blueContoursChoice", imgTmp);
  }
  else {
    cv::imshow("blueContours_NOT_FOUND", imgTmp);
  }
  
  std::cerr<<"ind="<<ind<<" nbContours1="<<nbContours1<<"\n";
#endif  //DEBUG_VISU
  
  if (ind != nbContours1) {
    std::swap(contours1[0], contours1[ind]);
    contours1.resize(1);
  }
  else {
    contours1.clear();
  }

}

/*
  Detect blue contour precisely on @a img.
  
  We use the given contour @a c that is an approximation of the blue contour 
  to get the blue and white colors of the marker on the image @a img.
  We then do a region growing algorithm starting from the center of this contour,
  and adding "more blue than white" pixels.
 */
static
void
findBlueContourWithRegionGrowing(const cv::Mat &img, 
				 const std::vector<cv::Point> &c, 
				 std::vector<cv::Point> &blueContour)
{
  cv::Point center;
  cv::Vec3b meanWhite, meanBlue;
  getMeanColorsOnContour(img, c, meanBlue, meanWhite, center);
  
  //const float thresholdDist0 = 0.85 * 1.2; 
  //const float thresholdDist1 = 0.85 * 1.56; //1.44;
  
  //thresholdDist0 : 0.85*1.2 val max for 131108325.png
  //thresholdDist0 : 0.85*1.2  too high for 131164499.png

  const float thresholdDistH = THRESHOLD_COLOR_DIST_SQUARE*1.05;

  //region growing
  
  cv::Mat maskTmp = cv::Mat::zeros(img.rows, img.cols, CV_8UC1);
  const std::vector<cv::Point> &ptsV = c;
  std::queue<cv::Point> ptsQueue;
  ptsQueue.push(center);
  for (size_t i=0; i<ptsV.size(); ++i)
    ptsQueue.push(ptsV[i]);
  
  while (! ptsQueue.empty()) {
    
    cv::Point pt = ptsQueue.front();
    ptsQueue.pop();
    
    unsigned char &m = maskTmp.at<unsigned char>(pt.y, pt.x);
    if (m == 0) {

      if (checkIfMarker(img.at<cv::Vec3b>(pt.y, pt.x), thresholdDistH) ||
	  isMoreBlueThanWhite(img.at<cv::Vec3b>(pt.y, pt.x), meanBlue, meanWhite)) {
	
	m = 255;
	if (pt.y > 0)
	  ptsQueue.push(cv::Point(pt.x, pt.y-1));
	if (pt.x > 0) 
	  ptsQueue.push(cv::Point(pt.x-1, pt.y));
	if (pt.y < img.rows-1)
	  ptsQueue.push(cv::Point(pt.x, pt.y+1));
	if (pt.x < img.cols-1)
	  ptsQueue.push(cv::Point(pt.x+1, pt.y));
      }
    }
  }

#if DEBUG_VISU
  {
    //cv::imshow("maskRegionGrowing", maskTmp);
  }
#endif

  std::vector<cv::Vec4i> hierarchy0b;
  std::vector<std::vector<cv::Point> > contours0b;
  cv::findContours(maskTmp, contours0b, hierarchy0b, cv::RETR_LIST, cv::CHAIN_APPROX_SIMPLE); //cv::RETR_TREE
  
  //std::cerr<<"contours0b.size()="<<contours0b.size()<<"\n";

  if (contours0b.size() == 0) {
#if DEBUG_VISU
    std::cerr<<"NO CONTOUR FOUND\n";
#endif //DEBUG_VISU

    blueContour.clear();
    return;
  }
  else {
    assert(contours0b.size() >= 1);

    if (contours0b.size() > 1) {
      //It is possible to have holes in maskTmp
      // and thus several contours
      //We keep the longest one
      
      size_t indMax = 0;
      size_t lenMax = contours0b[0].size();
      for (size_t i=0; i<contours0b.size(); ++i) {
	const size_t len = contours0b[i].size();
	if (len > lenMax) {
	  indMax = i;
	  lenMax = len;
	}
      }
      if (indMax != 0)
	std::swap(contours0b[0], contours0b[indMax]);
    }

    std::vector<std::vector<cv::Point> > contours0c(1);
    approxPolyDP(cv::Mat(contours0b[0]), contours0c[0], 5, true);
  
#if DEBUG_VISU
    {
      std::vector<std::vector<cv::Point> > contours1(1, c);

      cv::Mat imgTmp = img.clone();
      drawContours(imgTmp, contours1, 0, cv::Scalar(0, 255, 64)); 
      drawContours(imgTmp, contours0b, -1, cv::Scalar(0, 64, 255)); 
      drawContours(imgTmp, contours0c, -1, cv::Scalar(0, 255, 255)); 

      //cv::imshow("blueContours_1and0", imgTmp);
    }

#if 0
    {//DEBUG
      std::cerr<<"contours0b[0].size()="<<contours0b[0].size()<<"\n";
      for (int i=3; i<15; i+=2) {
	std::vector<std::vector<cv::Point> > contourApprox(1);
	approxPolyDP(cv::Mat(contours0b[0]), contourApprox[0], 7, true);
	std::cerr<<"epsilon="<<i<<" contourApprox[0].size()="<<contourApprox[0].size()<<"\n";
      }
    }//DEBUG
#endif

#endif //DEBUG_VISU
    
    blueContour = contours0c[0];
    return;
  }
}



/*
  find marker blue part countour in image.

*/
static
void
findBlueContour(const cv::Mat &img, std::vector<cv::Point> &blueContour)
{
  assert(img.type() == CV_8UC3);


  blueContour.clear();

  cv::Mat imgBin;
  binarizeForMarker(img, imgBin);


  {
    //remove crispy border...
    //We have then more chances to find contours in one piece with canny later !

    cv::Mat imgBin2;
    const int element2HalfSize = 7; 
    const cv::Mat element = cv::getStructuringElement(cv::MORPH_RECT,
						      cv::Size(2*element2HalfSize+1, 2*element2HalfSize+1), 
						      cv::Point(element2HalfSize, element2HalfSize));
    cv::morphologyEx(imgBin, imgBin2, cv::MORPH_CLOSE, element);
    imgBin = imgBin2;
  }


#if DEBUG_VISU
  {
    //cv::imshow("blue", imgBin);
  }
#endif  //DEBUG_VISU

  std::vector<std::vector<cv::Point> > contours1;
  findSquareContours(imgBin, contours1); 


#if DEBUG_VISU
  cv::Mat imgTmp = cv::Mat::zeros(img.rows, img.cols, CV_8UC3);
  drawContours(imgTmp, contours1, -1, cv::Scalar(255, 255, 255)); 
#endif 


  //We should find only one (blue) contour on imgBin !
  //Besides, correct blue contour must have "white" around it.

  const size_t nbContours1 = contours1.size();

#if DEBUG_VISU
  printContours(imgBin, contours1);
  //cv::imshow("contours", imgBin);
#endif  //DEBUG_VISU

  if (nbContours1 == 0) {
    blueContour.clear();
    return;
  }

  selectContourWithWhiteOutside(img, 
				contours1
#if DEBUG_VISU
				, imgTmp
#endif
				);


  if (contours1.size() != 1) {
    blueContour.clear();
    return;
  }


  //Now that we are more sure to have the correct (blue) contour,
  // we try to detect this contour again with a larger threshold on distance to blue color.
  // We should get a more precise contour.

  findBlueContourWithRegionGrowing(img, contours1[0], blueContour);

}

namespace {
  class MomentsComputer //: public girl::toger::pointIterator
  {
  public:
    explicit MomentsComputer(const cv::Mat &image)
      : m_image(image),
	m_m0(0), m_m1(), m_m2()
    {
      assert(image.type() == CV_8UC3);

      for (int i=0; i<3; ++i) {
	m_m1[i] = 0;
	m_m2[i] = 0;
      }

    }

    /*
    virtual void processPoint(const girl::point &pt)
    {
      const cv::Vec3b pix = m_image.at<cv::Vec3b>(pt.y(), pt.x());
      processPix(pix);
    }
    */

    inline void processPix(const cv::Vec3b pix)
    {
      ++m_m0;
      //BGR -> RGB
      m_m1[0] += pix[2];
      m_m1[1] += pix[1];
      m_m1[2] += pix[0];
      m_m2[0] += pix[2]*pix[2];
      m_m2[1] += pix[1]*pix[1];
      m_m2[2] += pix[0]*pix[0];
    }


    void getMeanStdDev(float mean[3], float &stdDev)
    {
      if (m_m0 > 0) {
	const float inv_m0 = 1.f / m_m0;
	for (int i=0; i<3; ++i)
	  mean[i] = m_m1[i] * inv_m0;
	
	const double sm1s = m_m1[0] * m_m1[0] + m_m1[1] * m_m1[1] + m_m1[2] * m_m1[2];
	const double sm2 = m_m2[0] + m_m2[1] + m_m2[2];
	stdDev = (sm2 - sm1s*inv_m0) * inv_m0;
	if (stdDev <= 0)
	  stdDev = 0;
	else 
	  stdDev = sqrt(stdDev);
      }
      else {
	mean[0] = mean[1] = mean[2] = 0;
	stdDev = 0;
      }
    }

    unsigned long getM0() const { return m_m0; }

  protected:
    const cv::Mat &m_image;
    unsigned long m_m0;
    unsigned long m_m1[3], m_m2[3];

  };

  class MomentsComputerDist //: public girl::toger::pointIterator
  {
  public:
    MomentsComputerDist(const cv::Mat &image, float mean[3], float maxDist)
      : m_image(image),
	m_mean(), m_dist(maxDist*maxDist), //store square maxDist
	m_m0(0), m_m1(), m_m2()
    {
      assert(image.type() == CV_8UC3);

      for (int i=0; i<3; ++i) {
	m_mean[i] = mean[i];
	m_m1[i] = 0;
	m_m2[i] = 0;
      }

    }

    void reset(float mean[3], float maxDist) 
    {
      m_m0 = 0;
      for (int i=0; i<3; ++i) {
	m_mean[i] = mean[i];
	m_m1[i] = 0;
	m_m2[i] = 0;
      }

      m_dist = maxDist*maxDist; //store square maxDist
    }

    float dist(float x, float y, float z, float x2, float y2, float z2)
    {
      return (x-x2)*(x-x2) + (y-y2)*(y-y2) + (z-z2)*(z-z2);
    }

    /*
    virtual void processPoint(const girl::point &pt)
    {
      const cv::Vec3b pix = m_image.at<cv::Vec3b>(pt.y(), pt.x());
      processPix(pix);
    }
    */

    inline void processPix(const cv::Vec3b pix)
    {
      //BGR -> RGB
      const float d = dist(pix[2], pix[1], pix[0], m_mean[0], m_mean[1], m_mean[2]);
      if (d < m_dist) {
	++m_m0;
	m_m1[0] += pix[2];
	m_m1[1] += pix[1];
	m_m1[2] += pix[0];
	m_m2[0] += pix[2]*pix[2];
	m_m2[1] += pix[1]*pix[1];
	m_m2[2] += pix[0]*pix[0];
      }
    }



    void getMeanStdDev(float mean[3], float &stdDev)
    {
      if (m_m0 > 0) {
	const float inv_m0 = 1.f / m_m0;
	for (int i=0; i<3; ++i)
	  mean[i] = m_m1[i] * inv_m0;
	
	const double sm1s = m_m1[0] * m_m1[0] + m_m1[1] * m_m1[1] + m_m1[2] * m_m1[2];
	const double sm2 = m_m2[0] + m_m2[1] + m_m2[2];
	stdDev = (sm2 - sm1s*inv_m0) * inv_m0;
	if (stdDev <= 0)
	  stdDev = 0;
	else 
	  stdDev = sqrtf(stdDev);
      }
      else {
	mean[0] = mean[1] = mean[2] = 0;
	stdDev = 0;
      }
    }

    unsigned long getM0() const { return m_m0; }

  protected:
    const cv::Mat &m_image;
    float m_mean[3];
    float m_dist;
    unsigned long m_m0;
    unsigned long m_m1[3], m_m2[3];

  };

}

/**
   Get mean white and blue colors and marker mask.

   @param[in] img input image
   @param[in] contours input contours (contours[0] is the white contour, contours[1] is the blue contour)
   @param[out] blue mean blue color
   @param[out] white mean white color
   @param[out] markerMask marker mask
 */
static
void
getColorsAndPaintMask(const cv::Mat &img, 
		      const std::vector<std::vector<cv::Point> > &contours,
		      cv::Vec3f &blue, cv::Vec3f &white, cv::Mat &markerMask)
{
  assert(contours.size() == 2);

  //- draw filled contours on markerMask
  
  int bigId = 0;
  int smallId = 1;

  markerMask = cv::Mat::zeros(img.rows, img.cols, CV_8UC1);

  const unsigned char colorBigUC = 255;
  const cv::Scalar colorBig(colorBigUC);
  drawContours(markerMask, contours, bigId, colorBig, -1);
  const unsigned char colorSmallUC = 128;
  const cv::Scalar colorSmall(colorSmallUC);
  drawContours(markerMask, contours, smallId, colorSmall, -1);

  //girl::toger tog(img.cols, img.rows);

  //- compute mean & stdDev
  
  MomentsComputer mcBlue(img);
  MomentsComputer mcWhite(img);


  int rows = img.rows;
  int cols = img.cols;
  if (img.isContinuous() && markerMask.isContinuous()) {
    cols *= rows;
    rows = 1;
  }
  for (int i=0; i<rows; ++i) {
    unsigned char *m = markerMask.ptr<unsigned char>(i);
    const cv::Vec3b *d = img.ptr<cv::Vec3b>(i);
    for (int j=0; j<cols; ++j) {
      if (*m == colorBigUC)
	mcWhite.processPix(*d);
      else if (*m == colorSmallUC)
	mcBlue.processPix(*d);
      ++m;
      ++d;
    }
  }
  
  float meanBlue[3], stdDevBlue;
  mcBlue.getMeanStdDev(meanBlue, stdDevBlue);
  float meanWhite[3], stdDevWhite;
  mcWhite.getMeanStdDev(meanWhite, stdDevWhite);

#ifdef VERBOSE_MARKER
  std::cerr<<"***0) blue: ("<<meanBlue[0]<<", "<<meanBlue[1]<<", "<<meanBlue[2]<<") stdDev="<<stdDevBlue<<" : m0="<<mcBlue.getM0()<<"\n";
  std::cerr<<"***0) white: ("<<meanWhite[0]<<", "<<meanWhite[1]<<", "<<meanWhite[2]<<") stdDev="<<stdDevWhite<<" : m0="<<mcWhite.getM0()<<"\n";
#endif //VERBOSE_MARKER

  //- recompute mean & stdDev keeping only pixels not too far from (first) mean
  
  MomentsComputerDist mcdBlue(img, meanBlue, stdDevBlue);
  MomentsComputerDist mcdWhite(img, meanWhite, stdDevWhite);

  //update colors
  for (int i=0; i<rows; ++i) {
    unsigned char *m = markerMask.ptr<unsigned char>(i);
    const cv::Vec3b *d = img.ptr<cv::Vec3b>(i);
    for (int j=0; j<cols; ++j) {
      if (*m == colorBigUC) {
	mcdWhite.processPix(*d);
      }
      else if (*m == colorSmallUC) {
	mcdBlue.processPix(*d);
      }
      ++m;
      ++d;
    }
  }


  mcdBlue.getMeanStdDev(meanBlue, stdDevBlue);
  mcdWhite.getMeanStdDev(meanWhite, stdDevWhite);

#ifdef VERBOSE_MARKER
  std::cerr<<"****1) blue: ("<<meanBlue[0]<<", "<<meanBlue[1]<<", "<<meanBlue[2]<<") stdDev="<<stdDevBlue<<" : m0="<<mcdBlue.getM0()<<"\n";
  std::cerr<<"****1) white: ("<<meanWhite[0]<<", "<<meanWhite[1]<<", "<<meanWhite[2]<<") stdDev="<<stdDevWhite<<" : m0="<<mcdWhite.getM0()<<"\n";
#endif //VERBOSE_MARKER

  //- Again recompute mean & stdDev keeping only pixels not too far from (second) mean

  mcdBlue.reset(meanBlue, stdDevBlue);
  mcdWhite.reset(meanWhite, stdDevWhite);

  //update colors & write mask
  for (int i=0; i<rows; ++i) {
    unsigned char *m = markerMask.ptr<unsigned char>(i);
    const cv::Vec3b *d = img.ptr<cv::Vec3b>(i);
    for (int j=0; j<cols; ++j) {
      if (*m == colorBigUC) {
	mcdWhite.processPix(*d);
	//*m = 255; //already white
      }
      else if (*m == colorSmallUC) {
	mcdBlue.processPix(*d);
	*m = 255;
      }
      ++m;
      ++d;
    }
  }


  mcdBlue.getMeanStdDev(meanBlue, stdDevBlue);
  mcdWhite.getMeanStdDev(meanWhite, stdDevWhite);

#ifdef VERBOSE_MARKER
  std::cerr<<"****2) blue: ("<<meanBlue[0]<<", "<<meanBlue[1]<<", "<<meanBlue[2]<<") stdDev="<<stdDevBlue<<" : m0="<<mcdBlue.getM0()<<"\n";
  std::cerr<<"****2) white: ("<<meanWhite[0]<<", "<<meanWhite[1]<<", "<<meanWhite[2]<<") stdDev="<<stdDevWhite<<" : m0="<<mcdWhite.getM0()<<"\n";
#endif //VERBOSE_MARKER
  
  for (int i=0; i<3; ++i) {
    blue[i] = meanBlue[i];
    white[i] = meanWhite[i];
  }


}



/*
  compute distance L2square between two points.

*/
static inline
float
distPts(cv::Point2f pt1, cv::Point2f pt2)
{
  return (pt1.x-pt2.x)*(pt1.x-pt2.x) + (pt1.y-pt2.y)*(pt1.y-pt2.y);
} 

/*
  Find the point of @a pts the closest to @a pt.
  This point is returned in @a cpt, and the corresponding (L2 square) disatnce in @a dist.
*/
static
void
findClosestPoint(cv::Point2f pt, std::vector<cv::Point> pts, cv::Point2f &cpt, float &dist)
{
  cpt = cv::Point2f(0, 0);
  dist = 0;
  if (pts.empty())
    return;

  cpt = pts[0];
  dist = distPts(cpt, pt);
  for (size_t i=1; i<pts.size(); ++i) { //start from 1
    const float d = distPts(pt, pts[i]);
    if (d < dist) {
      dist = d;
      cpt = pts[i];
    }

  }
}

/*
  Compute intersection point between AC & BD

  Return (-1, -1) if no intersection.
*/
static
cv::Point2f
intersection(cv::Point2f A, cv::Point2f C, cv::Point2f B, cv::Point2f D)
{
  cv::Point2f I1(-1, -1);
  const float det = (A.x-C.x)*(D.y-B.y) - (A.y-C.y)*(D.x-B.x);
  if (det == 0) {
    //lines are parallel
    //std::cerr<<"lines are parallel !!!\n";
    return I1;
  }
  const float inv_det = 1.f/det;
  
  I1.x = ((A.x*C.y - A.y*C.x)*(D.x-B.x) - (A.x-C.x)*(D.x*B.y - D.y*B.x))*inv_det;
  I1.y = ((A.x*C.y - A.y*C.x)*(D.y-B.y) - (A.y-C.y)*(D.x*B.y - D.y*B.x))*inv_det;

  return I1;
}

/*
  Find white and blue contours.

  @param[out] contours contains 2 contours : contours[0] is the white contour, contours[1] is the blue contour.
  Each contour has 4 points.

 */
static
void
findBothContours(const std::vector<cv::Point> &blueContour, 
		 std::vector<std::vector<cv::Point> > &contours)
{
  assert(blueContour.size() >= 4);


  cv::RotatedRect rr = cv::minAreaRect(blueContour);
  cv::Point2f rect_points[4]; 
  rr.points( rect_points );
  
  cv::Point2f corners[4];
  float distances[4];
  for (int j=0; j<4; ++j)
    findClosestPoint(rect_points[j], blueContour, corners[j], distances[j]);
  //ne trouve que 2 pts de juste !!!

  float dist0 = distances[0] + distances[2];
  float dist1 = distances[1] + distances[3];
  int cornersIndices[4];
  if (dist0 < dist1) {
    cornersIndices[0] = 0;
    cornersIndices[1] = 1;
    cornersIndices[2] = 2;
    cornersIndices[3] = 3;
  }
  else {
    cornersIndices[0] = 1;
    cornersIndices[1] = 2;
    cornersIndices[2] = 3;
    cornersIndices[3] = 0;
  }
  
  //cornersIndices[0] & cornersIndices[2] : are indices of two closest points
  // thus of the corners coinciding with oriented rectangle corners
  
  
  cv::Point2f A = rect_points[cornersIndices[0]];
  cv::Point2f B = rect_points[cornersIndices[1]];
  cv::Point2f C = rect_points[cornersIndices[2]];
  cv::Point2f D = rect_points[cornersIndices[3]];
  cv::Point2f B2 = corners[cornersIndices[1]];
  cv::Point2f D2 = corners[cornersIndices[3]];
  
  //1st case : borders of blueContour colinear with AD & BC
  cv::Point2f I1 = intersection(A, B2, B, C);	  
  cv::Point2f I2 = intersection(C, D2, D, A);
  
  //2nd case : borders of blueContour colinear with DC & AB
  cv::Point2f I1b = intersection(C, B2, A, B);
  cv::Point2f I2b = intersection(A, D2, D, C);
  
  cv::Point2f tmpPt;
  float distanceI1, distanceI2, distanceI1b, distanceI2b;
  findClosestPoint(I1, blueContour, tmpPt, distanceI1);
  findClosestPoint(I2, blueContour, tmpPt, distanceI2);
  findClosestPoint(I1b, blueContour, tmpPt, distanceI1b);
  findClosestPoint(I2b, blueContour, tmpPt, distanceI2b);
  
  if ((distanceI1+distanceI2) > (distanceI1b+distanceI2b)) {
    std::swap(I1, I1b);
    std::swap(I2, I2b);
  }
  
  std::vector<cv::Point2f> blueContour2;
  blueContour2.resize(4);
  blueContour2[0] = corners[cornersIndices[0]];
  blueContour2[1] = I1;
  blueContour2[2] = corners[cornersIndices[2]];
  blueContour2[3] = I2;

  cv::Point2f c = intersection(blueContour2[0], blueContour2[2], blueContour2[1], blueContour2[3]);
  
  std::vector<cv::Point> whiteContour(4);
  std::vector<cv::Point> blueContour3(4);

  const float scale = 1.85f; //should be in [1; 2] 
  //no need for the whiteContour to be too large

  for (int i=0; i<4; ++i) {
    whiteContour[i] = c + (blueContour2[i] - c)*scale;
    blueContour3[i] = blueContour2[i];
    /*
      std::cerr<<"blueContour2["<<i<<"]="<<blueContour2[i]<<" scaled to whiteContour[i]="<<whiteContour[i]<<"\n";
      std::cerr<<"blueContour3["<<i<<"]="<<blueContour3[i]<<"\n";
    */
  }
  contours.resize(2);
  contours[0] = whiteContour;
  contours[1] = blueContour3;


  assert(contours.size() == 2);
}


bool
detectMarker(const cv::Mat &img, float &pixelSize, cv::Mat &markerMask, cv::Vec3f &blue, cv::Vec3f &white)
{
  assert(img.type() == CV_8UC3);

#if DEBUG_VISU
  {
    //cv::imshow("img", img);
    std::cerr<<"img w="<<img.cols<<" h="<<img.rows<<"\n";
  }
#endif //DEBUG_VISU

  
  std::vector<cv::Point> blueContour;
  findBlueContour(img, blueContour);
  if (blueContour.empty()) {

#if DEBUG_VISU
    {
      std::cerr<<"Unable to find blue contour on image\n";
      cv::waitKey(0);
    }
#endif  //DEBUG_VISU

    return false;
  }
  


  std::vector<std::vector<cv::Point> > contours;
  findBothContours(blueContour, contours);

#if DEBUG_VISU
  {
    //cv::imshow("img", img);
    
    cv::Mat imgCtrs = img.clone();
    cv::drawContours(imgCtrs, contours, -1, cv::Scalar(0, 255, 0));
    cv::imshow("contours", imgCtrs);
    cv::waitKey(0);
  }
#endif //DEBUG_VISU


  //bool colorsOk = getColorsAndPaintMask_old(img, contours, blue, white, markerMask);
  getColorsAndPaintMask(img, contours, blue, white, markerMask);


  //-compute marker size

  //Currently, we use only interior/blue region size to compute marker size
  //(contours[0] is the white contour, contours[1] is the blue contour)
  
  //std::vector<cv::Point2f> corners0;
  //const bool ok0 = getCorners(contours[0], corners0);
  std::vector<cv::Point2f> corners1;
  const bool ok1 = getCorners(contours[1], corners1);
  if (! ok1) { //if (! ok0 || ! ok1) {
    std::cerr<<"unable to get corners\n";
    return false;
  }

  //mean of four sides
  float meanSideSize = 0;
  for (int i=0; i<4; ++i) {
    const float sideSize = sqrtf(distance(corners1[i], corners1[(i+1)%4]));
    meanSideSize += sideSize; 
  }
  meanSideSize /= 4.f;

  //mean of two diagonals/sqrt(2)
  const float diag1 = sqrtf(distance(corners1[0], corners1[2]));
  const float diag2 = sqrtf(distance(corners1[1], corners1[3]));
  const float meanSideSize2 = (diag1+diag2)/(2*sqrt(2));
 
  const float finalSideSize = (meanSideSize + meanSideSize2)*0.5f;

  pixelSize = MARKER_SIZE/finalSideSize;  


#if DEBUG_VISU
  {
    std::cerr<<"*** FINAL:\n";
    std::cerr<<"*** pixelSize="<<pixelSize<<"\n";
    std::cerr<<"*** blue=["<<blue[0]<<", "<<blue[1]<<", "<<blue[2]<<"]\n";
    std::cerr<<"*** white=["<<white[0]<<", "<<white[1]<<", "<<white[2]<<"]\n";
  }
#endif  //DEBUG_VISU

  return true;
}


bool 
intersection(const cv::Mat &mask1, const cv::Mat &mask2)
{
  assert(mask1.type() == CV_8UC1);
  assert(mask2.type() == CV_8UC1);
  assert(mask1.size() == mask2.size());

  int rows = mask1.rows;
  int cols = mask1.cols;
  if (mask1.isContinuous() && mask2.isContinuous()) {
    cols *= rows;
    rows = 1;
  }
  for (int i=0; i<rows; ++i) {
    const unsigned char *m1 = mask1.ptr<uchar>(i);
    const unsigned char *m2 = mask2.ptr<uchar>(i);
    for (int j=0; j<cols; ++j) {
      if (*m1 != 0 && *m2 != 0) 
	return true;
      ++m1;
      ++m2;
    }
  }

  return false;
}



bool
findMarkerContours(const cv::Mat &img,
		   std::vector<cv::Point> &blueContour, 
		   std::vector<cv::Point> &whiteContour)
{
  //some code diplication with detectMarker()
  
  assert(img.type() == CV_8UC3);

  findBlueContour(img, blueContour);
  if (blueContour.empty()) {
    return false;
  }

  std::vector<std::vector<cv::Point> > contours;
  findBothContours(blueContour, contours);

  whiteContour = contours[0];
  blueContour = contours[1];

  return true;
}

void
computeMarkerColorStats(const cv::Mat &img,
			const std::vector<cv::Point> &blueContour, 
			const std::vector<cv::Point> &whiteContour,
			float meanBlue[3], float &stdDevBlue,
			float meanWhite[3], float &stdDevWhite)
{
  //some code duplication with getColorsAndPaintMask()

  assert(img.type() == CV_8UC3);
  
  //- draw filled contours on markerMask
  
  cv::Mat markerMask = cv::Mat::zeros(img.rows, img.cols, CV_8UC1);

  std::vector< std::vector<cv::Point> > contours(1);
  contours[0] = whiteContour;
  const unsigned char colorBigUC = 255;
  const cv::Scalar colorBig(colorBigUC);
  drawContours(markerMask, contours, 0, colorBig, -1);
  contours[0] = blueContour;
  const unsigned char colorSmallUC = 128;
  const cv::Scalar colorSmall(colorSmallUC);
  drawContours(markerMask, contours, 0, colorSmall, -1);

  //- compute mean & stdDev
  
  MomentsComputer mcBlue(img);
  MomentsComputer mcWhite(img);


  int rows = img.rows;
  int cols = img.cols;
  if (img.isContinuous() && markerMask.isContinuous()) {
    cols *= rows;
    rows = 1;
  }
  for (int i=0; i<rows; ++i) {
    unsigned char *m = markerMask.ptr<unsigned char>(i);
    const cv::Vec3b *d = img.ptr<cv::Vec3b>(i);
    for (int j=0; j<cols; ++j) {
      if (*m == colorBigUC)
	mcWhite.processPix(*d);
      else if (*m == colorSmallUC)
	mcBlue.processPix(*d);
      ++m;
      ++d;
    }
  }
  
  mcBlue.getMeanStdDev(meanBlue, stdDevBlue);
  mcWhite.getMeanStdDev(meanWhite, stdDevWhite);

}


/*
  white is in RGB !

*/
cv::Mat
correctWhite(const cv::Mat &img, const cv::Vec3f &white)
{
  cv::Vec3f gamma;
  float log255 = logf(255.f);
  //RGB -> BGR
  gamma[0] = (white[2] > 0 ? log255/log(white[2]) : 1.f);
  gamma[1] = (white[1] > 0 ? log255/log(white[1]) : 1.f);
  gamma[2] = (white[0] > 0 ? log255/log(white[0]) : 1.f);

  cv::Mat imgC(img.rows, img.cols, CV_8UC3);

  int rows = img.rows;
  int cols = img.cols;

  if (img.isContinuous() && imgC.isContinuous()) {
    cols *= rows;
    rows = 1;
  }
  for (int i=0; i<rows; ++i) {
    const cv::Vec3b *src = img.ptr<cv::Vec3b>(i);
    cv::Vec3b *dst = imgC.ptr<cv::Vec3b>(i);
    for (int j=0; j<cols; ++j) {
      const cv::Vec3b &s = src[j];
      cv::Vec3b &d = dst[j];
      for (int k=0; k<3; ++k) {
	d[k] = cv::saturate_cast<unsigned char>(pow(s[k], gamma[k]));
      }
    }
  }
  
  return imgC;
}


float
computeMarkerSize(const cv::Mat &markerMask, float pixelSize)
{
  assert(markerMask.type() == CV_8UC1);

  int rows = markerMask.rows;
  int cols = markerMask.cols;
  if (markerMask.isContinuous()) {
    cols *= rows;
    rows = 1;
  }

  size_t area = 0;
  for (int i=0; i<rows; ++i) {
    const unsigned char *src = markerMask.ptr<unsigned char>(i);
    for (int j=0; j<cols; ++j) {
      if (src[j])
	++area;
    }
  }
  
  return area*pixelSize*pixelSize;
}
