#include "lab.hpp"

#include <cassert> 
#include <cmath> //cbrt


#define GAMMA

/*
//B

[1] 
"Image numérique couleur: de l'acquisition au traitement"
Alain Trémeau, Christine Fernandez-Maloigne, Pierre Bonton
Ed: Dunod

[2] 
"RGB coordinates of the Macbeth ColorChecker" Danny Pascale (2006)


( Xr Xg Xb )   ( R )   ( X )
( Yr Yg Yb ) * ( G ) = ( Y )
( Zr Zg Zb )   ( B )   ( Z )

Xw, Yw, Zw, Uw, Vw used in XYZ<->Luv/ab
correspond to: Xr, Yr, Zr, u'r, v'r in formulae at 
http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Lab.html
http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Luv.html
correspoding to : Xn, Yn, Zn in [2]
 Xn, Yn and Zn are the XYZ values of the reference white"


According to http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Luv.html
We have :
Uw = 4*Xw / (Xw + 15*Yw + 3*Zw)
Vw = 9*Yw / (Xw + 15*Yw + 3*Zw)

REM: it is not the same formulae in [1] p 51 : -3*Z instaed of +3*Z in denominator !! 
It is +3*Z everywhere else I have looked, for exemple here  http://en.wikipedia.org/wiki/CIELUV and even in [1] p100 !
Error in [1] ???


//For illuminant D65 :
======================

//B: the values given in OpenCV documentation
// http://docs.opencv.org/modules/imgproc/doc/miscellaneous_transformations.html#cvtcolor
// "CIE XYZ.Rec 709 with D65 white point" are more precise than in [2] ?
// In [2], we have the same values than OpenCV but rounded to 4 digits after comma in [2]
// p9, Table 6 "RGB (R709) to XYZ" cell
// The values at http://www.brucelindbloom.com/index.html?Eqn_RGB_XYZ_Matrix.html
// for sRGB D65 are slightly different
// We keep values of [2] (because we may also want to use the Bradford matrix from D50->D65 in [2])
//
// For the reference white, the values in OpenCV source 
// OpenCV-2.4.3/modules/imgproc/src/color.cpp in D65 array (times 100) are :
// 95.0456, 100.0, 108.8754
// The values in [2], p10, Table 7, are slightly different :
// 95.047, 100.0, 108.883
// http://www.brucelindbloom.com/index.html?Eqn_ChromAdapt.html gives the same values than [2]
// saying "come from ASTM E308-01"

//Uw & Vw are computed from Xw, Yw, Zw with previously given formulae.

*/

#define Xr 0.4125f
#define Yr 0.2127f
#define Zr 0.0193f

#define Xg 0.3576f 
#define Yg 0.7152f
#define Zg 0.1192f 

#define Xb 0.1804f
#define Yb 0.0722f
#define Zb 0.9503f 

#define Rx 3.2405
#define Gx -0.9693
#define Bx 0.0556

#define Ry -1.5371
#define Gy 1.8760
#define By -0.2040

#define Rz -0.4985
#define Gz 0.0416
#define Bz 1.0572

#define Xw 95.047f
#define Yw 100.0f
#define Zw 108.883f

#define Uw 0.19784f
#define Vw 0.468336f



//gamma corrections equations 
//found here : http://www.easyrgb.com/index.php?X=MATH&H=02#text2   in RGB->XYZ & XYZ->RGB
//They are also present in [2].
//In [2] it says for sRGB :
// offset=0.055, gamma=0.42, transition=0.003, slope=12.92
//
// ??? Is it the correct values ??? 1/0.42=2.380952 != 2.4   !!!???
// It is also 2.4 in OpenCV color.cpp

template <typename T>
static inline
float
gammaCorrected_to_linear(T R)
{
  //B:REM:Impl: using pow() & double is faster than using powf() & float on X86_64...
  
  float Rf = R / 255.f;
  if (Rf > 0.04045f)
    Rf = pow((( Rf + 0.055 ) / 1.055), 2.4);
  else
    Rf = Rf / 12.92;
  return Rf;
}

template <typename T>
void
RGBgammaCorrected_to_linear(T R, T G, T B, float &Rf, float &Gf, float &Bf)
{
  Rf = gammaCorrected_to_linear<T>(R);
  Gf = gammaCorrected_to_linear<T>(G);
  Bf = gammaCorrected_to_linear<T>(B);
}

void
RGBgammaCorrected_to_linear(unsigned char R, unsigned char G, unsigned char B, float &Rf, float &Gf, float &Bf)
{
  RGBgammaCorrected_to_linear<unsigned char>(R, G, B, Rf, Gf, Bf);
}

void
RGBgammaCorrected_to_linear(float R, float G, float B, float &Rf, float &Gf, float &Bf)
{
  RGBgammaCorrected_to_linear<float>(R, G, B, Rf, Gf, Bf);
}

static inline
unsigned char
linear_to_gammaCorrected(float Rf)
{
  if (Rf > 0.0031308)
    Rf = 1.055 * pow(Rf, 1/2.4) - 0.055;
  else 
    Rf *= 12.92;
  
  Rf *= 255.f;
    
  return CLAMP_TO_UCHAR(Rf);
}

void
RGBlinear_to_gammaCorrected(float Rf, float Gf, float Bf, 
			    unsigned char &R, unsigned char &G, unsigned char &B)
{
  R = linear_to_gammaCorrected(Rf);
  G = linear_to_gammaCorrected(Gf);
  B = linear_to_gammaCorrected(Bf);
}

//- faster versions with approximate pow function

//approximate versions from here :
//http://martin.ankerl.com/2012/01/25/optimized-approximative-pow-in-c-and-cpp/

//Warning: does not work with negative exponents !
// use pow(a, -b)= 1.0/pow(a, abs(b));
#if 0
static inline
double 
fastPrecisePow(double a, double b) { //fastPow(double a, double b) {
  union {
    double d;
    int x[2];
  } u = { a };
  u.x[1] = (int)(b * (u.x[1] - 1072632447) + 1072632447);
  u.x[0] = 0;
  return u.d;
}
#else
// should be much more precise with large b
static inline 
double
fastPrecisePow(double a, double b) {
  // calculate approximation with fraction of the exponent
  int e = (int) b;
  union {
    double d;
    int x[2];
  } u = { a };
  u.x[1] = (int)((b - e) * (u.x[1] - 1072632447) + 1072632447);
  u.x[0] = 0;
 
  // exponentiation by squaring with the exponent's integer part
  // double r = u.d makes everything much slower, not sure why
  double r = 1.0;
  while (e) {
    if (e & 1) {
      r *= a;
    }
    a *= a;
    e >>= 1;
  }
 
  return r * u.d;
}
#endif

void
RGBgammaCorrected_to_linear_fastP(unsigned char R, unsigned char G, unsigned char B, float &Rf, float &Gf, float &Bf)
{
  Rf = R / 255.f;
  Gf = G / 255.f;
  Bf = B / 255.f;

  if (Rf > 0.04045)
    Rf = fastPrecisePow((( Rf + 0.055 ) / 1.055), 2.4);
  else
    Rf = Rf / 12.92;
  if (Gf > 0.04045)
    Gf = fastPrecisePow((( Gf + 0.055 ) / 1.055), 2.4);
  else
    Gf = Gf / 12.92;
  if (Bf > 0.04045)
    Bf = fastPrecisePow((( Bf + 0.055 ) / 1.055), 2.4);
  else
    Bf = Bf / 12.92;
}

void
RGBlinear_to_gammaCorrected_fastP(float Rf, float Gf, float Bf, 
				  unsigned char &R, unsigned char &G, unsigned char &B)
{
  if (Rf > 0.0031308)
    Rf = 1.055 * fastPrecisePow(Rf, 1/2.4) - 0.055;
  else 
    Rf *= 12.92;
  if (Gf > 0.0031308)
    Gf = 1.055 * fastPrecisePow(Gf, 1/2.4) - 0.055;
  else 
    Gf *= 12.92;
  if (Bf > 0.0031308)
    Bf = 1.055 * fastPrecisePow(Bf, 1/2.4) - 0.055;
  else 
    Bf *= 12.92;

  Rf *= 255.f;
  Gf *= 255.f;
  Bf *= 255.f;
  
  R = CLAMP_TO_UCHAR(Rf);
  G = CLAMP_TO_UCHAR(Gf);
  B = CLAMP_TO_UCHAR(Bf);  
}






template <typename T>
static
void
P_RGB_to_XYZ(T R, T G, T B, float &X, float &Y, float &Z)
{
  
#ifdef GAMMA

  float Rf, Gf, Bf;
  RGBgammaCorrected_to_linear<T>(R, G, B, Rf, Gf, Bf);

  Rf *= 100.f;
  Gf *= 100.f;
  Bf *= 100.f;
  
  X = Xr*Rf + Xg*Gf + Xb*Bf;
  Y = Yr*Rf + Yg*Gf + Yb*Bf;
  Z = Zr*Rf + Zg*Gf + Zb*Bf;

#else

  X = Xr*R + Xg*G + Xb*B;
  Y = Yr*R + Yg*G + Yb*B;
  Z = Zr*R + Zg*G + Zb*B;

#endif

}

//OPTIMIZATion - code duplication with P_RGB_to_XYZ()
template <typename T>
static
void
P_RGB_to_Y(T R, T G, T B, float &Y)
{
  
#ifdef GAMMA

  float Rf, Gf, Bf;
  RGBgammaCorrected_to_linear<T>(R, G, B, Rf, Gf, Bf);

  Rf *= 100.f;
  Gf *= 100.f;
  Bf *= 100.f;
  
  Y = Yr*Rf + Yg*Gf + Yb*Bf;

#else

  Y = Yr*R + Yg*G + Yb*B;

#endif

}

static
void
P_XYZ_to_LAB(float X, float Y, float Z, float &L, float &a, float &b)
{
  
  // if ((X==0.0f)&&(Y==0.0f)&&(Z==0.0f)) {
  //   L = a = b = 0.0;
  //   return;
  // }
  
  const float XparXw = X/Xw;
  const float YparYw = Y/Yw;
  const float ZparZw = Z/Zw;
  
  float fx = 0.0f, fy = 0.0f, fz = 0.0f;
  // according to http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Lab.html & [1] p52
  const float Epsilon = 0.008856f; //static
  const float K = 903.3f; //static
  const float inv_116 = 1.0f / 116.0f; //static

  //B:impl: cbrtf seems slightly faster than cbrt 
  if (XparXw > Epsilon)
    fx = cbrtf(XparXw); 
  else
    fx = (K * XparXw + 16.0f) * inv_116;

  if (YparYw > Epsilon)
    fy = cbrtf(YparYw);
  else
    fy = (K * YparYw + 16.0f) * inv_116;

  if (ZparZw > Epsilon)
    fz = cbrtf(ZparZw);
  else
    fz = (K * ZparZw + 16.0f) * inv_116;
      
  L = 116.0f * fy  - 16.f;
  a = 500.f * (fx - fy);
  b = 200.f * (fy - fz);
}

//OPTIMIZATION: code duplication with P_XYZ_to_LAB()
static
void
P_Y_to_L(float Y, float &L)
{
  const float YparYw = Y/Yw;
  
  float fy = 0.0f;
  // according to http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Lab.html & [1] p52
  const float Epsilon = 0.008856f; //static
  const float K = 903.3f; //static
  const float inv_116 = 1.0f / 116.0f; //static

  //B:impl: cbrtf seems slightly faster than cbrt 
  if (YparYw > Epsilon)
    fy = cbrtf(YparYw);
  else
    fy = (K * YparYw + 16.0f) * inv_116;

  L = 116.0f * fy  - 16.f;
}




static
void
P_XYZ_to_LUV(float X, float Y, float Z, float &L, float &u, float &v)
{
  //http://www.easyrgb.com/index.php?X=MATH&H=16#text16
  //http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Luv.html & [1] p50/51

  if ((X==0.0f)&&(Y==0.0f)&&(Z==0.0f)) {
    L = u = v = 0.0;
    return;
  }
  
  float Lp = 0.0;
  
  const float YparYw = Y/Yw;
  if (YparYw > 0.008856f)
    Lp = 116.0f*cbrtf(YparYw) -16.0f; 
  else
    Lp = 903.3f*YparYw;
  
  L = Lp;
  
  const float up = 4.f*X/(X + 15.f*Y + 3.f*Z);
  u = 13.f*Lp*(up-Uw);
  
  const float vp = 9.f*Y/(X + 15.f*Y + 3.f*Z);
  v = 13.f*Lp*(vp-Vw);  
}

void
RGB_to_XYZ(float R, float G, float B, float &X, float &Y, float &Z)
{
  P_RGB_to_XYZ<float>(R, G, B, X, Y, Z);
}

void
RGB_to_XYZ(unsigned char R, unsigned char G, unsigned char B, float &X, float &Y, float &Z)
{
  P_RGB_to_XYZ<unsigned char>(R, G, B, X, Y, Z);
}

void
RGB_to_L(unsigned char R, unsigned char G, unsigned char B, float &L)
{
  float Y;
  P_RGB_to_Y<unsigned char>(R, G, B, Y);
  
  P_Y_to_L(Y, L);
}

template <typename T>
void
RGB_to_LAB(unsigned char R,
	   unsigned char G,
	   unsigned char B,
	   float &L,
	   float &a,
	   float &b)
{
  float X, Y, Z;
  P_RGB_to_XYZ<T>(R, G, B, X, Y, Z);
  
  P_XYZ_to_LAB(X, Y, Z, L, a, b);
}

void
RGB_to_LAB(unsigned char R,
	   unsigned char G,
	   unsigned char B,
	   float &L,
	   float &a,
	   float &b)
{
  RGB_to_LAB<unsigned char>(R, G, B, L, a, b);
}

void
RGB_to_LAB(float R,
	   float G,
	   float B,
	   float &L,
	   float &a,
	   float &b)
{
  RGB_to_LAB<float>(R, G, B, L, a, b);
}


template <typename T>
void
RGB_to_LUV(unsigned char R,
	   unsigned char G,
	   unsigned char B,
	   float &L,
	   float &u,
	   float &v)
{
  float X, Y, Z;
  P_RGB_to_XYZ<T>(R, G, B, X, Y, Z);
  
  P_XYZ_to_LUV(X, Y, Z, L, u, v);
}

void
RGB_to_LUV(unsigned char R,
	   unsigned char G,
	   unsigned char B,
	   float &L,
	   float &u,
	   float &v)
{
  RGB_to_LUV<unsigned char>(R, G, B, L, u, v);
}

void
RGB_to_LUV(float R,
	   float G,
	   float B,
	   float &L,
	   float &u,
	   float &v)
{
  RGB_to_LUV<float>(R, G, B, L, u, v);
}





template <typename T>
void
convertRGBtoLabD65(const T rgb[][3], float lab[][3], size_t size)
{
  for (size_t i=0; i<size; ++i) {
    RGB_to_LAB<T>(rgb[i][0], rgb[i][1], rgb[i][2],
		  lab[i][0], lab[i][1], lab[i][2]);
  }
}

void
convertRGBtoLabD65(const unsigned char rgb[][3], float lab[][3], size_t size)
{
  convertRGBtoLabD65<unsigned char>(rgb, lab, size);
}

void
convertRGBtoLabD65(const float rgb[][3], float lab[][3], size_t size)
{
  convertRGBtoLabD65<float>(rgb, lab, size);
}



static
void
P_LAB_to_XYZ(float L, float a, float b, float &X, float &Y, float &Z)
{
  const float fy = (L + 16.0f) / 116.0f;
  const float fx = a / 500.0f + fy;
  const float fz = fy - b / 200.0f;
  //TODO:OPTIM: replace "divide by constant" by "multiply by constant'=inv(constant)" !?
  
  const float fx_3 = fx*fx*fx;
  const float K = 903.3f;//static ?
  const float Epsilon = 0.008856f;//static ?
  const float inv_K = 1.0f / K;//static ?
  if (fx_3 > Epsilon)
    X = Xw * fx_3;
  else
    X = Xw * (116.0f * fx - 16.0f)*inv_K;
  
  const float KEpsilon = K * Epsilon; //static ?
  if (L > KEpsilon) {
    const float Lp = (L+16)/116.0f; //TODO:OPTIM: use inv_116 ?
    Y = Yw * (Lp*Lp*Lp);
  }
  else {
    Y = Yw * L * inv_K;
  }
  
  const float fz_3 = fz*fz*fz;
  if (fz_3 > Epsilon)
    Z = Zw * fz_3;
  else
    Z = Zw * (116.0f * fz - 16.0f)*inv_K;
}


static
void
P_XYZ_to_RGB(float X, float Y, float Z, unsigned char &R, unsigned char &G, unsigned char &B)
{
#ifdef GAMMA

  X *= 0.01f;
  Y *= 0.01f;
  Z *= 0.01f;

  float Rf = Rx*X + Ry*Y + Rz*Z;
  float Gf = Gx*X + Gy*Y + Gz*Z;
  float Bf = Bx*X + By*Y + Bz*Z;
  
  RGBlinear_to_gammaCorrected(Rf, Gf, Bf, R, G, B);

#else
  const float Rf = Rx*X + Ry*Y + Rz*Z + 0.5f;
  const float Gf = Gx*X + Gy*Y + Gz*Z + 0.5f;
  const float Bf = Bx*X + By*Y + Bz*Z + 0.5f;
  R = CLAMP_TO_UCHAR(Rf);
  G = CLAMP_TO_UCHAR(Gf);
  B = CLAMP_TO_UCHAR(Bf);
#endif
}


void
LAB_to_RGB(float L, float a, float b,
	   unsigned char &R, unsigned char &G, unsigned char &B)
{
  float X, Y, Z;
  P_LAB_to_XYZ(L, a, b, X, Y, Z);

  P_XYZ_to_RGB(X, Y, Z, R, G, B);
}


void
convertLabD65toRGB(const float lab[][3], unsigned char rgb[][3], size_t size)
{
  for (size_t i=0; i<size; ++i) {

    LAB_to_RGB(lab[i][0], lab[i][1], lab[i][2],
	       rgb[i][0], rgb[i][1], rgb[i][2]);
    
  }

}




//values from [2], p10, Table 7
#define Xw_D50 96.422f
#define Yw_D50 100.0f
#define Zw_D50 82.521f


static
void
P_LabD50_to_XYZD50(float L, float a, float b, float &X, float &Y, float &Z)
{
  //formula from http://www.brucelindbloom.com/index.html?Eqn_Lab_to_XYZ.html

  const float fy = (L + 16.0f) / 116.0f;
  const float fx = a / 500.0f + fy;
  const float fz = fy - b / 200.0f;
  
  const float fx_3 = fx*fx*fx;
  const float K = 903.3f;//static ?
  const float Epsilon = 0.008856f;//static ?
  const float inv_K = 1.0f / K;//static ?
  if (fx_3 > Epsilon)
    X = Xw_D50 * fx_3;
  else
    X = Xw_D50 * (116.0f * fx - 16.0f)*inv_K;

  const float KEpsilon = K * Epsilon; //static ?
  if (L > KEpsilon) {
    const float Lp = (L+16)/116.0f; //TODO:OPTIM: use inv_116 ?
    Y = Yw_D50 * (Lp*Lp*Lp);
  }
  else {
    Y = Yw_D50 * L * inv_K;
  }

  const float fz_3 = fz*fz*fz;
  if (fz_3 > Epsilon)
    Z = Zw_D50 * fz_3;
  else
    Z = Zw_D50 * (116.0f * fz - 16.0f)*inv_K;
 
}

static
void
P_XYZD50_to_XYZD65(float X1, float Y1, float Z1, float &X2, float &Y2, float &Z2)
{
  // [2] p10 Table 7 : Bradford matrix from D50 to D65
  
  X2 =  0.9556*X1 + -0.0230*Y1 + 0.0632*Z1;
  Y2 = -0.0283*X1 +  1.0099*Y1 + 0.0210*Z1;
  Z2 =  0.0123*X1 + -0.0205*Y1 + 1.3299*Z1;
}


void
convertLabD50toLabD65(const float labD50[][3], float labD65[][3], size_t size)
{
  //Bradford, or BFD for short, chromatic adaptation transform
  //[2] gives the formula (6) to transform between any given pair of illuminant
  //(values of (4) & (5) are also available here http://www.brucelindbloom.com/index.html?Eqn_ChromAdapt.html

  for (size_t i=0; i<size; ++i) {
  
    float X_D50, Y_D50, Z_D50;
    P_LabD50_to_XYZD50(labD50[i][0], labD50[i][1], labD50[i][2], X_D50, Y_D50, Z_D50);
    float X_D65, Y_D65, Z_D65;    
    P_XYZD50_to_XYZD65(X_D50, Y_D50, Z_D50, X_D65, Y_D65, Z_D65);
    
    P_XYZ_to_LAB(X_D65, Y_D65, Z_D65, labD65[i][0], labD65[i][1], labD65[i][2]);
  }
  
}
