#include "insideWoundClassification.hpp"

#include <opencv2/ml/ml.hpp>
#include <girl/girlToger.hpp>
#include <girlTools/girlToolsRegionInfos.hpp>
#include <girlTools/girlToolsPainter.hpp>
#include <ise/iseSegmentationClustering.hpp>
#include <ise/iseSegmentationContext.hpp>
#include <ise/iseSplitAlgorithmDefault.hpp>
#include <ise/iseSplitDriverDefault.hpp>
#include <ise/iseSegmentationClusteringGonogo1.hpp>

#include "utilsSeg.hpp"
#include "utilsClassifier.hpp"
#include "defineUrgoEye.hpp"

#if DEBUG
#include <opencv2/highgui/highgui.hpp>
#endif //DEBUG

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
typedef CvSVM *IWClassifier;
#else
typedef cv::Ptr<cv::ml::SVM> IWClassifier;
#endif

static
cv::Vec3b
getColorFromProba(float p)
{
  assert(p>=0 && p<=1);

  const unsigned char r = 255;
  const unsigned char g = cv::saturate_cast<unsigned char>(p * 255);
  const unsigned char b = 0;

  return cv::Vec3b(b, g, r);  //BGR !!!
}

namespace {

  class QuantificationThresholdsIMGMask : public ise::SegmentationClustering
  {
  public:
  
    QuantificationThresholdsIMGMask(ise::SegmentationContext *segContext, const cv::Mat &imgCS, const cv::Mat &mask)
      : ise::SegmentationClustering(segContext),
	m_imgCS(imgCS),
	m_mask(mask)
    {
      assert(m_imgCS.type() == CV_32FC3);
      assert(m_mask.type() == CV_8UC1);
    }

    void setParams(const std::vector< std::vector<float> > &thresholds)
    {
      assert(thresholds.size() <= 3);
      m_thresholds = thresholds;
      //std::cerr<<"(thresholds[0])[0]="<<(m_thresholds[0])[0]<<" (thresholds[2])[1]="<<(m_thresholds[2])[1]<<"\n";
    }
  
    virtual void init(const girl::regionDeque  &/* regions */, int /* level */) {} 
  
    virtual void init(girl::region /* r */, int /* level */) {}

    virtual bool same(const girl::point &p, girl::direction dir)
    {
      girl::point p2 = p;
      p2.move(dir);

   
      const unsigned char &c1 = m_mask.at<unsigned char>(p.y(), p.x());
      const unsigned char &c2 = m_mask.at<unsigned char>(p2.y(), p2.x());
      if (c1 != c2) {
	return false;
      }
      if (c1 == c2 && c1 == 1) {
	return true;
      }
   
      const cv::Vec3f &pix1 = m_imgCS.at<cv::Vec3f>(p.y(), p.x());
      const cv::Vec3f &pix2 = m_imgCS.at<cv::Vec3f>(p2.y(), p2.x());
    
      assert(m_thresholds.size() <= 3);
      for (size_t k=0; k<m_thresholds.size(); ++k) 
	{
	  const std::vector<float> &thresholds = m_thresholds[k];
	  if (! thresholds.empty()) {
	
	    const int v1 = pix1[k];
	    const int v2 = pix2[k];

	    //	  std::cerr<<"v1"<< v1 << "v2" << v2 << "\n";

	    int level1, level2;
	    level1 = level2 = std::numeric_limits<int>::min();

	    for (size_t i=0; i<thresholds.size(); ++i)
	      {
		if (v1 > thresholds[i])
		  level1 = thresholds[i];
		if (v2 > thresholds[i])
		  level2 = thresholds[i];
	      }
	    if (level1 != level2)
	      return false;
	  }
	}
      return true;
    }

  protected:
    const cv::Mat &m_imgCS;
    const cv::Mat &m_mask;
    std::vector<std::vector<float> > m_thresholds;
  };


  class RegionMomentsComputer : public girl::toger::pointIterator
  {
  public:

    explicit RegionMomentsComputer(cv::Mat &img)
      : m_img(img),
	m_m0(0), m_m1(0,0,0), m_m2(0,0,0)
    {
      assert(img.type() == CV_32FC3);
    }

    virtual void processPoint(const girl::point &pt)
    { 
      ++m_m0;
      const cv::Vec3f &pix = m_img.at<cv::Vec3f>(pt.y(), pt.x());
      for (int k=0; k<3; ++k) {
	m_m1[k] += pix[k];
	m_m2[k] += pix[k]*pix[k];
      }
    }

  public:
    cv::Mat &m_img;
    size_t m_m0;
    cv::Vec3f m_m1;
    cv::Vec3f m_m2;
  };


  class PainterProbaMean : public girl::toger::pointIterator
  {
  public:

    explicit PainterProbaMean(cv::Mat &outImg)
      : m_outImg(outImg)
    {
      assert(outImg.type() == CV_8UC3);
    }

    virtual void processPoint(const girl::point &pt)
    { 
      m_outImg.at<cv::Vec3b>(pt.y(), pt.x()) = m_color;
    }

    void setColor(const cv::Vec3b &c)
    {
      m_color = c;
    }

    void setProba(float p)
    {
      m_color = getColorFromProba(p);
    }

  protected:
    cv::Mat &m_outImg;
    cv::Vec3b m_color;
  };

} //end anonymous namespace 


static 
void
splitNecrose(ise::SegmentationContext &segContext, 
	     const cv::Mat &imgCS, 
	     const cv::Mat &markerMask,
	     const girl::regionDeque &regs,
	     girl::regionDeque &nonNecroseRegions, girl::regionDeque &necroseRegions)
{



  assert(imgCS.type() == CV_32FC3);
  assert(markerMask.type() == CV_8UC1);
  
  girl::toger &tog = *(segContext.getToger());

  assert(imgCS.rows == tog.height() && imgCS.cols == tog.width());
  
  ise::SplitAlgorithmDefault splitAlgo(&segContext);
  ise::SplitDriverDefault    splitDriver(&segContext);
  QuantificationThresholdsIMGMask segClust1(&segContext, imgCS, markerMask);
  std::vector< std::vector<float> > thresholds(3);
  std::vector<float> &thresholdsL = thresholds[0];
  thresholdsL.push_back(28);
  std::vector<float> &thresholdsA = thresholds[1];
  thresholdsA.push_back(-10);
  thresholdsA.push_back(25);
  std::vector<float> &thresholdsB = thresholds[2];
  thresholdsB.push_back(-10);
  thresholdsB.push_back(25);

  segClust1.setParams(thresholds);
  girl::regionDeque createdRegions1; 
  splitAlgo.apply(regs, splitDriver, segClust1, 0, createdRegions1);
  
  for (girl::regionDeque::const_iterator it = createdRegions1.begin();
       it != createdRegions1.end();
       ++it) {
    const girl::region r = *it;
    
    girl::point pt = tog.anyPoint(r);
    const cv::Vec3f &pix = imgCS.at<cv::Vec3f>(pt.y(), pt.x());
    if (pix[0] < thresholdsL[0]
	&& pix[1] >= thresholdsA[0] && pix[1] <= thresholdsA[1]
	&& pix[2] >= thresholdsB[0] && pix[2] <= thresholdsB[1]) {
      necroseRegions.push_back(r);
    }
    else {
      nonNecroseRegions.push_back(r);
    }
  }
  
}


//#define IW_BOOST_PROBA 1

#ifdef IW_BOOST_PROBA
static
float 
disc(float p)
{
  #if 0
  //DISC3
  /**/
  if (p < 0.3) {
    p = 0;
  }
  else if (p < 0.4) {
    p = 0.3;
  }
  else if (p < 0.6) {
    p = 0.5;
  }
  else if (p < 0.8) {
    p = 0.8;
  }
  else if (p < 0.9) {
    p = 1;
  }
  else {
    p = 1;
  }
  /**/
#else
  if (p < 0.5) {
    p = 0;
  }
  else {
    p = 1;
  }
#endif //0

  return p;
}
#endif //IW_BOOST_PROBA

static
void
detectInsideWound(const std::string &iwModelFilename,
		  const cv::Mat &inputImage,
		  const cv::Mat &inputGTImage,
		  const cv::Mat &inputImageOrig,
		  const std::string &iwProbaSigmoidFilename,
		  ColorSpace cs,
		  int /*patchSize*/, PatchType patchType,
		  int /*smallSize*/,
		  int nbClasses,
		  bool useL, 
		  const Normalizer &n,
		  std::vector<float> &percentages,
		  cv::Mat &outputImage)
{
  
  assert(inputImage.rows == inputGTImage.rows && inputImage.cols == inputGTImage.cols);
  
  assert(patchType == eMEAN_STDDEV); //Only this type is handled for Now !!!
  

  const cv::Mat &img = inputImage;
  
#ifdef TIMING
  double t = (double)cv::getTickCount();
#endif

  //- create toger and select wound regions 

  girl::toger tog(img.cols, img.rows);

  cv::Mat mask = inputGTImage;
  assert(mask.type() == CV_8UC1);

  girl::regionDeque regs;
  girl::tools::getTogerFromImageLabel(mask, tog, regs);
  assert(regs.size() == tog.numRegions());

  girl::regionDeque regionsWound;
  
  for (girl::regionDeque::const_iterator it = regs.begin();
       it != regs.end();
       ++it) {

    const girl::point pt = tog.anyPoint(*it);
    assert(pt.x() < mask.cols && pt.y() < mask.rows);
    const unsigned char m = mask.at<unsigned char>(pt.y(), pt.x());
    if (m) {
      //region is in mask -> region correspond to wound

      regionsWound.push_back(*it);
    }
  }
  assert(regionsWound.size() <= regs.size());


  
  girl::tools::Image gtImg;
  BGR2girlToolsImage(img, gtImg);
  ise::SegmentationContext segContext(&tog, &gtImg);

  //split for necrose
  
  cv::Mat imgCS;
  convert(img, eLAB, imgCS);
  cv::Mat markerMask = cv::Mat::zeros(img.rows, img.cols, CV_8UC1);
  girl::regionDeque nonNecroseRegions, necroseRegions;
  splitNecrose(segContext, imgCS, markerMask, regionsWound, nonNecroseRegions, necroseRegions);
  
  if (cs != eLAB)
    convert(img, cs, imgCS);
  assert(imgCS.type() == CV_32FC3);



  //- split with gonogo


  ise::SplitAlgorithmDefault splitAlgo(&segContext);
  ise::SplitDriverDefault    splitDriver(&segContext);
#ifdef VERBOSE
  const float regionAreaPercent = 0.002; //0.008;
  const int regionArea = static_cast<int>(regionAreaPercent * tog.width()*tog.height());
  std::cerr<<" regionArea="<<regionAreaPercent<<"*"<<tog.width()*tog.height()<<"="<<regionArea<<" for image w="<<tog.width()<<" h="<<tog.height()<<"\n";
#endif  

    

  
#ifdef VERBOSE
  std::cerr<<"GONOGO1  nbClasses="<<nbClasses<<"\n";
#endif  

#if 1
  //- with Gonogo1
  ise::SegmentationClusteringGonogo1 g(&segContext);
  const int nbClasses1 = 26; //12;
  const float regionAreaPercent1 = 0.000001;
  const int regionArea1 = static_cast<int>(regionAreaPercent1 * img.rows*img.cols);
  //std::cerr<<" regionArea="<<regionArea<<" for image w="<<img.cols<<" h="<<img.rows<<"\n";
  const int flashLuminanceThreshold1 = 80;
  g.setParameters(ISE_RGB_to_Lab, nbClasses1, regionArea1, flashLuminanceThreshold1);
#else
  //-with NbColors
  const int nbColors = 8; //26; //12;
  ise::QuantificationNbColors g(&segContext, ISE_RGB_to_Lab, nbColors);

#endif //0
    

  girl::regionDeque createdRegions; 

  ise::SegmentationClustering &segClust = g;

  splitAlgo.apply(nonNecroseRegions, splitDriver, segClust, 0, createdRegions);



#if 0 //def VERBOSE
  std::cerr<<"wound regs="<<regionsWound.size()<<" necroseRegions="<<necroseRegions.size()<<" nonNecroseRegions="<<nonNecroseRegions.size()<<" splitted in "<<createdRegions.size()<<" regions\n";
#endif  


  
  const int numRegions = createdRegions.size();

  assert((size_t)numRegions <= tog.numRegions());
  assert(numRegions > 0);


  //- gather data of regions

  int dim = (useL ? 3 : 2);
  int start = 3 - dim;

  int cols = getLABsSize(1, patchType, useL); //patchSize==1

  cv::Mat testData(createdRegions.size(), cols, CV_32F);


  std::vector<size_t> m0s(numRegions);
  size_t area = 0;

  int i=0;


  for (girl::regionDeque::const_iterator it = createdRegions.begin();
       it != createdRegions.end();
       ++it, ++i) {

    RegionMomentsComputer rmc(imgCS);
    tog.iterateOnRegionDomain(*it, rmc);

    m0s[i] = rmc.m_m0;
    area += rmc.m_m0;

    float *d = testData.ptr<float>(i);

    //some CODE CUPLICATION with utilsConcepts.hpp : getLABMts()
    if (patchType == eMTS) {
      if (useL) {
	d[0] = rmc.m_m0;
	d[1] = rmc.m_m1[0]; d[2] = rmc.m_m1[1]; d[3] = rmc.m_m1[2];
	d[4] = rmc.m_m2[0]; d[5] = rmc.m_m2[1]; d[6] = rmc.m_m2[2];
      }
      else {
	d[0] = rmc.m_m0;
	d[1] = rmc.m_m1[1]; d[2] = rmc.m_m1[2];
	d[3] = rmc.m_m2[1]; d[4] = rmc.m_m2[2]; 
      }
    }
    else if (patchType == eMEAN_VAR || patchType == eMEAN_STDDEV) {
      assert(rmc.m_m0 > 0);
      const float inv_m0 = 1.f / rmc.m_m0;
      if (useL) {
	d[0] = rmc.m_m1[0]*inv_m0;
	d[1] = rmc.m_m1[1]*inv_m0;
	d[2] = rmc.m_m1[2]*inv_m0;
	d[3] = std::max((rmc.m_m2[0] - rmc.m_m1[0]*rmc.m_m1[0]*inv_m0)*inv_m0, 0.f);
	d[4] = std::max((rmc.m_m2[1] - rmc.m_m1[1]*rmc.m_m1[1]*inv_m0)*inv_m0, 0.f);
	d[5] = std::max((rmc.m_m2[2] - rmc.m_m1[2]*rmc.m_m1[2]*inv_m0)*inv_m0, 0.f); 
      }
      else {
	d[0] = rmc.m_m1[1]*inv_m0;
	d[1] = rmc.m_m1[2]*inv_m0;
	d[2] = std::max((rmc.m_m2[1] - rmc.m_m1[1]*rmc.m_m1[1]*inv_m0)*inv_m0, 0.f);
	d[3] = std::max((rmc.m_m2[2] - rmc.m_m1[2]*rmc.m_m1[2]*inv_m0)*inv_m0, 0.f); 
      }
      if (patchType == eMEAN_STDDEV) {
	if (useL) {
	  d[3] = sqrt(d[3]);
	  d[4] = sqrt(d[4]);
	  d[5] = sqrt(d[5]);
	}
	else {
	  d[2] = sqrt(d[2]);
	  d[3] = sqrt(d[3]);
	}
      }
    }
    else if (patchType == eMEAN || patchType == eLAB_VALUES || patchType == eLAB_VALUES_UNSORTED || patchType == eLA_SORTED) {
      const float inv_m0 = 1.f / rmc.m_m0;
      if (useL) {
	d[0] = rmc.m_m1[0]*inv_m0;
	d[1] = rmc.m_m1[1]*inv_m0;
	d[2] = rmc.m_m1[2]*inv_m0;
      }
      else {
	d[0] = rmc.m_m1[1]*inv_m0;
	d[1] = rmc.m_m1[2]*inv_m0;
      }
      if (patchType == eLAB_VALUES || patchType == eLAB_VALUES_UNSORTED || patchType == eLA_SORTED) {
	std::cerr<<"WARNING: insideWoundCalssification: patchType=eLAB_VALUES|eLAB_VALUES_UNSORTED|eLA_SORTED : will use mean LAB of region\n";
      }
    }
    else {
      std::cerr<<"ERROR: insideWoundCalssification: unhandled patchType\n";
      assert(false);
      exit(-1);
    }
    
  }


  size_t areaNecrose = 0;
  for (girl::regionDeque::const_iterator it = necroseRegions.begin();
       it != necroseRegions.end();
       ++it, ++i) {
    areaNecrose += girl::tools::computeArea(tog, *it);
  }
  area += areaNecrose;
  //std::cerr<<"areaNecrose="<<areaNecrose<<" area="<<area<<"\n";

  //- apply classifier
  
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
  CvSVM clas;
  clas.load(iwModelFilename.c_str());
  IWClassifier classifier = &clas;
#else
  IWClassifier classifier = cv::ml::StatModel::load<cv::ml::SVM>(iwModelFilename.c_str());
#endif


  double sigmoid_A=0, sigmoid_B=0;
  if (! iwProbaSigmoidFilename.empty())
    readProbaSigmoidFile(iwProbaSigmoidFilename, sigmoid_A, sigmoid_B);

  n.normalizeMat(testData); //do nothing if iwNormalizeFilename was empty


  cv::Mat res;
  const bool returnDFVal = true;
  const bool normalize = true;
  classifyData(classifier, testData, res, returnDFVal, normalize, sigmoid_A, sigmoid_B);
  //warning: if (sigmoid_A==sigmoid_B==0) : We do not have a proba : we have a distance !!!

  assert(res.rows == testData.rows);
  assert(res.rows == numRegions);

  //std::cerr<<"sigmoid_A="<<sigmoid_A<<" sigmoid_B="<<sigmoid_B<<" area="<<area<<" testData r="<<testData.rows<<" c="<<testData.cols<<"\n";//DEBUG



#if DEBUG
  {
    std::cerr<<"returnDFVal="<<returnDFVal<<" sigmoid A="<<sigmoid_A<<" B="<<sigmoid_B<<"\n";
    
    cv::Mat probasImage = inputImageOrig.clone();

    //paint fibrine/bourgeon according to proba
    PainterProbaMean pr(probasImage);
    const float *p = res.ptr<float>(0);
    int i=0;
    for (girl::regionDeque::const_iterator it = createdRegions.begin();
	 it != createdRegions.end();
	 ++it, ++i) {
      assert(i < numRegions);
      //const float proba = *p;
      const unsigned char g = cv::saturate_cast<unsigned char>(*p * 255);
      cv::Vec3b color(g, g, g);  //BGR !!!
      pr.setColor(color);
      tog.iterateOnRegionDomain(*it, pr);

      const size_t area = girl::tools::computeArea(tog, *it);
      if (area > 32) {
	std::cerr<<i<<") region "<<*it<<" area="<<area<<" :  proba="<<*p;
#ifdef IW_BOOST_PROBA
	std::cerr<<" disc(p)="<<disc(*p);
#endif //IW_BOOST_PROBA
	std::cerr<<"  color="<<(int)g<<"\n";

	std::cerr<<"  data: ";
	const float *p = testData.ptr<float>(i);
	for (int i=0; i<testData.cols; ++i)
	  std::cerr<<p[i]<<" ";
	std::cerr<<"\n";
      }
      
      ++p;
    }
    //paint necrose regions
    const cv::Scalar COLOR_NECROSE(0, 0, 0); //BGR
    girl::tools::Painter<cv::Vec3b> ptr(probasImage, COLOR_NECROSE);
    for (girl::regionDeque::const_iterator it = necroseRegions.begin();
	 it != necroseRegions.end();
	 ++it) {
      std::cerr<<"region "<<*it<<" is necrose\n";
      tog.iterateOnRegionDomain(*it, ptr);
    }
  

    cv::imshow("probas", probasImage);
    cv::waitKey(0);
  }
#endif //DEBUG

  
  percentages.resize(IW_PERCENT_SIZE);

  float proportionFibrine = 0.0f;

  const float inv_area = 1.f / area;
  const float *p = res.ptr<float>(0);
  for (int i=0; i<numRegions; ++i) {
    float proba = *p;

    //std::cerr<<"p="<<proba<<" ";
    
#ifdef IW_BOOST_PROBA
    //Boost
    proba = disc(proba);
#endif //IW_BOOST_PROBA
    
    proportionFibrine += proba * m0s[i] * inv_area;

    ++p;
  }


#ifdef TIMING
  t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
  std::cout<<"***Time for processImage: "<<t<<"s"<<std::endl;
#endif //TIMING
  std::cerr<<"proportionFibrine="<<proportionFibrine<<"\n";//DEBUG

  const float percentNecrose = areaNecrose/(float)area*100.f;
  //const float percentFibrine = static_cast<int>(proportionFibrine*(100.f-percentNecrose) + 0.5f);
  const float percentFibrine = proportionFibrine*(100.f-percentNecrose);
  const float percentBourgeon = 100.f-percentNecrose-percentFibrine;

  percentages[IW_PERCENT_FIBRINE] = percentFibrine;
  percentages[IW_PERCENT_BOURGEON] = percentBourgeon;
  percentages[IW_PERCENT_NECROSE] = percentNecrose;
  

  //outputImage = cv::Mat::zeros(img.rows, img.cols, CV_8UC3);
  //paint on original image
  outputImage = inputImageOrig.clone();
  {
    //paint fibrine/bourgeon according to proba
    PainterProbaMean pr(outputImage);
    const float *p = res.ptr<float>(0);
    int i=0;
    for (girl::regionDeque::const_iterator it = createdRegions.begin();
	 it != createdRegions.end();
	 ++it, ++i) {
      float proba = *p;
#ifdef IW_BOOST_PROBA
      //Boost
      proba = disc(proba);
#endif //IW_BOOST_PROBA
      
      pr.setProba(proba);
      tog.iterateOnRegionDomain(*it, pr);
      
      ++p;
    }
    //paint necrose regions
    static const cv::Scalar COLOR_NECROSE(0, 0, 0); //BGR
    girl::tools::Painter<cv::Vec3b> ptr(outputImage, COLOR_NECROSE);
    for (girl::regionDeque::const_iterator it = necroseRegions.begin();
	 it != necroseRegions.end();
	 ++it) {
      tog.iterateOnRegionDomain(*it, ptr);
    }
  } 
  
}

//NEW VERSION


namespace { //anonymous namespace 

  class PatchRegionMomentsComputer : public girl::toger::pointIterator
  {
  public:

    explicit PatchRegionMomentsComputer(cv::Mat &img, int patchSize, PatchType patchType, float lSaturatedThreshold, bool useL, cv::Mat &data)
      : m_img(img),
	m_patchSize(patchSize), m_patchType(patchType), m_lSaturatedThreshold(lSaturatedThreshold), m_useL(useL),
	m_data(data),
	m_row(0)
    {
      assert(img.type() == CV_32FC3);
    }

    virtual void processPoint(const girl::point &pt)
    {
      assert(m_row < m_data.rows);
      cv::Mat r = m_data.row(m_row);
      getLABMts(pt.x(), pt.y(), m_img, m_patchSize, m_patchType, m_lSaturatedThreshold, m_useL, r);
      ++m_row;
    }

  public:
    cv::Mat &m_img;
    int m_patchSize;
    PatchType m_patchType;
    float m_lSaturatedThreshold;
    bool m_useL;
    cv::Mat &m_data;
    int m_row; //current row
  };

  class ProbaPixelPainter : public girl::toger::pointIterator
  {
  public:

    explicit ProbaPixelPainter(cv::Mat &img, const cv::Mat &proba)
      : m_img(img),
	m_proba(proba),
	m_row(0)
    {
      assert(m_img.type() == CV_8UC3);
      assert(m_proba.type() == CV_32FC1);
      assert(m_proba.cols == 1);
    }
    
    virtual void processPoint(const girl::point &pt)
    {
      assert(m_row < m_proba.rows);
      const float proba = m_proba.at<float>(m_row, 0);
      const unsigned char g = cv::saturate_cast<unsigned char>(proba * 255);
      cv::Vec3b color(g, g, g); //BGR
      assert(pt.y() < m_img.rows && pt.x() < m_img.cols);
      m_img.at<cv::Vec3b>(pt.y(), pt.x()) = color;

      ++m_row;
    }

    cv::Mat &m_img;
    const cv::Mat &m_proba;
    int m_row;
  };

  class FibrineProbaPixelPainter : public girl::toger::pointIterator
  {
  public:

    explicit FibrineProbaPixelPainter(cv::Mat &img, const cv::Mat &proba)
      : m_img(img),
	m_proba(proba),
	m_row(0)
    {
      assert(m_img.type() == CV_8UC3);
      assert(m_proba.type() == CV_32FC1);
      assert(m_proba.cols == 1);
    }
    
    virtual void processPoint(const girl::point &pt)
    {
      assert(m_row < m_proba.rows);
      float proba = m_proba.at<float>(m_row, 0);
#ifdef IW_BOOST_PROBA
    //Boost
    proba = disc(proba);
#endif //IW_BOOST_PROBA
      cv::Vec3b color = getColorFromProba(proba);
      assert(pt.y() < m_img.rows && pt.x() < m_img.cols);
      m_img.at<cv::Vec3b>(pt.y(), pt.x()) = color;

      ++m_row;
    }

    cv::Mat &m_img;
    const cv::Mat &m_proba;
    int m_row;
  };

  
} //end anonymous namespace 


static
void
detectInsideWound_NEW(const std::string &iwModelFilename,
		      const cv::Mat &inputImage,
		      const cv::Mat &inputGTImage,
		      const cv::Mat &inputImageOrig,
		      const std::string &iwProbaSigmoidFilename,
		      ColorSpace cs,
		      int patchSize, PatchType patchType,
		      int /*smallSize*/,
		      int nbClasses,
		      bool useL, 
		      const Normalizer &n,
		      std::vector<float> &percentages,
		      cv::Mat &outputImage)
{
  
  assert(inputImage.rows == inputGTImage.rows && inputImage.cols == inputGTImage.cols);
  
  // assert(patchType == eMEAN_STDDEV); //Only this type is handled for Now !!!
  

  const cv::Mat &img = inputImage;
  
#ifdef TIMING
  double t = (double)cv::getTickCount();
#endif

  //- create toger and select wound regions 

  girl::toger tog(img.cols, img.rows);

  cv::Mat mask = inputGTImage;
  assert(mask.type() == CV_8UC1);

  girl::regionDeque regs;
  girl::tools::getTogerFromImageLabel(mask, tog, regs);
  assert(regs.size() == tog.numRegions());

  girl::regionDeque regionsWound;
  
  for (girl::regionDeque::const_iterator it = regs.begin();
       it != regs.end();
       ++it) {

    const girl::point pt = tog.anyPoint(*it);
    assert(pt.x() < mask.cols && pt.y() < mask.rows);
    const unsigned char m = mask.at<unsigned char>(pt.y(), pt.x());
    if (m) {
      //region is in mask -> region correspond to wound

      regionsWound.push_back(*it);
    }
  }
  assert(regionsWound.size() <= regs.size());


  
  girl::tools::Image gtImg;
  BGR2girlToolsImage(img, gtImg);
  ise::SegmentationContext segContext(&tog, &gtImg);

  //split for necrose
  
  cv::Mat imgCS;
  convert(img, eLAB, imgCS);
  cv::Mat markerMask = cv::Mat::zeros(img.rows, img.cols, CV_8UC1);
  girl::regionDeque nonNecroseRegions, necroseRegions;
  splitNecrose(segContext, imgCS, markerMask, regionsWound, nonNecroseRegions, necroseRegions);
  
  if (cs != eLAB)
    convert(img, cs, imgCS);
  assert(imgCS.type() == CV_32FC3);


  //process nonNecroseRegions
  
  const int numRegions = nonNecroseRegions.size();
  assert((size_t)numRegions <= tog.numRegions());
  assert(numRegions > 0);

  //- gather data of regions
  size_t totalAreaNonNecrose = 0;
  for (girl::regionDeque::const_iterator it = nonNecroseRegions.begin();
       it != nonNecroseRegions.end();
       ++it) {
    const size_t area = girl::tools::computeArea(tog, *it);
    totalAreaNonNecrose += area;
  }

  int cols = getLABsSize(patchSize, patchType, useL);
  cv::Mat testData(totalAreaNonNecrose, cols, CV_32F);

  float lSaturatedThreshold = 100; //TODO
  
  PatchRegionMomentsComputer rmc(imgCS, patchSize, patchType, lSaturatedThreshold, useL, testData);
  
  int i=0;
  for (girl::regionDeque::const_iterator it = nonNecroseRegions.begin();
       it != nonNecroseRegions.end();
       ++it, ++i) {

    tog.iterateOnRegionDomain(*it, rmc);

  }

  assert(rmc.m_row == totalAreaNonNecrose);

  size_t area = totalAreaNonNecrose;
  size_t areaNecrose = 0;
  for (girl::regionDeque::const_iterator it = necroseRegions.begin();
       it != necroseRegions.end();
       ++it, ++i) {
    areaNecrose += girl::tools::computeArea(tog, *it);
  }
  area += areaNecrose;
  //std::cerr<<"areaNecrose="<<areaNecrose<<" area="<<area<<"\n";

  
  //- apply classifier
  
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
  CvSVM clas;
  clas.load(iwModelFilename.c_str());
  IWClassifier classifier = &clas;
#else
  IWClassifier classifier = cv::ml::StatModel::load<cv::ml::SVM>(iwModelFilename.c_str());
#endif

  double sigmoid_A=0, sigmoid_B=0;
  if (! iwProbaSigmoidFilename.empty())
    readProbaSigmoidFile(iwProbaSigmoidFilename, sigmoid_A, sigmoid_B);

  n.normalizeMat(testData); //do nothing if iwNormalizeFilename was empty

  cv::Mat res; //probas
  const bool returnDFVal = true;
  const bool normalize = true;
  classifyData(classifier, testData, res, returnDFVal, normalize, sigmoid_A, sigmoid_B);
  //warning: if (sigmoid_A==sigmoid_B==0) : We do not have a proba : we have a distance !!!

  assert(res.rows == testData.rows);
  assert(res.rows == numRegions);

  //std::cerr<<"sigmoid_A="<<sigmoid_A<<" sigmoid_B="<<sigmoid_B<<" area="<<area<<" testData r="<<testData.rows<<" c="<<testData.cols<<"\n";//DEBUG

#if DEBUG
  {
    std::cerr<<"returnDFVal="<<returnDFVal<<" sigmoid A="<<sigmoid_A<<" B="<<sigmoid_B<<"\n";
    
    cv::Mat probasImage = inputImageOrig.clone();

    //paint non necrose regions
    ProbaPixelPainter ppp(probasImage, res);
    
    int i=0;
    for (girl::regionDeque::const_iterator it = nonNecroseRegions.begin();
	 it != nonNecroseRegions.end();
	 ++it, ++i) {

      tog.iterateOnRegionDomain(*it, ppp);
    }
    
    //paint necrose regions
    const cv::Scalar COLOR_NECROSE(0, 0, 0); //BGR
    girl::tools::Painter<cv::Vec3b> ptr(probasImage, COLOR_NECROSE);
    for (girl::regionDeque::const_iterator it = necroseRegions.begin();
	 it != necroseRegions.end();
	 ++it) {
      std::cerr<<"region "<<*it<<" is necrose\n";
      tog.iterateOnRegionDomain(*it, ptr);
    }
  
    cv::imshow("probas", probasImage);
    cv::waitKey(0);
  }
#endif //DEBUG

  size_t dbg_nbInf = 0;
  size_t dbg_nbInf_b = 0;
  
  
  float proportionFibrine = 0.0f;
  float *p = res.ptr<float>(0);
  for (int i=0; i<res.rows; ++i) {
    float proba = *p;

    if (proba <= 0.5)
      ++dbg_nbInf;
    
#ifdef IW_BOOST_PROBA
    //Boost
    proba = disc(proba);
#endif //IW_BOOST_PROBA

    if (proba <= 0.5)
      ++dbg_nbInf_b;

    
    proportionFibrine += proba;
    ++p;
  }

  std::cerr<<"dbg_nbInf="<<dbg_nbInf<<" dbg_nbInf_b="<<dbg_nbInf_b<<" proportionFibrine="<<proportionFibrine<<" m="<<proportionFibrine/res.rows<<"\n";
  
  proportionFibrine /= (float)res.rows;
  
#ifdef TIMING
  t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
  std::cout<<"***Time for processImage: "<<t<<"s"<<std::endl;
#endif //TIMING
  std::cerr<<"proportionFibrine="<<proportionFibrine<<"\n";//DEBUG

  const float percentNecrose = areaNecrose/(float)area*100.f;
  //const float percentFibrine = static_cast<int>(proportionFibrine*(100.f-percentNecrose) + 0.5f);
  const float percentFibrine = proportionFibrine*(100.f-percentNecrose);
  const float percentBourgeon = 100.f-percentNecrose-percentFibrine;

  percentages[IW_PERCENT_FIBRINE] = percentFibrine;
  percentages[IW_PERCENT_BOURGEON] = percentBourgeon;
  percentages[IW_PERCENT_NECROSE] = percentNecrose;
  

  //outputImage = cv::Mat::zeros(img.rows, img.cols, CV_8UC3);
  //paint on original image
  outputImage = inputImageOrig.clone();
  {
    //paint fibrine/bourgeon according to proba

    FibrineProbaPixelPainter fppp(outputImage, res);
    
    for (girl::regionDeque::const_iterator it = nonNecroseRegions.begin();
	 it != nonNecroseRegions.end();
	 ++it) {

      tog.iterateOnRegionDomain(*it, fppp);
      
    }
    //paint necrose regions
    static const cv::Scalar COLOR_NECROSE(0, 0, 0); //BGR
    girl::tools::Painter<cv::Vec3b> ptr(outputImage, COLOR_NECROSE);
    for (girl::regionDeque::const_iterator it = necroseRegions.begin();
	 it != necroseRegions.end();
	 ++it) {
      tog.iterateOnRegionDomain(*it, ptr);
    }
  } 
  
}







void
classifyInsideWound(const cv::Mat &img,  //image with color correction
		    const cv::Mat &imgGT,
		    const cv::Mat &imgOrig, //original image, without color correction
		    const std::string &iwModelFilename, 
		    const std::string &iwNormalizeFilename, 
		    const std::string &iwProbaSigmoidFilename,
		    int smallSize,
		    int nbClasses,
		    ColorSpace cs,
		    int patchSize,
		    PatchType patchType, 
		    bool useL, 
		    std::vector<float> &percentages,
		    cv::Mat &outputImage)
{
  Normalizer n(iwNormalizeFilename);

#ifdef USE_CLASSIF_PATCH

  cv::Mat imgBlur;
#ifdef USE_BLUR
  const int kSize = 3;
  cv::GaussianBlur(img, imgBlur, cv::Size(kSize, kSize), 0, 0, cv::BORDER_DEFAULT );
#else
  imgBlur = img;
#endif
  
  detectInsideWound_NEW(iwModelFilename, imgBlur, imgGT, imgOrig,
			iwProbaSigmoidFilename, 
			cs, patchSize, patchType, smallSize, nbClasses, useL, n, 
			percentages, outputImage);
#else
  detectInsideWound(iwModelFilename, img, imgGT, imgOrig,
		    iwProbaSigmoidFilename, 
		    cs, patchSize, patchType, smallSize, nbClasses, useL, n, 
		    percentages, outputImage);

#endif //USE_CLASSIF_PATCH
  

  
}
