#include <iostream>
#include <iomanip>
#include <sstream>
#include <fstream>
#include <vector>

#include <unistd.h>
#include <cassert>
#include <getopt.h>

#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgproc/imgproc.hpp>

#include "utilsClassifier.hpp"
#include "pngUtils.hpp"
#include "pointsUtils.hpp"
#include "marker.hpp"


//#define DEBUG_VISU 1
//#define SAVE_PROBA 1
//#define TIMING 1

#define GONOGO2 0

#define GONOGO_SKIP_FINAL_MERGE 0



#include "utilsSeg.hpp"

#include "utilsConcepts.hpp"

#include <girlTools/girlToolsRegionInfos.hpp>


#define DETECT_MARKER 1




enum ERRORS {
  NO_ERROR=0,
  MARKER_NOT_FOUND_ERROR=1,
  INVALID_IMAGE_ERROR=2,
  INVALID_MASK_ERROR=3,
  INVALID_SKIN_COORDS_ERROR=4,
  INVALID_WOUND_COORDS_ERROR=5,
  WRITE_PROCESSED_IMAGE_ERROR=6,
  WRITE_MASK_ERROR=7,
  WRITE_INFO_ERROR=8,
  OTHER_ERROR=9
};


#include "defineUrgoEye.hpp"



static
void
usage(char* argv0)
{
  std::cerr<<"Usage: "<<argv0<<" -i inputImageFilename [--im inputMaskFilename] -s xSkin,ySkin -w xWound,yWound --om outputMaskFilename --oi outputAnnotatedImageFilename --ot outputInfoFilename [--model1 inputModelFile1  [--normalizeFile1 inputNormalizeFile1]  --model2 inputModelFile2  [--normalizeFile2 inputNormalizeFile2]  [--model3 inputModelFile3  [--normalizeFile3 inputNormalizeFile3] ] [--help] [--nbClasses N] [--useL onoff] [--colorSpace cs] [--patchSize p] [--patchType t] [--saveInColors onoff] [--size smallSize]]  [--iwmodel inputIWModelFile  [--iwnormalizeFile inputIWNormalizeFile] [--inputIWProbaSigmoid iwprobaSigmoidFilename] [--iwnbClasses N] [--iwuseL onoff] [--iwcolorSpace cs] [--iwpatchSize p] [--iwpatchType t] [--size smallSize]]"<<std::endl;

  std::cerr<<"--help : print this help and exit\n";
  
  std::cerr<<"-i inputImageFilename: specify input image filename\n";
  std::cerr<<"--im inputMaskFilename: specify input mask image filename\n";

  std::cerr<<"-s skinX,skinY : specify input point for skin touch. It is possible to specify several points with x1,y1:x2,y2:..:xn,yn\n";
  std::cerr<<"-w woundX,woundY : specify input point for wound touch. It is possible to specify several points with x1,y1:x2,y2:..:xn,yn\n";
  std::cerr<<"--om outputMaskFilename: specify output image filename\n";
  std::cerr<<"--ot outputInfoFilename: specify output info filename\n";
  std::cerr<<"--oi outputAnnotatedImageFilename: specify output image filename\n";

  std::cerr<<"--nbClasses N : specify number of classes to use for segmentation. Default is "<< DEFAULT_NBCLASSES<<"\n";
  std::cerr<<"--model1 inputModelFile1 : specify input model file for Fibrine\n";
  std::cerr<<"--model2 inputModelFile2 : specify input model file for Bourgeon\n";
  std::cerr<<"--model3 inputModelFile3 : specify input model file for Skin\n";
  std::cerr<<"--useL onoff : specify if L of LAB is used. Default is "<<DEFAULT_USE_L<<"\n";
  std::cerr<<"--doWB onoff : specify if white balance correction is done. Default is "<<DEFAULT_DO_WHITEBALANCE<<"\n";
  std::cerr<<"--normalizeFile1 inputNormalizeFile1 : specify an input normalization filename for model1\n";
  std::cerr<<"--normalizeFile2 inputNormalizeFile2 : specify an input normalization filename for model2\n";
  std::cerr<<"--normalizeFile3 inputNormalizeFile3 : specify an input normalization filename for model3\n";

  std::cerr<<"--colorSpace cs : set color space. ";
  {
    const int numCS = sizeof(colorSpaceStr)/sizeof(const char *);
    for (int i=0; i<numCS-1; ++i)
      std::cerr<<(int)i<<" is for "<<colorSpaceStr[i]<<", ";
    if (numCS > 0)
      std::cerr<<(int)(numCS-1)<<" is for "<<colorSpaceStr[numCS-1]<<".";
    std::cerr<<" Default is "<<DEFAULT_COLORSPACE<<"\n";
  }

  std::cerr<<"--patchSize p : set patchSize to p. Default is "<<DEFAULT_PATCH_SIZE<<"\n";
  std::cerr<<"--patchType t : set patch type. ";
  const int numPatchesTypes = sizeof(patchTypeStr)/sizeof(const char *);
  for (int i=0; i<numPatchesTypes-1; ++i)
    std::cerr<<(int)i<<" is for "<<patchTypeStr[i]<<", ";
  if (numPatchesTypes > 0)
    std::cerr<<(int)(numPatchesTypes-1)<<" is for "<<patchTypeStr[numPatchesTypes-1]<<".";
  std::cerr<<" Default is "<<DEFAULT_PATCH_TYPE<<"\n";

  std::cerr<<"--saveInColors onoff : specify if output image must be saved in colors. Default is "<<DEFAULT_SAVE_IN_COLORS<<"\n";

  std::cerr<<"--size smallSize: specify small size to which input image is resized (for classification). Default is "<<DEFAULT_SMALL_SIZE<<"\n";
  std::cerr<<"--iwmodel inputModelFile : specify input model file for inside wound\n";
  std::cerr<<"--iwnormalizeFile inputNormalizeFile : specify an input normalization filename for inside wound\n";

  std::cerr<<"--inputIWProbaSigmoid probaSigmoidFilename : specify input file with sigmoid parameters for SVM probabilities computation for inside wound\n";


  std::cerr<<"--iwuseL onoff : specify if L of LAB is used for inside wound. Default is "<<DEFAULT_INSIDEWOUND__USE_L<<"\n";

  std::cerr<<"--iwcolorSpace cs : set color space for inside wound. ";
  {
    const int numCS = sizeof(colorSpaceStr)/sizeof(const char *);
    for (int i=0; i<numCS-1; ++i)
      std::cerr<<(int)i<<" is for "<<colorSpaceStr[i]<<", ";
    if (numCS > 0)
      std::cerr<<(int)(numCS-1)<<" is for "<<colorSpaceStr[numCS-1]<<".";
    std::cerr<<" Default is "<<DEFAULT_INSIDEWOUND__COLORSPACE<<"\n";
  }

  std::cerr<<"--iwpatchSize p : set patchSize to p for inside wound. Default is "<<DEFAULT_INSIDEWOUND__PATCH_SIZE<<"\n";
  std::cerr<<"--iwpatchType t : set patch type for inside wound. ";
  //const int numPatchesTypes = sizeof(patchTypeStr)/sizeof(const char *);
  for (int i=0; i<numPatchesTypes-1; ++i)
    std::cerr<<(int)i<<" is for "<<patchTypeStr[i]<<", ";
  if (numPatchesTypes > 0)
    std::cerr<<(int)(numPatchesTypes-1)<<" is for "<<patchTypeStr[numPatchesTypes-1]<<".";
  std::cerr<<" Default is "<<DEFAULT_INSIDEWOUND__PATCH_TYPE<<"\n";

  std::cerr<<"--iwnbClasses N : specify number of classes to use for segmentation of inside wound. Default is "<< DEFAULT_INSIDEWOUND__NBCLASSES<<"\n";

  std::cerr<<"--iwsize smallSize: specify small size to which input image is resized (for classification of inside wound). Default is "<<DEFAULT_INSIDEWOUND__SMALL_SIZE<<"\n";


  std::cerr<<"\n";

}



static
void
processArgs(int argc, char *argv[],
	    std::string &inputFilename,
	    std::string &inputMaskFilename,
	    std::string &outputMaskFilename,
	    std::string &outputAnnotatedImageFilename,
	    std::string &outputInfoFilename,
	    std::string &modelFilename1,
	    std::string &normalizeFile1,
	    std::string &modelFilename2,
	    std::string &normalizeFile2,
	    std::string &modelFilename3,
	    std::string &normalizeFile3,
	    std::vector<std::vector<cv::Point> > &inputSkinPts,
	    std::vector<std::vector<cv::Point> > &inputWoundPts,
	    ColorSpace &cs,
	    int &patchSize,
	    PatchType &patchType,
	    //ClassifierType &classifierType,
	    int &smallSize,
	    int &nbClasses,
	    bool &useL, 
	    bool &saveInColors,
	    std::string &iwModelFilename, 
	    std::string &iwNormalizeFilename, 
	    std::string &iwProbaSigmoidFilename, 
	    ColorSpace &iw_cs, 
	    int &iw_patchSize, 
	    PatchType &iw_patchType, 
	    int &iw_smallSize, 
	    int &iw_nbClasses, 
	    bool &iw_use_L,
	    bool &do_wb)
{

  static struct option long_options[] = {
    {"oi",       required_argument,       0, 'b'},
    {"saveInColors",      required_argument,     0, 'c'},
    {"colorSpace",        required_argument,       0, 'e'},
    {"input",        required_argument,       0, 'i'},
    {"om",       required_argument,       0, 'g'},
    {"help",         no_argument,       0, 'h'},
    {"model1",        required_argument,     0, 'f'},
    {"model2",        required_argument,     0, 'j'},
    {"model3",        required_argument,     0, 'k'},
    {"im",    required_argument,       0, 'm'},
    {"nbClasses",    required_argument,       0, 'n'},
    {"patchSize",    required_argument,     0, 'p'},
    {"patchType",    required_argument,     0, 'q'},
    {"normalizeFile1",    required_argument,     0, 'r'},
    {"normalizeFile2",    required_argument,     0, 'x'},
    {"normalizeFile3",    required_argument,     0, 't'},
    {"size",    required_argument,       0, 'u'},
    {"useL",      required_argument,     0, 'y'},
    {"ot",      required_argument,     0, 'z'},

    {"iwmodel", required_argument,     0, 'D'},
    {"iwnormalizeFile", required_argument,     0, 'E'},
    {"inputIWProbaSigmoid", required_argument,     0, 'F'},
    {"iwuseL", required_argument,     0, 'J'},
    {"iwcolorSpace", required_argument,     0, 'K'},
    {"iwpatchSize", required_argument,     0, 'L'},
    {"iwpatchType", required_argument,     0, 'M'},
    {"iwnbClasses", required_argument,     0, 'N'},
    {"iwsize", required_argument,     0, 'O'},
    {"doWB", required_argument,     0, 'W'},

    {0, 0, 0, 0}
  };

  std::string skinCoordsString;
  std::string woundCoordsString;

  while (1) {
    int option_index=0;
    int c = getopt_long(argc, argv, "b:c:e:f:g:hi:j:k:m:n:p:q:r:s:t:u:w:x:y:z:D:E:F:J:K:L:M:N:O:W:", long_options, &option_index);

    if (c == -1)
      break;

    switch (c) {
    case 'b':
      outputAnnotatedImageFilename = optarg;
      break;
    case 'c':
      saveInColors = atoi(optarg);
      break;
    case 'e':
      cs = (ColorSpace)atoi(optarg);
      break;
    case 'f':
      modelFilename1 = optarg;
      break;
    case 'g':
      outputMaskFilename = optarg;
      break;
    case 'h':
      usage(argv[0]);
      exit(NO_ERROR);
      break;
    case 'i':
      inputFilename = optarg;
      break;
    case 'j':
      modelFilename2 = optarg;
      break;
    case 'k':
      modelFilename3 = optarg;
      break;
    case 'm':
      inputMaskFilename = optarg;
      break;
    case 'n':
      nbClasses = atoi(optarg);
      break;
    case 'p':
      patchSize = atoi(optarg);
      break;
    case 'q':
      patchType = (PatchType)atoi(optarg);
      break;
    case 'r':
      normalizeFile1 = optarg;
      break;
    case 's':
      skinCoordsString = optarg;
      break;
    case 't':
      normalizeFile3 = optarg;
      break;
    case 'u':
      smallSize = atoi(optarg);
      break;
    case 'w':
      woundCoordsString = optarg;
      break;
    case 'x':
      normalizeFile2 = optarg;
      break;
    case 'y':
      useL = atoi(optarg);
      break;
    case 'z':
      outputInfoFilename = optarg;
      break;
    case 'D':
      iwModelFilename = optarg;
      break;
    case 'E':
      iwNormalizeFilename = optarg;
      break;
    case 'F':
      iwProbaSigmoidFilename = optarg;
      break;
    case 'J':
      iw_use_L = atoi(optarg);
      break;
    case 'K':
      iw_cs = (ColorSpace)atoi(optarg);
      break;
    case 'L':
      iw_patchSize = atoi(optarg);
      break;
    case 'M':
      iw_patchType = (PatchType)atoi(optarg);
      break;
    case 'N':
      iw_nbClasses = atoi(optarg);
      break;
    case 'O':
      iw_smallSize = atoi(optarg);
      break;
    case 'W':
      do_wb = atoi(optarg);
      break;

    case -1: //end of list
      break;
    default:
      exit(OTHER_ERROR);
      break;
    }
  }

  if (inputFilename.empty()) {
    std::cerr << "Error: unspecified input filename\n";
    usage(argv[0]);
    exit(INVALID_IMAGE_ERROR);
  }
  if (inputMaskFilename.empty() && outputMaskFilename.empty()) {
    std::cerr << "Error: unspecified input mask filename\n";
    usage(argv[0]);
    exit(INVALID_MASK_ERROR);
  }

  const bool parsingSkinCoordsOk = parsePoints(skinCoordsString, inputSkinPts);
  if (! parsingSkinCoordsOk) {
    std::cerr<<"Error: invalid input skin coords\n";
    exit(INVALID_SKIN_COORDS_ERROR);
  }
  const bool parsingWoundCoordsOk = parsePoints(woundCoordsString, inputWoundPts);
  if (! parsingWoundCoordsOk) {
    std::cerr<<"Error: invalid input wound coords\n";
    exit(INVALID_WOUND_COORDS_ERROR);
  }
  

  if (outputMaskFilename.empty() && outputAnnotatedImageFilename.empty()) {
    std::cerr << "Error: no ouput specified\n";
    usage (argv[0]);
    if (! inputMaskFilename.empty())
      exit(WRITE_PROCESSED_IMAGE_ERROR); //???
    else
      exit(WRITE_MASK_ERROR);//???
  }

  if (inputMaskFilename.empty()) {
    if (modelFilename1.empty()) {
      std::cerr<<"Error: no input model 1 filename specified\n";
      usage(argv[0]);
      exit(OTHER_ERROR);
    }
    if (modelFilename2.empty()) {
      std::cerr<<"Error: no input model 2 filename specified\n";
      usage(argv[0]);
      exit(OTHER_ERROR);
    }
    /*
      if (modelFilename3.empty()) {
      std::cerr<<"Error: no input model 3 filename specified\n";
      usage(argv[0]);
      exit(OTHER_ERROR);
      }
    */
  }
  if (! inputMaskFilename.empty()) {
    if (iwModelFilename.empty()) {
      std::cerr<<"Error: no input iw model filename specified\n";
      usage(argv[0]);
      exit(OTHER_ERROR);
    }
  }

  if (patchType <0 || patchType>=sizeof(patchTypeStr)/sizeof(const char *)) {
    std::cerr<<"Error: invalid patch type\n";
    usage(argv[0]);
    exit(OTHER_ERROR);
  }


}



static
void
computeAreaOfMask(cv::Mat mask, float pixelSize, float &woundArea, float &woundWidth, float &woundHeight)
{
  assert(mask.type() == CV_8UC1);
  assert(mask.isContinuous());
  float area = 0.f;
  const unsigned char*m = mask.ptr<unsigned char>(0);

  int minX = mask.rows;
  int maxX = 0;
  int minY = mask.cols;
  int maxY = 0;
    
  for (int i=0; i<mask.rows; ++i) {
    for (int j=0; j<mask.cols; ++j) {
      if (*m != 0) {
	++area;
	if (j < minX)
	  minX = j;
	if (j > maxX)
	  maxX= j;
	if (i < minY)
	  minY = i;
	if (i > maxY)
	  maxY = i;
      }
      ++m;
    }
  }

  const int w = maxX-minX+1; 
  const int h = maxY-minY+1; 
    
  woundArea = area * pixelSize * pixelSize;
  woundWidth = w * pixelSize;
  woundHeight = h * pixelSize;
}




static
void
drawPts(cv::Mat &img, const std::vector<cv::Point> &pts, cv::Scalar color)
{
  assert(mask.type() == CV_8UC1);

  const int radius = 1;
  
  const size_t nbPts = pts.size();
  for (size_t i=0; i<nbPts; ++i) {

    cv::circle(img, pts[i], radius, color);
  }
}

static
void
drawPts(cv::Mat &img, const std::vector<std::vector<cv::Point> >&ptsGroup, const cv::Scalar color)
{
  for (std::vector<std::vector<cv::Point> >::const_iterator itG = ptsGroup.begin();
       itG != ptsGroup.end(); ++itG)
    drawPts(img, *itG, color);
}

static
void
drawPts(cv::Mat &img, const std::vector<std::vector<cv::Point> >&ptsSkin, const std::vector<std::vector<cv::Point> >&ptsWound)
{
  drawPts(img, ptsSkin, cv::Scalar(0, 255, 0));
  drawPts(img, ptsWound, cv::Scalar(0, 0, 255));
}






int
main(int argc, char *argv[])
{
  srand(time(NULL));

  std::string inputFilename, 
    inputMaskFilename, 
    outputMaskFilename, 
    outputAnnotatedImageFilename,
    outputInfoFilename,
    modelFilename1, normalizeFilename1, 
    modelFilename2, normalizeFilename2, 
    modelFilename3, normalizeFilename3;
  ColorSpace cs = DEFAULT_COLORSPACE;
  int patchSize = DEFAULT_PATCH_SIZE;
  PatchType patchType = DEFAULT_PATCH_TYPE;
  //ClassifierType classifierType = DEFAULT_CLASSIFIER;
  int smallSize = DEFAULT_SMALL_SIZE;
  int nbClasses = DEFAULT_NBCLASSES;
  bool use_L = DEFAULT_USE_L;
  bool saveInColors = DEFAULT_SAVE_IN_COLORS;

  std::vector<std::vector<cv::Point> > inputSkinPts;
  std::vector<std::vector<cv::Point> > inputWoundPts;

  std::string iwModelFilename, iwNormalizeFilename, iwProbaSigmoidFilename;
  ColorSpace iw_cs = DEFAULT_INSIDEWOUND__COLORSPACE;
  int iw_patchSize = DEFAULT_INSIDEWOUND__PATCH_SIZE;
  PatchType iw_patchType = DEFAULT_INSIDEWOUND__PATCH_TYPE;
  int iw_smallSize = DEFAULT_INSIDEWOUND__SMALL_SIZE;
  int iw_nbClasses = DEFAULT_INSIDEWOUND__NBCLASSES;
  bool iw_use_L = DEFAULT_INSIDEWOUND__USE_L;
  bool do_wb = DEFAULT_DO_WHITEBALANCE;

  modelFilename1=MODELS_DIR"/model1_SVM.xml";
  normalizeFilename1=MODELS_DIR"/norm1_SVM.txt";
  modelFilename2=MODELS_DIR"/model2_SVM.xml";
  normalizeFilename2=MODELS_DIR"/norm2_SVM.txt";
#ifdef USE_3CLASSIFIERS
  modelFilename3=MODELS_DIR"/model3_SVM.xml";
  normalizeFilename3=MODELS_DIR"/norm3_SVM.txt";
#endif //USE_3CLASSIFIERS


  iwModelFilename=MODELS_DIR"/model_FB.xml";
  iwNormalizeFilename=MODELS_DIR"/norm_FB.txt";
  iwProbaSigmoidFilename=MODELS_DIR"/probaSigmoid_FB.txt";
  
  std::string minMaxDistFilename1, minMaxDistFilename2, minMaxDistFilename3;
  minMaxDistFilename1=MODELS_DIR"/minMax1_SVM.txt";
  minMaxDistFilename2=MODELS_DIR"/minMax2_SVM.txt";
  minMaxDistFilename3=MODELS_DIR"/minMax3_SVM.txt";
  //std::string iwMinMaxFilename;
  //iwMinMaxFilename=MODELS_DIR"/minMaxFB.txt";


  processArgs(argc, argv, 
	      inputFilename, 
	      inputMaskFilename, 
	      outputMaskFilename,
	      outputAnnotatedImageFilename,
	      outputInfoFilename,
	      modelFilename1, normalizeFilename1, 
	      modelFilename2, normalizeFilename2, 
	      modelFilename3, normalizeFilename3, 
	      inputSkinPts, inputWoundPts,
	      cs, patchSize, patchType, smallSize, nbClasses, use_L, saveInColors,
	      iwModelFilename, iwNormalizeFilename, iwProbaSigmoidFilename, 
	      iw_cs, iw_patchSize, iw_patchType, iw_smallSize, iw_nbClasses, 
	      iw_use_L, do_wb);
  
  cv::Mat img = cv::imread(inputFilename);
  if (img.empty()) {
    std::cerr<<"Error: unable to load image file: "<<inputFilename<<"\n";
    exit(INVALID_IMAGE_ERROR);
  }

  //  std::cout<<" FileName " << inputFilename << "\n";
  cv::Mat imgC = img;

  const bool woundMaskProvided = ! inputMaskFilename.empty();
  cv::Mat woundMask;
  if (woundMaskProvided) {
#if 0
    const bool readOk = readBinaryPNG(inputMaskFilename, woundMask);
#else
    //OpenCV imread is able to load binary PNG images.
    woundMask = cv::imread(inputMaskFilename, 0); //loaded as one channel
    const bool readOk = ! woundMask.empty();
#endif
    if (! readOk) {
      std::cerr<<"Error: unable to load mask file: "<<inputMaskFilename<<"\n";
      exit(INVALID_MASK_ERROR);
    }
    assert(woundMask.type() == CV_8UC1);
  }
  

  //detection of marker on non color-corrected image
  float pixelSize = 0;
  cv::Mat markerMask;
  cv::Vec3f blue, white;
#if DETECT_MARKER
  const bool markerDetectionOk = detectMarker(img, pixelSize, markerMask, blue, white);
  if (! markerDetectionOk) {
    std::cerr<<"Error: unable to detect marker on image "<<inputFilename<<"\n";
    exit(MARKER_NOT_FOUND_ERROR);
  }

  imgC = correctWhite(img, white);

#else
  markerMask = cv::Mat::zeros(img.rows, img.cols, CV_8UC1);
  pixelSize = 1;
#endif    

#ifdef SAVE_SEG
  std::cout<<outputMaskFilename<<"\n";
  std::cout<<"marker size= "<<computeMarkerSize(markerMask, pixelSize)<<"\n";
#endif //SAVE_SEG


  drawPts(imgC, inputSkinPts, inputWoundPts);

  cv::imshow("img with pts", imgC);
  cv::waitKey(0);

  return 0;
}


