#include <iostream>
#include <iomanip>
#include <sstream>
#include <fstream>
#include <vector>
#include <limits>

#include <unistd.h>
#include <cassert>
#include <getopt.h>

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/ml/ml.hpp>

#include "marker.hpp"



//#define DEBUG_VISU 1
//#define SAVE_PROBA 1
//#define TIMING 1



static
void
usage(char* argv0)
{
  std::cerr<<"Usage: "<<argv0<<" inputImageFilename"<<std::endl;
}



static
void
processArgs(int argc, char *argv[],
	    std::string &inputFilename)
{
  if (argc != 2) {
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }
  inputFilename = argv[1];

  if (inputFilename.empty()) {
    std::cerr << "Error: unspecified input filename\n";
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }


}



int
main(int argc, char *argv[])
{
  srand(time(NULL));

  std::string inputFilename;

  processArgs(argc, argv, 
	      inputFilename);
  
  cv::Mat img = cv::imread(inputFilename);
  if (img.empty()) {
    std::cerr<<"Error: unable to load image file: "<<inputFilename<<"\n";
    exit(EXIT_FAILURE);
  }

  std::vector<cv::Point> blueContour, whiteContour;
  const bool markerDetectionOk = findMarkerContours(img, blueContour, whiteContour);
  
  if (! markerDetectionOk) {
    std::cerr<<"Error: unable to detect marker on image "<<inputFilename<<"\n";
    exit(EXIT_FAILURE);
  }

  cv::Mat markerMask = cv::Mat::zeros(img.rows, img.cols, CV_8UC1);

  std::vector< std::vector<cv::Point> > contours(1);
  contours[0] = whiteContour;
  const unsigned char colorBigUC = 255;
  const cv::Scalar colorBig(colorBigUC);
  drawContours(markerMask, contours, 0, colorBig, -1);

  size_t markerSize = 0;
  int rows = markerMask.rows;
  int cols = markerMask.cols;
  if (markerMask.isContinuous()) {
    cols = rows * cols;
    rows = 1;
  }
  for (int i=0; i<rows; ++i) {
    const uchar *m = markerMask.ptr<unsigned char>(i);
    for (int j=0; j<cols; ++j) {
      if (m[j]) {
	++markerSize;
      }
    }
  }

  //Warning: white contour is approximate so we get an approximate marker size 

  const size_t imageSize = img.rows*img.cols;

  const float ratioMarkerImg = markerSize / (float)imageSize;
  
  const float thresholdMarkerImg = 0.0245; 
  
  if (ratioMarkerImg < thresholdMarkerImg) {
    std::cerr<<inputFilename<<" : small marker: "<<markerSize<<"/"<<imageSize<<"="<<ratioMarkerImg<<" < "<<"thresholdMarkerImg="<<thresholdMarkerImg<<"\n";
    cv::imshow(inputFilename, img);
    cv::waitKey(0);
  }
  

  return 0;
}


