#include <iostream>
#include <sstream>

#include <cassert>
#include <getopt.h>
#include <fstream>

#if defined(_MSC_VER)  || ( defined(__clang__) && __clang_major__ >= 5 )
#include <unordered_set>
#else
#include <tr1/unordered_set>
#endif

#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/ml/ml.hpp>
#include <opencv2/imgproc/imgproc.hpp>


#include "utilsConceptsTrain.hpp"
#include "utilsClassifier.hpp" //sigmoid_predict, readFileLibSVM
#include "lab.hpp"


#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
typedef CvSVM *Classifier;
#else
typedef cv::Ptr<cv::ml::SVM> Classifier;
#endif


void
computeSigmoid(const cv::Mat &responsesD, 
	       const cv::Mat &responses, 
	       double prior1,
	       double prior0,
	       double &sigmoid_A, 
	       double &sigmoid_B);

#define DEBUG_VISU 1



typedef enum {NBC=0, KNN, SVM_CSVC_POLY, SVM_NUSVC_POLY, SVM_CSVC_RBF, SVM_NUSVC_RBF, SVM_CSVC_SIGMOID, SVM_NUSVC_SIGMOID, DT, GBT, RF, ERT, ANN, EM} ClassifierType;
const char *classifierNames[] ={"NormalBayessianClassifier", "KNearestNeighbors", "SVM_CSVC_POLY", "SVM_NUSVC_POLY", "SVM_CSVC_RBF", "SVM_NUSVC_RBF", "SVM_CSVC_SIGMOID", "SVM_NUSVC_SIGMOID", "DecisionTree", "GradientBoostedTree", "RandomForest", "ExtremelyRandomizedTrees", "ArtificialNeuralNetworks", "ExpectationMaximization"};

static const ClassifierType DEFAULT_CLASSIFIER = SVM_CSVC_POLY;

static const size_t DEFAULT_MAX_NB_SAMPLES = 30000;

static const bool DEFAULT_AUTOTEST = true; //false;
static const bool DEFAULT_AUTOTEST_ALL = true; //false;

static const bool DEFAULT_USE_BACKGROUND = false;
static const char *BACKGROUND_STR="fond";
static const char *BACKGROUND_STR2="";

static const bool DEFAULT_USE_L = true;

static const bool DEFAULT_AUTOTRAIN = false; //??


static const int DEFAULT_PATCH_SIZE = 3;
static const PatchType DEFAULT_PATCH_TYPE = eLAB_VALUES;

static const bool DEFAULT_RESIZE_IMAGES = false;
static const int DEFAULT_SMALL_SIZE = 180;

static const bool DEFAULT_BALANCE = true;

static const ColorSpace DEFAULT_COLORSPACE = eLAB;


typedef enum {eMinMax=0, eMeans, eMedians} MinMaxType;
const char *minMaxTypeStr[] = {"MinMax", "Means", "Medians"};
static const MinMaxType DEFAULT_MINMAX_TYPE = eMinMax;

#if SAVE_PROBA
std::string g_trainStatsDistFilename = "";
std::string g_trainDistHistoFilename = "";

std::string g_trainStatsProbaFilename = "";
std::string g_trainProbaHistoFilename = "";
#endif //SAVE_PROBA


static
void
usage(char* argv0)
{
  std::cerr<<"Usage: "<<argv0<<"[--autoTest onoff] [--background b] [--help] [--classifier c] (--train trainFilename | --readData inputDataFilename | --readDataLibSVM inputDataLibSVMFilename) [--saveData outputDataFilename] [--saveDataLibSVM outputDataFilenameLibSVM] [--saveDataCSV outputDataFilenameCSV] --concept conceptFilename --outputModel outputFilename [--max maxNbOfSamplesForTrain] [--useL onoff] [--autoTrain onoff] [--normalize onoff] [--colorSpace cs] [--patchSize p] [--patchType t] [--normalizeFile normalizationOutputFile] [--doResize onoff [--size smallSize]] [--balance onoff] [--merge mergeFilename]* [--outputAutoTestMinMax outputAutoTestMinMaxFilename [--minMaxType type]] [--outputProbaSigmoid outputProbaSigmoidFilename]\n"<<std::endl;
  {
    std::cerr<<"--classifier c : choose classifiers to train on pixels of concepts. Choices are:\n";
    const int numClassifiers = sizeof(classifierNames)/sizeof(const char *);
    for (int i=0; i<numClassifiers-1; ++i)
      std::cerr<<i<<" for "<<classifierNames[i]<<", ";
    std::cerr<<numClassifiers-1<<" for "<<classifierNames[numClassifiers-1]<<".\n";
    std::cerr<<"Default is "<<DEFAULT_CLASSIFIER<<"\n";
  }
  
  std::cerr<<"--autoTest onoff: if a is 1, train data will be used a test data. Default is "<<DEFAULT_AUTOTEST<<"\n";
  std::cerr<<"--background b: if b is 0 background class will not be used, otherwise it will be used. Default is "<<DEFAULT_USE_BACKGROUND<<"\n";
  std::cerr<<"--help : print this help and exit\n";
  std::cerr<<"--train trainFilename: specify input file for training, listing on each line: an image and an image label filenames [rest of the line is ignored]\n";
  std::cerr<<"--readData inputDataFilename : specify input data file for training [will use this instead of images in trainFilename]\n";
  std::cerr<<"--readDataLibSVM inputDataLibSVMFilename: specify input data file for training in LibSVM format [will use this instead of images in trainFilename]\n";
  std::cerr<<"--saveData outputDataFilename: specify output data file\n";
  std::cerr<<"--saveDataLibSVM outputDataFilenameLibSVM : specify output data file in libSVM format\n";
  std::cerr<<"--saveDataCSV outputDataFilenameCSV : specify output data file in CSV format\n";
  std::cerr<<"--concept conceptFilename : specify input concept filename (qise CSV format). Warning: only concepts/colors listed in this file are considered in GT image files.\n";
  std::cerr<<"--outputModel outputFilename : specify output filename for model\n";
  std::cerr<<"--max maxNbOfSamplesForTrain: specify maximum number of samples to use for training per concept. Default is "<<DEFAULT_MAX_NB_SAMPLES<<"\n";
  std::cerr<<"--useL onoff : specify if L of LAB is used. Default is "<<DEFAULT_USE_L<<"\n";
  std::cerr<<"--autoTrain onoff : specify if auto train must be done. Default is "<<DEFAULT_AUTOTRAIN<<"\n";
  std::cerr<<"--normalizeFile normalizeOutputFile : if a file is given, normalization will be done and normalization factors save to file\n";

  std::cerr<<"--colorSpace cs : set color space. ";
  {
    const int numCS = sizeof(colorSpaceStr)/sizeof(const char *);
    for (int i=0; i<numCS-1; ++i)
      std::cerr<<(int)i<<" is for "<<colorSpaceStr[i]<<", ";
    if (numCS > 0)
      std::cerr<<(int)(numCS-1)<<" is for "<<colorSpaceStr[numCS-1]<<".";
    std::cerr<<" Default is "<<DEFAULT_COLORSPACE<<"\n";
  }

  std::cerr<<"--patchSize p : set patchSize to p. Default is "<<DEFAULT_PATCH_SIZE<<"\n";
  std::cerr<<"--patchType t : set patch type. ";
  const int numPatchesTypes = sizeof(patchTypeStr)/sizeof(const char *);
  for (int i=0; i<numPatchesTypes-1; ++i)
    std::cerr<<(int)i<<" is for "<<patchTypeStr[i]<<", ";
  if (numPatchesTypes > 0)
    std::cerr<<(int)(numPatchesTypes-1)<<" is for "<<patchTypeStr[numPatchesTypes-1]<<".";
  std::cerr<<" Default is "<<DEFAULT_PATCH_TYPE<<"\n";

  std::cerr<<"--doResize onoff: specify if input images must be resized. Default is "<<DEFAULT_RESIZE_IMAGES<<"\n";
  std::cerr<<"--size smallSize: specify size to which input images must be resized (if --doResize is enabled). Default is "<<DEFAULT_SMALL_SIZE<<"\n";

  std::cerr<<"--balance onoff: specify if the number of samples per concept must be balanced/equal. Default is "<<DEFAULT_BALANCE<<"\n";

  std::cerr<<"--merge mergeFilename: specify a merge filename in CSV format with on each line a concept. Only the first concept will be kept. All other conceptss will be merged with the first one. Contrary to file specified for --concept, last value of the line indicates the percentage of concept to garantee in final mix. The value is a float in [0; 1]. If it is 0, there is no garantee. This percentage is taken into account only for merged concepts (not for the kept one). This option can be used several times to make several merges. However concepts listed in different files must all be different. Otherwise percentage in the final mix is not garanteed. If at least one merge file is specified, only first concepts listed in these merge files will be kept for training. The other will be removed.\n";

  std::cerr<<"--outputAutoTestMinMax outputAutoTestMinMaxFilename : specify that min/max on autotest of train data must be save to file outputAutoTestMinMaxFilename\n";
  std::cerr<<"--minMaxType type : specify which min/max is saved (when --outputAutoTestMinMax is specified). ";
  {
    const int numT = sizeof(minMaxTypeStr)/sizeof(const char *);
    for (int i=0; i<numT-1; ++i)
      std::cerr<<(int)i<<" is for "<<minMaxTypeStr[i]<<", ";
    if (numT > 0)
      std::cerr<<(int)(numT-1)<<" is for "<<minMaxTypeStr[numT-1]<<".";
    std::cerr<<" Default is "<<DEFAULT_MINMAX_TYPE<<"\n";
  }
  
  std::cerr<<"--outputProbaSigmoid outputProbaSigmoidFilename : specify the output filename to save sigmoid parameters for probability computation\n";

  std::cerr<<"\n";
}

static
void
processArgs(int argc, char *argv[],
	    std::string &trainFilename,
	    std::string &conceptFilename,
	    std::string &inputDataFilename,
	    std::string &inputDataFilenameLibSVM,
	    std::string &outputDataFilename,
	    std::string &outputDataFilenameLibSVM,
	    std::string &outputDataFilenameCSV,
	    std::string &outputModelFilename,
	    std::string &outputNormalizeFilename,
	    std::vector<std::string> &mergeFilenames,
	    size_t &maxNbSamples,
	    ColorSpace &cs,
	    int &patchSize,
	    PatchType &patchType,
	    ClassifierType &classifierType,
	    bool &autoTrain,
	    bool &autoTest,
	    bool &autoTestAll,
	    std::string &outputProbaSigmoidFilename,
	    std::string &outputAutoTestMinMaxFilename,
	    MinMaxType &minMaxType,
	    bool &useBackground,
	    bool &useL, 
	    bool &doResize,
	    int &smallSize,
	    bool &balance)
{

  static struct option long_options[] = {
    {"autoTest",     required_argument,     0, 'a'},
    {"background",   required_argument,     0, 'b'},
    {"classifier",   required_argument,     0, 'c'},
    {"doResize",   required_argument,     0, 'd'},
    {"merge",   required_argument,     0, 'e'},
    {"balance", required_argument,     0, 'f'},
    {"outputAutoTestMinMax", required_argument,     0, 'g'},
    {"help",         no_argument,     0, 'h'},
    {"train",        required_argument,     0, 'i'},
    {"concept",      required_argument,     0, 'j'},
    {"readData",     required_argument,     0, 'k'},
    {"saveData",     required_argument,     0, 'l'},
    {"max",          required_argument,     0, 'm'},
    {"colorSpace",   required_argument,     0, 'n'},
    {"outputModel",  required_argument,     0, 'o'},
    {"patchSize",    required_argument,     0, 'p'},
    {"patchType",    required_argument,     0, 'q'},
    {"saveDataLibSVM",     required_argument,     0, 'r'},
    {"saveDataCSV",     required_argument,     0, 's'},
    {"minMaxType", required_argument,     0, 't'},
    {"size", required_argument,     0, 'u'},
    {"outputProbaSigmoid", required_argument,     0, 'v'},
    {"useL",         required_argument,     0, 'w'},
    {"autoTrain",    required_argument,     0, 'x'},
    {"normalizeFile",    required_argument,     0, 'y'},
    {"readDataLibSVM",     required_argument,     0, 'A'},
    {"autoTestAll",     required_argument,     0, 'Z'},
    {0, 0, 0, 0}
  };

  while (1) {
    int option_index=0;
    int c = getopt_long(argc, argv, "abc:d:e:f:g:hi:j:k:l:m:n:o:p:q:r:s:t:u:v:w:x:y:z:A:Z:", long_options, &option_index);

    if (c == -1)
      break;

    switch (c) {
    case 'a':
      autoTest = atoi(optarg);
      break;
    case 'b':
      useBackground = atoi(optarg);
      break;
    case 'c':
      {
	const int c = atoi(optarg);
	const int numClassifiers = sizeof(classifierNames)/sizeof(const char *);
	if (c < 0 || c >= numClassifiers) {
	  std::cerr<<"Error: invalid value for classifier: "<<c<<". It must be in [0; "<<numClassifiers-1<<"]\n";
	  exit(EXIT_FAILURE);
	}
	classifierType = (ClassifierType)c;
      }
      break;
    case 'd':
      doResize = atoi(optarg);
      break;
    case 'e':
      mergeFilenames.push_back(optarg);
      break;
    case 'f':
      balance = atoi(optarg);
      break;
    case 'g':
      outputAutoTestMinMaxFilename = optarg;
      break;
    case 'h':
      usage(argv[0]);
      exit(EXIT_SUCCESS);
      break;
    case 'i':
      trainFilename = optarg;
      break;
    case 'j':
      conceptFilename = optarg;
      break;
    case 'k':
      inputDataFilename = optarg;
      break;
    case 'l':
      outputDataFilename = optarg;
      break;
    case 'm':
      maxNbSamples = atoi(optarg);
      break;
    case 'n':
      cs = (ColorSpace)atoi(optarg);
      break;
    case 'o':
      outputModelFilename = optarg;
      break;
    case 'p':
      patchSize = atoi(optarg);
      break;
    case 'q':
      patchType = (PatchType)atoi(optarg);
      break;
    case 'r':
      outputDataFilenameLibSVM = optarg;
      break;
    case 's':
      outputDataFilenameCSV = optarg;
      break;
    case 't':
      minMaxType = (MinMaxType)atoi(optarg);
      break;
    case 'u':
      smallSize = atoi(optarg);
      break;
    case 'v':
      outputProbaSigmoidFilename = optarg;
      break;
    case 'w':
      useL = atoi(optarg);
      break;
    case 'x':
      autoTrain = atoi(optarg);
      break;
    case 'y':
      outputNormalizeFilename = optarg;
      break;
    case 'A':
      inputDataFilenameLibSVM = optarg;
      break;
    case 'Z':
      autoTestAll = atoi(optarg);
      break;
    case -1: //end of list
      break;
    default:
      exit(EXIT_FAILURE);
      break;
    }
  }

  if (trainFilename.empty() && inputDataFilename.empty() && inputDataFilenameLibSVM.empty()) {
    std::cerr<<"Error: no input train filename or input data filename specified\n";
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }
  if (! trainFilename.empty() && ! inputDataFilename.empty()) {
    std::cerr<<"Error: both input train filename and input data filename specified\n";
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }
  if (! trainFilename.empty() && ! inputDataFilenameLibSVM.empty()) {
    std::cerr<<"Error: both input train filename and input data filename (in LibSVM format) specified\n";
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }
  if (! inputDataFilename.empty() && ! inputDataFilenameLibSVM.empty()) {
    std::cerr<<"Error: both input data filename and input data filename (in LibSVM format) specified\n";
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }



  if (conceptFilename.empty() && inputDataFilenameLibSVM.empty()) {
    std::cerr<<"Error: no input concept filename specified\n";
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }
  if (outputModelFilename.empty()) {
    std::cerr<<"Error: no output model filename specified\n";
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }

  if (patchType <0 || patchType>=sizeof(patchTypeStr)/sizeof(const char *)) {
    std::cerr<<"Error: invalid patch type\n";
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }
    
  if (maxNbSamples == 0) {
    std::cerr<<"ERROR: max number of samples must be > 0\n";
    exit(EXIT_FAILURE);
  }

}

typedef std::vector<LConcept> LConceptVector;
typedef std::vector<LConceptVector> LConceptVV;


  
static
void
readConceptsFromCSV(const std::string &csvFilename,
		    std::vector<LConcept> &concepts,
		    bool readWithFloat = false)
{
  concepts.clear();

  std::ifstream in(csvFilename.c_str());
  if (! in) {
    std::cerr<<"Error: unable to open CSV annotation file: "<<csvFilename<<std::endl;
    exit(EXIT_FAILURE);
  }

  const char SEPARATOR=';';

  bool firstDone = false;
  size_t count = 0;

  while (! in.eof()) {
    std::string str;
    std::getline(in, str);
    if (in.eof())
      break;
    if (str.empty())
      continue;
    //std::cerr<<"line="<<str<<"\n";

    std::istringstream iss(str);
    
    std::string token;
    if (! firstDone) {
      count = 0;
      while ( getline(iss, token, SEPARATOR) ) {
	++count;
      }
      if (count != 5) {
	std::cerr<<"Error: unable to parse CSV file: "<<csvFilename<<". It should have 5 fields on each line"<<std::endl;
	exit(EXIT_FAILURE);
      }

      firstDone= true;
    }
    else {
      std::string name;
      char sep;
      int r=0, g=0, b=0, defaultV=0;
      float p=0;
      getline(iss, name, SEPARATOR);
      iss >> r;
      iss >> sep;
      iss >> g;
      iss >> sep;
      iss >> b;
      iss >> sep;
      if (! readWithFloat)
	iss >> defaultV;
      else
	iss >> p;
      if (iss.bad()) {
	std::cerr<<"Error: unable to parse CSV file: "<<csvFilename<<" not enough fields on one line"<<std::endl;
	exit(EXIT_FAILURE);
      }
      if (sep != SEPARATOR) {
	std::cerr<<"ERROR: separator="<<sep<<" != "<<SEPARATOR<<"\n";
	exit(EXIT_FAILURE);
      }
      //std::cerr<<"name="<<name<<" r="<<(int)r<<" g="<<(int)g<<" b="<<(int)b<<" defaultV="<<(int)defaultV<<std::endl;

      if (! readWithFloat) {
	if (defaultV == 0) { //We do not keep pixels marked as default !
	  concepts.push_back(LConcept(name, cv::Vec3b(r, g, b)));
	}
      }
      else {
	if (p<0 || p>1) {
	  std::cerr<<"ERROR: file:"<<csvFilename<<" concept "<<name<<" with color ["<<r<<", "<<g<<", "<<b<<"] has a percentage to keep not in [0; 1]\n";
	  exit(EXIT_FAILURE);
	}
	concepts.push_back(LConcept(name, cv::Vec3b(r, g, b), p));
      }

    }
    
  }

}





//just for code factorization for classifiers with a 'simple' predict() method
// (predict() method is not virtual in base class because it can not be unified for all classifiers)
template <typename C>
void
doClassifierAutoTest(C &classifier, const std::string &s,
		     const cv::Mat &trainData, const cv::Mat &responses)
{
  
#if 1 //predict one by one
    {
      double t = (double)cv::getTickCount();
      int nbCorrects = 0;
      const int *resps = responses.ptr<int>(0);
      for (int i=0; i<trainData.rows; ++i) {
	const int response = (int)classifier.predict( trainData.row(i) );
	if (response == *resps)
	  ++nbCorrects;
	++resps;
      }
      t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
      std::cerr<<"Time for autoTest["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";
      std::cerr<<"nbCorrects="<<nbCorrects<<"/"<<trainData.rows<<"="<<(trainData.rows>0 ? 100*nbCorrects/(double)trainData.rows : 0)<<"%\n\n";    
    }
#else
    {
      double t = (double)cv::getTickCount();
      int nbCorrects = 0;
      cv::Mat responses2(responses.rows, 1, CV_32F);
      classifier.predict( trainData, responses2 );
      const int *resps = responses.ptr<int>(0);
      const float *resps2 = responses2.ptr<float>(0);
      for (int i=0; i<trainData.rows; ++i) {
	if ((int)*resps2 == *resps)
	  ++nbCorrects;
	++resps;
	++resps2;
      }
      t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
      std::cerr<<"Time for autoTest["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";
      std::cerr<<"nbCorrects="<<nbCorrects<<"/"<<trainData.rows<<"="<<(trainData.rows>0 ? 100*nbCorrects/(double)trainData.rows : 0)<<"%\n\n";    
    }
#endif //0
  
}


template <>
void
doClassifierAutoTest(Classifier &classifier, const std::string &s,
		     const cv::Mat &trainData, const cv::Mat &responses)
{
  if (trainData.rows != responses.rows) {
    std::cerr<<"ERROR: doClassifierAutoTest<>: trainData.rows="<<trainData.rows<<" != responses.rows="<<responses.rows<<" !!!\n";
    exit(10);
  }
  
  
#if 0
    {
      //predict row by row
      double t = (double)cv::getTickCount();
      int nbCorrects = 0;
      const int *resps = responses.ptr<int>(0);
      for (int i=0; i<trainData.rows; ++i) {
	const int response = (int)classifier->predict( trainData.row(i) );
	if (response == *resps)
	  ++nbCorrects;
	++resps;
      }
      t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
      std::cerr<<"Time for autoTest["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";
      std::cerr<<"nbCorrects="<<nbCorrects<<"/"<<trainData.rows<<"="<<(trainData.rows>0 ? 100*nbCorrects/(double)trainData.rows : 0)<<"%\n\n";    
    }
#elif 0 //#else
    {
      //predict all rows at once (parallelized)
      double t = (double)cv::getTickCount();
      int nbCorrects = 0;
      cv::Mat responses2(responses.rows, 1, CV_32F);
      classifier->predict( trainData, responses2 );
      const int *resps = responses.ptr<int>(0);
      const float *resps2 = responses2.ptr<float>(0);
      for (int i=0; i<trainData.rows; ++i) {

	if (i==trainData.rows/4 || i==3*trainData.rows/4) { //DEBUG
	  std::cerr<<"  row "<<i<<" resp="<<*resps2<<" GT="<<*resps<<"\n";
	}
	
	if ((int)*resps2 == *resps)
	  ++nbCorrects;
	++resps;
	++resps2;
      }
      t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
      std::cerr<<"Time for autoTest["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";
      std::cerr<<"nbCorrects="<<nbCorrects<<"/"<<trainData.rows<<"="<<(trainData.rows>0 ? 100*nbCorrects/(double)trainData.rows : 0)<<"%\n\n";    
    }
#endif //0

    {
      double sigmoid_A, sigmoid_B;
      double prior0 = 0;
      double prior1 = 0;
      cv::Mat responses2 = cv::Mat::zeros(responses.rows, 1, CV_32F);
      {
	assert(responses.isContinuous());
	const int *r = responses.ptr<int>(0);
	for (int i=0; i<responses.rows; ++i) {
	  if (*r > 0)
	    prior1 += 1;
	  else
	    prior0 += 1;
	  ++r;
	}
      }
      double t = (double)cv::getTickCount();
      const bool returnDFVal = true;
      //Form OpenCV doc:
      // returnDFVal – Specifies a type of the return value. If true and the problem is 2-class classification then the method returns the decision function value that is signed distance to the margin, else the function returns a class label (classification) or estimated function value (regression).
      
#if ! (CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300)
      int flags = returnDFVal ? cv::ml::StatModel::RAW_OUTPUT : 0;
#endif

      classifier->predict( trainData, responses2, 
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300			   
			   returnDFVal //parallelized version, taking returnDFVal, is only available in OpenCV 2.4.x patched version
#else
			   flags
#endif
			   );
     t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
     std::cerr<<"Time for autoTest["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";
    
      computeSigmoid(responses2, 
		     responses, 
		     prior1,
		     prior0,
		     sigmoid_A, 
		     sigmoid_B);

      int nbCorrectsA = 0;
      int nbCorrects = 0;

      int numProbas1 = 0;
      float sumProba1 = 0;
      int numProbas0 = 0;
      float sumProba0 = 0;

      int numProbas1c = 0;
      float sumProba1c = 0;
      int numProbas0c = 0;
      float sumProba0c = 0;
      
      int numProbas1ic = 0;
      float sumProba1ic = 0;
      int numProbas0ic = 0;
      float sumProba0ic = 0;

      const int *resps = responses.ptr<int>(0);
      const float *resps2 = responses2.ptr<float>(0);
      for (int i=0; i<trainData.rows; ++i) {
	float proba = sigmoid_predict(*resps2, sigmoid_A, sigmoid_B);

	//std::cerr<<"label="<<*resps<<" resp="<<*resps2<<" respSig="<<proba<<"\n";
	
	if (proba > 0.5) {
	  sumProba1 += proba;
	  ++numProbas1;
	}
	else {
	  sumProba0 += proba;
	  ++numProbas0;
	}

	if (proba > 0.5) {

	  if (*resps > 0) {
	    ++nbCorrects;
	    
	    ++numProbas1c;
	    sumProba1c += proba;
	  }
	  else {
	    ++numProbas0ic;
	    sumProba0ic += proba;
	  }
	  
	}
	else { //if (proba <= 0.5) {

	  if (*resps<=0) {
	    ++nbCorrects;
	    
	    ++numProbas0c;
	    sumProba0c += proba;
	  }
	  else {
	    ++numProbas1ic;
	    sumProba1ic += proba;
	  }
	}

	if ((int)*resps2 == *resps)
	  ++nbCorrectsA;

	
	++resps;
	++resps2;
      }
    
      std::cerr<<"nbCorrects="<<nbCorrects<<"/"<<trainData.rows<<"="<<(trainData.rows>0 ? 100*nbCorrects/(double)trainData.rows : 0)<<"%\n\n";         std::cerr<<"nbCorrects[sigmoid]="<<nbCorrects<<"/"<<trainData.rows<<"="<<(trainData.rows>0 ? 100*nbCorrects/(double)trainData.rows : 0)<<"%\n"; 
      std::cerr<<"[all] "<<numProbas1<<" probas > 0.5 : mean="<<sumProba1/numProbas1<<"\n";
      std::cerr<<"[all] "<<numProbas0<<" probas <= 0.5 : mean="<<sumProba0/numProbas0<<"\n";
      std::cerr<<"[correct] "<<numProbas1c<<" probas > 0.5 : mean="<<sumProba1c/numProbas1c<<"\n";
      std::cerr<<"[correct] "<<numProbas0c<<" probas <= 0.5 : mean="<<sumProba0c/numProbas0c<<"\n";
      std::cerr<<"[incorrect] "<<numProbas1ic<<" probas > 0.5 : mean="<<sumProba1ic/numProbas1ic<<"\n";
      std::cerr<<"[incorrect] "<<numProbas0ic<<" probas <= 0.5 : mean="<<sumProba0ic/numProbas0ic<<"\n";
    }
} 



static
void 
normalizeMatAndGetMinsMaxs(cv::Mat &mat, std::vector<float> &mins, std::vector<float> &maxs)
{
  int size = mat.cols;
  mins.clear(); maxs.clear();
  mins.resize(size, std::numeric_limits<float>::max());
  maxs.resize(size, -std::numeric_limits<float>::max());
  assert(mat.cols == size);
  for (int i=0; i<mat.rows; ++i) {
    float *p = mat.ptr<float>(i);
    for (int j=0; j<mat.cols; ++j) {
      mins[j] = std::min(mins[j], p[j]);
      maxs[j] = std::max(maxs[j], p[j]);
    }
  }
  
  for (int i=0; i<mat.rows; ++i) {
    float *p = mat.ptr<float>(i);
    for (int j=0; j<mat.cols; ++j)
      p[j] = (p[j]-mins[j])/(maxs[j]-mins[j]);
  }
  
}

static
void
writeMinMaxs(const std::string &outputFilename, 
	     const std::vector<float> &mins, 
	     const std::vector<float> &maxs)
{
  assert(mins.size() == maxs.size());

  std::ofstream out(outputFilename.c_str());
  if (! out) {
    std::cerr<<"ERROR: unable to open output file: "<<outputFilename<<"\n";
    exit(10);
  }
  int size = mins.size();
  out<<size<<"\n";
  for (int i=0; i<size; ++i)
    out<<mins[i]<<" ";
  out<<"\n";
  for (int i=0; i<size; ++i)
    out<<maxs[i]<<" ";
  out<<"\n";
}



static
void
saveAutoTestMinMax(Classifier &classifier,
		   const cv::Mat &testData,
		   const cv::Mat &responsesGT, 
		   const std::string &outputFilename,
		   MinMaxType minMaxType)
{
  assert(responsesGT.type() == CV_32S); //classes

  std::ofstream out(outputFilename.c_str());
  if (! out) {
    std::cerr<<"ERROR: unable to open output file: "<<outputFilename<<"\n";
    exit(EXIT_FAILURE);
  }

  cv::Mat responses(testData.rows, 1, CV_32F);

  float maxR = std::numeric_limits<int>::min();
  float minR = std::numeric_limits<int>::max();

  const bool returnDFVal = true;

  classifier->predict(testData, responses, returnDFVal); //parallelized version, taking returnDFVal, is only available with my patch in 2.4.x ...

  const float *d1 = responses.ptr<float>(0);
  for (int i=0; i<responses.rows; ++i) {
    const float response = *d1;
    if(response > maxR)
      maxR = response;
    if (response < minR)
      minR = response;
    ++d1;
  }


  //Use median or mean
  {
    std::vector<float> responsesC0, responsesC1, responsesC0All, responsesC1All;
    responsesC0.reserve(responses.rows);
    responsesC1.reserve(responses.rows);
    responsesC0All.reserve(responses.rows);
    responsesC1All.reserve(responses.rows);
    const float *d1 = responses.ptr<float>(0);
    const int *c = responsesGT.ptr<int>(0);
    int wrongC0 = 0, wrongC1=0;
    assert(responses.rows == responsesGT.rows);
    for (int i=0; i<responses.rows; ++i) {
      if (*d1 > 0) {
	if (*c == 0) { 
	  responsesC0.push_back(*d1);
	}
	else {
	  ++wrongC0;
	}
	responsesC0All.push_back(*d1);
      }
      else if (*d1 < 0) {
	if (*c == 1) {
	  responsesC1.push_back(*d1);
	}
	else {
	  ++wrongC1;
	}
	responsesC1All.push_back(*d1);
      }
      ++d1;
      ++c;
    }
  
    std::cerr<<"Dists:\n";
    std::cerr<<"responsesC0.size()="<<responsesC0.size()<<" wrongC0="<<wrongC0<<" / "<<responses.rows/2<<" = "<<2*wrongC0/(float)responses.rows<<"\n";
    std::cerr<<"responsesC1.size()="<<responsesC1.size()<<" wrongC1="<<wrongC1<<" / "<<responses.rows/2<<" = "<<2*wrongC1/(float)responses.rows<<"\n";
    std::cerr<<"wrongC0+wrongC1="<<wrongC0+wrongC1<<" / "<<responses.rows<<" = "<<( wrongC0+wrongC1)/(float)responses.rows<<"\n";

    std::vector<float>::iterator itC0 = responsesC0.begin()+responsesC0.size()/2;
    std::nth_element(responsesC0.begin(), itC0, responsesC0.end());
    float medC0 = maxR;
    if (itC0 != responsesC0.end())
      medC0 = *itC0;

    std::vector<float>::iterator itC1 = responsesC1.begin()+responsesC1.size()/2;
    std::nth_element(responsesC1.begin(), itC1, responsesC1.end());
    float medC1 = minR;
    if (itC1 != responsesC1.end())
      medC1 = *itC1;

    std::cerr<<"medC1="<<medC1<<" ; medC0="<<medC0<<"\n";
    assert(medC1<=medC0);

    float meanC0=0;
    for (std::vector<float>::iterator it=responsesC0.begin(); it!=responsesC0.end(); ++it)
      meanC0 += *it;
    if (! responsesC0.empty())
      meanC0 /= responsesC0.size();
    float meanC1=0;
    for (std::vector<float>::iterator it=responsesC1.begin(); it!=responsesC1.end(); ++it)
      meanC1 += *it;
    if (! responsesC1.empty())
      meanC1 /= responsesC1.size();
    std::cerr<<"meanC1="<<meanC1<<" ; meanC0="<<meanC0<<"\n";
    assert(meanC1<=meanC0);
    std::cerr<<"minR="<<minR<<" ; maxR="<<maxR<<"\n";

    int rightC0Mean = 0, rightC1Mean = 0;
    int wrongC0Mean = 0, wrongC1Mean = 0;
    {
      const float *d1 = responses.ptr<float>(0);
      const int *c = responsesGT.ptr<int>(0);
      assert(responses.rows == responsesGT.rows);
      for (int i=0; i<responses.rows; ++i) {
	if (*d1 > meanC0) {
	  if (*c == 0) { 
	    ++rightC0Mean;
	  }
	  else {
	    ++wrongC0Mean;
	  }
	}
	else if (*d1 < meanC1) {
	  if (*c == 1) {
	    ++rightC1Mean;
	  }
	  else {
	    ++wrongC1Mean;
	  }
	}
	++d1;
	++c;
      }
      std::cerr<<"Outside Mean: ["<<meanC1<<"; "<<meanC0<<"]\n";
      std::cerr<<"wrongC0Mean="<<wrongC0Mean<<" / "<<(rightC0Mean+wrongC0Mean)<<" = "<<wrongC0Mean/(float)(rightC0Mean+wrongC0Mean)<<" ; rightC0Mean="<<rightC0Mean<<" / "<<(rightC0Mean+wrongC0Mean)<<" = "<<rightC0Mean/(float)(rightC0Mean+wrongC0Mean)<<"\n";
      std::cerr<<"wrongC1Mean="<<wrongC1Mean<<" / "<<(rightC1Mean+wrongC1Mean)<<" = "<<wrongC1Mean/(float)(rightC1Mean+wrongC1Mean)<<" ; rightC1Mean="<<rightC1Mean<<" / "<<(rightC1Mean+wrongC1Mean)<<" = "<<rightC1Mean/(float)(rightC1Mean+wrongC1Mean)<<"\n";
    }

    int rightC0Median = 0, rightC1Median = 0;
    int wrongC0Median = 0, wrongC1Median = 0;
    {
      const float *d1 = responses.ptr<float>(0);
      const int *c = responsesGT.ptr<int>(0);
      assert(responses.rows == responsesGT.rows);
      for (int i=0; i<responses.rows; ++i) {
	if (*d1 > medC0) {
	  if (*c == 0) { 
	    ++rightC0Median;
	  }
	  else {
	    ++wrongC0Median;
	  }
	}
	else if (*d1 < medC1) {
	  if (*c == 1) {
	    ++rightC1Median;
	  }
	  else {
	    ++wrongC1Median;
	  }
	}
	++d1;
	++c;
      }
      std::cerr<<"Outside Median: ["<<medC1<<"; "<<medC0<<"]\n";
      std::cerr<<"wrongC0Median="<<wrongC0Median<<" / "<<(rightC0Median+wrongC0Median)<<" = "<<wrongC0Median/(float)(rightC0Median+wrongC0Median)<<" ; rightC0Median="<<rightC0Median<<" / "<<(rightC0Median+wrongC0Median)<<" = "<<rightC0Median/(float)(rightC0Median+wrongC0Median)<<"\n";
      std::cerr<<"wrongC1Median="<<wrongC1Median<<" / "<<(rightC1Median+wrongC1Median)<<" = "<<wrongC1Median/(float)(rightC1Median+wrongC1Median)<<" ; rightC1Median="<<rightC1Median<<" / "<<(rightC1Median+wrongC1Median)<<" = "<<rightC1Median/(float)(rightC1Median+wrongC1Median)<<"\n";


      std::cerr<<"minMaxType="<<(int)minMaxType<<"\n";
    }
     




    switch (minMaxType) {
    case eMinMax:
      //nothing to do : minR/maxR already set correctly
      break;
    case eMeans:
      minR = meanC1;
      maxR = meanC0;
      break;
    case eMedians:
      minR = medC1;
      maxR = medC0;
      break;
    default:
      std::cerr<<"Error: unknow MinMaxType "<<(int)minMaxType<<"\n";
      exit(EXIT_FAILURE);
    }

    {

#if SAVE_PROBA
      {
	std::ofstream out(g_trainStatsDistFilename.c_str());
	if (! out) {
	  std::cerr<<"ERROR: unable to open output file: "<<g_trainStatsDistFilename<<"\n";
	  exit(EXIT_FAILURE);
	}
	out<<"responsesC0.size()="<<responsesC0.size()<<" wrongC0="<<wrongC0<<" / "<<responses.rows/2<<" = "<<2*wrongC0/(float)responses.rows<<"\n";
	out<<"responsesC1.size()="<<responsesC1.size()<<" wrongC1="<<wrongC1<<" / "<<responses.rows/2<<" = "<<2*wrongC1/(float)responses.rows<<"\n";
	out<<"wrongC0+wrongC1="<<wrongC0+wrongC1<<" / "<<responses.rows<<" = "<<( wrongC0+wrongC1)/(float)responses.rows<<"\n";
	out<<"medC1="<<medC1<<" ; medC0="<<medC0<<"\n";
	out<<"meanC1="<<meanC1<<" ; meanC0="<<meanC0<<"\n";
	out<<"minR="<<minR<<" ; maxR="<<maxR<<"\n";
	out<<"Outside Mean: ["<<meanC1<<"; "<<meanC0<<"]\n";
	out<<"wrongC0Mean="<<wrongC0Mean<<" / "<<(rightC0Mean+wrongC0Mean)<<" = "<<wrongC0Mean/(float)(rightC0Mean+wrongC0Mean)<<" ; rightC0Mean="<<rightC0Mean<<" / "<<(rightC0Mean+wrongC0Mean)<<" = "<<rightC0Mean/(float)(rightC0Mean+wrongC0Mean)<<"\n";
	out<<"wrongC1Mean="<<wrongC1Mean<<" / "<<(rightC1Mean+wrongC1Mean)<<" = "<<wrongC1Mean/(float)(rightC1Mean+wrongC1Mean)<<" ; rightC1Mean="<<rightC1Mean<<" / "<<(rightC1Mean+wrongC1Mean)<<" = "<<rightC1Mean/(float)(rightC1Mean+wrongC1Mean)<<"\n";
	out<<"Outside Median: ["<<medC1<<"; "<<medC0<<"]\n";
	out<<"wrongC0Median="<<wrongC0Median<<" / "<<(rightC0Median+wrongC0Median)<<" = "<<wrongC0Median/(float)(rightC0Median+wrongC0Median)<<" ; rightC0Median="<<rightC0Median<<" / "<<(rightC0Median+wrongC0Median)<<" = "<<rightC0Median/(float)(rightC0Median+wrongC0Median)<<"\n";
	out<<"wrongC1Median="<<wrongC1Median<<" / "<<(rightC1Median+wrongC1Median)<<" = "<<wrongC1Median/(float)(rightC1Median+wrongC1Median)<<" ; rightC1Median="<<rightC1Median<<" / "<<(rightC1Median+wrongC1Median)<<" = "<<rightC1Median/(float)(rightC1Median+wrongC1Median)<<"\n";
      }


      //visualization histogram des distances
      {
	int histSize = 512; //nb of bins

	float range[] = {minR, maxR};
	cv::Mat responsesC0M(responsesC0All.size(), 1, CV_32F, responsesC0All.data());
	cv::Mat responsesC1M(responsesC1All.size(), 1, CV_32F, responsesC1All.data());
	const float *histRange = {range};
	bool uniform = true;
	bool accumulate = false;
	cv::Mat histC0, histC1;
	cv::calcHist(&responsesC0M, 1, 0, cv::Mat(), histC0, 1, &histSize, &histRange, uniform, accumulate);
	cv::calcHist(&responsesC1M, 1, 0, cv::Mat(), histC1, 1, &histSize, &histRange, uniform, accumulate);
      
	//Make image
	int w = 512, h=512;
	int bin_w = cvRound( (double)w/histSize);
	cv::Mat histImage(h, w, CV_8UC3, cv::Scalar(0,0,0));
	// normalize the result to [0; histImage.rows]
	//cv::normalize(histC0, histC0, 0, histImage.rows, cv::NORM_MINMAX, -1, cv::Mat());
	//cv::normalize(histC1, histC1, 0, histImage.rows, cv::NORM_MINMAX, -1, cv::Mat());
	//normalize two histograms with same max
	{
	  assert(histC0.isContinuous());
	  assert(histC1.isContinuous());
	  float *h0 = histC0.ptr<float>(0);
	  float *h1 = histC1.ptr<float>(0);
	  //float minH = 0; 
	  float maxH = 0;
	  for (int i=0; i<histC0.rows; ++i, ++h0) 
	    if (*h0 > maxH)
	      maxH = *h0;
	  for (int i=0; i<histC1.rows; ++i, ++h1) 
	    if (*h1 > maxH)
	      maxH = *h1;
	  if (maxH > 0) {
	    const float c = histImage.rows/maxH;
	    h0 = histC0.ptr<float>(0);
	    h1 = histC1.ptr<float>(0);
	    for (int i=0; i<histC0.rows; ++i, ++h0) 
	      *h0 *= c;
	    for (int i=0; i<histC1.rows; ++i, ++h1) 
	      *h1 *= c;
	  }
	}

      

	// draw 
	for (int i=1; i<histSize; ++i) {//start from 1
	  cv::line(histImage, 
		   cv::Point(bin_w*(i-1), h - cvRound(histC0.at<float>(i-1))),
		   cv::Point(bin_w*(i), h - cvRound(histC0.at<float>(i))),
		   cv::Scalar(255, 0, 0), 2, 8, 0 );
	  cv::line(histImage, 
		   cv::Point(bin_w*(i-1), h - cvRound(histC1.at<float>(i-1))),
		   cv::Point(bin_w*(i), h - cvRound(histC1.at<float>(i))),
		   cv::Scalar(0, 255, 0), 2, 8, 0 );
	}
      
	cv::imwrite(g_trainDistHistoFilename, histImage);
      }

      std::cerr<<"\n";
#endif //SAVE_PROBA



    }
    
  }

  out << minR <<" "<<maxR<<"\n"; 

  out.close();
}

#if SAVE_PROBA

static
void
saveStatsResponses(const cv::Mat &responsesD, 
		   const cv::Mat &responsesGT, 
		   double sigmoid_A, 
		   double sigmoid_B)
{
  cv::Mat responses = responsesD.clone();
  {
    float *d1 = responses.ptr<float>(0);
    for (int i=0; i<responses.rows; ++i, ++d1)
      *d1 = sigmoid_predict(*d1, sigmoid_A, sigmoid_B);
  }
  
  
  {
    assert(responses.rows == responsesGT.rows);
    assert(responses.type() == CV_32FC1);
    assert(responsesGT.type() == CV_32SC1);

    std::vector<float> responsesC0, responsesC1, responsesC0All, responsesC1All;
    responsesC0.reserve(responses.rows);
    responsesC1.reserve(responses.rows);
    responsesC0All.reserve(responses.rows);
    responsesC1All.reserve(responses.rows);
    const float *d1 = responses.ptr<float>(0);
    const int *c = responsesGT.ptr<int>(0);
    int wrongC0 = 0, wrongC1=0;
    assert(responses.rows == responsesGT.rows);

    float minR = std::numeric_limits<float>::max();
    float maxR = -std::numeric_limits<float>::max();

    int nb0 = 0, nb1 = 0;
    int nbGood0=0, nbGood1=0;
    int nbReallyGood0=0, nbReallyGood1=0;
    const float thReallyGood = 0.65;

    for (int i=0; i<responses.rows; ++i) {

      if (*c == 1) {
	++nb1;
	if (*d1 > 0.5) {
	  ++nbGood1;
	}
	if (*d1 > thReallyGood) {
	  ++nbReallyGood1;
	}
      }
      else if (*c == 0) {
	++nb0;
	if (*d1 < 0.5) {
	  ++nbGood0;
	}
	if (*d1 < (1-thReallyGood)) {
	  ++nbReallyGood0;
	}
      }

      if (*d1 <= 0.5) {
	if (*c == 0) { 
	  responsesC0.push_back(*d1);
	}
	else {
	  ++wrongC0;
	}
	responsesC0All.push_back(*d1);
      }
      else {
	assert(*d1 > 0.5);
	if (*c == 1) {
	  responsesC1.push_back(*d1);
	}
	else {
	  ++wrongC1;
	}
	responsesC1All.push_back(*d1);
      }

      if (*d1 < minR)
	minR = *d1;
      if (*d1 > maxR)
	maxR = *d1;

      ++d1;
      ++c;
    }
  
    std::cerr<<"Probas: \n";
    std::cerr<<"responsesC0.size()="<<responsesC0.size()<<" wrongC0="<<wrongC0<<" / "<<responses.rows/2<<" = "<<2*wrongC0/(float)responses.rows<<"\n";
    std::cerr<<"responsesC1.size()="<<responsesC1.size()<<" wrongC1="<<wrongC1<<" / "<<responses.rows/2<<" = "<<2*wrongC1/(float)responses.rows<<"\n";
    std::cerr<<"wrongC0+wrongC1="<<wrongC0+wrongC1<<" / "<<responses.rows<<" = "<<( wrongC0+wrongC1)/(float)responses.rows<<"\n";

    std::cerr<<" classe0: good="<<nbGood0<<"/"<<nb0<<"="<<(nb0 > 0 ? nbGood0/(float)nb0 : 0)<<" ; really good[th="<<thReallyGood<<"]="<<nbReallyGood0<<"/"<<nb0<<"="<<(nb0 > 0 ? nbReallyGood0/(float)nb0 : 0)<<" \n";
    std::cerr<<" classe1: good="<<nbGood1<<"/"<<nb1<<"="<<(nb1 > 0 ? nbGood1/(float)nb1 : 0)<<" ; really good[th="<<thReallyGood<<"]="<<nbReallyGood1<<"/"<<nb1<<"="<<(nb1 > 0 ? nbReallyGood1/(float)nb1 : 0)<<"\n";
    std::cerr<<" total: good="<<nbGood0+nbGood1<<"/"<<nb0+nb1<<"="<<(nb0+nb1 > 0 ? (nbGood0+nbGood1)/(float)(nb0+nb1) : 0)<<" ; really good[th="<<thReallyGood<<"]="<<nbReallyGood0+nbReallyGood1<<"/"<<nb0+nb1<<"="<<(nb0+nb1 > 0 ? (nbReallyGood0+nbReallyGood1)/(float)(nb0+nb1) : 0)<<"\n";


    std::vector<float>::iterator itC0 = responsesC0.begin()+responsesC0.size()/2;
    std::nth_element(responsesC0.begin(), itC0, responsesC0.end());
    float medC0 = -1;
    if (itC0 != responsesC0.end())
      medC0 = *itC0;

    std::vector<float>::iterator itC1 = responsesC1.begin()+responsesC1.size()/2;
    std::nth_element(responsesC1.begin(), itC1, responsesC1.end());
    float medC1 = -1;
    if (itC1 != responsesC1.end())
      medC1 = *itC1;

    std::cerr<<"medC0="<<medC0<<" ; medC1="<<medC1<<"\n";
    assert(medC1>=medC0);

    float meanC0=0;
    for (std::vector<float>::iterator it=responsesC0.begin(); it!=responsesC0.end(); ++it)
      meanC0 += *it;
    if (! responsesC0.empty())
      meanC0 /= responsesC0.size();
    float meanC1=0;
    for (std::vector<float>::iterator it=responsesC1.begin(); it!=responsesC1.end(); ++it)
      meanC1 += *it;
    if (! responsesC1.empty())
      meanC1 /= responsesC1.size();
    std::cerr<<"meanC0="<<meanC0<<"; meanC1="<<meanC1<<"\n";
    assert(meanC1>=meanC0);
    std::cerr<<"minR="<<minR<<" ; maxR="<<maxR<<"\n";

    {
      std::ofstream out(g_trainStatsProbaFilename.c_str());
      if (! out) {
	std::cerr<<"ERROR: unable to open output file: "<<g_trainStatsProbaFilename<<"\n";
	exit(EXIT_FAILURE);
      }
      out<<"responsesC0.size()="<<responsesC0.size()<<" wrongC0="<<wrongC0<<" / "<<responses.rows/2<<" = "<<2*wrongC0/(float)responses.rows<<"\n";
      out<<"responsesC1.size()="<<responsesC1.size()<<" wrongC1="<<wrongC1<<" / "<<responses.rows/2<<" = "<<2*wrongC1/(float)responses.rows<<"\n";
      out<<"wrongC0+wrongC1="<<wrongC0+wrongC1<<" / "<<responses.rows<<" = "<<( wrongC0+wrongC1)/(float)responses.rows<<"\n";
      out<<"medC0="<<medC0<<" ; medC1="<<medC1<<"\n";
      out<<"meanC0="<<meanC0<<" ; meanC1="<<meanC1<<"\n";
      out<<"minR="<<minR<<" ; maxR="<<maxR<<"\n";
    }

    //visualization histogram des distances
    {
      int histSize = 512; //nb of bins

      float range[] = {0, 1};
      cv::Mat responsesC0M(responsesC0All.size(), 1, CV_32F, responsesC0All.data());
      cv::Mat responsesC1M(responsesC1All.size(), 1, CV_32F, responsesC1All.data());
      const float *histRange = {range};
      bool uniform = true;
      bool accumulate = false;
      cv::Mat histC0, histC1;
      cv::calcHist(&responsesC0M, 1, 0, cv::Mat(), histC0, 1, &histSize, &histRange, uniform, accumulate);
      cv::calcHist(&responsesC1M, 1, 0, cv::Mat(), histC1, 1, &histSize, &histRange, uniform, accumulate);
      
      //Make image
      int w = 512, h=512;
      int bin_w = cvRound( (double)w/histSize);
      cv::Mat histImage(h, w, CV_8UC3, cv::Scalar(0,0,0));
      // normalize the result to [0; histImage.rows]
      //cv::normalize(histC0, histC0, 0, histImage.rows, cv::NORM_MINMAX, -1, cv::Mat());
      //cv::normalize(histC1, histC1, 0, histImage.rows, cv::NORM_MINMAX, -1, cv::Mat());
      //normalize two histograms with same max
      {
	assert(histC0.isContinuous());
	assert(histC1.isContinuous());
	float *h0 = histC0.ptr<float>(0);
	float *h1 = histC1.ptr<float>(0);
	//float minH = 0; 
	float maxH = 0;
	for (int i=0; i<histC0.rows; ++i, ++h0) 
	  if (*h0 > maxH)
	    maxH = *h0;
	for (int i=0; i<histC1.rows; ++i, ++h1) 
	  if (*h1 > maxH)
	    maxH = *h1;
	if (maxH > 0) {
	  const float c = histImage.rows/maxH;
	  h0 = histC0.ptr<float>(0);
	  h1 = histC1.ptr<float>(0);
	  for (int i=0; i<histC0.rows; ++i, ++h0) 
	    *h0 *= c;
	  for (int i=0; i<histC1.rows; ++i, ++h1) 
	    *h1 *= c;
	}
      }

      // draw 
      for (int i=1; i<histSize; ++i) {//start from 1
	cv::line(histImage, 
		 cv::Point(bin_w*(i-1), h - cvRound(histC0.at<float>(i-1))),
		 cv::Point(bin_w*(i), h - cvRound(histC0.at<float>(i))),
		 cv::Scalar(255, 0, 0), 2, 8, 0 );
	cv::line(histImage, 
		 cv::Point(bin_w*(i-1), h - cvRound(histC1.at<float>(i-1))),
		 cv::Point(bin_w*(i), h - cvRound(histC1.at<float>(i))),
		 cv::Scalar(0, 255, 0), 2, 8, 0 );
      }
      
      //const char *n = "histogram distances";
      //cv::namedWindow(n, CV_WINDOW_AUTOSIZE);
      //cv::imshow(n, histImage);
      //cv::waitKey(0);

      cv::imwrite(g_trainProbaHistoFilename, histImage);
    }
  }
}

#endif //SAVE_PROBA


/*
  Inspired from pseudo-code available in Annexe C of 
  "A Note on Platt's Probabilistic Outputs for Support Vector Machines"
  Hsuan-Tien Lin, Chih-Jen Lin & Ruby C. Weng

  and implementation in libsvm-3.17

*/
void
computeSigmoid(const cv::Mat &responsesD, 
	       const cv::Mat &responses, 
	       double prior1,
	       double prior0,
	       double &sigmoid_A, 
	       double &sigmoid_B)
{
  assert(responsesD.type() == CV_32F);
  assert(responses.type() == CV_32S);
  assert(responses.rows == responsesD.rows);

  /*
    responsesD = deci : array of SVM decision values
    responses = label : array of labels
    prior1 = number of positive examples
    prior2 = number of negative examples

  */

  int max_iter=100;	 // Maximal number of iterations
  double min_step=1e-10; // Minimal step taken in line search
  double sigma=1e-12;	 // For numerically strict PD of Hessian
  
  const double eps = 1e-5;

  double hiTarget=(prior1+1.0)/(prior1+2.0);
  double loTarget=1/(prior0+2.0);
  
  const int len = responses.rows;
  assert(len == prior0 + prior1);
  std::vector<double> t(len);
  
  // Initial Point and Initial Fun Value
  double A=0.0; 
  double B=log((prior0+1.0)/(prior1+1.0));
  double fval = 0.0;

  const int *r = responses.ptr<int>(0);
  const float *rd = responsesD.ptr<float>(0);
  for (int i=0; i<len; ++i) {
    if (r[i] > 0)
      t[i] = hiTarget;
    else
      t[i] = loTarget;
    
    const double fApB = rd[i] * A + B;
    if (fApB >= 0)
      fval += t[i]*fApB + log(1 + exp(-fApB));
    else
      fval += (t[i] - 1)*fApB + log(1 + exp(fApB));
  }

  int iter = 0;
  for ( iter=0; iter<max_iter; ++iter) {
    // Update Gradient and Hessian (use H' = H + sigma I)
    double h11=sigma; // numerically ensures strict PD
    double h22=sigma;
    double h21=0.0;
    double g1=0.0;
    double g2=0.0;

    for (int i=0; i<len; ++i) {
      
      const double fApB = rd[i]*A + B;
      double p, q;
      if (fApB >= 0) {
	p = exp(-fApB)/(1.0+exp(-fApB));
	q = 1.0/(1.0+exp(-fApB));
      }
      else {
	p = 1.0/(1.0+exp(fApB));
	q = exp(fApB)/(1.0+exp(fApB));
      }
      const double d2 = p*q;
      h11 += rd[i]*rd[i]*d2;
      h22 += d2;
      h21 += rd[i]*d2;
      const double d1 = t[i]-p;
      g1 += rd[i]*d1;
      g2 += d1;

    }

    // Stopping Criteria
    if (fabs(g1)<eps && fabs(g2)<eps)
      break;
    
    // Finding Newton direction: -inv(H') * g
    double det = h11*h22 - h21*h21;
    double dA = -(h22*g1 - h21*g2) / det;
    double dB = -(-h21*g1 + h11*g2) / det;
    double gd = g1*dA + g2*dB;
    
    //Line search
    double stepsize = 1; 
    while (stepsize >= min_step) {
      double newA = A + stepsize * dA;
      double newB = B + stepsize * dB;

      // New function value
      double newf = 0.0;
      for (int i=0; i<len; ++i) {
	double fApB = rd[i]*newA + newB;
	if (fApB >= 0)
	  newf += t[i]*fApB + log(1+exp(-fApB));
	else
	  newf += (t[i] - 1)*fApB +log(1+exp(fApB));
      }
      
      //Check sufficient decrease
      if (newf < fval+0.0001*stepsize*gd) {
	A = newA;
	B = newB;
	fval = newf;
	break; //Sufficient decrease satisfied
      }
      else {
	stepsize = stepsize / 2.0;
      }

    }

    if (stepsize < min_step) {
      std::cerr<<"Warning: line search fails in two-class probability estimates\n";
      break;
    }

  }

  if (iter >= max_iter) {
    std::cerr<<"Warning: Reaching maximal iterations in two-class probability estimates\n";
  }

  sigmoid_A = A;
  sigmoid_B = B;

  std::cerr<<"computeSigmoid: final (iter="<<iter<<") A="<<A<<" B="<<B<<"\n";
}

void
computeSigmoid(Classifier &classifier, 
	       const cv::Mat &data, 
	       const cv::Mat &responses, 
	       double &sigmoid_A, 
	       double &sigmoid_B)
{
  assert(responses.rows == data.rows);

  assert(responses.type() == CV_32S);
  double prior0 = 0;
  double prior1 = 0;
  {
    assert(responses.isContinuous());
    const int *r = responses.ptr<int>(0);
    for (int i=0; i<responses.rows; ++i) {
      if (*r > 0)
	prior1 += 1;
      else
	prior0 += 1;
      ++r;
    }
  }


  cv::Mat responsesD(data.rows, 1, CV_32F);
  const bool returnDFVal = true;

#if ! (CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300)
  int flags = returnDFVal ? cv::ml::StatModel::RAW_OUTPUT : 0;
#endif

  classifier->predict(data, responsesD, 
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
		      returnDFVal //parallelized version, taking returnDFVal, is only available in OpenCV 2.4.x patched version
#else
		      flags
#endif
		      ); 


  computeSigmoid(responsesD, 
		 responses, 
		 prior1,
		 prior0,
		 sigmoid_A, 
		 sigmoid_B);

#if SAVE_PROBA
  saveStatsResponses(responsesD, 
		     responses, 
		     sigmoid_A, 
		     sigmoid_B);
#endif //SAVE_PROBA
}

void
saveProbaSigmoid(const std::string &outputProbaSigmoidFilename, 
		 double sigmoid_A, double sigmoid_B)
{
  std::ofstream out(outputProbaSigmoidFilename.c_str());
  if (! out) {
    std::cerr<<"ERROR: unable to open output file: "<<outputProbaSigmoidFilename<<"\n";
    exit(EXIT_FAILURE);
  }
  out << sigmoid_A << " " << sigmoid_B;
  
}

	   
void
trainAndTestSVM(const cv::Mat &trainData,
		const cv::Mat &responses, 
		const cv::Mat &trainDataAll,
		const cv::Mat &responsesAll, 
		const std::string &outputModelFilename, 
		bool autoTrain,
		bool autoTest,
		bool autoTestAll,
		const std::string &outputProbaSigmoidFilename, 
		const std::string &outputAutoTestMinMaxFilename,
		MinMaxType minMaxType, 
		const std::string &s="SVM_CSVC_POLY", 
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
		int svm_type = CvSVM::C_SVC, 
		int kernel_type=CvSVM::POLY
#else
		int svm_type = cv::ml::SVM::C_SVC, 
		int kernel_type = cv::ml::SVM::POLY
#endif
		)
{
  assert(trainData.rows > 0);
  assert(trainData.type() == CV_32F);
  assert(responses.type() == CV_32S);
  
  //const std::string s="SVM";
  assert(trainData.type() == CV_32F);

  /*
  {//DEBUG

    std::cerr<<"trainData "<<trainData.rows<<"x"<<trainData.cols<<"\n";
    for (int i=0; i<std::min(trainData.rows, 3); ++i) {
      std::cerr<<"trainData row "<<i<<"\n";
      const float *d = trainData.ptr<float>(i);
      for (int j=0; j<trainData.cols; ++j) {
	std::cerr<<d[j]<<" ";
      }
      std::cerr<<"\n";
    }
    exit(10);
  }
  */
  
  
  double t = (double)cv::getTickCount();

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
  CvSVM clas;
  Classifier svmClassifier = &clas;

  CvSVMParams params;
  params.svm_type = svm_type;
  params.kernel_type = kernel_type; //CvSVM::POLY; //CvSVM::LINEAR;
  params.degree = 0.5;
  params.gamma = 1;
  params.coef0 = 1;
  params.C = 1;
  params.nu = 0.5;
  params.p = 0;
  params.term_crit = cvTermCriteria(CV_TERMCRIT_ITER, 1000, 0.01);

#else
  Classifier svmClassifier = cv::ml::SVM::create();

  svmClassifier->setType(svm_type);
  svmClassifier->setKernel(kernel_type);
  svmClassifier->setDegree(0.5);
  svmClassifier->setGamma(1);
  svmClassifier->setCoef0(1);
  svmClassifier->setC(1);
  svmClassifier->setNu(0.5);
  svmClassifier->setP(0);
  svmClassifier->setTermCriteria(cv::TermCriteria(cv::TermCriteria::COUNT, 1000, 0.01));

#endif

  if (autoTrain) {

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
    svmClassifier->train_auto(trainData, responses, cv::Mat(), cv::Mat(), params);
#else
    //TODO: use Ptr<TrainData>
    std::cerr<<"autoTrain not handled !!!\n";
    exit(EXIT_FAILURE);
#endif

  }
  else {

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
    svmClassifier->train(trainData, responses, cv::Mat(), cv::Mat(), params);
#else
    svmClassifier->train(trainData, cv::ml::ROW_SAMPLE, responses);
#endif

  }
  
  t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();

  std::cerr<<"Time for training["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";

  svmClassifier->save(outputModelFilename.c_str());

  if (autoTest)
    doClassifierAutoTest<Classifier>(svmClassifier, s, trainData, responses);

#if SAVE_PROBA
  g_trainStatsDistFilename = outputModelFilename + "_distStats.txt";
  g_trainDistHistoFilename = outputModelFilename + "_distHisto.png";
  g_trainStatsProbaFilename = outputModelFilename + "_probaStats.txt";
  g_trainProbaHistoFilename = outputModelFilename + "_probaHisto.png";
#endif //SAVE_PROBA

  if (! outputAutoTestMinMaxFilename.empty())
    saveAutoTestMinMax(svmClassifier, trainData, responses, outputAutoTestMinMaxFilename, minMaxType);
 
  if (! outputProbaSigmoidFilename.empty()) {
    double sigmoid_A, sigmoid_B;
    computeSigmoid(svmClassifier, trainData, responses, sigmoid_A, sigmoid_B);
    saveProbaSigmoid(outputProbaSigmoidFilename, sigmoid_A, sigmoid_B);
  }    


  if (autoTestAll) {
    std::cerr<<"\n\n*** AutoTestAll: ***\n";
    doClassifierAutoTest<Classifier>(svmClassifier, s, trainDataAll, responsesAll);
  }

}



#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300

void
trainAndTestGBT(const cv::Mat &trainData, const cv::Mat &responses, 
		const std::string &outputModelFilename, 
		bool autoTest)
{
  const std::string s="GBT";

  double t = (double)cv::getTickCount();

  CvGBTrees gbtrees;
  
  cv::Mat var_types( 1, trainData.cols + 1, CV_8UC1, cv::Scalar(CV_VAR_ORDERED) );
  var_types.at<unsigned char>( trainData.cols ) = CV_VAR_CATEGORICAL;
  
  CvGBTreesParams params( CvGBTrees::DEVIANCE_LOSS, // loss_function_type
			  100, // weak_count //B: gros impact sur le temps de calcul
			  0.8f, //0.1f, // shrinkage
			  1.0f, // subsample_portion
			  3, //2, // max_depth //B: que jusqu'à 4 ? 
			  false // use_surrogates )
			  );

  gbtrees.train( trainData, CV_ROW_SAMPLE, responses, cv::Mat(), cv::Mat(), var_types, cv::Mat(), params );

  t = ((double)cv::getTickCount() - t)/cv::getTickFrequency();
  std::cerr<<"Time for training["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";

  gbtrees.save(outputModelFilename.c_str());

  if (autoTest)
    doClassifierAutoTest<CvGBTrees>(gbtrees, s, trainData, responses);
  
}
#endif //CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
void
trainAndTestRF(const cv::Mat &trainData, const cv::Mat &responses, 
	       bool autoTest)
{
  const std::string s="RF";

  double t = (double)cv::getTickCount();

  CvRTrees rtrees;
  CvRTParams params( 4, // max_depth,
		     2, // min_sample_count,
		     0.f, // regression_accuracy,
		     false, // use_surrogates,
		     16, // max_categories,
		     0, // priors,
		     false, // calc_var_importance,
		     1, // nactive_vars,
		     5, // max_num_of_trees_in_the_forest,
		     0, // forest_accuracy,
		     CV_TERMCRIT_ITER // termcrit_type
		     );
  
  rtrees.train( trainData, CV_ROW_SAMPLE, responses, cv::Mat(), cv::Mat(), cv::Mat(), cv::Mat(), params );
  
  std::cerr<<"Time for training["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";

  if (autoTest)
    doClassifierAutoTest<CvRTrees>(rtrees, s, trainData, responses);

}
#endif //CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300


#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
void
trainAndTestERT(const cv::Mat &trainData, const cv::Mat &responses, 
		bool autoTest)
{
  const std::string s="ERT";

  double t = (double)cv::getTickCount();

  CvERTrees ertrees;
  
  cv::Mat var_types( 1, trainData.cols + 1, CV_8UC1, cv::Scalar(CV_VAR_ORDERED) );
  var_types.at<unsigned char>( trainData.cols ) = CV_VAR_CATEGORICAL;
  
  CvRTParams params( 4, // max_depth,
		     2, // min_sample_count,
		     0.f, // regression_accuracy,
		     false, // use_surrogates,
		     16, // max_categories,
		     0, // priors,
		     false, // calc_var_importance,
		     1, // nactive_vars,
		     5, // max_num_of_trees_in_the_forest,
		     0, // forest_accuracy,
		     CV_TERMCRIT_ITER // termcrit_type
		     );
  
  ertrees.train( trainData, CV_ROW_SAMPLE, responses, cv::Mat(), cv::Mat(), var_types, cv::Mat(), params );
  
  std::cerr<<"Time for training["<<s<<"]: "<<t<<"s on "<<trainData.rows<<" samples\n";

  if (autoTest)
    doClassifierAutoTest<CvERTrees>(ertrees, s, trainData, responses);


}
#endif //CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300



namespace {

  class AnnotationAccumulator
  {
    //typedef std::vector<LConcept> LConceptVector;
    LConceptVector m_concepts;

    ColorSpace m_cs;
    int m_patchSize;
    PatchType m_patchType;
    bool m_useL;
    bool m_doResize;
    int m_smallSize;
    bool m_balance;

    LConceptVV m_conceptsToMergeV;

  public:


    void readMergeFilenames(std::vector<std::string> &mergeFilenames)
    {
      const size_t nbMerges = mergeFilenames.size();
      m_conceptsToMergeV.reserve(nbMerges);
      for (size_t i=0; i<nbMerges; ++i) {
	LConceptVector conceptsToMerge;
	const bool readWithFloat = true;
	readConceptsFromCSV(mergeFilenames[i], conceptsToMerge, readWithFloat);

	const size_t nbc = conceptsToMerge.size();

	//check that we have more than one concept
	if (nbc == 0) {
	  std::cerr<<"ERROR: merge file: "<<mergeFilenames[i]<<" lists no concept\n";
	  exit(EXIT_FAILURE);
	}
	if (nbc == 1) {
	  //std::cerr<<"ERROR: merge file: "<<mergeFilenames[i]<<" lists only one concept\n";
	  //exit(EXIT_FAILURE);
	  std::cerr<<"Warning: merge file: "<<mergeFilenames[i]<<" lists only one concept\n"; //this can be on purpose : to keep this concept !
	}
	assert(nbc >= 1);
      
	//check that specified percentages sum to less than 1
	float sum = 0;
	for (size_t j=0; j<nbc; ++j) {
	  const float p = conceptsToMerge[j].percent;
	  assert(p >= 0 && p <=1);
	  sum += p;
	}
	if (sum > 1) {
	  std::cerr<<"ERROR: merge file: "<<mergeFilenames[i]<<": concepts sum to more than 1\n";
	  exit(EXIT_FAILURE);
	}
	assert(sum>=0 && sum<=1);

	//check that concepts are in m_concepts
	//Warning: we just check the color (not that they have the same name)
	for (size_t j=0; j<nbc; ++j) {
	  const LConcept &c = conceptsToMerge[j];
	  MapType::iterator itC = findConceptByColor(c.color);
	  if (itC == m_conceptsPixels.end()) {
	  
	    std::cerr<<"ERROR: merge file: "<<mergeFilenames[i]<<": concept "<<c.name<<" with color=["<<(int)c.color[0]<<", "<<(int)c.color[1]<<", "<<(int)c.color[2]<<"] is not in considered concepts\n";
	    {
	      for (MapType::iterator it = m_conceptsPixels.begin();
		   it != m_conceptsPixels.end();
		   ++it) {
		std::cerr<<it->first.name<<" ["<<(int)(it->first.color[0])<<", "<<(int)(it->first.color[1])<<", "<<(int)(it->first.color[2])<<"] : accumulated "<<it->second.rows<<"\n";
	      
	      }   
	      std::cerr<<"\n";
	    }
	    exit(EXIT_FAILURE);
	  }
	}


	m_conceptsToMergeV.push_back(conceptsToMerge);
      }
      assert(m_conceptsToMergeV.size() <= nbMerges);


    }

    AnnotationAccumulator(const std::string &conceptsFilename, bool readConcepts,
			  ColorSpace cs, int patchSize, PatchType patchType, bool useL, 
			  bool doResize, int smallSize, 
			  bool balance, 
			  std::vector<std::string> &mergeFilenames)
      : m_cs(cs), m_patchSize(patchSize), m_patchType(patchType), m_useL(useL), 
	m_doResize(doResize), m_smallSize(smallSize),
	m_balance(balance)
    {
      if (readConcepts) {
	readConceptsFromCSV(conceptsFilename, m_concepts);
	//std::cerr<<"Nb concepts="<<m_concepts.size()<<"\n\n";
      
	//fill map with empty matrix for each concept
	const int cols = getCols();
	for (LConceptVector::const_iterator it = m_concepts.begin();
	     it != m_concepts.end();
	     ++it) {
	
	  const LConcept &lc = *it;
	
	  if (m_conceptsPixels.find(lc) != m_conceptsPixels.end()) {
	    std::cerr<<"Error: concept with color="<<lc.color<<" & name="<<lc.name<<" is present twice in annotation file: "<<conceptsFilename<<"\n";
	    exit(EXIT_FAILURE);
	  }
      
	  m_conceptsPixels[lc] = cv::Mat(0, cols, CV_32F);
	}    
      
	readMergeFilenames(mergeFilenames); //must be done after initialization of m_conceptsPixels
      }


    }

    int getCols() const 
    {
      return getLABsSize(m_patchSize, m_patchType, m_useL);
    }
  
    void removeConcept(const std::string &conceptName)
    {
      for (MapType::iterator it = m_conceptsPixels.begin();
	   it != m_conceptsPixels.end();
	   ++it) {

	if (it->first.name == conceptName) {
	  it->second = cv::Mat(0, getCols(), CV_32F);  //We cannot remove a concept from the list (it would change the classe numbers !!!)
	  return;
	}

      }

      //std::cerr<<"Warning: no concept with name="<<conceptName<<" found\n";
      //std::cerr<<"ERROR: no concept with name="<<conceptName<<" found\n";
      //exit(EXIT_FAILURE);
    }

    void reset() 
    {
      const int cols = getCols();
      for (MapType::iterator it = m_conceptsPixels.begin();
	   it != m_conceptsPixels.end();
	   ++it) {
	it->second = cv::Mat(0, cols, CV_32F);
      }    
    }

										
    void readData(const std::string &inputDataFilename)
    {
      //reset all matrices (in case we do not read all concepts in file)
      reset();

      std::ifstream in(inputDataFilename.c_str(), std::ios::in|std::ios::binary);
      if (! in) {
	std::cerr<<"Error: unable to open file: "<<inputDataFilename<<"\n";
	exit(EXIT_FAILURE);
      }
      int nbConcepts = 0;
      in.read((char *)&nbConcepts, sizeof(nbConcepts));
      for (int i=0; i<nbConcepts; ++i) {
	int len;
	in.read((char *)&len, sizeof(int)); 
	std::vector<char> sv(len);
	in.read((char *)&sv[0], len*sizeof(char));
	std::string name(&sv[0], len);
      
	cv::Vec3b color;
	in.read((char *)&color, sizeof(cv::Vec3b));
      
	LConcept lc(name, color);

	int rows=0, cols=0;
	in.read((char *)&rows, sizeof(rows));
	in.read((char *)&cols, sizeof(cols));
      
	//std::cerr<<"readData name="<<name<<" rows="<<rows<<" cols="<<cols<<"\n";

	cv::Mat m;
	m.create(rows, cols, CV_32F);
	assert(m.isContinuous());
	const size_t size = rows*cols*sizeof(float);
	in.read((char *)m.ptr<float>(0), size);

	MapType::iterator it = m_conceptsPixels.find(lc);
	if (it == m_conceptsPixels.end()) {
	  std::cerr<<"ERROR: concept ["<<lc.name<<", ("<<lc.color[0]<<", "<<lc.color[1]<<", "<<lc.color[2]<<")] found in input data file: "<<inputDataFilename<<" did not exist in read concepts\n";
	  exit(10);
	}
	else {
	  it->second = m;
	}
      }
    
    }

    void writeData(const std::string &outputDataFilename) const
    {
      std::ofstream out(outputDataFilename.c_str(), std::ios::out|std::ios::binary);
      if (! out) {
	std::cerr<<"Error: unable to open file: "<<outputDataFilename<<"\n";
	exit(EXIT_FAILURE);
      }
    
      int nbConcepts = m_conceptsPixels.size();
      out.write((const char *)&nbConcepts, sizeof(nbConcepts));
      for (MapType::const_iterator it = m_conceptsPixels.begin();
	   it != m_conceptsPixels.end();
	   ++it) {
	const LConcept &lc = it->first;

	int len = lc.name.length();
	out.write((const char *)&len, sizeof(int)); 
	out.write((const char *)lc.name.c_str(), len*sizeof(char));
      
	out.write((const char *)&lc.color, sizeof(cv::Vec3b));
      
	const cv::Mat &m = it->second;

	int rows=m.rows, cols=m.cols;
	out.write((const char *)&rows, sizeof(rows));
	out.write((const char *)&cols, sizeof(cols));
	assert(CV_MAKETYPE(cv::DataDepth<float>::value, 1) == m.type()); 
	assert(m.isContinuous());
	const size_t size = m.rows*m.cols;
	out.write((const char *)m.ptr<float>(0), size*sizeof(float));
      }
    
    }


    void process(const std::string &imageFilename, 
		 const std::string &imageLabelFilename)
    {
      //std::cerr<<"image="<<imageFilename<<"\n";

      cv::Mat img = cv::imread(imageFilename);
      if (img.empty()) {
	std::cerr<<"Error: unable to load image file: "<<imageFilename<<"\n";
	exit(EXIT_FAILURE);
      }
      cv::Mat labelImg = cv::imread(imageLabelFilename);
      if (labelImg.empty()) {
	std::cerr<<"Error: unable to load image file: "<<imageLabelFilename<<"\n";
	exit(EXIT_FAILURE);
      }

      if (img.cols != labelImg.cols || img.rows != labelImg.rows) {
	std::cerr<<"Error: original image "<<imageFilename<<" and label image "<<imageLabelFilename<<" have not the same size: ("<<img.cols<<"x"<<img.rows<<") vs ("<<labelImg.cols<<"x"<<labelImg.rows<<")\n";
	exit(EXIT_FAILURE);
      }
    
      bool removeSaturated = true;
      float lSaturatedThreshold = 84; //threhsold on "L" of Lab above which patches are discarded if removeStaurated==true
      if (m_cs != 0) {
	removeSaturated = false;  //TODO: => what would be a good threshold ???
	lSaturatedThreshold = -1000;
      }

      if (m_doResize) {
	//Reduce size of img !!!                                                                                                                                                                    
	int dstLargeSide = m_smallSize;
	int srcLargeSide = img.cols;
	int srcSmallSide = img.rows;
	if (img.rows > img.cols) {
	  srcLargeSide = img.rows;
	  srcSmallSide = img.cols;
	}
	int dstSmallSide = static_cast<int>(srcSmallSide*dstLargeSide/(float)srcLargeSide + 0.5);
	if (img.rows > img.cols)
	  std::swap(dstLargeSide, dstSmallSide);

	cv::Mat imgSmall, labelImgSmall;

	cv::resize(img, imgSmall, cv::Size(dstLargeSide, dstSmallSide), 0.0, 0.0, cv::INTER_AREA);
	cv::resize(labelImg, labelImgSmall, cv::Size(dstLargeSide, dstSmallSide), 0.0, 0.0, cv::INTER_NEAREST);


	accumulateImagePatch(imgSmall, labelImgSmall, m_conceptsPixels, m_cs, m_patchSize, m_patchType, lSaturatedThreshold, m_useL, removeSaturated);
      }
      else {
	accumulateImagePatch(img, labelImg, m_conceptsPixels, m_cs, m_patchSize, m_patchType, lSaturatedThreshold, m_useL, removeSaturated);
      }

    } 


    void process(const std::string &trainFilename, int maxNbSamples) 
    {
      std::ifstream in(trainFilename.c_str());
      if (! in) {
	std::cerr<<"Error: unable to open file: "<<trainFilename<<"\n";
	exit(EXIT_FAILURE);
      }
      std::string imageFilename, imageLabelFilename;
      size_t nbImages=0;
      while (in.good() && ! in.eof()) {
      
	std::string line;
	std::getline(in, line);
	std::stringstream ss;
	ss<<line;
      
	ss >> imageFilename;
	if (ss.bad() || ss.eof())
	  break;
	ss >> imageLabelFilename;
	if (ss.bad()) {
	  std::cerr<<"Error: while reading input file: "<<trainFilename<<"\n";
	  exit(EXIT_FAILURE);
	}
      
	process(imageFilename, imageLabelFilename);
      
	++nbImages;
      }
      //std::cerr<<nbImages<<" images\n\n";

      {
	std::cerr<<"before merge maxNbSamples="<<maxNbSamples<<"\n";
	for (MapType::iterator it = m_conceptsPixels.begin();
	     it != m_conceptsPixels.end();
	     ++it) {
	  std::cerr<<it->first.name<<" ["<<(int)(it->first.color[0])<<", "<<(int)(it->first.color[1])<<", "<<(int)(it->first.color[2])<<"] : accumulated "<<it->second.rows<<"\n";
	
	}   
	std::cerr<<"\n";
      }

      mergeConcepts(maxNbSamples);

      {
	std::cerr<<"after merge maxNbSamples="<<maxNbSamples<<"\n";
	for (MapType::iterator it = m_conceptsPixels.begin();
	     it != m_conceptsPixels.end();
	     ++it) {
	  std::cerr<<it->first.name<<" ["<<(int)(it->first.color[0])<<", "<<(int)(it->first.color[1])<<", "<<(int)(it->first.color[2])<<"] : accumulated "<<it->second.rows<<"\n";
	
	}   
	std::cerr<<"\n";
      }

      subsample(maxNbSamples);
    }

    MapType::iterator findConceptByColor(const cv::Vec3b &color)
    {
      MapType::iterator itR = m_conceptsPixels.end();
      for (MapType::iterator it = m_conceptsPixels.begin();
	   it != m_conceptsPixels.end();
	   ++it) {
	if (it->first.color == color) {
	  itR = it;
	  break;
	}
      }
      return itR;
    }

    void mergeConcepts(MapType::iterator itKept, MapType::iterator itDel)
    {
      assert(itKept != m_conceptsPixels.end());
      if (itDel != m_conceptsPixels.end()) {
	//std::cerr<<"merge itKept->second.rows="<<itKept->second.rows<<" itDel->second.rows="<<itDel->second.rows<<" => ";
	itKept->second.push_back(itDel->second);
	itDel->second.resize(0);
	//std::cerr<<itKept->second.rows<<"\n";
	assert(itDel->second.rows == 0);
	m_conceptsPixels.erase(itDel);
      }
    
    }

    void mergeConcepts(MapType::iterator itKept, cv::Vec3b delColor, const std::string &s)
    {
      assert(itKept != m_conceptsPixels.end());
      MapType::iterator itDel = findConceptByColor(delColor);
      //if (itDel == m_conceptsPixels.end()) { std::cerr<<"warning: concept "<<s<<" not found\n"; }
      mergeConcepts(itKept, itDel);
    }

    void mergeConcepts(int maxNbSample)
    {
      int lMaxNbSample = maxNbSample;

      for (LConceptVV::const_iterator it = m_conceptsToMergeV.begin();
	   it != m_conceptsToMergeV.end();
	   ++it) {
	const LConceptVector &conceptsToMerge = *it;
	assert(conceptsToMerge.size()>=1);
	if (conceptsToMerge.size()>1) {

	  LConceptVector::const_iterator itKeepM = conceptsToMerge.begin(); 
	  MapType::iterator itKeep = findConceptByColor(itKeepM->color);

	  float sum = 0;
	  //float minNbElts = std::numeric_limits<float>::max(); //minimum nb of elements for concept for which p > 0;
	  //float minNbEltsP = 0;
	  for (LConceptVector::const_iterator itDelM = conceptsToMerge.begin()+1;
	       itDelM != conceptsToMerge.end();
	       ++itDelM) {
	    const float p = itDelM->percent;
	    sum += p;
	    if (p > 0) {
	      MapType::iterator itC = findConceptByColor(itDelM->color);
	      assert(itC != m_conceptsPixels.end());
	      const int nbElts = itC->second.rows;

	      const float maxNbSampleC = nbElts/p; //maxNbSample specific for this concept at this percentage
	      lMaxNbSample = std::min(lMaxNbSample*p, maxNbSampleC);
	  
	      /*
		if (nbElts < minNbElts) {
		minNbElts = nbElts;
		minNbEltsP = p;
		}
	      */
	  
	    }
	  }
	  if (sum == 0.f) {
	    //no percentage to garantee

	    for (LConceptVector::const_iterator itDelM = conceptsToMerge.begin()+1;
		 itDelM != conceptsToMerge.end();
		 ++itDelM) {
	      mergeConcepts(itKeep, itDelM->color, itDelM->name);
	    }
	  }
	  else {
	    assert(sum > 0.f && sum <= 1);
	
	    //garantees on merged concepts
	    //Warning: percentage of kept concept is NOT taken into account.

	    //WARNING: If final kept concepts are rally unbalanced
	    // (that is lMaxNbSamples < maxNbSamples in subsample() )
	    // there is no real garantee !!!

	    //WARNING: If a percentage is specified, we keep exactly this percentage !!!
	    //Thus if a high percentage is applied to a concept with a low number of elements, it may change lMaxNbSamples and then lead to unbalanced concepts later, and thus void garantee on percentages...


	    if (lMaxNbSample != maxNbSample) {
	      assert(lMaxNbSample < maxNbSample);
	      std::cerr<<"max nb sample reduced for merge : from "<<maxNbSample<<" to "<<lMaxNbSample<<"\n";
	    }

	    float p0 = 1-sum;
	    const int nbElts0Final = p0*lMaxNbSample;
	    subsampleRows(itKeep->second, nbElts0Final, itKeep->second);
	    int sumNbElts = itKeep->second.rows;
	    for (LConceptVector::const_iterator itDelM = conceptsToMerge.begin()+1;
		 itDelM != conceptsToMerge.end();
		 ++itDelM) {
	      const float p = itDelM->percent;
	      const int nbEltsFinal = p*lMaxNbSample;
	      MapType::iterator itD = findConceptByColor(itDelM->color);
	      assert(itD != m_conceptsPixels.end());
	      subsampleRows(itD->second, nbEltsFinal, itD->second);
	      sumNbElts += itD->second.rows;
	      mergeConcepts(itKeep, itD);
	    }
	
	    assert(sumNbElts <= lMaxNbSample);


	  }
	

	}

      }

      if (! m_conceptsToMergeV.empty()) {
	//At least one merge file was specified
	//=> We remove all concepts that to not appear in any merge file as a concept to keep (i.e., as first concept).
	for (MapType::iterator it = m_conceptsPixels.begin();
	     it != m_conceptsPixels.end();
	     ) { //no increment
	  bool toKeep = false;
	  for (LConceptVV::const_iterator itV = m_conceptsToMergeV.begin();
	       itV != m_conceptsToMergeV.end();
	       ++itV) {
	    const LConceptVector &conceptsToMerge = *itV;
	    assert(conceptsToMerge.size()>=1);
	    //just check the first one
	    LConceptVector::const_iterator itKeepM = conceptsToMerge.begin();
	    if (itKeepM->color == it->first.color) {
	      toKeep = true;
	      break;
	    }
	  }
	  if (toKeep) 
	    ++it;
	  else 
	    it = m_conceptsPixels.erase(it);
	}
      } 
    
    }

    void subsample(int maxNbSamples)
    {
      int lMaxNbSamples = maxNbSamples;
      if (m_balance) {
	int minRows = std::numeric_limits<int>::max();
	for (MapType::const_iterator it = m_conceptsPixels.begin();
	     it != m_conceptsPixels.end();
	     ++it) {
	  const cv::Mat &mat = it->second;
	  if (mat.rows < minRows)
	    minRows = mat.rows;
	}
	lMaxNbSamples = std::min(maxNbSamples, minRows);

	if (lMaxNbSamples < maxNbSamples && ! m_conceptsToMergeV.empty()) {
	  std::cerr<<"*** WARNING: As final concepts are unbalanced we need to subsample. The specified percentages for merged concepts will not be honored\n!!!";
	}
      
      }

      for (MapType::const_iterator it = m_conceptsPixels.begin();
	   it != m_conceptsPixels.end();
	   ++it) {
	cv::Mat matSS;
	subsampleRows(it->second, lMaxNbSamples, matSS);
	m_conceptsPixelsSub[it->first] = matSS;
      }
    }

    void trainAndTest(ClassifierType classifierType,
		      const std::string &inputDataFilenameLibSVM,
		      const std::string &outputModelFilename,
		      const std::string &outputDataFilenameLibSVM,
		      const std::string &outputDataFilenameCSV,
		      const std::string & outputNormalizeFilename, 
		      bool autoTrain, 
		      bool autoTest, //auto-test on train data (that is sub-sampled data)
		      bool autoTestAll, //auto-test on all data (that is without sub-sampling)
		      const std::string &outputProbaSigmoidFilename, 
		      const std::string &outputAutoTestMinMaxFilename,
		      MinMaxType minMaxType) const
    {
      //trainData is form sub-sampled data
      //trainDataAll is all available data
      
      cv::Mat trainData(0, getCols(), CV_32FC1);
      cv::Mat responses(0, 1, CV_32SC1);
      cv::Mat trainDataAll(0, getCols(), CV_32FC1);
      cv::Mat responsesAll(0, 1, CV_32SC1);


      int classe = 0;
      if (inputDataFilenameLibSVM.empty()) {

	int nbNonEmptyClasses = 0;
      
	//TODO: ne pas prendre en compte le default concept (background) : car très bruité ???

	//std::cerr<<"m_conceptsPixels.size()="<<m_conceptsPixels.size()<<"\n";

	//REM: we need order here for class incrementation !!!! (cannot traverse unordered_map  m_conceptsPixels)
	for (LConceptVector::const_iterator it=m_concepts.begin();
	     it != m_concepts.end();
	     ++it) {

	  MapType::const_iterator itP = m_conceptsPixelsSub.find(*it);
	  if (itP != m_conceptsPixelsSub.end()) { //merge can remove concepts... 

	    const cv::Mat &mat = itP->second;
	
	    trainData.push_back(mat);
	    assert(trainData.type() == CV_32F);
	
	    cv::Mat resps(mat.rows, 1, CV_32SC1);
	    resps = classe; //assign to all members
	
	    responses.push_back(resps);
	
	    if (mat.rows > 0) {
	      std::cerr<<"concept: "<<it->name<<" rows="<<mat.rows<<"\n";
	      ++nbNonEmptyClasses;
	    }

	    if (autoTestAll) {
	      MapType::const_iterator itPa = m_conceptsPixels.find(*it);
	      assert(itPa != m_conceptsPixels.end());
	      const cv::Mat &matA = itPa->second;
	      trainDataAll.push_back(matA);

	      cv::Mat respsA(matA.rows, 1, CV_32SC1);
	      respsA = classe; //assign to all members
	      responsesAll.push_back(respsA);
	    }

	    ++classe;
	  }

	}

	if (trainData.rows == 0) {
	  std::cerr<<"Error: classifierTrainLAB_J_patch: No train data !\n";
	  exit(EXIT_FAILURE);
	}
	if (nbNonEmptyClasses == 1) {
	  std::cerr<<"Error: classifierTrainLAB_J_patch: data only for one class !\n";
	  exit(EXIT_FAILURE);
	}
	assert(! (autoTestAll && trainDataAll.rows == 0));
	assert(! autoTestAll || trainDataAll.type() == CV_32F);
	assert(! autoTestAll || responsesAll.type() == CV_32S);
	assert(! autoTestAll || trainData.cols == trainDataAll.cols);
	assert(! autoTestAll || trainData.rows <= trainDataAll.rows);
	assert(! autoTestAll || trainDataAll.rows == responsesAll.rows);
	if (autoTestAll) {
	  if (trainDataAll.rows != responsesAll.rows) {
	    std::cerr<<"ERROR: trainDataAll.rows="<<trainDataAll.rows<<" != responsesAll.rows="<<responsesAll.rows<<"\n";
	  }
	}
	
	/*
	//Boris: 
	//OpenCV 2.4.6
	//It seems that if nb of elements per class is not balanced 
	// training of CvSVM failed (it terminates very quickly) and when we attempt to save such a model it core-dumps !!!
	*/

      }
      else {

	readFileLibSVM(inputDataFilenameLibSVM.c_str(), trainData, responses);

	assert(responses.type() == CV_32S && responses.cols == 1);
	const int *r = responses.ptr<int>(0);
	classe = 0;
	for (int i=0; i<responses.rows; ++i) {
	  if (*r > classe)
	    classe = *r;
	  ++r;
	}
	classe += 1;

	autoTestAll = false;
      }
    
      cv::Mat trainDataP = trainData;

      if (! outputDataFilenameLibSVM.empty())
	saveLibSVMFormat(trainDataP, responses, outputDataFilenameLibSVM);
      if (! outputDataFilenameCSV.empty())
	saveCSVFormat(trainDataP, responses, outputDataFilenameCSV);

      if (! outputNormalizeFilename.empty()) {
	std::vector<float> minV, maxV;
	normalizeMatAndGetMinsMaxs(trainDataP, minV, maxV);
	writeMinMaxs(outputNormalizeFilename, minV, maxV);

	assert(minV.size() == maxV.size());
	for (size_t i=0; i<minV.size(); ++i) {
	  assert(minV[i] != maxV[i]);
	}
	
	if (autoTestAll) {
	  P_normalizeMat(minV, maxV, trainDataAll);
	}
      }
    
      //assert(m_concepts.size() == m_conceptsPixels.size());
      assert(trainData.type() == CV_32F);


#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
      int SVM_type_C_SVC = CvSVM::C_SVC; 
      int SVM_type_NU_SVC = CvSVM::NU_SVC; 
      int SVM_kernel_type_POLY = CvSVM::POLY;
      int SVM_kernel_type_RBF = CvSVM::RBF;
      int SVM_kernel_type_SIGMOID = CvSVM::SIGMOID;
#else
      int SVM_type_C_SVC = cv::ml::SVM::C_SVC; 
      int SVM_type_NU_SVC = cv::ml::SVM::NU_SVC; 
      int SVM_kernel_type_POLY = cv::ml::SVM::POLY;
      int SVM_kernel_type_RBF = cv::ml::SVM::RBF;
      int SVM_kernel_type_SIGMOID = cv::ml::SVM::SIGMOID;
#endif //CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300

      switch (classifierType) {
      case SVM_CSVC_POLY:
	trainAndTestSVM(trainDataP, responses, trainDataAll, responsesAll, outputModelFilename, autoTrain, autoTest, autoTestAll, outputProbaSigmoidFilename, outputAutoTestMinMaxFilename, minMaxType, "SVM_CSVC_POLY", SVM_type_C_SVC, SVM_kernel_type_POLY);
	break;
      case SVM_NUSVC_POLY:
	trainAndTestSVM(trainDataP, responses, trainDataAll, responsesAll, outputModelFilename, autoTrain, autoTest, autoTestAll, outputProbaSigmoidFilename, outputAutoTestMinMaxFilename, minMaxType, "SVM_NUSVC_POLY", SVM_type_NU_SVC, SVM_kernel_type_POLY);
	break;
      case SVM_CSVC_RBF:
	trainAndTestSVM(trainDataP, responses, trainDataAll, responsesAll, outputModelFilename, autoTrain, autoTest, autoTestAll, outputProbaSigmoidFilename, outputAutoTestMinMaxFilename, minMaxType, "SVM_CSVC_RBF", SVM_type_C_SVC, SVM_kernel_type_RBF);
	break;
      case SVM_NUSVC_RBF:
	trainAndTestSVM(trainDataP, responses, trainDataAll, responsesAll, outputModelFilename, autoTrain, autoTest, autoTestAll, outputProbaSigmoidFilename, outputAutoTestMinMaxFilename, minMaxType, "SVM_NUSVC_RBF", SVM_type_NU_SVC, SVM_kernel_type_RBF);
	break;
      case SVM_CSVC_SIGMOID:
	trainAndTestSVM(trainDataP, responses, trainDataAll, responsesAll, outputModelFilename, autoTrain, autoTest, autoTestAll, outputProbaSigmoidFilename, outputAutoTestMinMaxFilename, minMaxType, "SVM_CSVC_SIGMOID", SVM_type_C_SVC, SVM_kernel_type_SIGMOID);
	break;
      case SVM_NUSVC_SIGMOID:
	trainAndTestSVM(trainDataP, responses, trainDataAll, responsesAll, outputModelFilename, autoTrain, autoTest, autoTestAll, outputProbaSigmoidFilename, outputAutoTestMinMaxFilename, minMaxType, "SVM_NUSVC_SIGMOID", SVM_type_NU_SVC, SVM_kernel_type_SIGMOID);
	break;
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
      case GBT:
	trainAndTestGBT(trainDataP, responses, outputModelFilename, autoTest);
	break;
      case RF:
	trainAndTestRF(trainDataP, responses, autoTest);
	break;
      case ERT:
	trainAndTestERT(trainDataP, responses, autoTest);
	break;
      case NBC:
      case DT:
      case ANN:
      case EM:
      case KNN:
	std::cerr<<"Error: unhandled classifier !\n";
	break;
#endif //CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 300
      default:
	std::cerr<<"Error: unknow classifier !\n";
	break;
      }

    }

  protected:

    MapType m_conceptsPixels; //merged concepts
    MapType m_conceptsPixelsSub; //merged concepts subsampled 
    
  };

} //end anonymous namespace


int
main(int argc, char *argv[])
{
  srand(time(NULL));

  std::string trainFilename, conceptFilename, inputDataFilename, inputDataFilenameLibSVM, outputDataFilename, outputDataFilenameLibSVM, outputDataFilenameCSV, outputModelFilename, outputNormalizeFilename;
  size_t maxNbSamples = DEFAULT_MAX_NB_SAMPLES;
  ColorSpace cs = DEFAULT_COLORSPACE;
  int patchSize = DEFAULT_PATCH_SIZE;
  PatchType patchType = DEFAULT_PATCH_TYPE;
  ClassifierType classifierType = DEFAULT_CLASSIFIER;
  bool autoTrain = DEFAULT_AUTOTRAIN;
  bool autoTest = DEFAULT_AUTOTEST;
  bool autoTestAll = DEFAULT_AUTOTEST_ALL;
  bool useBackground = DEFAULT_USE_BACKGROUND;
  bool use_L = DEFAULT_USE_L;
  bool doResize = DEFAULT_RESIZE_IMAGES;
  int smallSize = DEFAULT_SMALL_SIZE;
  bool balance = DEFAULT_BALANCE;
  std::vector<std::string> mergeFilenames;
  std::string outputProbaSigmoidFilename;
  std::string outputAutoTestMinMaxFilename;
  MinMaxType minMaxType = DEFAULT_MINMAX_TYPE;

  processArgs(argc, argv, trainFilename, conceptFilename, inputDataFilename, inputDataFilenameLibSVM, outputDataFilename, outputDataFilenameLibSVM, outputDataFilenameCSV, outputModelFilename, outputNormalizeFilename, mergeFilenames, maxNbSamples, cs, patchSize, patchType, classifierType, autoTrain, autoTest, autoTestAll, outputProbaSigmoidFilename, outputAutoTestMinMaxFilename, minMaxType, useBackground, use_L, doResize, smallSize, balance);

  const bool readConcepts = inputDataFilenameLibSVM.empty();

  AnnotationAccumulator aa(conceptFilename, readConcepts, cs, patchSize, patchType, use_L, doResize, smallSize, balance, mergeFilenames);

  if (! useBackground) {
    aa.removeConcept(BACKGROUND_STR);
    aa.removeConcept(BACKGROUND_STR2);
    //aa.removeConcept("epithelialisation");
  }

  if (! inputDataFilename.empty()) {

    aa.readData(inputDataFilename);

  }
  else if (inputDataFilenameLibSVM.empty()) {
    
    aa.process(trainFilename, maxNbSamples);
  }

  if (! outputDataFilename.empty())
    aa.writeData(outputDataFilename);
  

 
  aa.trainAndTest(classifierType, inputDataFilenameLibSVM, outputModelFilename, outputDataFilenameLibSVM, outputDataFilenameCSV, outputNormalizeFilename, autoTrain, autoTest, autoTestAll, outputProbaSigmoidFilename, outputAutoTestMinMaxFilename, minMaxType);


  return 0;
}
