#include <queue>

class Splitter : public girl::toger::sameRegion
{
  const cv::Mat &m_imgP;

public:
  explicit Splitter(const cv::Mat &imgP)
    : m_imgP(imgP)
  {
    assert(imgP.type() == CV_8UC1);
  }

  virtual bool same(const girl::point &p, girl::direction dir)
  {
    girl::point p2 = p;
    p2.move(dir);

    const bool b1 = ( m_imgP.at<unsigned char>(p.y(), p.x()) < 128 );
    const bool b2 = ( m_imgP.at<unsigned char>(p2.y(), p2.x()) < 128 );
    
    return b1 == b2;
  }
		
};


static 
void 
traverseComponent(const USet &woundRegionsPts, 
		  girl::regionDeque &woundKeepedRegions, 
		  const RegionMeanProbaH &regionsProbas4, 
		  const girl::toger &tog, 
		  float threshold, 
		  const USet &skinRegions)
{
  USet processedRegions;
  
  for (USet::const_iterator itP = woundRegionsPts.begin(); itP!=woundRegionsPts.end(); ++itP) 
    {
      const girl::region seed = *itP;

      float rMin = std::numeric_limits<float>::max();
      float rMax = std::numeric_limits<float>::min();
      
      girl::regionDeque tmp;
      girl::regionDeque tmpKept;

      woundKeepedRegions.push_back(seed);
      tmp.push_back(seed);

      //RegionMeanProbaH::const_iterator itH_ALL1 = regionsProbas4.find(*itP);
      //const float meanReg1 = itH_ALL1->second.getMean();

      while( ! tmp.empty())
	{
	  girl::region rcur = tmp.front();
	  tmp.pop_front();

	  if ((processedRegions.find(rcur) == processedRegions.end()) && (skinRegions.find(rcur) == skinRegions.end()))
	    {
	      processedRegions.insert(rcur); 

	      girl::regionDeque neighbors;
	      tog.regionNeighbourhood(rcur, girl::toger::NEIGHBOUR_ANY , girl::toger::MARK_NONE, neighbors);
	      
	      for (girl::regionDeque::const_iterator it = neighbors.begin();
		   it != neighbors.end();
		   ++it)
		{
		  RegionMeanProbaH::const_iterator itH_ALL = regionsProbas4.find(*it);
		  const float meanReg = itH_ALL->second.getMean();

		  if((meanReg >= threshold) && (skinRegions.find(*it) == skinRegions.end()))
		    {
		      tmp.push_back(*it);

		      // tmpKept.push_back(*it);

		      woundKeepedRegions.push_back(*it);

		      if(meanReg < rMin)
			rMin = meanReg;
		      if(meanReg > rMax)
			rMax = meanReg;
		    }
		}
	    }
	}      

    }
}

static
void
selectRegionsWound(const RegionMeanProbaH &regionsProbas4,  
		   girl::regionDeque &woundRegions, 
		   const std::vector< std::vector<cv::Point> > &woundPts,  
		   girl::toger tog,
		   const USet &skinRegions)
{

  girl::regionDeque woundKeepedRegions;
  USet woundRegionsPts;
  
  for (PointVecVec::const_iterator itVV = woundPts.begin();
       itVV != woundPts.end(); 
       ++itVV) {
    const PointVec &pts = *itVV; //points for one touch
    for (PointVec::const_iterator itV = pts.begin();
	 itV != pts.end(); 
	 ++itV) {      
      const girl::region r = tog.findRegion(girl::point(itV->x, itV->y));
      assert(r != 0);
      woundRegionsPts.insert(r);
    }
  }

  traverseComponent(woundRegionsPts, woundKeepedRegions, regionsProbas4, tog, thresholdWoundMerge, skinRegions);   

  woundRegions.clear();
  woundRegions = woundKeepedRegions;

}



/*
  createdMask is 255 where is marker, 0 otherwise.

  @a regions si filled with regions corresponding to 0.
 */
// CURRENT INSERT

static
void
splitWithMask(const cv::Mat &markerMask, 
	      girl::toger &tog, 
	      girl::regionDeque &maskRegions,
	      girl::regionDeque &nonMaskRegions
)
{
  maskRegions.clear();
  nonMaskRegions.clear();

  girl::regionDeque createdRegions;
  Splitter s(markerMask);
  tog.splitRegion(1, s, createdRegions);
  
  for (girl::regionDeque::const_iterator it = createdRegions.begin();
       it != createdRegions.end(); ++it) {
    const girl::region r = *it;
    const girl::point pt = tog.anyPoint(r);
    const unsigned char pix = markerMask.at<unsigned char>(pt.y(), pt.x());
    if (pix < 128) {
      nonMaskRegions.push_back(r);
    }
    else {
      maskRegions.push_back(r);
    }
  }
  
}



/*
  compute mean skin color @a meanSkinColor from patches around @a skinPts in @a imgLAB.

  Patch rectangles are saved in @a skinPatches.
 */
static 
void
computeMeanSkinLAB(const cv::Mat &imgLAB,
			cv::Vec3f &meanSkinColor,
			const std::vector<cv::Point>  &skinPts,
			std::vector<cv::Rect > &skinPatches)
{
  assert(imgLAB.type() == CV_32FC3);

  const int halfSizeRect = sizeRectSkinTouch/2;
  
  unsigned int nbPoints = 0;
  float meanColorSkin[3];
  meanColorSkin[0] = meanColorSkin[1] = meanColorSkin[2] = 0.f;
 
  
  for (PointVec::const_iterator itV = skinPts.begin();
       itV != skinPts.end(); 
       ++itV) 
    {
      const int roi_topLeft_x = std::max(itV->x - halfSizeRect, 0);
      const int roi_topLeft_y = std::max(itV->y - halfSizeRect, 0);
      const int roi_botRight_x = std::min(itV->x + halfSizeRect, imgLAB.cols-1);
      const int roi_botRight_y = std::min(itV->y + halfSizeRect, imgLAB.rows-1);
      
      const cv::Rect r(roi_topLeft_x, roi_topLeft_y, roi_botRight_x-roi_topLeft_x+1, roi_botRight_y-roi_topLeft_y+1);
      const cv::Mat roiImg = imgLAB(r);   
      
      const int rows = roiImg.rows;
      const int cols = roiImg.cols;
      
      for (int i=0; i<rows; ++i) {
	const cv::Vec3f *s = roiImg.ptr<cv::Vec3f>(i);
	for (int j=0; j<cols; ++j) { 
	  const cv::Vec3f &intensity = s[j];
	  meanColorSkin[0] += intensity[0];
	  meanColorSkin[1] += intensity[1];
	  meanColorSkin[2] += intensity[2];
	  ++nbPoints;
	}
      }
      
      skinPatches.push_back(r);
    }

  
  assert(nbPoints > 0);
  const float inv_nbPoints = 1.0f / nbPoints;
  for (int k=0; k<3; ++k) {
    meanSkinColor[k] = meanColorSkin[k] * inv_nbPoints;
  }
  
}





static
void
modifyMaskAccordingToL(const cv::Mat &imgLAB,
		       const float thresholdL,
		       cv::Mat &mask)
{
  assert(imgLAB.type() == CV_32FC3);
  assert(mask.type() == CV_8UC1);
  assert(mask.size() == imgLAB.size());

  int rows = imgLAB.rows;
  int cols = imgLAB.cols;
  if (mask.isContinuous() && imgLAB.isContinuous()) {
    cols *= rows;
    rows = 1;
  }
  for (int i=0; i<rows; ++i) {
    unsigned char *dP = mask.ptr<unsigned char>(i);
    const cv::Vec3f *labP = imgLAB.ptr<cv::Vec3f>(i);
    for (int j=0; j<cols; ++j) {
      const cv::Vec3f &lab = labP[j];
      if (lab[0] > thresholdL) {
	dP[j] = 255;
      }
    }
  } 
}
  

static
void      
extendSkinPatches(const cv::Mat &imgLAB, 
		  const std::vector< cv::Rect > &skinPatches,
		  const cv::Vec3f &meanSkinColor,
		  cv::Mat &markerMask)
{
  assert(imgLAB.type() == CV_32FC3);
  assert(markerMask.type() == CV_8UC1);
  
  std::queue<cv::Point> ptsQueue;

  const size_t numPatches = skinPatches.size();
  for (size_t i=0; i<numPatches; ++i) 
    {
      const cv::Rect &r = skinPatches[i];
    
      const int y1 = r.y + r.height;
      const int x1 = r.x + r.width;    
      for (int y=r.y; y<y1; ++y) {
	for (int x=r.x; x<x1; ++x) { 
	  ptsQueue.push(cv::Point(x, y));
	}
      }
    }
  
  const float low_l  = meanSkinColor[0] - skinThresholdExtendL;
  const float high_l = meanSkinColor[0] + skinThresholdExtendL;
  const float low_a  = meanSkinColor[1] - skinThresholdExtendA;
  const float high_a = meanSkinColor[1] + skinThresholdExtendA;
  const float low_b  = meanSkinColor[2] - skinThresholdExtendB;
  const float high_b = meanSkinColor[2] + skinThresholdExtendB;

  while (! ptsQueue.empty()) 
    {
      
      cv::Point pt = ptsQueue.front();
      ptsQueue.pop();
      
      unsigned char &m = markerMask.at<unsigned char>(pt.y, pt.x);
      if (m == 0) {
	const cv::Vec3f &lab = imgLAB.at<cv::Vec3f>(pt.y, pt.x);
	if (low_l<=lab[0] && lab[0]<=high_l && low_a<=lab[1] && lab[1]<=high_a && low_b<=lab[2] && lab[2]<=high_b) 
	  // if (low_a<=lab[1] && lab[1]<=high_a && low_b<=lab[2] && lab[2]<=high_b) 
	  {
	    m = 255;
	    if (pt.y > 0)
	      ptsQueue.push(cv::Point(pt.x, pt.y-1));
	    if (pt.x > 0) 
	      ptsQueue.push(cv::Point(pt.x-1, pt.y));
	    if (pt.y < imgLAB.rows-1)
	      ptsQueue.push(cv::Point(pt.x, pt.y+1));
	    if (pt.x < imgLAB.cols-1)
	      ptsQueue.push(cv::Point(pt.x+1, pt.y));
	  }
	
      }
      
    }
}

/*
  Here, we suppose that we have "skin" around the markerMask.
  If the lab color is not too far from the mean skin color found around the marker mask,
  we extend the marker mask by region growing.
 */
static 
void
increaseAroundMask(const cv::Mat &imgSmallLAB, 
		   cv::Mat &markerMask)

{

  std::queue<cv::Point> ptsQueue;
  
  float meanColorSkin[3] = {0.f};
  
  unsigned int nbPoints = 0;
  
  for (int j=0; j<markerMask.rows; ++j) 
    {
      const unsigned char *m = markerMask.ptr<unsigned char>(j);
      for (int i=0; i<markerMask.cols-1; ++i) 
	{ 
	  if (m[i] != m[i+1]) //border of mask
	    {
	      //case: m[i] == 0 && m[i+1] == 255: skin is at m[i]
	      int row = i;
	      int col = j;

	      if (m[i] == 255) {
		//case: m[i] == 255 && m[i+1] == 0 : skin is at m[i+1]
		row = i+1;
	      }

	      const cv::Vec3f &intensity = imgSmallLAB.at<cv::Vec3f>(row, col);
	      meanColorSkin[0] += intensity[0];
	      meanColorSkin[1] += intensity[1];
	      meanColorSkin[2] += intensity[2];
	      ++nbPoints;
	      ptsQueue.push(cv::Point(row, col));
	    }	
	} 
    }
  
  if (nbPoints > 0) {
    const float inv = 1.f / (float)nbPoints;
    meanColorSkin[0] *= inv;
    meanColorSkin[1] *= inv;
    meanColorSkin[2] *= inv;
  }
 
  const float low_l  = meanColorSkin[0] - skinMaskThresholdExtendL;
  const float high_l = meanColorSkin[0] + skinMaskThresholdExtendL;
  const float low_a  = meanColorSkin[1] - skinMaskThresholdExtendA;
  const float high_a = meanColorSkin[1] + skinMaskThresholdExtendA;
  const float low_b  = meanColorSkin[2] - skinMaskThresholdExtendB;
  const float high_b = meanColorSkin[2] + skinMaskThresholdExtendB;
  while (! ptsQueue.empty()) 
    {
      
      cv::Point pt = ptsQueue.front();
      ptsQueue.pop();
      
      unsigned char &m = markerMask.at<unsigned char>(pt.y, pt.x);
      if (m == 0) 
	{
	  const cv::Vec3f &lab = imgSmallLAB.at<cv::Vec3f>(pt.y, pt.x);
	  if (low_l<=lab[0] && lab[0]<=high_l && low_a<=lab[1] && lab[1]<=high_a && low_b<=lab[2] && lab[2]<=high_b) 
	    {
	      m = 255;
	      if (pt.y > 0)
		ptsQueue.push(cv::Point(pt.x, pt.y-1));
	      if (pt.x > 0) 
		ptsQueue.push(cv::Point(pt.x-1, pt.y));
	      if (pt.y < imgSmallLAB.rows-1)
		ptsQueue.push(cv::Point(pt.x, pt.y+1));
	      if (pt.x < imgSmallLAB.cols-1)
		ptsQueue.push(cv::Point(pt.x+1, pt.y));
	      //++nbPoints;
	    }
	}
    }
}





/*
  @param[in, out] markerMask
  @param[out] skinRegions
 */
static void increaseMaskWithSkinLAB(const cv::Mat &imgSmallLAB, 
				    cv::Mat &markerMaskSmall,
				    const std::vector<std::vector<cv::Point> >&smallSkinPts)
{
  assert(imgSmallLAB.type() == CV_32FC3);

  increaseAroundMask(imgSmallLAB, markerMaskSmall);

  for (PointVecVec::const_iterator itVV = smallSkinPts.begin();
       itVV != smallSkinPts.end(); 
       ++itVV) 
    {
      cv::Vec3f meanColorSkin;
      std::vector< cv::Rect > skinPatches;
      
      const PointVec &pts = *itVV; 
      computeMeanSkinLAB(imgSmallLAB, meanColorSkin, pts, skinPatches);
      extendSkinPatches (imgSmallLAB,  skinPatches, meanColorSkin, markerMaskSmall);
    }

  
#ifdef SAVE_PROBA
  {
    const std::string filename = g_outputGrowMarkerFilename;
    const bool writeOk = cv::imwrite(filename, markerMaskSmall);
    if (! writeOk) {
      std::cerr<<"Error: unable to write proba image file: "<<filename<<"\n";
      exit(EXIT_FAILURE);
    }
  }
#endif //SAVE_PROBA

}




//param[in,out] outImgC
static
void
filterKeepOnlyRegionsWithWoundPts(cv::Mat &outImgC, const std::vector< std::vector<cv::Point> > &woundPts)
{
  assert(outImgC.type() == CV_8UC1);

  girl::toger tog(outImgC.cols, outImgC.rows);

  girl::regionDeque regions;
  Splitter s(outImgC);
  tog.splitRegion(1, s, regions);
  
  USet woundRegions;
  for (std::vector< std::vector<cv::Point> >::const_iterator it  = woundPts.begin();
       it != woundPts.end(); 
       ++it) {
    
    const std::vector<cv::Point> &pts = *it;
    
    cv::Point pt = pts[pts.size()/2];
    const girl::region r = tog.findRegion(girl::point(pt.x, pt.y));
    assert(r != 0);
    woundRegions.insert(r); 
  }

}

static
void
increaseMaskWithNonWoundColors(const cv::Mat &imgLAB, cv::Mat &markerMask)
{
  assert(imgLAB.type() == CV_32FC3);
  assert(markerMask.type() == CV_8UC1);

  assert(imgLAB.size() == markerMask.size());


  //const float absMin_l = 0;
  //const float absMax_l = 100.004;
  const float absMin_a = -86.1846;
  const float absMax_a = 98.2157;
  const float absMin_b = -107.85;
  //const float absMax_b = 94.4895;

  const float L_LOW = 3;
  const float L_HIGH = 99;
  const float A_LOW = -12;
  const float A_HIGH = 78;
  const float B_LOW = -29;
  const float B_HIGH = 59;

  const float s1_a0 = absMin_a;
  const float s1_b0 = 160.f;
  const float s1_a1 = 56.f;
  const float s1_b1 = absMin_b;
  const float s1_a = (s1_b1-s1_b0)/(s1_a1-s1_a0);
  const float s1_b = s1_b0 - s1_a*s1_a0;
  //line y=s1_a*x + s1_b

  const float s2_a0 = -5.f;
  const float s2_b0 = absMin_b;
  const float s2_a1 = absMax_a;
  const float s2_b1 = 6.f;
  const float s2_a = (s2_b1-s2_b0)/(s2_a1-s2_a0);
  const float s2_b = s2_b0 - s2_a*s2_a0;

  const float s3_a0 = absMin_a;
  const float s3_b0 = 45.f;
  const float s3_a1 = 180.f;
  const float s3_b1 = absMin_b;
  const float s3_a = (s3_b1-s3_b0)/(s3_a1-s3_a0);
  const float s3_b = s3_b0 - s3_a*s3_a0;


  int rows = imgLAB.rows;
  int cols = imgLAB.cols;
  if (imgLAB.isContinuous() && markerMask.isContinuous()) {
    cols *= rows;
    rows = 1;
  }
  for (int i=0; i<rows; ++i) {
    const cv::Vec3f *src = imgLAB.ptr<cv::Vec3f>(i);
    unsigned char *dst = markerMask.ptr<unsigned char>(i);
    for (int j=0; j<cols; ++j) {
      const cv::Vec3f &s = src[j];
      float cL=s[0], cA=s[1], cB=s[2];
      
      if (L_LOW >= cL || cL >= L_HIGH || A_LOW >= cA || cA >= A_HIGH || B_LOW >= cB || cB >= B_HIGH) {
	dst[j] = 255;
      }
      else {
	const float s1_y = s1_a * cA  + s1_b;
	const float s2_y = s2_a * cA  + s2_b;
	const float s3_y = s3_a * cA  + s3_b;
	if (cB <= s1_y || cB <= s2_y || cB <= s3_y) {
	  dst[j] = 255;
	}
      }
    }
  }

  cv::Mat markerMask2;
  const cv::Mat element = cv::getStructuringElement(cv::MORPH_RECT, cv::Size(3, 3));
  cv::morphologyEx(markerMask, markerMask2, cv::MORPH_CLOSE, element);

  markerMask = markerMask2;
}
