/*
  Templated vector type.
  Copyright (C) 2005-2009 Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
 * @file girlExtVector.hpp
 * @brief Generic vector class.
 */
#ifndef GIRLEXT_VECTOR_HPP
#define GIRLEXT_VECTOR_HPP

#include <cassert>

namespace girlExt {

  template <class T, size_t N>
  class vector 
  {
  public:

    /** 
     * @brief Default constructor.
     *
     * @warning elements are not initialized.
     */
    vector()
    {

    }

    /**
     * @brief Constructor.
     *
     * Initialize all elements with @a t.
     */
    explicit vector(const T &t)
    {
      for (size_t i=0; i<N; ++i)
	m_data[i] = t;
    }

    /**
     * @brief Const accessor.
     */
    const T &operator[](size_t i) const
    {
      assert(i<N);
      return m_data[i];
    }
    
    /**
     * @brief Accessor.
     */
    T &operator[](size_t i)
    {
      assert(i<N);
      return m_data[i];
    }
    
    /**
     * @brief Assignment operator : assign @a t to all elements of vector.
     */
    vector &operator=(const T &t)
    {
      for (size_t i=0; i<N; ++i)
	m_data[i] = t;
      return *this;

    }


    /**
     * @brief Add @a t to each element of vector.
     */
    vector &operator+=(const T &t)
    {
      for (size_t i=0; i<N; ++i)
	m_data[i] += t;
      return *this;
    }
    
    /**
     * @brief Add @a v to vector.
     */
    vector &operator+=(const vector &v)
    {
      for (size_t i=0; i<N; ++i)
	m_data[i] += v.m_data[i];
      return *this;
    }
    
    
    /**
     * @brief Subtract @a t to each element of vector.
     */
    vector &operator-=(const T &t)
    {
      for (size_t i=0; i<N; ++i)
	m_data[i] -= t;
      return *this;
    }
    
    /**
     * @brief Subtract @a v to vector.
     */
    vector &operator-=(const vector &v)
    {
      for (size_t i=0; i<N; ++i)
	m_data[i] -= v.m_data[i];
      return *this;
    }
    
    /**
     * @brief Multiply by @a t each element of vector.
     */
    vector &operator*=(const T &t)
    {
      for (size_t i=0; i<N; ++i)
	m_data[i] *= t;
      return *this;
    }
    
    /**
     * @brief Divide by @a t each element of vector.
     */
    vector &operator/=(const T &t)
    {
      for (size_t i=0; i<N; ++i)
	m_data[i] /= t;
      return *this;
    }
    

  protected:
    T m_data[N];

  };


  template <class T, size_t N>
  vector<T, N> operator+(const vector<T, N> &v1, const vector<T, N> &v2)
  {
    vector<T, N> v(v1);
    v += v2;
    return v;
  }

  template <class T, size_t N>
  vector<T, N> operator-(const vector<T, N> &v1, const vector<T, N> &v2)
  {
    vector<T, N> v(v1);
    v -= v2;
    return v;
  }

  template <class T, size_t N>
  vector<T, N> operator*(const vector<T, N> &v1, const T &t)
  {
    vector<T, N> v(v1);
    v *= t;
    return v;
  }


  template <class T, size_t N>
  vector<T, N> operator/(const vector<T, N> &v1, const T &t)
  {
    vector<T, N> v(v1);
    v /= t;
    return v;
  }


  template <class T, size_t N>
  vector<T, N> operator*(const T &t, const vector<T, N> &v1)
  {
    return v1*t;
  }


  template <class T, size_t N>
  vector<T, N> operator*(const vector<T, N> &v1, const vector<T, N> &v2)
  {
    vector<T, N> res(v1);

    for (size_t i = 0; i < N; ++i)
      res[i] *= v2[i];

    return res;
  }

  template <class T, size_t N>
  T dotProduct(const vector<T, N> &v1, const vector<T, N> &v2)
  {
    assert(N>0);

    T res = v1[0]*v2[0];
    for (size_t i = 1; i < N; ++i) //start from 1
      res += v1[i]*v2[i];

    return res;
  }











  typedef double Flt;

  class Flt2 : public vector<Flt, 2>
  {
  public:

    Flt2(Flt a, Flt b)
    {
      m_data[0] = a;
      m_data[1] = b;
    }

    Flt2(const vector<Flt, 2> &v)
      : vector<Flt, 2>(v)
    {
    }


  };

  class Flt3 : public vector<Flt, 3>
  {
  public:
    
    Flt3(Flt a, Flt b, Flt c)
    {
      m_data[0] = a;
      m_data[1] = b;
      m_data[2] = c;
    }

    Flt3(const vector<Flt, 3> &v)
      : vector<Flt, 3>(v)
    {
    }

  };

  typedef int Int;
  
  class Int2 : public vector<Int, 2>
  {
  public:
    
    Int2(Int a, Int b)
    {
      m_data[0] = a;
      m_data[1] = b;
    }

    Int2(const vector<Int, 2> &v)
      : vector<Int, 2>(v)
    {
    }

  };
  
  
}

#endif /* ! GIRLEXT_VECTOR_HPP */
