/*
  Functions to compute various measures on moments.
  Copyright (C) 2005-2009 Jean-Philippe Domenger, Achille Braquelaire, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "iseStat.hpp"

static const double EPS=0.000000001;


static inline
double 
P_iseStat_iseComputeSQE(double inv_m0, const girl::tools::Flt3 &m1, const girl::tools::Flt3 &m2)
{
  const double sm1s = m1[0] * m1[0] + m1[1] * m1[1] + m1[2] * m1[2];
  const double sm2 = m2[0] + m2[1] + m2[2];
  
  return (sm2 - sm1s*inv_m0);
}

static inline
double
P_iseStat_iseComputeSQE(double inv_m0, const double *m1, const double *m2, size_t size)
{
  double sm1s = 0.0;
  double sm2 = 0.0;
  
  for (size_t i = 0 ; i < size ; ++i) {
    sm1s += m1[i]*m1[i];
    sm2 += m2[i];
  }
  
  return (sm2 - sm1s*inv_m0);
}

double
iseStat_iseComputeMean(double m0, double m1)
{
  if (GIRL_likely(m0 > EPS)) {
    return m1 / m0;
  }
  else {
    return 0.0;
  }
}

double 
iseStat_iseComputeSQE(double m0, const girl::tools::Flt3 &m1, const girl::tools::Flt3 &m2)
{
  if (GIRL_likely(m0 > EPS)) {
    const double inv_m0 = 1.0 / m0; 
    
    return P_iseStat_iseComputeSQE(inv_m0, m1, m2);
  }
  else {
    return 0.0;
  }
}

double
iseStat_iseComputeSQE(double m0, const double *m1, const double *m2, size_t size)
{
  if (GIRL_likely(m0 > EPS)) {
    const double inv_m0 = 1.0 / m0;

    return P_iseStat_iseComputeSQE(inv_m0, m1, m2, size);
  }
  else {
    return 0.0;
  }
}

double
iseStat_iseComputeVariance(double m0, const girl::tools::Flt3 &m1, const girl::tools::Flt3 &m2)
{
  if (GIRL_likely(m0 > EPS)) {
    const double inv_m0 = 1.0 / m0; 
    
    return P_iseStat_iseComputeSQE(inv_m0, m1, m2) * inv_m0;
  }
  else {
    return 0.0;
  }
}

double
iseStat_iseComputeVariance(double m0, const double *m1, const double *m2, size_t size)
{
  if (GIRL_likely(m0 > EPS)) {
    const double inv_m0 = 1.0 / m0; 

    return P_iseStat_iseComputeSQE(inv_m0, m1, m2, size) * inv_m0;
  }
  else {
    return 0.0;
  }
}



static inline
double 
P_iseStat_iseComputeSQE(double inv_m0, double m1c, double m2c)
{
  return (m2c - m1c*m1c*inv_m0);
}

double 
iseStat_iseComputeSQE(int channel,
		      double m0, const girl::tools::Flt3 &m1, const girl::tools::Flt3 &m2)
{
  assert(channel>=0 && channel<3);

  if (GIRL_likely(m0 > EPS)) {
    const double inv_m0 = 1.0 / m0; 
    
    return P_iseStat_iseComputeSQE(inv_m0, m1[channel], m2[channel]);
  }
  else {
    return 0.0;
  }
}

double
iseStat_iseComputeVariance(int channel,
			   double m0, const girl::tools::Flt3 &m1, const girl::tools::Flt3 &m2)
{
  assert(channel>=0 && channel<3);

  if (GIRL_likely(m0 > EPS)) {
    const double inv_m0 = 1.0 / m0; 
    
    return P_iseStat_iseComputeSQE(inv_m0, m1[channel], m2[channel]) * inv_m0;
  }
  else {
    return 0.0;
  }
}

double
iseStat_iseComputeVariance(double m0, double m1, double m2)
{
  if (GIRL_likely(m0 > EPS)) {
    const double inv_m0 = 1.0 / m0; 
    return P_iseStat_iseComputeSQE(inv_m0, m1, m2) * inv_m0;
  }
  else {
    return 0.0;
  }
}
