/*
  1D Histogram class.
  Copyright (C) 2005-2011 Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef ISE_SPARSE_HISTOGRAM_1D_HPP
#define ISE_SPARSE_HISTOGRAM_1D_HPP

#include <cstddef>//size_t
#include <map>

namespace ise {

  /**
   * Generic 1D Histogram class.
   * It associates a value (of type KEY) and a frequency (of type T).
   */
  template <typename KEY, typename T>
  class SparseHistogram1D
  {
  public:
    
    typedef typename std::map<KEY, T> Map;
    typedef typename Map::iterator iterator;
    typedef typename Map::const_iterator const_iterator;
    typedef typename Map::reverse_iterator reverse_iterator;
    typedef typename Map::const_reverse_iterator const_reverse_iterator;

    /**
     * @brief Empty histogram then fill it with pairs from @a m.
     */
    void populate(const Map &m);

    /**
     * @brief 
     */
    void add(KEY k, T v) { m_map[k] += v; } //if m_map[k] did not exist, default constructor is first called.

    /**
     * @brief Empty histogram.
     */
    void clear() { m_map.clear(); }

    /**
     * @brief Return number of values of histogram.
     */
    size_t size() const { return m_map.size(); }

    /**
     * @brief Return if histogram is empty.
     */
    bool empty() const { return m_map.empty(); }

    /**
     * @brief 
     */
    iterator begin() { return m_map.begin(); }
    /**
     * @brief 
     */
    iterator end() { return m_map.end(); }
    
    /**
     * @brief 
     */
    reverse_iterator rbegin() { return m_map.rbegin(); }
    /**
     * @brief 
     */
    reverse_iterator rend() { return m_map.rend(); }
    
    /**
     * @brief 
     */
    const_iterator begin() const { return m_map.begin(); }

    /**
     * @brief 
     */
    const_iterator end() const { return m_map.end(); }
    
    /**
     * @brief 
     */
    const_reverse_iterator rbegin() const { return m_map.rbegin(); }
    /**
     * @brief 
     */
    const_reverse_iterator rend() const { return m_map.rend(); }


    /**
     * @brief 
     */
    T operator[](KEY k) const
    {
      const_iterator it = m_map.find(k);
      if (it != m_map.end())
        return it->second;
      else
        return T();
    }


    /**
     * @brief return mean value of histogram
     */
    KEY getMean() const;

    /**
     * @brief Standard deviation of histogram.
     *
     * @param mean mean value of histogram.
     *
     * @sa getMean.
     */
    KEY getStandardDeviation(KEY mean) const;

  protected:

    Map m_map;

  };

  
  template <typename KEY, typename T>
  void SparseHistogram1D<KEY, T>::populate(const Map &m)
  {
    clear();

    const_iterator it = m.begin();
    const const_iterator itEnd = m.end();
    for ( ; it != itEnd; ++it) {
      add(it->first, it->second);
    }
  }

  template <typename KEY, typename T>
  KEY SparseHistogram1D<KEY, T>::getMean() const
  {
    KEY sum = 0;
    T freq = 0;
    typename Map::const_iterator it = m_map.begin();
    typename Map::const_iterator itEnd = m_map.end();
    for (; it != itEnd; ++it) {
      sum += it->first*it->second;
      freq += it->second;
    }
    return (freq > 0) ? sum/freq : 0;
  }

  template <typename KEY, typename T>
  KEY SparseHistogram1D<KEY, T>::getStandardDeviation(KEY mean) const
  {
    KEY stdD = 0.0;
  
    if (m_map.size() > 1) {
    //const float m = mean + m_shift;

      KEY s = 0.0;
      T n = 0;
      typename Map::const_iterator it = m_map.begin();
      typename Map::const_iterator itEnd = m_map.end();
      for (; it != itEnd; ++it) { 
        const float t = it->first - mean;
        const double freq = it->second;
        s += freq * t*t;
        n += freq;
      }
      if (n>0)
        stdD = sqrt( 1.0f/n * s);
    }

    return stdD;
  }



class SparseHistogram1DD : public ise::SparseHistogram1D<double, double> {};


}

#endif /* ! ISE_SPARSE_HISTOGRAM_1D_HPP */
