/*
  Segmentation context: toger, image, region features, ...
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Achille Braquelaire, Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef ISESEGMENTATIONCONTEXT_HPP
#define ISESEGMENTATIONCONTEXT_HPP

#include <map>

#include <girl/girlCommon.hpp>
#include <girl/girlToger.hpp>
#include <girl/girlSharedPtr.hpp>

#include <girlTools/girlToolsImage.hpp>
#include <girlTools/girlToolsImageFeature.hpp>

#include "iseImageFeatureComputerDefault.hpp"

#include "iseConversion.hpp"
#include "iseFeatureComputerHandler.hpp"
#include "iseImageFeatureCollection.hpp"
#include "iseImageFeatureComputer.hpp"
#include "iseRegionSetInterface.hpp"
#include "iseRegionTools.hpp"

namespace ise {

  typedef std::map<int, girl::tools::ImageFeature*> FeaturesMap;

  class GIRL_API SegmentationContext : public girl::toger::observer
  {
  public:

    /**
     * @brief Constructor.
     *
     */
    SegmentationContext(girl::toger *tog=NULL,
			girl::tools::Image *img=NULL,
			void *userData=NULL);

    /**
     * @brief Destructor.
     *
     */
    ~SegmentationContext();


    inline girl::toger *getToger() const { return m_tog; }
    inline void setToger(girl::toger *tog) { 
      setToger_aux(tog);
      fillRegSet();
    }

    inline girl::tools::Image *getImage() const { return m_img; }
    inline void setImage(girl::tools::Image *img) {
      setImage_aux(img);
      fillRegSet();
      
      // Loadind default image feature
      // girl::shared_ptr<ise::ImageFeatureComputer> imageFeatureComputerDefault = 
      // 	girl::shared_ptr<ise::ImageFeatureComputer>(new ise::ImageFeatureComputerDefault(this));
      // imageFeatureComputerDefault->compute();
      // addImageFeatureComputer(imageFeatureComputerDefault);
      
      // assert(getImageFeature(ise::ImageFeatureComputerDefault::DEFAULT_IMAGE_FEATURE));
    }

    inline void setImageToger(girl::tools::Image *img, girl::toger *tog) {
      setToger_aux(tog);
      setImage_aux(img);
      fillRegSet(); //done only once.
    }

  //  inline const ise::ImageLabelToger* getImageLabel() const { return m_imageLabel; }



    /****************************************************************/
    /*                Functions to handle region sets               */
    /****************************************************************/
    /**
     * Add the region set to the segmentation context
     * @param rs: the region set to add
     */
    void addRegionSet(girl::shared_ptr<RegionSetInterface> rs);

    /**
     * Remove the region set from the segmentation context
     * @param uri: uri of the region set to remove
     */
    inline void removeRegionSet(const std::string &title, bool removeDep = false)
    {
      FeatureComputer::FeatureComputerListType dep;
      m_regSets.remove(title, removeDep, dep);
    }

    /**
     * Get a region set
     * @param uri: uri of the region set to retrieve
     * @return: the region set or NULL if the region set is not in the seg context
     */
    const girl::shared_ptr<RegionSetInterface>
    getRegionSet(const std::string &uri) const;
    /**
     * Check if a region set is in the segmentation context
     * @param uri: uri of the region set to check
     * @return: true if it is in the segmentation context, false otherwise
     */
    inline bool
    regionSetExists(const std::string &uri)
    {
      return m_regSets.exists(uri);
    }

    /**
     * Get the list of region sets stored in the segmentation context
     * @param list: filled with the titles of the region sets in the segmentation context
     */
    inline void getRegionSetUris(ise::FeatureComputer::FeatureComputerListType &list)
    {
    //  m_regSets.getURIs(list);
      m_regSets.getComputerList<RegionSetInterface>(list);

    }

    /**
     * @return: the number of regions in the region set (the first region set if several)
     */
  //  inline size_t getNbRegFromRegSet() { return m_regSets.nbReg(); }


    /**
     * Get the description of the region set, i.e. what feature is computed and
     * a description of the feature dimension semantics
     * @param uri uri of the region set
     * @param title is filled with the name of the feature
     * @param dimDesc is filled with a description of the semantics of each
     * dimension of the feature vector
     * @return false if region set was not found in the collection(hence params are not filled), true otherwise
     */
    // R: is it used?
//    inline bool getFeatureDescription(const std::string &uri, std::string &title, std::vector<std::string>& dimDesc) const
//    {
//      return m_regSets.getFeatureDescription(uri, title, dimDesc);
//    }

    /**
     * @param uri uri of the region set
     * @param title is filled with the name of the feature
     * @return false if region set was not found in the collection(hence params are not filled), true otherwise
     */
      //R: is it used?
//    inline bool getFeatureTitle(const std::string &uri, std::string &title) const
//    {
//      return m_regSets.getFeatureTitle(uri, title);
//    }

    /**
     * Return a semantic description of each dimension of the feature vector
     * Default behaviour: return the dimension index
     * @param uri uri of the region set
     * @param dimDesc is filled with a description of the semantics of each
     * dimension of the feature vector
     * @return false if region set was not found in the collection(hence params are not filled), true otherwise
     */
    // R: is it used?
//    inline bool getFeatureDimDescription(const std::string &uri, std::vector<std::string> &dimDesc) const
//    {
//      return m_regSets.getFeatureDimDescription(uri, dimDesc);
//    }


    /**
     * Rename the computer
     * TODO:improve possible cause of bug if the user get a feature computer and call the function
     * setTitle directly on this computer, then the ComputerHandler is not up-to-date because it
     * is not notified of the name change.
     */
//    inline void
//    renameComputer(const std::string &oldName, const std::string &newName)
//    {
//      m_regSets.renameComputer(oldName, newName);
//    }

    virtual bool doNewRegion() const { return true; }
    virtual void newRegion(const girl::regionDeque &regions);
    virtual bool doUpdateRegion() const { return true; }
    virtual void updateRegion(const girl::regionDeque &regions);
    virtual bool doMergeRegions() const { return true; }
    virtual void mergeRegions(girl::region keptRegion, girl::region deletedRegion);
    virtual bool doDeleteRegion() const { return true; }
    virtual void deleteRegion(girl::region r);
    virtual bool doRenameRegion() const { return true; }
    virtual void renameRegion(girl::region oldRegion, girl::region newRegion);



    /****************************************************************/
    /*                Functions to handle image feature             */
    /****************************************************************/
    /**
     * @brief Add the image feature computer @a imgf to the segmentation context
     * @param imgf : the image feature computer to add
     */
    void
    addImageFeatureComputer(girl::shared_ptr<ise::ImageFeatureComputer> imgF);


    /**
     * @brief Remove the image feature computer from the segmentation context
     * @param uri : uri of image feature computer to remove
     */
    void
    removeImageFeature(const std::string &uri, bool removeDep = false);


    /**
     * @brief Get an image feature
     * @param title: title of the image feature to retrieve
     * @return : the image feature or NULL if the image feature is not in the seg context
     */
    const girl::shared_ptr<girl::tools::ImageFeature>
    getImageFeature(const std::string &title);

     /**
     * @brief Get an image feature computer
     * @param title: title of the image feature computer to retrieve
     * @return : the image feature computer or NULL if the image feature computer is not in the seg context
     */  
    const girl::shared_ptr<ImageFeatureComputer>
    getImageFeatureComputer(const std::string &title);

    /**
     * @brief Get the list of image features uris stored in the segmentation context
     *        with her image feature computer type uri is @a imageFeatureComputerUri
     * @param list:  filled with the uris of the image features in the segmentation context
     * @param imageFeatureComputerUri: the image feature computer type uri 
     */
    void
    getImageFeaturesUri(std::vector<std::string> &list,
        const std::string &imageFeatureComputerUri = "") const;

    /**
     * @brief Get the list of images features titles stored in 
     *        the segmentation with her base image feature computer type uri
     *        is @a imageFeatureComputerUri
     * @param list: filled with the titles of the image features computer
     *              in the segmentation context
     * @param imageFeatureComputerUri: the image feature computer type uri
     */
    void
    getImageFeaturesBaseUri(std::vector<std::string> &list,
        const std::string &imageFeatureComputerUri = "") const;

    /**
     * @brief Get the computerUri's list
     */
    void
    getImageFeaturesComputerUri(std::set<std::string> &set) const;


    /**
     * Check if an image feature computer is in the segmentation context
     * @param title: title of the region set to check
     * @return: true if it is in the segmentation context, false otherwise
     */
    bool
    imageFeatureComputerExists(const std::string &title) const;


    // R: for debugging purposes
   // inline void checkAllRegions() { m_regSets.checkAllRegions(); }


    const FeatureComputerHandler&
    getFeatureComputerHandler() const
    {
      return m_regSets;
    }

    void
    setFeatureComputerHandler(const FeatureComputerHandler& handler)
    {
      // R: is it the right way to do it? But operator= will launch the computation
      // of features, and it seems a good time to do it since when = is performed
      // we know the new computers, etc
      m_regSets.update(handler);
    }


  protected:

    inline void setToger_aux(girl::toger *tog) { 
      m_tog = tog; 
    }
    inline void setImage_aux(girl::tools::Image *img) {
      m_img = img;
      if (img->width() && img->height())
        fillLabelImage();      
    }

    void fillRegSet();


    inline void fillLabelImage()
    {
//      if (m_imageLabel) {
//        delete m_imageLabel;
//      }
//      m_imageLabel = new ise::ImageLabelToger(m_img->width(), m_img->height());
    }

  protected:

 //   ImageLabelToger *m_imageLabel;
  //  RegionSetCollection m_regSets;
    FeatureComputerHandler m_regSets;
//    ImageFeatureCollection m_imageFeatures;
    girl::toger *m_tog;
    girl::tools::Image *m_img;
    FeaturesMap m_featuresMap;

  };

}

#endif /* ! ISESEGMENTATIONCONTEXT_HPP*/
