/*
  Copyright (C) 2005-2009 Roigt Julian, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "iseSegmentationClusteringImageFeatureClusterer.hpp"
#include <girlTools/girlToolsImageFeature.hpp>
#include <girl/girlToger.hpp>

#include <limits>

//#define TIMING 1

#if TIMING
#include <girl/girlTimer.hpp>
#endif

using namespace ise;

namespace {

  class Adder : public girl::toger::pointIterator
  {
  public:
    Adder(const cv::Mat &src, cv::Mat &data)
      : m_src(src),
	m_data(data),
	m_size(0),
	m_xMin(std::numeric_limits<girl::POINT_TYPE>::max()),
	m_yMin(std::numeric_limits<girl::POINT_TYPE>::max()),
	m_xMax(std::numeric_limits<girl::POINT_TYPE>::min()),
	m_yMax(std::numeric_limits<girl::POINT_TYPE>::min())
    {
      assert(data.rows == src.rows*src.cols);
      assert(data.cols == src.channels());
      assert(data.type() == CV_MAKETYPE(cv::DataDepth<girl::tools::ImageFeature::type>::value, 1));
    }

    virtual void processPoint(const girl::point& p)
    {
      const int channels = m_src.channels();
      const girl::tools::ImageFeature::type *buf = m_src.ptr<girl::tools::ImageFeature::type>(p.y()) + p.x() * channels;
      
      assert(m_size < m_data.rows);
      memcpy(m_data.ptr<girl::tools::ImageFeature::type>(m_size), buf, channels*sizeof(girl::tools::ImageFeature::type));
      ++m_size;

      if (p.x() < m_xMin)
	m_xMin = p.x();
      if (p.x() > m_xMax)
	m_xMax = p.x();
      if (p.y() < m_yMin)
	m_yMin = p.y();
      if (p.y() > m_yMax)
	m_yMax = p.y();

    }

    void finalize()
    {
      assert(m_size>0);
      m_data.resize(m_size);
    }

    girl::point getMinPt() const
    {
      return girl::point(m_xMin, m_yMin);
    }

    girl::point getMaxPt() const
    {
      return girl::point(m_xMax, m_yMax);
    }

  private:
    const cv::Mat &m_src;
    cv::Mat &m_data;
    int m_size;

    girl::POINT_TYPE m_xMin;
    girl::POINT_TYPE m_yMin;
    girl::POINT_TYPE m_xMax;
    girl::POINT_TYPE m_yMax;
  };


  class Labeler : public girl::toger::pointIterator
  {
  public:
    Labeler(const cv::Mat &labels, 
	    const girl::point &shift, 
	    cv::Mat &imageLabel)
      : m_labels(labels),
	m_shift(shift),
	m_imageLabel(imageLabel),
	m_index(0)
    {
      assert(labels.channels()==1);
      assert(imageLabel.channels()==1);
      assert(labels.type() == imageLabel.type());
      assert(labels.rows*labels.cols <= imageLabel.rows*imageLabel.cols);
    }

    virtual void processPoint(const girl::point& pt)
    {
      assert(m_index < m_labels.rows);
      const int label = m_labels.at<int>(m_index, 0);

      assert((pt.x() >= m_shift.x()) && (pt.x() < m_shift.x() + m_imageLabel.cols) 
	     && (pt.y() >= m_shift.y()) && (pt.y() < m_shift.y() + m_imageLabel.rows));
      
      
      m_imageLabel.at<int>(pt.y()-m_shift.y(), pt.x()-m_shift.x()) = label;

      ++m_index;
    }      
    
  protected:
    
    const cv::Mat &m_labels;
    girl::point m_shift;
    cv::Mat &m_imageLabel;
    int m_index;
  };

} //end anonymous namespace

SegmentationClusteringImageFeatureClusterer::SegmentationClusteringImageFeatureClusterer(ise::SegmentationContext *segContext)
  : SegmentationClustering(segContext)
  , m_clusterer(girl::shared_ptr<ise::Clusterer>())
  , m_algo()
{
}

static
bool
areAllRegionsSelected(const girl::toger &tog, const girl::regionDeque &regions)
{
  const size_t numRegions = tog.numRegions(); //O(N) but rather fast
  assert(regions.size() <= numRegions);
  return numRegions == regions.size();
}

void
SegmentationClusteringImageFeatureClusterer::init(const girl::regionDeque &regions, int)
{
  assert(m_segContext);
  assert(m_segContext->getToger());
  assert(m_algo->getImageFeature());

  girl::shared_ptr<girl::tools::ImageFeature> imageFeature = m_algo->getImageFeature();

  cv::Mat data;

#if TIMING
  girl::Timer timer;
  timer.start();
#endif //TIMING


  const girl::toger &tog = *(m_segContext->getToger());
  const int width = tog.width();
  const int height = tog.height();

  const bool allRegionsSelected = areAllRegionsSelected(tog, regions);

  if (! allRegionsSelected) {
    const cv::Mat &src = *(imageFeature->cvMat());
    data.create(src.rows, src.cols, src.type());
    data = data.reshape(1, width*height); //1 channel, several rows : will only have src.channels column
    Adder a(src, data);
    girl::regionDeque::const_iterator it = regions.begin();
    const girl::regionDeque::const_iterator itEnd = regions.end();
    assert(! regions.empty());
    for ( ; it != itEnd; ++it) 
      tog.iterateOnRegionDomain(*it, a);
    a.finalize();

    const girl::point minPt = a.getMinPt();
    const girl::point maxPt = a.getMaxPt();
    assert(minPt.x()<=maxPt.x());
    assert(minPt.y()<=maxPt.y());

    m_shift = minPt;

    m_imageLabel.create(maxPt.y()-minPt.y()+1, maxPt.x()-minPt.x()+1, CV_32S);
  }
  else {
    data = *(imageFeature->cvMat()); //no copy (juste header)
    data = data.reshape(1, width*height); //1 channel, several rows

    m_shift = girl::point(0, 0);

    //m_imageLabel.create(height, width, CV_32S);
  }

  //B: do we initialize each pixel of label image to invalid value ???

#if TIMING
  {
    const size_t time = timer.stop();
    std::cerr<<"Time to gather data="<<time<<"ms\n";
    timer.restart();
  }
#endif //TIMING


  assert(data.channels() == 1);

  cv::Mat labels;

#if 0
    {//DEBUG
      std::cerr<<"SegmentationClusteringImageFeatureClusterer::init() : data rows="<<data.rows<<" cols="<<data.cols<<" :\n";
      if (data.rows > 0) {
	for (int i=0; i<std::min(4, data.rows); ++i) {
	  const double *r = data.ptr<double>(i);
	  std::cerr<<i<<") ";
	  for (int j=0; j<std::min(6, data.cols); ++j)
	    std::cerr<<r[j]<<" ";
	  std::cerr<<"\n";
	}
	std::cerr<<"...\n";
	for (int i=std::max(0, data.rows-4); i<data.rows; ++i) {
	  const double *r = data.ptr<double>(i);
	  std::cerr<<i<<") ";
	  for (int j=0; j<std::min(6, data.cols); ++j)
	    std::cerr<<r[j]<<" ";
	  std::cerr<<"\n";
	}
      }

    }
#endif //0

  assert(m_clusterer);
  m_clusterer->compute(data, labels);


#if 0
    {//DEBUG
      if ( labels.rows > 0) {
	std::cerr<<"labels: \n";
	for (int i=0; i<std::min(4, labels.rows); ++i) {
	  const int *r = labels.ptr<int>(i);
	  for (int j=0; j<std::min(6, labels.cols); ++j)
	    std::cerr<<r[j]<<" ";
	  std::cerr<<"\n";
	}
	std::cerr<<"...\n";
	for (int i=std::max(0, labels.rows-4); i<labels.rows; ++i) {
	  const int *r = labels.ptr<int>(i);
	  for (int j=0; j<std::min(6, labels.cols); ++j)
	    std::cerr<<r[j]<<" ";
	  std::cerr<<"\n";
	}
      }

      std::vector<size_t> h(m_clusterer->numberClasses(), 0);
      for (int i=0; i<labels.rows; ++i) {
	const int r = *(labels.ptr<int>(i));
	if (r < 0 || r >= m_clusterer->numberClasses()) {
	  std::cerr<<"Error: for row "<<i<<" label="<<r<<" not in [0; "<<m_clusterer->numberClasses()<<"[\n";
	}
	else {
	  h[r] += 1;
	}
      }
      size_t minH = m_clusterer->numberClasses()+1;
      size_t maxH = 0; 
      size_t maxH2 = 0; 
      size_t minHIndex = 0;
      size_t maxHIndex = 0;
      size_t maxHIndex2 = 0;
      size_t nbNN = 0;
      for (int i=0; i<h.size(); ++i) {
	const size_t f = h[i];
	if (f > 0) {
	  ++nbNN;
	  if (f > maxH) {
	    maxH2 = maxH;
	    maxHIndex2 = maxHIndex;
	    maxH = f;
	    maxHIndex = i;
	  }
	  else if (f > maxH2) {
	    maxH2 = f;
	    maxHIndex2 = i;
	  }
	}
      }
      std::cerr<<nbNN<<"/"<<h.size()<<" non null labels\n";
      std::cerr<<maxH<<" elts labeled "<<maxHIndex<<"\n";
      std::cerr<<maxH2<<" elts labeled "<<maxHIndex2<<"\n";

    }
#endif //0




  if (! allRegionsSelected) {
    Labeler a(labels, m_shift, m_imageLabel);
    girl::regionDeque::const_iterator it = regions.begin();
    const girl::regionDeque::const_iterator itEnd = regions.end();
    for ( ; it != itEnd; ++it) 
      tog.iterateOnRegionDomain(*it, a);
  }
  else {
    labels = labels.reshape(1, height);
    assert(labels.rows == height && labels.cols == width);
    m_imageLabel = labels;
  }

  
#if TIMING
  {
    const size_t time = timer.stop();
    std::cerr<<"Time of clustering="<<time<<"ms\n";
    timer.restart();
  }
#endif //TIMING

  /*
  std::vector<double*> data;
  data.reserve(tog->width()*tog->height());
  
  Adder add(&data, imageFeature);
  size_t size = regions.size();
  for (size_t i = 0 ; i < size ; ++i) {
    tog->iterateOnRegionDomain(regions[i], add);
  }

  m_clusterer->compute(data, imageFeature->channels());
  */
}

void
SegmentationClusteringImageFeatureClusterer::init(girl::region, int) {}

void
SegmentationClusteringImageFeatureClusterer::setParameters(ise::ImageFeatureComputerPtr algo, girl::shared_ptr<ise::Clusterer> clusterer)
{
  m_algo = algo;
  m_clusterer = clusterer;
}

void
SegmentationClusteringImageFeatureClusterer::getParameters(ise::ImageFeatureComputerPtr algo, girl::shared_ptr<ise::Clusterer> clusterer) const
{
  algo = m_algo;
  clusterer = m_clusterer;
}

bool
SegmentationClusteringImageFeatureClusterer::same(const girl::point &pt1, girl::direction dir)
{
  assert(m_segContext);
  assert(m_algo->getImageFeature());

  girl::point pt2 = pt1;
  pt2.move(dir);

#if 0

  const cv::Mat &imgf = *(m_algo->getImageFeature()->cvMat());

  const int channels = imgf.channels();
  const girl::tools::ImageFeature::type *buf1 = imgf.ptr<girl::tools::ImageFeature::type>(pt1.y()) + channels * pt1.x();
  const girl::tools::ImageFeature::type *buf2 = imgf.ptr<girl::tools::ImageFeature::type>(pt2.y()) + channels * pt2.x();

  const int cluster1 = m_clusterer->getCluster(buf1, channels);
  const int cluster2 = m_clusterer->getCluster(buf2, channels);

#else

  assert((pt1.x() >= m_shift.x()) && (pt1.x() < m_shift.x() + m_imageLabel.cols) 
	 && (pt1.y() >= m_shift.y()) && (pt1.y() < m_shift.y() + m_imageLabel.rows));
  const int cluster1 = m_imageLabel.at<int>(pt1.y()-m_shift.y(), pt1.x()-m_shift.x());

  assert((pt2.x() >= m_shift.x()) && (pt2.x() < m_shift.x() + m_imageLabel.cols) 
	 && (pt2.y() >= m_shift.y()) && (pt2.y() < m_shift.y() + m_imageLabel.rows));
  const int cluster2 = m_imageLabel.at<int>(pt2.y()-m_shift.y(), pt2.x()-m_shift.x());

#endif

  return ( cluster1 == cluster2);
}
