/*
  Segmentation clustering: Gonogo1
  Copyright (C) 20013 Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "iseSegmentationClusteringGonogo1.hpp"

#include <cassert>
#include <limits>

#include <ise/iseConversion.hpp>
#include <ise/iseImageFeatureColorSpace.hpp>
#include <ise/iseClustererGenericKMeans.hpp>
#include <ise/iseSegmentationClusteringImageFeatureClusterer.hpp>
#include <ise/iseSplitAlgorithmDefault.hpp>
#include <ise/iseSplitDriverDefault.hpp>
#include <ise/iseRegionSetInterfaceMean.hpp>
#include <ise/iseMergeAlgorithmUntil.hpp>
#include <ise/iseMergeDriverMeta.hpp>

#include <ise/iseRegionSetInterfaceMean.hpp>
#include <ise/iseRegionBinaryPredicateBoolCombine.hpp>
#include <ise/iseMergeDriverMeta.hpp>
#include <ise/iseImageFeatureColorSpace.hpp>
#include <ise/iseRegionBinaryFunctionWithRegSetEuclideanDistance.hpp>
#include <ise/iseRegionUnaryFunctionWithRegSetIdentity.hpp>
#include <ise/iseRegionBinaryPredicateFromUnaryPredicate.hpp>
#include <ise/iseRegionUnaryPredicateThreshold.hpp>
#include <ise/iseQuantificationThresholds.hpp>
#include <ise/iseHistogram1D.hpp>
#include <ise/iseQuantificationThresholderMeanPlusTwoStdDev.hpp>

#define DBG_VERBOSE 0
#define DBG_VISU 0

//#define TIMING 1

#if DBG_VISU
#include <girlTools/girlToolsPainter.hpp>
#include <opencv2/highgui/highgui.hpp>
static
int 
dbg_bounded_rand(int min, int max)
{
  return static_cast<int>( min + (static_cast<float>(rand()) / RAND_MAX * (max - min + 1)) );
}
#endif //DBG_VISU


using namespace ise;

static const int DEFAULT_COLORSPACE = ISE_RGB_to_Lab;
static const int DEFAULT_NB_CLASSES = 6;
static const int DEFAULT_REGION_AREA = 144;
static const double DEFAULT_MEAN_COLOR_THRESHOLD = 57;

SegmentationClusteringGonogo1::SegmentationClusteringGonogo1(SegmentationContext *segContext)
  : SegmentationClustering(segContext),
    m_colorSpace(DEFAULT_COLORSPACE),
    m_nbClasses(DEFAULT_NB_CLASSES),
    m_regionArea(DEFAULT_REGION_AREA),
    m_flashLuminanceThreshold(DEFAULT_MEAN_COLOR_THRESHOLD)
{
  setSegContext(segContext);
}

void
SegmentationClusteringGonogo1::setSegContext(ise::SegmentationContext *s) 
{
  SegmentationClustering::setSegContext(s);
}
void 
SegmentationClusteringGonogo1::setParameters(int colorSpace, int nbClasses, int regionArea, double flashLuminanceThreshold)
{
  m_colorSpace = colorSpace;
  m_nbClasses = nbClasses;
  m_regionArea = regionArea;
  m_flashLuminanceThreshold = flashLuminanceThreshold;
} 

void 
SegmentationClusteringGonogo1::getParameters(int &colorSpace, int &nbClasses, int &regionArea, double &flashLuminanceThreshold) const
{
  colorSpace = m_colorSpace;
  nbClasses = m_nbClasses;
  regionArea = m_regionArea;
  flashLuminanceThreshold = m_flashLuminanceThreshold;
} 

static
void
mergeSmallRegionsButNotFlash(girl::regionDeque &regionsToMerge,
			     ise::SegmentationContext &segContext,
			     girl::shared_ptr<ise::RegionSetInterfaceMoments> rsimts,
			     girl::shared_ptr<ise::RegionSetInterfaceMean> rsim,
			     int regionArea,
			     double flashLuminanceThreshold)
{
  //- merge small regions (which area is inferior to @a regionArea)
  // without merging regions of too different luminances :
  //  we can only merge two regions which luminance is both inferior to @a flashLuminanceThreshold, or both superior to @a flashLuminanceThreshold]
  //
  // The binary predicate is 
  // (r1.area < regionArea || r2.area < regionArea) 
  // && ( (r1.lum < flashLuminanceThreshold && r2.lum < flashLuminanceThreshold) 
  //       || (r1.lum >= flashLuminanceThreshold && r2.lum >= flashLuminanceThreshold) )


  girl::shared_ptr<ise::RegionBinaryFunctionWithRegSetEuclideanDistance> distF(new ise::RegionBinaryFunctionWithRegSetEuclideanDistance(&segContext));
  distF->setRegSet(rsim);//B:API: why named setRegSet() and not setParameters() instead ???
  //B:TODO:OPTIM: no need of EuclideanDistance. EuclideanDistanceSquare would suffice !? We would spare a sqrt() !!!


  /// unary predicate ruptA1=(m0 < regionArea)
  girl::shared_ptr<ise::RegionUnaryFunctionWithRegSetIdentity> rufwrsiA1(new ise::RegionUnaryFunctionWithRegSetIdentity(&segContext));
  rufwrsiA1->setParameters(rsimts, 0); //m0 from moments
  girl::shared_ptr<ise::RegionUnaryPredicateThreshold> ruptA1(new ise::RegionUnaryPredicateThreshold(&segContext));
  ruptA1->setRegionUnaryFunction(rufwrsiA1);
  ruptA1->setComparisonOperator(ise::LESS);
  ruptA1->setThreshold(regionArea);
  /// binary predicate rbpfupA1 = ruptA1 applied to first region of the two
  girl::shared_ptr<ise::RegionBinaryPredicateFromUnaryPredicate> rbpfupA_1(new ise::RegionBinaryPredicateFromUnaryPredicate(&segContext));  
  rbpfupA_1->setRegionUnaryPredicate(ruptA1);
  rbpfupA_1->setSelectedRegion(0);
  /// binary predicate rbpfupA2 = ruptA1 applied to second region of the two
  girl::shared_ptr<ise::RegionBinaryPredicateFromUnaryPredicate> rbpfupA_2(new ise::RegionBinaryPredicateFromUnaryPredicate(&segContext));
  rbpfupA_2->setRegionUnaryPredicate(ruptA1);
  rbpfupA_2->setSelectedRegion(1);  
  
  /// binary predicate rbpbcA = (rbpfupA_1 || rbpfupA_2)
  girl::shared_ptr<ise::RegionBinaryPredicateBoolCombine> rbpbcA(new ise::RegionBinaryPredicateBoolCombine(&segContext));
  rbpbcA->setFirstPredicate(rbpfupA_1);
  rbpbcA->setSecondPredicate(rbpfupA_2);
  rbpbcA->setBoolOperator(ise::RegionBinaryPredicateBoolCombine::OR);



  /// unary predicate ruptB1=(mean0=L < flashLuminanceThreshold)
  girl::shared_ptr<ise::RegionUnaryFunctionWithRegSetIdentity> rufwrsiB1(new ise::RegionUnaryFunctionWithRegSetIdentity(&segContext));
  rufwrsiB1->setParameters(rsim, 0); //mean0=L from mean(moments(Lab))
  girl::shared_ptr<ise::RegionUnaryPredicateThreshold> ruptB1(new ise::RegionUnaryPredicateThreshold(&segContext));
  ruptB1->setRegionUnaryFunction(rufwrsiB1);
  ruptB1->setComparisonOperator(ise::LESS);
  ruptB1->setThreshold(flashLuminanceThreshold);
  /// binary predicate rbpfupB1 = ruptB1 applied to first region of the two
  girl::shared_ptr<ise::RegionBinaryPredicateFromUnaryPredicate> rbpfupB1_1(new ise::RegionBinaryPredicateFromUnaryPredicate(&segContext));  
  rbpfupB1_1->setRegionUnaryPredicate(ruptB1);
  rbpfupB1_1->setSelectedRegion(0);
  /// binary predicate rbpfupB1_2 = ruptB1 applied to second region of the two
  girl::shared_ptr<ise::RegionBinaryPredicateFromUnaryPredicate> rbpfupB1_2(new ise::RegionBinaryPredicateFromUnaryPredicate(&segContext));
  rbpfupB1_2->setRegionUnaryPredicate(ruptB1);
  rbpfupB1_2->setSelectedRegion(1);  
  
  /// binary predicate rbpbcB1 = (rbpfupB1_1 && rbpfupB1_2)
  girl::shared_ptr<ise::RegionBinaryPredicateBoolCombine> rbpbcB1(new ise::RegionBinaryPredicateBoolCombine(&segContext));
  rbpbcB1->setFirstPredicate(rbpfupB1_1);
  rbpbcB1->setSecondPredicate(rbpfupB1_2);
  rbpbcB1->setBoolOperator(ise::RegionBinaryPredicateBoolCombine::AND);



  /// unary predicate ruptB2=(mean0=L >= flashLuminanceThreshold)
  girl::shared_ptr<ise::RegionUnaryFunctionWithRegSetIdentity> rufwrsiB2(new ise::RegionUnaryFunctionWithRegSetIdentity(&segContext));
  rufwrsiB2->setParameters(rsim, 0); //mean0=L from mean(moments(Lab))
  girl::shared_ptr<ise::RegionUnaryPredicateThreshold> ruptB2(new ise::RegionUnaryPredicateThreshold(&segContext));
  ruptB2->setRegionUnaryFunction(rufwrsiB2);
  ruptB2->setComparisonOperator(ise::GREATER_OR_EQUAL);
  ruptB2->setThreshold(flashLuminanceThreshold);
  /// binary predicate rbpfupB2 = ruptB2 applied to first region of the two
  girl::shared_ptr<ise::RegionBinaryPredicateFromUnaryPredicate> rbpfupB2_1(new ise::RegionBinaryPredicateFromUnaryPredicate(&segContext));  
  rbpfupB2_1->setRegionUnaryPredicate(ruptB2);
  rbpfupB2_1->setSelectedRegion(0);
  /// binary predicate rbpfup2 = ruptB2 applied to second region of the two
  girl::shared_ptr<ise::RegionBinaryPredicateFromUnaryPredicate> rbpfupB2_2(new ise::RegionBinaryPredicateFromUnaryPredicate(&segContext));
  rbpfupB2_2->setRegionUnaryPredicate(ruptB2);
  rbpfupB2_2->setSelectedRegion(1);  
  
  /// binary predicate rbpbcB2 = (rbpfupB2_1 && rbpfupB2_2)
  girl::shared_ptr<ise::RegionBinaryPredicateBoolCombine> rbpbcB2(new ise::RegionBinaryPredicateBoolCombine(&segContext));
  rbpbcB2->setFirstPredicate(rbpfupB2_1);
  rbpbcB2->setSecondPredicate(rbpfupB2_2);
  rbpbcB2->setBoolOperator(ise::RegionBinaryPredicateBoolCombine::AND);
  

  girl::shared_ptr<ise::RegionBinaryPredicateBoolCombine> rbpbcB(new ise::RegionBinaryPredicateBoolCombine(&segContext));
  rbpbcB->setFirstPredicate(rbpbcB1);
  rbpbcB->setSecondPredicate(rbpbcB2);
  rbpbcB->setBoolOperator(ise::RegionBinaryPredicateBoolCombine::OR);
  
  girl::shared_ptr<ise::RegionBinaryPredicateBoolCombine> rbpbc(new ise::RegionBinaryPredicateBoolCombine(&segContext));
  rbpbc->setFirstPredicate(rbpbcA);
  rbpbc->setSecondPredicate(rbpbcB);
  rbpbc->setBoolOperator(ise::RegionBinaryPredicateBoolCombine::AND);



  ise::MergeDriverMeta md(&segContext);
  md.setDistanceFunction(distF);
  md.setAllowMergePredicate(rbpbc);
  
  ise::MergeAlgorithmUntil mergeAlgo(&segContext);
  girl::regionDeque createdRegions;
  mergeAlgo.apply(regionsToMerge, md, 1, createdRegions);
  
#if DBG_VERBOSE
  std::cerr<<"merge non-flash : "<<createdRegions.size()<<" regions  [toger all regions="<<segContext.getToger()->numRegions()<<"]\n";
#endif //DBG_VERBOSE


}


#if DBG_VISU
static
void
dbg_paintToger(const std::string &name, 
	       const girl::toger &toger, 
	       const girl::regionDeque &regions)
{
  cv::Mat imgP = cv::Mat::zeros(toger.height(), toger.width(), CV_8UC3); 
  for (girl::regionDeque::const_iterator it = regions.begin();
       it != regions.end(); ++it) {
    girl::tools::Painter<cv::Vec3b> p(imgP, CV_RGB(dbg_bounded_rand(0, 255), dbg_bounded_rand(0, 255), dbg_bounded_rand(0, 255)));
    
    toger.iterateOnRegionDomain(*it, p);
  }
  cv::namedWindow(name, 0);
  cv::imshow(name, imgP);
  cv::waitKey();
}

static
void
dbg_paintToger(const std::string &name, 
	       const girl::toger &toger)
{
  girl::regionDeque regions;
  toger.allRegions(regions);

  dbg_paintToger(name, toger, regions);
}
#endif //DBG_VISU


#include <ise/iseSparseHistogram1D.hpp>

//TODO: put in a common place !!! header ? in SparseHistogram directly ?
static
void
resetSparseHistogram(const girl::tools::ImageFeature &imgF, int channel, ise::SparseHistogram1DD &h)
{
  const int channels = imgF.channels();
  assert(channel < channels);
  const int height = imgF.height();
  const int width = imgF.width();
  for (int i=0; i<height; ++i) {
    const girl::tools::ImageFeature::type *imgFBuf = imgF.scanLine(i);
    for (int j=0; j<width; ++j) {
      const girl::tools::ImageFeature::type k = imgFBuf[channel];
      h.add(k, 1);
      imgFBuf += channels;
    }
  }
}

namespace {
  class RegionSparseHistoComputer : public girl::toger::pointIterator
  {
  public:  
    RegionSparseHistoComputer(const girl::tools::ImageFeature &imgF, int channel, 
			      ise::SparseHistogram1DD &h)
      :
      m_imgF(imgF),
      m_numChannels(imgF.channels()),
      m_channel(channel),
      m_h(h)
    {
    }
  
    virtual void processPoint(const girl::point &pt)
    {
      assert(pt.x()<m_imgF.width() && pt.y()<m_imgF.height());

      const girl::tools::ImageFeature::type *imgFBuf = m_imgF.scanLine(pt.y()) + pt.x()*m_numChannels;
      const girl::tools::ImageFeature::type k = imgFBuf[m_channel];
      m_h.add(k, 1);
    }

  protected:
    const girl::tools::ImageFeature &m_imgF;
    int m_numChannels;
    int m_channel;
    ise::SparseHistogram1DD &m_h;
  };
}

static
void
resetSparseHistogram(const girl::tools::ImageFeature &imgF, int channel, 
		     const girl::toger &tog, const girl::regionDeque &regions,
		     ise::SparseHistogram1DD &h)
{
  if (tog.numRegions() == regions.size()) {
    //all regions
    // => faster to do reset on whole image
    resetSparseHistogram(imgF, channel, h);
  }
  else {
    RegionSparseHistoComputer rshc(imgF, channel, h);
    
    girl::regionDeque::const_iterator it = regions.begin();
    const girl::regionDeque::const_iterator itEnd = regions.end();
    for ( ; it != itEnd; ++it)
      tog.iterateOnRegionDomain(*it, rshc);
  }

}

//namespace { //anonymous

//namespace ise {
namespace {

  class QuantificationThresholdsIF : public ise::SegmentationClustering
  {
  public:

    explicit QuantificationThresholdsIF(SegmentationContext *segContext);

    void setParams(girl::shared_ptr<girl::tools::ImageFeature> imgF, const std::vector< std::vector<double> > &thresholds);

    void setParams(girl::shared_ptr<girl::tools::ImageFeature> imgF, int channel, const std::vector<double> &thresholds);

    //TODO
    //getParams

    virtual void init(const girl::regionDeque  &/* regions */, int /* level */) {} 
    virtual void init(girl::region /* r */, int /* level */) {}

    virtual bool same(const girl::point &p, girl::direction dir);

  protected:
    
    girl::shared_ptr<const girl::tools::ImageFeature> m_imgF;
    std::vector< std::vector<double> > m_thresholds;
    int m_channel;

  };

  QuantificationThresholdsIF::QuantificationThresholdsIF(ise::SegmentationContext *segContext)
    : ise::SegmentationClustering(segContext),
      m_channel(0)
  {}

  //TODO: use a girl::shared_ptr<const girl::tools::ImageFeature> imgF ?
  void
  QuantificationThresholdsIF::setParams(girl::shared_ptr<girl::tools::ImageFeature> imgF, const std::vector< std::vector<double> > &thresholds)
  {
    assert(thresholds.size() < static_cast<size_t>(imgF->channels()));

    m_imgF = imgF;
    m_thresholds = thresholds;
    m_channel = std::max(thresholds.size(), (size_t)imgF->channels()); //invalid value on purpose.
  }

  //TODO: use a girl::shared_ptr<const girl::tools::ImageFeature> imgF ?
  void
  QuantificationThresholdsIF::setParams(girl::shared_ptr<girl::tools::ImageFeature> imgF, int channel, const std::vector<double> &thresholds)
  {
    assert(channel < imgF->channels());
#ifndef NDEBUG
    //- check that thresholds are in increasing order.
    for (size_t i=1; i<thresholds.size(); ++i) //start from 1
      assert(thresholds[i] > thresholds[i-1]);
#endif

    m_imgF = imgF;
    m_thresholds.clear();
    m_thresholds.resize(channel+1);
    m_thresholds[channel] = thresholds;
  }

  bool 
  QuantificationThresholdsIF::same(const girl::point &p1, girl::direction dir)
  {
    girl::point p2 = p1;
    p2.move(dir);

    assert(p1.x()<m_imgF->width() && p1.y()<m_imgF->height());
    assert(p2.x()<m_imgF->width() && p2.y()<m_imgF->height());
    
    const int numChannels = m_imgF->channels();

    const girl::tools::ImageFeature::type *pix1 = m_imgF->scanLine(p1.y()) + p1.x()*numChannels;
    const girl::tools::ImageFeature::type *pix2 = m_imgF->scanLine(p2.y()) + p2.x()*numChannels;
   
    assert(m_thresholds.size() <= static_cast<size_t>(numChannels));
    for (size_t k=0; k<m_thresholds.size(); ++k) {
    
      const std::vector<double> &thresholds = m_thresholds[k];
      const girl::tools::ImageFeature::type v1 = pix1[k];
      const girl::tools::ImageFeature::type v2 = pix2[k];
      
      double level1, level2;
      level1 = level2 = std::numeric_limits<double>::min();
      for (size_t i=0; i<thresholds.size(); ++i) {
	if (v1 > thresholds[i])
	  level1 = thresholds[i];
	if (v2 > thresholds[i])
	  level2 = thresholds[i];
      }
      if (level1 != level2)
	return false;
    }
    return true;    
  }
  

} //end of ise namespace



//Optimized version : only one threshold on first channel
//namespace ise {
namespace {

  class QuantificationThresholdsIF_0 : public ise::SegmentationClustering
  {
  public:

    explicit QuantificationThresholdsIF_0(SegmentationContext *segContext);

    void setParams(girl::shared_ptr<girl::tools::ImageFeature> imgF, double threshold);

    //TODO
    //getParams

    virtual void init(const girl::regionDeque  &/* regions */, int /* level */) {} 
    virtual void init(girl::region /* r */, int /* level */) {}

    virtual bool same(const girl::point &p, girl::direction dir);

  protected:
    
    cv::Mat m_mat;
    double m_threshold;
  };

  QuantificationThresholdsIF_0::QuantificationThresholdsIF_0(ise::SegmentationContext *segContext)
    : ise::SegmentationClustering(segContext),
      m_threshold(0)
  {}

  //TODO: use a girl::shared_ptr<const girl::tools::ImageFeature> imgF ?
  void
  QuantificationThresholdsIF_0::setParams(girl::shared_ptr<girl::tools::ImageFeature> imgF, double threshold)
  {
    m_mat = *(imgF->cvMat());
    m_threshold = threshold;
  }

  bool 
  QuantificationThresholdsIF_0::same(const girl::point &p1, girl::direction dir)
  {
    girl::point p2 = p1;
    p2.move(dir);

    assert(p1.x()<m_mat.cols && p1.y()<m_mat.rows);
    assert(p2.x()<m_mat.cols && p2.y()<m_mat.rows);
    
    const int numChannels = m_mat.channels();

    const girl::tools::ImageFeature::type *pix1 = m_mat.ptr<girl::tools::ImageFeature::type>(p1.y()) + p1.x()*numChannels;
    const girl::tools::ImageFeature::type *pix2 = m_mat.ptr<girl::tools::ImageFeature::type>(p2.y()) + p2.x()*numChannels;
    
    return (pix1[0] > m_threshold) == (pix2[0] > m_threshold);
  }
  

} //end of ise namespace




void 
SegmentationClusteringGonogo1::init(const girl::regionDeque &regions, int /* level */)
{
  assert(m_segContext->getToger());
  assert(m_segContext->getImage());

#ifdef TIMING
  double t1 = (double)cv::getTickCount();
#endif //TIMING

  m_toger = *(m_segContext->getToger()); //girl::toger(m_segContext->getImage()->width(), m_segContext->getImage()->height()); 

  ise::SegmentationContext segContext(&m_toger, m_segContext->getImage());

#ifdef TIMING
  double tf = (double)cv::getTickCount();
#endif //TIMING

  //- compute ImageFeature on m_colorSpace
  girl::shared_ptr<ise::ImageFeatureColorSpace> ifc(new ise::ImageFeatureColorSpace(&segContext, m_colorSpace));
  ifc->compute();
  //girl::shared_ptr<girl::tools::ImageFeature> m_imageF = computer.getImageFeature();
  //TODO:OPTIM: it would be faster if we could compute ImageFeatures only of pixels of selected regions...
  segContext.addImageFeatureComputer(ifc);

  girl::shared_ptr<ise::ImageFeatureColorSpace> ifcLab = ifc;
  if (m_colorSpace != ISE_RGB_to_Lab) {
    ifcLab.reset(new ise::ImageFeatureColorSpace(&segContext, ISE_RGB_to_Lab));
    ifcLab->compute();
    segContext.addImageFeatureComputer(ifcLab);
  }

  //- compute moments & mean on ImageFeature (on m_colorSpace)
  girl::shared_ptr<ise::RegionSetInterfaceMoments> rsimts(new ise::RegionSetInterfaceMoments(&segContext, ise::RegionSetInterface::ALWAYS, ifcLab));
  girl::shared_ptr<ise::RegionSetInterfaceMean> rsim(new ise::RegionSetInterfaceMean(&segContext, rsimts, ise::RegionSetInterface::ALWAYS));
  
  segContext.addRegionSet(rsimts);
  segContext.addRegionSet(rsim);


  const int chanL = 0;

#ifdef TIMING
  tf = ((double)cv::getTickCount() - tf)/cv::getTickFrequency();
  std::cout<<"SegmentationClusteringGonogo1::init():  feature computation time="<<tf<<"s"<<std::endl;
  double tfl = (double)cv::getTickCount();
#endif //TIMING

  //- compute threshold for flash
#if 0
  //TODO:OPTIM: useless conversion to Lab in Histogram1D::reset() to L

  ise::Histogram1D h;
  h.reset(*(segContext.getImage()), *(segContext.getToger()), ISE_RGB_to_Lab, chanL, regions);

  std::vector<double> thresholds;
  ise::QuantificationThresholderMeanPlusTwoStdDev().getThresholds(h, thresholds);

#else

  ise::SparseHistogram1DD sh;
  resetSparseHistogram(*(ifcLab->getImageFeature()), chanL, *(segContext.getToger()), regions, sh);
    
  std::vector<double> thresholds;
  ise::QuantificationThresholderMeanPlusTwoStdDev().getThresholds(sh, thresholds);

#endif //0

  m_flashLuminanceThreshold = std::max(thresholds[0], m_flashLuminanceThreshold);

#if DBG_VERBOSE
  std::cerr<<"threshold for flash  : "<<m_flashLuminanceThreshold<<" (mean+2stddev="<<thresholds[0]<<")\n";
#endif //DBG_VERBOSE

#ifdef TIMING
  tfl = ((double)cv::getTickCount() - tfl)/cv::getTickFrequency();
  std::cout<<"SegmentationClusteringGonogo1::init():  flash threshold time="<<tfl<<"s"<<std::endl;
  double ts = (double)cv::getTickCount();
#endif //TIMING

  //- split selected region @a regions with QuantificationThresholds to separate flash/non flash regions
#if 0

  ise::QuantificationThresholds quantif(&segContext);
  std::vector<int> levels;
  levels.push_back(static_cast<int>(m_flashLuminanceThreshold+0.5));
  quantif.setParams(ISE_RGB_to_Lab, chanL, levels);

#else

#if 0
  //ise::
  QuantificationThresholdsIF quantif(&segContext);
  quantif.setParams(ifcLab->getImageFeature(), chanL, thresholds);
#else
  //ise::
  QuantificationThresholdsIF_0 quantif(&segContext); //optimized version : only one threshold on first channel !
  quantif.setParams(ifcLab->getImageFeature(), thresholds[0]);  
#endif //0

#endif //0


  ise::SplitAlgorithmDefault splitAlgo(&segContext);
  ise::SplitDriverDefault splitDriver(&segContext);
  girl::regionDeque createdRegions;
  assert(! regions.empty());
  splitAlgo.apply(regions, splitDriver, quantif, 0, createdRegions);

#ifdef TIMING
  ts = ((double)cv::getTickCount() - ts)/cv::getTickFrequency();
  std::cout<<"SegmentationClusteringGonogo1::init(): time split [QuantificationThresholds] ="<<ts<<"s"<<std::endl;

  std::cerr<<" createdRegions.size()="<<createdRegions.size()<<"\n";

  double trf = (double)cv::getTickCount();
#endif //TIMING


#if DBG_VERBOSE
  std::cerr<<"split flash/non-flash : "<<createdRegions.size()<<" regions [toger all regions="<<m_toger.numRegions()<<"]\n";
#endif //DBG_VERBOSE
#if DBG_VISU
  dbg_paintToger("split0", m_toger, createdRegions);
#endif //DBG_VISU

  //- get regions of non flash in nonFlashRegions & remove them from createdRegions
  std::vector<ise::RegionFeature_Data_Type> buffer(rsim->getFeatureDim(), 0);
  ise::RegionFeature rf(&buffer[0]);
  const size_t L_index = 0;

  girl::regionDeque nonFlashRegions;
  for (girl::regionDeque::iterator it = createdRegions.begin(); it != createdRegions.end();  ) {
    rsim->getRegionFeatures(*it, rf);
    if (rf[L_index] < m_flashLuminanceThreshold) {
      nonFlashRegions.push_back(*it);
      it = createdRegions.erase(it);
    }
    else {
      ++it;
    }
  }
  
#if DBG_VERBOSE
  std::cerr<<"non-flash regions: "<<nonFlashRegions.size()<<" regions  (flash regions="<<createdRegions.size()<<") [toger all regions="<<m_toger.numRegions()<<"]\n";
#endif //DBG_VERBOSE

#ifdef TIMING
  trf = ((double)cv::getTickCount() - trf)/cv::getTickFrequency();
  std::cout<<"SegmentationClusteringGonogo1::init(): time remove flash regions="<<trf<<"s"<<std::endl;
  double tsk = (double)cv::getTickCount();
#endif //TIMING

  if (! nonFlashRegions.empty()) {

    //- split non flash regions nonFlashRegions with Generic K-Means in m_nbClasses on ImageFeature on m_colorSpace
    girl::shared_ptr<ise::Clusterer> clusterer(new ise::ClustererGenericKMeans(m_nbClasses));
    ise::SegmentationClusteringImageFeatureClusterer segClustKMeans(&segContext);  
    segClustKMeans.setParameters(ifc, clusterer);
    
    ise::SplitAlgorithmDefault splitAlgo2(&segContext);
    ise::SplitDriverDefault splitDriver2(&segContext);
    
    splitAlgo2.apply(nonFlashRegions, splitDriver2, segClustKMeans, 0, createdRegions); //appended in createdRegions to be able to do the merge!
    
#if DBG_VERBOSE
    std::cerr<<"split non-flash : "<<createdRegions.size()<<" regions  [toger all regions="<<m_toger.numRegions()<<"]\n";
#endif //DBG_VERBOSE
    
#if DBG_VISU
    dbg_paintToger("split", m_toger, createdRegions);
#endif //DBG_VISU
    

    mergeSmallRegionsButNotFlash(createdRegions, segContext,
				 rsimts, rsim,
				 m_regionArea,
				 m_flashLuminanceThreshold);
    
  }
  else {
    std::cerr<<"Warning: gonogo1: nonFlashRegions.size()="<<nonFlashRegions.size()<<"\n";
    
    m_toger = *(m_segContext->getToger());
  }

#ifdef TIMING
  tsk = ((double)cv::getTickCount() - tsk)/cv::getTickFrequency();
  std::cout<<"SegmentationClusteringGonogo1::init(): time split kmeans="<<tsk<<"s"<<std::endl;
#endif //TIMING



#if DBG_VISU
  dbg_paintToger("merge", m_toger);
#endif //DBG_VISU


#ifdef TIMING
    t1 = ((double)cv::getTickCount() - t1)/cv::getTickFrequency();
    std::cout<<"SegmentationClusteringGonogo1::init(): total time "<<t1<<"s"<<std::endl;
#endif //TIMING
}




bool
SegmentationClusteringGonogo1::same(const girl::point &p1, girl::direction dir)
{
  return m_toger.inSameRegion(p1, dir);
}
