/*
  Set of functions to compute features on regions.
  Copyright (C) 2005-2009 Jean-Philippe Domenger, Achille Braquelaire, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "iseRegionTools.hpp"

#include <cassert>
#include <limits>

#include <girlTools/girlToolsVector.hpp>

#include <girl/euclideanPath.hpp>

#include "iseRegionColorimetricFeatures.hpp"
#include "iseStat.hpp"
//#include "iseGenericImage.hpp"

namespace ise {


  void RegionTools_computeColorimetricMoments(const girl::toger &tog,
					      const girl::tools::ImageFeature *img,
					      girl::region regId,
					      double &m0,
					      double *m1,
					      double *m2)
  {
    ise::MomentsComputerIF it(img, &m0, m1, m2); //will set m0, m1, m2 to zero.
  
    tog.iterateOnRegionDomain(regId, it); //will fill m0, m1 & m2.
  }


  ise::RegionColorimetricFeatures *
  RegionTools_createRegionFeatures(const girl::toger &tog,
				   const girl::tools::ImageFeature *img,
				   girl::region regId)
  {
    std::vector<double> m(img->channels()*2+1); //B:TODO:OPTIM: use std::tr1::array ?
    double *m0 = &m[0];
    double *m1 = &m[1];
    double *m2 = &m[4];

    RegionTools_computeColorimetricMoments(tog, img, regId, *m0, m1, m2); //fill m0, m1, m2.
  
    return new ise::RegionColorimetricFeatures(regId, *m0, m1, m2, img->channels()); //will copy m1, m0, m2;
  }

  

  double
  RegionTools_computeTangentVarianceOnOuterBoundary(const girl::toger &tog, 
						    girl::region regId)
  {
    girl::contour c;
    tog.regionOuterBoundary(regId, c);

    double tangent_m0 = 0.0;
    double tangent_m1 = 0.0;
    double tangent_m2 = 0.0;

    girl::euclideanPath::computeTangentMoments(c,
					       tangent_m0,
					       tangent_m1,
					       tangent_m2);

    return iseStat_iseComputeVariance(tangent_m0, tangent_m1, tangent_m2);
  }

  double
  RegionTools_computeTangentMeanOnOuterBoundary(const girl::toger &tog,
						girl::region regId)
  {
    girl::contour c;
    tog.regionOuterBoundary(regId, c);

    double tangent_m0 = 0.0;
    double tangent_m1 = 0.0;
    double tangent_m2 = 0.0;

    girl::euclideanPath::computeTangentMoments(c,
					       tangent_m0,
					       tangent_m1,
					       tangent_m2);

    return iseStat_iseComputeMean(tangent_m0, tangent_m1);
  }


  /*
    Search index where point @a p and direction @a d
    are found in contour @a c, starting after @a startIndex moves, at point @a cpt of countour.
  */
  static
  size_t
  P_searchIndex(const girl::contour &c, 
		size_t startIndex, 
		girl::point startPt, 
		girl::point p, girl::direction d)
  {
    assert(startIndex <= c.length()); 
    size_t i = startIndex;
    girl::point cpt = startPt;
    girl::contour::const_dir_iterator itD = c.begin() + startIndex;
    const girl::contour::const_dir_iterator itDEnd = c.end();
    for ( ; itD != itDEnd; ++itD) {
      if (cpt == p && *itD == d)
	return i;
      cpt.move(*itD);
      ++i;
    }
    return i;
  }

#ifndef NDEBUG
  /*
    Check that path @a p is found in contour @c, starting after @a startIndex moves.
  */
  static
  bool
  P_checkPathInContour(const girl::contour &c, 
		       size_t startIndex, 
		       const girl::path &p)
  {
    assert(p.length() > 0);

    assert(P_searchIndex(c, 0, c.startPt(), p.startPt(), p[0]) == startIndex);//check searchIndex from 0 == startIndex
    assert(startIndex + p.length() <= c.length()); 

    girl::contour::const_dir_iterator itc = c.begin() + startIndex;
    girl::contour::const_dir_iterator itp = p.begin();
    const girl::contour::const_dir_iterator itpEnd = p.end();
    for ( ; itp != itpEnd; ++itp, ++itc) {
      if (*itp != *itc)
	return false;
    }
    return true;
  }
#endif


  double
  RegionTools_computeTangentVarianceOnCommonBoundary(const girl::toger &tog, 
						     girl::region reg1Id,
						     girl::region reg2Id)
  {
    girl::pathDeque b;

    tog.regionsCommonBoundary(reg1Id, reg2Id, b);


    girl::pathDeque::const_iterator it = b.begin();
    const girl::pathDeque::const_iterator itEnd = b.end();


    double sTangent_m0 = 0.0;
    double sTangent_m1 = 0.0;
    double sTangent_m2 = 0.0;
  
  
    assert(! b.empty());

    const girl::path &firstPth = *it;
    const girl::point pt = firstPth.startPt();
    const girl::dart d = tog.getDart(pt, *(firstPth.begin()));
    assert(d != 0);
    const girl::face f = tog.lambda(d);
    girl::region r = f;
    const girl::face fm = tog.parent(f);
    if (fm != 0) //f is infinite
      r = tog.lambda(tog.alpha(d));
    assert(r == reg1Id || r == reg2Id);

    girl::contour c1;
    tog.regionOuterBoundary(r, c1);

    int nbDirs = 0;

    size_t startIndex = 0;
    girl::point startPt = c1.startPt();

    for ( ; it != itEnd; ++it) {
      const girl::path &pth = *it;

      const size_t len = pth.length();
      nbDirs += len;

      double tangent_m0 = 0.0;
      double tangent_m1 = 0.0;
      double tangent_m2 = 0.0;

      //We want to find start index of each path pth in contour c1.
      //We consider that paths returned by toger::regionsCommonBoundary() are ordered in the contour c1
      // thus we seek each path from the end of the previously found one [and not from the start of the contour].
      
      startIndex = P_searchIndex(c1, startIndex, startPt, pth.startPt(), pth[0]);
      assert(P_checkPathInContour(c1, startIndex, pth));

      girl::euclideanPath::computeTangentMoments(c1, startIndex, len,
						 tangent_m0,
						 tangent_m1,
						 tangent_m2);
      
      startIndex += len;
      startPt = pth.endPt(); //O(N)

      assert(tangent_m0 >= 0.0);
      assert(tangent_m2 >= 0.0);

      sTangent_m0 += tangent_m0;
      sTangent_m1 += tangent_m1;
      sTangent_m2 += tangent_m2;

    }

  
    if (nbDirs == 0) {
      assert(sTangent_m0 < std::numeric_limits<double>::epsilon());
      return 0.0;
    }

    assert(sTangent_m0 > 0.0);

  

    return iseStat_iseComputeVariance(sTangent_m0, sTangent_m1, sTangent_m2);
  }

#if 0

  static 
  float
  P_getGradient(girl::direction d,
		int &x, int &y, //may be modified
		const ise::GenericImage<float, 1> &normalizedGradModuleX,
		const ise::GenericImage<float, 1> &normalizedGradModuleY)
  {

    float gradient = 0.0f;

    switch (d)
      {
      case girl::RIGHT:
	gradient += normalizedGradModuleY.getPixel(x+1, y+1)[0];
	++x;
	break;
      case girl::UP:
	gradient += normalizedGradModuleX.getPixel(x+1, y)[0];
	--y;
	break;
      case girl::LEFT:
	gradient += normalizedGradModuleY.getPixel(x, y)[0];
	--x;
	break;
      case girl::DOWN:
	gradient += normalizedGradModuleX.getPixel(x, y+1)[0];
	++y;
	break;
      case girl::INVALID:
      default:
	assert(false);
	break;
      }
  
    return gradient;
  }


  double
  RegionTools_computeMeanGradientOnOuterBoundary(const girl::toger &tog, 
						 girl::region regId,
						 const ise::GenericImage<float, 1> &normalizedGradModuleX,
						 const ise::GenericImage<float, 1> &normalizedGradModuleY)
  {

    girl::contour c;
    tog.regionOuterBoundary(regId, c);


    size_t nbPoint = c.length();
  
    double sumGradient = 0;
  
    const girl::point startPoint = c.startPt();

    int x = startPoint.x();
    int y = startPoint.y();
  
    girl::path::const_dir_iterator itD = c.begin();
    const girl::path::const_dir_iterator itDEnd = c.end();
  
    for ( ; itD != itDEnd; ++itD)
      {
	const girl::direction d = *itD;
      
	sumGradient += P_getGradient(d, x, y, normalizedGradModuleX, normalizedGradModuleY); //may modify x & y
      }

    const double meanGradient = sumGradient / static_cast<double>(nbPoint);
  
    return meanGradient;
  }

#endif //0

  size_t
  RegionTools_computeLengthOfCommonBoundary(const girl::toger &tog,
                                     girl::region reg1Id,
                                     girl::region reg2Id)
  {
    girl::pathDeque paths;
    tog.regionsCommonBoundary(reg1Id, reg2Id, paths);
    size_t res = 0;
    girl::pathDeque::const_iterator it = paths.begin();
    const girl::pathDeque::const_iterator itEnd = paths.end();
    for (; it != itEnd; ++it)
      res += it->length();
    return res;
  }


}//end ise namespace
