/*
  Copyright (C) 2005-2010 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef ISEREGIONSETINTERFACEVARIANCE_HPP_
#define ISEREGIONSETINTERFACEVARIANCE_HPP_

#include <girl/girlCommon.hpp>

#include "iseRegionSetInterface.hpp"
#include "iseRegionSetInterfaceMoments.hpp"
namespace ise
{

  class GIRL_API RegionSetInterfaceVariance : public RegionSetInterface
  {
    static const RegionSetInterface::CacheType DEFAULT_CACHE_STRATEGY = RegionSetInterface::NONE;

  public:

    typedef girl::shared_ptr<ise::RegionSetInterfaceMoments> MomentsPtr;

    RegionSetInterfaceVariance(ise::SegmentationContext* segContext,
			       ise::RegionSetInterface::CacheType cacheType = DEFAULT_CACHE_STRATEGY)
    : RegionSetInterface(segContext),
      m_regSetMoment(),
      m_momentMemory(NULL)
    {
      setCacheState(cacheType);
    }

    RegionSetInterfaceVariance(ise::SegmentationContext* segContext,
			       MomentsPtr regSetMoment,
			       ise::RegionSetInterface::CacheType cacheType = DEFAULT_CACHE_STRATEGY)
    : RegionSetInterface(segContext),
      m_regSetMoment(regSetMoment),
      m_momentMemory(NULL)
    {
      if (m_regSetMoment)
        allocateMemory();

      setCacheState(cacheType);
    }

    virtual
    ~RegionSetInterfaceVariance()
    {
      freeMemory();
    }

    
    inline const
      std::string uri() const
    {
      assert(m_regSetMoment);
      std::stringstream ss(std::ios::in|std::ios::out);
      ss << "Variance" << URI_SEPARATOR << m_regSetMoment->uri();
      return ss.str();
    }
    

    inline size_t
    getFeatureDim() const
    {
      assert(m_regSetMoment);
      return (m_regSetMoment->getFeatureDim() - 1) / 2;
    }

    /*
    inline void
    getFeatureTitle(std::string& title) const
    {
      title = "Variance";
    }
    */

    inline void
    getDimDescription(std::vector<std::string>& dimDesc) const
    {
      dimDesc.clear();
      std::stringstream ss;
      const size_t s = getFeatureDim();
      dimDesc.resize(s);
      for (size_t i = 0; i < s; ++i) {
	ss.clear();
	ss.str("");
	ss<<"Variance ["<<i<<"]";
	dimDesc[i] = ss.str();
      }
    }

    void
    setParameters(MomentsPtr regSetMoment)
    {
      assert(regSetMoment);
      m_regSetMoment = regSetMoment;
      allocateMemory();
    }

    void
      getRequiredComputers(FeatureComputerListType &req) const
    {
      assert(m_regSetMoment);
      req.insert(m_regSetMoment->uri());
    }

    const std::string
    help() const
    {
      return
	"Computes the variance value for pixel features in the region"
	"Parameter: the ImageFeature the statistics are collected from";
    }

  protected:

    void
    compute(const girl::region& id, ise::RegionFeature& feature)
    {
      assert(m_regSetMoment);
      const size_t size = getFeatureDim();
      ise::RegionFeature m0(m_momentMemory);
      ise::RegionFeature m1((&m_momentMemory[1]));
      ise::RegionFeature m2((&m_momentMemory[1+size]));
      m_regSetMoment->getMoments(id, m0, m1, m2);
      if (GIRL_likely(m0[0] > 0)) {
	const double inv_m0 = 1. / m0[0];
	for (size_t i = 0; i < size; ++i)
	  feature[i] = (m2[i] - m1[i]*m1[i]*inv_m0)*inv_m0;
      }
      else {
	for (size_t i = 0; i < size; ++i)
	  feature[i] = 0.;
      }      
    }

    void
    mergeFeatures(RegionFeature& kept,
		  RegionFeature& /*del*/,
		  const girl::region& keptId,
		  const girl::region& /*delId*/)
    {

      // TODO:OPTIM What is better, merge (requires 2 get, 2*N multiplications and 1 addition
      // or recompute (1 get, N divisions).
      compute(keptId, kept);
    }

    inline void
    allocateMemory()
    {
      freeMemory();
      assert(m_regSetMoment);
      m_momentMemory = new RegionFeature_Data_Type[m_regSetMoment->getFeatureDim()];
    }

    inline void
    freeMemory()
    {
      delete [] m_momentMemory;
    }

  protected:

    MomentsPtr m_regSetMoment;

    /**
     * Used for optimization (avoiding multiple allocation/deallocation at each
     * call of compute() function). Allocate memory once at the construction, use
     * this memory for all calls and release at the destruction
     */
    RegionFeature_Data_Type *m_momentMemory;

  };

}

#endif /* ISEREGIONSETINTERFACEVARIANCE_HPP_ */
