/*
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "iseRegionSetInterfaceGeometric.hpp"

#include "iseGeometricComputer.hpp"
#include <girlTools/girlToolsRegionInfos.hpp>


namespace ise
{

  RegionSetInterfaceGeometric::RegionSetInterfaceGeometric(ise::SegmentationContext* segContext,
							   ise::RegionSetInterface::CacheType cacheType)
  : RegionSetInterface(segContext)
  {
    setCacheState(cacheType);
  }

  RegionSetInterfaceGeometric::~RegionSetInterfaceGeometric() {}

  
  const std::string
  RegionSetInterfaceGeometric::uri() const
  {
    std::stringstream ss(std::ios::in|std::ios::out);
    ss << __FILE__ << std::endl;
    return ss.str();
  }
  

  inline size_t
  RegionSetInterfaceGeometric::getFeatureDim() const
  {
    return FEATURE_DIMENSION;
  }

  inline void
  RegionSetInterfaceGeometric::getDimDescription(std::vector<std::string>& dimDesc) const
  {
    dimDesc.resize(FEATURE_DIMENSION);
    
    dimDesc[AREA] = "Area";
    dimDesc[PERIMETER] = "Perimeter";
    dimDesc[COMPACTNESS] = "Compactness";
    dimDesc[X_COORD_CENTER] = "X coordinate center ";
    dimDesc[Y_COORD_CENTER] = "Y coordinate center";
    dimDesc[MAJOR_AXIS_LENGTH] = "Major axis length";
    dimDesc[MINOR_AXIS_LENGTH] = "Minor axis length";
    dimDesc[ANGLE_MAJOR_AXIS_ABSCICE] = "Angle Major axis - absice";
    dimDesc[MAJOR_MINOR_AXIS_RATIO] = "Major/Minor axis";
  }

  const std::string
  RegionSetInterfaceGeometric::help() const
  {
    return "Computes the following geometric region descriptor: area, perimeter, compactness";
  }

  void
  RegionSetInterfaceGeometric::compute(const girl::region &id, ise::RegionFeature &feature)
  {
    assert(m_segContext);
    girl::toger *tog = m_segContext->getToger();
    assert(tog);

    ise::GeometricComputer gc;
    gc.process(*tog, id);
    gc.finalize();

    const double area = gc.getArea();
    assert(area > 0);
    const double perimeter = static_cast<double>(girl::tools::computePerimeter(*tog, id));
    const double compactness = perimeter*perimeter/area;
    feature[AREA] = area;
    feature[PERIMETER] = perimeter;
    feature[COMPACTNESS] = compactness;

    feature[X_COORD_CENTER] = gc.getXCoordCenter();
    feature[Y_COORD_CENTER] = gc.getYCoordCenter();

    feature[MAJOR_AXIS_LENGTH] = gc.getMajorAxisLength();
    feature[MINOR_AXIS_LENGTH] = gc.getMinorAxisLength();

    feature[ANGLE_MAJOR_AXIS_ABSCICE] = gc.getAngleMajorAxisAbscice();
    
    if (feature[MINOR_AXIS_LENGTH] > 0)
      feature[MAJOR_MINOR_AXIS_RATIO] = feature[MAJOR_AXIS_LENGTH]/feature[MINOR_AXIS_LENGTH]; ////sqrt(feature[MAJOR_AXIS_LENGTH]/feature[MINOR_AXIS_LENGTH]);
    else 
      feature[MAJOR_MINOR_AXIS_RATIO] = 0;
  }

  void
  RegionSetInterfaceGeometric::mergeFeatures(RegionFeature &kept, RegionFeature &/*del*/,
      const girl::region &keptReg, const girl::region &/*delId*/)
  {
    compute(keptReg, kept);
  }


}
