/*
 * iseRegionSetInterfaceClustering.cpp
 *
 *  Created on: Apr 15, 2011
 *      Author: vieux
 */

#include "iseRegionSetInterfaceClustering.hpp"

#include "iseRegionFeatureSelector.hpp"

namespace ise
{

  static const RegionFeature_Data_Type UNCLASSIFIED = -1.0;

#ifndef NDEBUG
  static bool
  P_checkParameters(const std::vector<girl::shared_ptr<RegionSetInterface> > &regSets,
      const std::vector<bool> &mask,
      const girl::regionDeque &regs)
  {
    size_t dim = 0;
    for (size_t i = 0; i < regSets.size(); ++i) {
        dim += regSets[i]->getFeatureDim();
    }
    bool ok = (dim == mask.size());
    ok &= (regs.size() > 0);
    return ok;
  }
#endif //NDEBUG

  RegionSetInterfaceClustering::RegionSetInterfaceClustering(ise::SegmentationContext *segContext,
      FeatureComputer::CacheType /*cache*/) :
      RegionSetInterface(segContext),
      m_regSets(),
      m_regs(),
      m_regsToClusterMap(),
      m_mask(),
      m_clusterer()

  {
  }

  RegionSetInterfaceClustering::~RegionSetInterfaceClustering()
  {
  }

  const std::string
  RegionSetInterfaceClustering::uri() const
  {
    std::stringstream ss(std::ios::in|std::ios::out);
    ss << __FILE__ << m_clusterer->uri();
    for (size_t i = 0; i < m_regSets.size(); ++i) {
      ss << m_regSets[i]->uri();
    }
    for (size_t i = 0; i < m_mask.size(); ++i) {
      ss << m_mask[i];
    }
    // TODO: add uri of clusterer
    return ss.str();
  }

  size_t
  RegionSetInterfaceClustering::getFeatureDim() const
  {
    return 1;
  }

  void
  RegionSetInterfaceClustering::getDimDescription(std::vector<std::string> &dimDesc) const
  {
    dimDesc.clear();
    dimDesc.push_back("Index of cluster");
  }

  void
  RegionSetInterfaceClustering::setParameters(const std::vector<girl::shared_ptr<RegionSetInterface> > &regSets,
					      const std::vector<bool> &mask,
					      const girl::regionDeque &regs,
					      const girl::shared_ptr<ise::Clusterer> &clusterer)
  {
    m_regSets = regSets;
    m_mask = mask;
    m_regs = regs;
    m_clusterer = clusterer;
    assert(P_checkParameters(regSets, mask, regs));
  }

  void
  RegionSetInterfaceClustering::getRequiredComputers(FeatureComputerListType& list) const
  {
    std::vector<girl::shared_ptr<RegionSetInterface> >::const_iterator it = m_regSets.begin();
    const std::vector<girl::shared_ptr<RegionSetInterface> >::const_iterator itEnd = m_regSets.end();
    for (; it != itEnd; ++it) {
        list.insert((*it)->uri());
    }
  }


  const std::string
  RegionSetInterfaceClustering::help() const
  {
    return "Compute clustering of regions based on the user-chosen features and user-chosen clustering algorithm";
  }

  bool
  RegionSetInterfaceClustering::computeClustering(void)
  {
    assert(m_segContext);
    assert(m_regSets.size() > 0);
    assert(m_regs.size() > 0);
    assert(m_mask.size() > 0);
    assert(m_clusterer);

    cv::Mat data;
    cv::Mat labels;
    RegionFeatureSelector::selectFeatures(m_regs, m_regSets, m_mask, data);
    m_clusterer->compute(data, labels);
    m_regsToClusterMap.clear();
    assert(labels.type() == CV_32S);
    girl::regionDeque::const_iterator regIt = m_regs.begin();
    const girl::regionDeque::const_iterator regItEnd = m_regs.end();
    for (size_t i = 0; regIt != regItEnd; ++regIt, ++i) {
        double cIndex = static_cast<double>(*labels.ptr<int>(i));
        m_regsToClusterMap[*regIt] = cIndex;
    }

    return true;
  }

  void
  RegionSetInterfaceClustering::compute(const girl::region &id, ise::RegionFeature &feature)
  {
    if (m_regsToClusterMap.size() == 0) {
#ifndef NDEBUG
      const bool ok = 
#endif
	computeClustering();
      assert(ok);
    }
    assert(m_regsToClusterMap.size() > 0);
    girl::hashMap<girl::region, double>::iterator it = m_regsToClusterMap.find(id);
    if (it != m_regsToClusterMap.end())
      feature[0] = it->second;
    else
      feature[0] = UNCLASSIFIED;
  }

  void
  RegionSetInterfaceClustering::mergeFeatures(RegionFeature& kept,
      RegionFeature& ,
      const girl::region&,
      const girl::region& )
  {
    kept[0] = UNCLASSIFIED;
  }

}
