/*
  Copyright (C) 2005-2010 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef ISEREGIONSETINTERFACE_HPP_
#define ISEREGIONSETINTERFACE_HPP_

#include <iostream>
#include <string>


#include <girl/girlCommon.hpp>
#include <girl/girlSharedPtr.hpp>
#include <girl/girlToger.hpp>

#include "iseFeatureComputer.hpp"
#include "iseRegionFeature.hpp"
#include "iseRegionFeatureAllocator.hpp"
#include "iseRegionFeatureCacheState.hpp"
#include "iseRegionFeatureStorage.hpp"
//#include "iseURI.hpp"

namespace ise {

  class RegionFeatureCacheState;
  class RegionFeatureStorage;
  class SegmentationContext;

  /**
   * Interface for all the Region Sets. Each region set holds a particular
   * Region Feature. The interface defines basic handling of the features
   * (insert, replace, remove, delete). The actual feature computation, merging,
   * etc are delegated to the FeatureComputer algorithm.
   */
  class GIRL_API RegionSetInterface : public FeatureComputer {

    /**
     * The following classes are declared as friends because they need to access
     * to the compute and mergeRegions functions defined as protected in this interface.
     */
    friend class RegionFeatureCacheStateLazy;
    friend class RegionFeatureCacheStateAlwaysStore;
    friend class RegionFeatureCacheStateNeverStore;
    friend class RegionFeatureStorageHash;

  public:
    /**
     * Use this separator in URI function to isolate the different parts of the URI
     * such as the parameters
     */
    static const char URI_SEPARATOR = '#';



  public:

    /**
     * By default the cache strategy is set to NONE during construction
     * Use setCacheState to change this behavior
     */
    RegionSetInterface(ise::SegmentationContext *segContext);

    virtual
    ~RegionSetInterface();


    virtual void
    setCacheState(FeatureComputer::CacheType cacheType);


    /**
     * Empty the region set, i.e. release all the region features contained in the set
     */
    virtual void
    reset();


    /**
     * Return a copy of the features of region @id in the pointer @rf
     *
     * @warning: memory must be preallocated, features are copied.
     */
    virtual void
    getRegionFeatures(const girl::region &id, RegionFeature& rf);


    //B:HACK 
    void setRegionFeatures(const girl::region &id, RegionFeature& rf);


//    RegionFeature_Data_Type
//    getFeature(const girl::region id, const std::string &featureDescription);

//    RegionFeature_Data_Type
//    getFeature(const RegionFeature& rf, const std::string &featureDescription);

    /**
      * Return the index of the feature matching the description given in @param featureDescription
      * return -1 otherwise
      */
    //size_t getFeatureIndex(const std::string &featureDescription);


    /**
     * Insert region @param id in the region set.
     * @param toger the segmentation toger.

     * This function will compute the features for the region @param id
     * and store them in the region set.
     *
     * @warning region @a id must not already be in RegionSet.
     */
    void
    insertRegion(const girl::region &id);

    /**
     * Will insert the region feature in the representation. Region feature could
     * have been obtained elsewhere, e.g. from a rename region operation.
     * @param features: the region features to insert
     * @param id: id of the region the features belong to
     *
     * @warning region @a id must not already be in RegionSet.
     */
    //   virtual void insertRegion(RegionFeature *features, const girl::region &id) = 0;

    /**
     * Update the features for the region @param id.
     *
     * If region @a id was already there, its previous RegionFeature is freed 
     * and replaced by a new one.
     * If region @a id was not already there, it is inserted.
     */
    void
    replaceRegion(const girl::region &id);

    /**
     * Remove features associated with @param reg for region @a id from region set,
     * and return it.
     *
     * @warning @a id must be in regSet
     * @warning it is client responsibility to remove returned region features.
     *
     * @sa deleteRegion.
     *
     * TODO:UNSURE Is this function necessary?? Why not a rename region?
     */
    // virtual RegionFeature* removeRegion(girl::region id) = 0;


    void
    renameRegion(girl::region oldId, girl::region newId);

    /**
     * Delete the region feature associated with region @param id
     * and remove the region from the region set
     */
    void
    deleteRegion(girl::region id);

    /**
     * Merge features of regions @param keptReg and @param delReg into @param keptReg
     */
    void
    mergeRegions(girl::region keptReg, girl::region delReg);



    bool
    isTogerObserver() const
    { return true; }

    // functions of toger::observer
    void
    newRegion(const girl::regionDeque&);

    void
    updateRegion(const girl::regionDeque&);

    /**
     * Debugging function
     */
    void
    checkAllRegions();




    // VIRTUAL FUNCTIONS
  public:
    /**
     * @return URI unique identifier of the RegionSetInterface
     */
    //TODO:REMOVE
    // virtual const URI uri() const = 0;

    /**
     * Initialize the content by calling insertRegion for all regions currently
     * present in the toger.
     * WARNING: This function should be called only once
     */
    //R: moved it to public because I need it in my project
    // IncrementalClustering/src/main.cpp function mergeSmallRegions
    // We must check why ...
    virtual void
    compute();


  protected:

    void
    update();



    /**
     * Actual algorithm which compute the feature for region @id
     */
    virtual void
    compute(const girl::region &id,
        ise::RegionFeature &feature) = 0;

    /**
     * Merge two region features. The actual merging is let to the concrete implementation,
     * since some features might be easily merged (e.g. added) while some can not.
     * Implementation must guarantee that keptFeatures contains the merged features, whether
     * it is recomputed or added. If recomputed, it means that the regions have already been
     * merged in the toger, and we call compute using @keptId. We need both the features and
     * id of merged regions in case merge is based on dependent features that must be
     * retrieved (e.g. mean)
     */
    virtual void
    mergeFeatures(RegionFeature &keptFeatures,
        RegionFeature& delFeatures,
        const girl::region &keptId,
        const girl::region &delId) = 0;

  protected:

    ise::RegionFeatureCacheState* m_cacheStrategy;

    ise::RegionFeatureStorage*    m_storage;

    ise::RegionFeatureAllocator*  m_allocator;



  };

  typedef girl::shared_ptr<ise::RegionSetInterface> RegionSetInterfacePtr;
}

#endif /* ISEREGIONSETINTERFACE_HPP_ */
