/*
  Copyright (C) 2005-2011 Remi Vieux remi.vieux@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "iseRegionSetInterface.hpp"

#include "iseRegionFeatureAllocatorStack.hpp"
#include "iseRegionFeatureCacheStateAlwaysStore.hpp"
#include "iseRegionFeatureCacheStateLazy.hpp"
#include "iseRegionFeatureCacheStateNeverStore.hpp"
#include "iseRegionFeatureStorageHash.hpp"
#include "iseSegmentationContext.hpp"

namespace ise
{
  RegionSetInterface::RegionSetInterface(ise::SegmentationContext* segContext) :
      FeatureComputer(segContext),
      m_cacheStrategy(NULL),
      m_storage(NULL),
      m_allocator(NULL)
  {
    m_allocator = new RegionFeatureAllocatorStack();
    m_cacheStrategy = new RegionFeatureCacheStateNeverStore;
  }

  RegionSetInterface::~RegionSetInterface()
  {
    delete m_storage;
    delete m_cacheStrategy;
    delete m_allocator;
  }

  void
  RegionSetInterface::setCacheState(CacheType cacheType)
  {
    switch (cacheType)
    {
    case NONE:
      {

        switch(m_cacheType)
        {
        case NONE: // NONE->NONE
          break;
        case LAZY: // LAZY->NONE
        case ALWAYS: // ALWAYS->NONE
          delete m_storage;
          m_storage = NULL;
          delete m_cacheStrategy;
          m_cacheStrategy = new RegionFeatureCacheStateNeverStore();
          break;
        default:
          assert(false);
          break;
        }

        break;
      }
    case LAZY:
      {
        switch(m_cacheType)
        {
        case NONE: // NONE->LAZY
          // there should be no storage
          m_storage = new RegionFeatureStorageHash(*m_allocator);
          delete m_cacheStrategy;
          m_cacheStrategy = new RegionFeatureCacheStateLazy;
          break;
        case LAZY: // LAZY->LAZY
          break;
        case ALWAYS: // ALWAYS->LAZY
          // TODO: should we not remove all features? Otherwise not sure
          // that features are up to date when calling get
          // m_storage->reset();
          delete m_cacheStrategy;
          m_cacheStrategy = new RegionFeatureCacheStateLazy;
          break;
        default:
          assert(false);
          break;
        }
        break;
      }

    case ALWAYS:
      {
        switch(m_cacheType)
        {
        case ALWAYS: // ALWAYS->ALWAYS
          {
            break;
          }
        default:
          {
            assert(false);
            break;
          }
        case NONE: // NONE->ALWAYS
          assert(!m_storage);
          m_storage = new RegionFeatureStorageHash(*m_allocator);
	  //REM: no break !
        case LAZY: // LAZY->ALWAYS
          // populate
          delete m_cacheStrategy;
          m_cacheStrategy = new RegionFeatureCacheStateAlwaysStore();
	  update(); // ??? //       compute();
          break;

        }
        break;
      }

    default:
      {
        assert(false);
        break;
      }
    }

    m_cacheType = cacheType;
  }

  void
  RegionSetInterface::reset()
  {
    if (m_storage)
      m_storage->reset();
  }

  void
  RegionSetInterface::getRegionFeatures(const girl::region &id, RegionFeature& rf)
  {
    assert(m_cacheStrategy);
    m_cacheStrategy->getRegionFeatures(id, m_storage, *this, m_allocator, rf);
  }

  //B:HACK
  void 
  RegionSetInterface::setRegionFeatures(const girl::region &id, RegionFeature& rf)
  {
    assert(m_cacheStrategy);
    
    m_cacheStrategy->setRegionFeatures(id, m_storage, *this, m_allocator, rf);
  }

  /*
  size_t
  RegionSetInterface::getFeatureIndex(const std::string &featureDescription)
  {
   std::vector<std::string> dimDesc;
   getDimDescription(dimDesc);
   std::vector<std::string>::const_iterator it = std::find(dimDesc.begin(), dimDesc.end(), featureDescription);
   return it == dimDesc.end() ? -1 : it - dimDesc.begin();
  }
  */

  void
  RegionSetInterface::insertRegion(const girl::region &id)
  {
    assert(m_cacheStrategy);
    m_cacheStrategy->insertRegion(id, m_storage, *this, m_allocator);
  }


  void
  RegionSetInterface::replaceRegion(const girl::region &id)
  {
    assert(m_cacheStrategy);
    m_cacheStrategy->replaceRegion(id, m_storage, *this, m_allocator);
  }


  void
  RegionSetInterface::renameRegion(girl::region oldId, girl::region newId)
  {
    assert(m_cacheStrategy);
    m_cacheStrategy->renameRegion(oldId, newId, m_storage);
  }

  void
  RegionSetInterface::deleteRegion(girl::region id)
  {
    assert(m_cacheStrategy);
    //   assert(m_storage == NULL || m_storage->exists(id));
    m_cacheStrategy->deleteRegion(id, m_storage);
  }



  void
  RegionSetInterface::mergeRegions(girl::region keptReg, girl::region delReg)
  {
    assert(m_cacheStrategy);
    // wrong assert when cachestate = lazy, regions can not exist
    //  assert(m_storage == NULL || (m_storage->exists(keptReg) && m_storage->exists(delReg)));
    m_cacheStrategy->mergeRegions(keptReg, delReg, m_storage, *this);
  }

  void
  RegionSetInterface::newRegion(const girl::regionDeque &regions)
  {
    girl::regionDeque::const_iterator it = regions.begin();
    const girl::regionDeque::const_iterator itEnd = regions.end();
    for (; it != itEnd; ++it) {
        insertRegion(*it);
    }
  }

  void
  RegionSetInterface::updateRegion(const girl::regionDeque &regions)
  {
    girl::regionDeque::const_iterator it = regions.begin();
    const girl::regionDeque::const_iterator itEnd = regions.end();
    for (; it != itEnd; ++it) {
        replaceRegion(*it);
    }
  }


  void
  RegionSetInterface::checkAllRegions()
  {
    assert(m_segContext);
    assert(m_segContext->getToger());
    girl::regionDeque regs;
    m_segContext->getToger()->allRegions(regs);
    girl::regionDeque::const_iterator it = regs.begin();
    const girl::regionDeque::const_iterator itEnd = regs.end();
    std::vector<RegionFeature_Data_Type> values(getFeatureDim(), 0);
    ise::RegionFeature rf(&(values[0]));
    std::cout << "Region ";
    for ( ; it != itEnd; ++it) {
        std::cout << *it << ": ";
        //m_cacheStrategy->getRegionFeatures(*it, m_storage, *this, m_allocator, rf);
        getRegionFeatures(*it, rf);
        for (size_t j = 0; j < getFeatureDim(); ++j) {
            std::cout << rf[j] << " ";
        }
        std::cout << std::endl;
    }
  }

  void
  RegionSetInterface::compute()
  {
    if (!m_storage)
      return;
    assert(m_storage);
    assert(m_segContext);
    assert(m_segContext->getToger());
    assert(m_cacheStrategy);
    girl::regionDeque regs;
    m_segContext->getToger()->allRegions(regs);
 //   m_segContext->newRegion(regs);
    // this version and not the above to not recompute already existing features (i.e. when switching from lazy to always)
    girl::regionDeque::const_iterator regIt = regs.begin();
    const girl::regionDeque::const_iterator regItEnd = regs.end();
    for ( ; regIt != regItEnd; ++regIt) {
      if (!m_storage->exists(*regIt)) {
	m_cacheStrategy->insertRegion(*regIt, m_storage, *this, m_allocator); //REM: will only do something if cacheStrategy is always...
      }
    }
  }

  void
  RegionSetInterface::update()
  {
      compute();
  }


}
