/*
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef ISEREGIONFEATURETOOLS_HPP_
#define ISEREGIONFEATURETOOLS_HPP_

#include <cmath>

#include "iseRegionFeature.hpp"

namespace ise {
/**
 * Define in this header the functions that can be computed on the RegionFeature
 * R: is it possible to use functors? What should change in the interface?
 */


  /**
   * Interface for function objects that return the norm of the region feature,
   * i.e. it's distance from the origin in a vector space.
   */
  class NormFunctor : public std::unary_function<RegionFeature, double>
  {
  public:

    virtual ~NormFunctor() {}
    
    /**
     * Class Constructor.
     * @parameter size: the size of the region feature we are dealing with,
     * i.e. the size of the array, since it is not stored in the RegionFeature
     * itself
     */
    NormFunctor(size_t size) : m_featureSize(size) {}

    virtual double operator()(const RegionFeature &rf) = 0;

  protected:
    size_t m_featureSize;
  };


  /**
   * Interface for function object that compute a distance between RegionFeatures
   */
  class DistanceFunctor : public std::binary_function<RegionFeature, RegionFeature, double>
  {
  public:

    virtual ~DistanceFunctor() {}
    
    /**
     * Class Constructor
     * @parameter size: size of the array of the region feature
     */
    DistanceFunctor(size_t size) : m_featureSize(size) {}
    virtual double operator()(const RegionFeature&, const RegionFeature&) = 0;

  protected:
    size_t m_featureSize;
  };







  class RegionFeatureL1Norm : public NormFunctor
  {
  public:
    explicit RegionFeatureL1Norm(size_t size): NormFunctor(size) {}

    inline double operator()(const RegionFeature &rf)
    {
      double res = 0;
      for (size_t i = 0; i < m_featureSize; ++i) {
        res += fabs(rf[i]);
      }
      return res;
    }
  };
  
  class RegionFeatureL1Dist : public DistanceFunctor
  {
  public:
    explicit RegionFeatureL1Dist(size_t size) : DistanceFunctor(size) {}

    inline double operator()(const RegionFeature &a, const RegionFeature &b)
    {
      // TODO: rewrite with accumulate? better for inlining
      double res = 0;
      for (size_t i = 0; i < m_featureSize; ++i) {
        res += fabs(a[i] - b[i]);
      }
      return res;
    }
  };

  class RegionFeatureL2Norm : public NormFunctor
  {
  public:
    explicit RegionFeatureL2Norm(size_t size) : NormFunctor(size) {}
    
    inline double operator()(const RegionFeature &rf)
    {
      double res = 0;
      for (size_t i = 0; i < m_featureSize; ++i) {
	res += rf[i]*rf[i];
      }
      return sqrt(res);
    }
  };
  
  class RegionFeatureL2Dist : public DistanceFunctor
  {
  public:
    explicit RegionFeatureL2Dist(size_t size) : DistanceFunctor(size) {}

    inline double operator()(const RegionFeature &a, const RegionFeature &b)
    {
      double res = 0;
      for (size_t i = 0; i < m_featureSize; ++i) {
        double val = a[i] - b[i];
        res += val*val;
      }
      return sqrt(res);
    }
  };

  class RegionFeatureL2DistSq : public DistanceFunctor
   {
   public:
     explicit RegionFeatureL2DistSq(size_t size) : DistanceFunctor(size) {}

     inline double operator()(const RegionFeature &a, const RegionFeature &b)
     {
       double res = 0;
       for (size_t i = 0; i < m_featureSize; ++i) {
         double val = a[i] - b[i];
         res += val*val;
       }
       return res;
     }
   };
  


}

#endif /* ISEREGIONFEATURETOOLS_HPP_ */
