/*
  Copyright (C) 2005-2010 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef ISEREGIONFEATURECACHELAZY_HPP_
#define ISEREGIONFEATURECACHELAZY_HPP_


#include <girl/girlCommon.hpp>

#include "iseRegionFeatureCacheState.hpp"

/**
 * Implementation of CacheState using lazy initialization. Feature
 * are only computed and stored when performing get operations
 */
namespace ise
{
  class GIRL_API RegionFeatureCacheStateLazy : public ise::RegionFeatureCacheState
  {
  public:

    /**
     * In lazy mode, do the computation and store the feature
     * afterward
     */
    void
    getRegionFeatures(const girl::region& id,
        ise::RegionFeatureStorage* memStorage,
        ise::RegionSetInterface& regSet,
        ise::RegionFeatureAllocator* allocator,
        ise::RegionFeature &feature);

    /**
     * Do nothing in lazy mode
     */
    inline void
    insertRegion(const girl::region&,
        ise::RegionFeatureStorage*,
        ise::RegionSetInterface&,
        ise::RegionFeatureAllocator*)
    {}
    
    /**
     * In lazy mode, only remove previous features. They will be recomputed on demand.
     */
    inline void
    replaceRegion(const girl::region &id,
		    ise::RegionFeatureStorage *storage,
		    ise::RegionSetInterface &,
		    ise::RegionFeatureAllocator *) 
    {
      deleteRegion(id, storage);
    }

    inline void
    deleteRegion(const girl::region& id,
        ise::RegionFeatureStorage* storage)
    {
      assert(storage);
     // TODO:OPTIM
     // with hashmap, 2 find are done, one for exists and one for erase
      if (storage->exists(id))
	storage->erase(id);
    }


    /**
     * Rename if @oldId is already in the storage, otherwise do nothing
     */
    inline void
    renameRegion(const girl::region& oldId,
        const girl::region& newId,
        ise::RegionFeatureStorage* storage)
    {
      assert(storage);
      if (storage->exists(oldId)) {
	storage->rename(oldId, newId);
      }
    }

    /**
     * Invalidate the kept region. Will be recomputed on demand
     * The deleted region will be deleted by a call to delete
     */
    inline void
    mergeRegions(const girl::region& kept,
        const girl::region&,
        ise::RegionFeatureStorage* storage,
        ise::RegionSetInterface&)
    {
      assert(storage);
      if (storage->exists(kept))
        storage->erase(kept);

    }


    /**
     * Do nothing in lazy mode
     */
    inline void
    mergeRegions(const girl::region&,
        const girl::region&,
        ise::RegionFeatureStorage*)
    {}


  };

}


#endif /* ISEREGIONFEATURECACHELAZY_HPP_ */
