/*
  Copyright (C) 2005-2010 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef REGIONBINARYFUNCTIONWITHREGSET_HPP_
#define REGIONBINARYFUNCTIONWITHREGSET_HPP_

#include <girl/girlSharedPtr.hpp>

#include "iseRegionBinaryFunction.hpp"
#include "iseRegionSetInterface.hpp"

/**
 * Interface for all the functions whose definition domain is two regions
 * and that return a real, e.g. eval functions, and are bounded to a region set,
 * that is they ask the region set for the region features.
 * Operator() must be subclassed to do the actual computation of the function.
 */

namespace ise
{

  class GIRL_API RegionBinaryFunctionWithRegSet : public RegionBinaryFunction
  {

  public:
    explicit RegionBinaryFunctionWithRegSet(ise::SegmentationContext *segContext)
    : RegionBinaryFunction(segContext),
      m_regSet(),
      m_memory1(NULL),
      m_memory2(NULL)
    {}

    virtual
    ~RegionBinaryFunctionWithRegSet() {
      freeMemory();
    }

    void
    setRegSet(girl::shared_ptr<ise::RegionSetInterface> regSet)
    {
      assert(regSet);
      m_regSet = regSet;
      allocateMemory();
    }

  protected:

    void
    allocateMemory()
    {
      freeMemory();
      assert(m_regSet);
      //REM: allocate both at once.
      m_memory1 = new ise::RegionFeature_Data_Type[m_regSet->getFeatureDim() * 2];
      m_memory2 = m_memory1 + m_regSet->getFeatureDim();
    }
    
    void
    freeMemory()
    {
      //REM: free both at once
      delete [] m_memory1;
    }


    protected:

    girl::shared_ptr<RegionSetInterface> m_regSet;

    //Memory allocated for querying region features, 
    //since they must be copied from the reg set.
    //One per region
    RegionFeature_Data_Type* m_memory1;
    RegionFeature_Data_Type* m_memory2;

  };

  typedef girl::shared_ptr<RegionSetInterface> RegionSetInterfacePtr;

}

#endif /* REGIONBINARYFUNCTIONWITHREGSET_HPP_ */
