/*
  Merge driver clusterer implementation.
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Achille Braquelaire, Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "iseMergeDriverClusterer.hpp"

#include "iseSegmentationContext.hpp"

#include "iseClusterer.hpp"

#include "iseRegionSetInterface.hpp"

using namespace ise;

MergeDriverClusterer::MergeDriverClusterer(ise::SegmentationContext *segContext)
: MergeDriver(segContext),
  m_clusterer(),
  m_regSet(),
  m_regToIndexMap(),
  m_labels()
{
}

MergeDriverClusterer::~MergeDriverClusterer()
{

}

void
MergeDriverClusterer::init(const girl::regionDeque &regionsToMerge, int)
{
  assert(m_clusterer);
  assert(m_regSet);

  m_regToIndexMap.clear();
  //std::vector<double*> data;
  int sizeDeque = regionsToMerge.size();

  cv::Mat data(sizeDeque, m_regSet->getFeatureDim(), CV_MAKETYPE(cv::DataDepth<RegionFeature_Data_Type>::value, 1));
  m_labels = cv::Mat::zeros(sizeDeque, 1, CV_32S);

  girl::regionDeque::const_iterator rIt = regionsToMerge.begin();
  const girl::regionDeque::const_iterator rItEnd = regionsToMerge.end();
  for (int i = 0 ; rIt != rItEnd ; ++i, ++rIt) {
      //std::vector<RegionFeature_Data_Type> buff(m_regSet->getFeatureDim(), 0);
      ise::RegionFeature feature(data.ptr<RegionFeature_Data_Type>(i));
      m_regSet->getRegionFeatures(*rIt, feature);
      m_regToIndexMap[*rIt] = i;
  }

  m_clusterer->compute(data, m_labels);
}

void 
MergeDriverClusterer::filter(girl::regionDeque &/*regionsToMerge*/, int /*level*/)
{

}

double 
MergeDriverClusterer::distance(girl::region /*reg1Id*/, girl::region /*reg2Id*/)
{
  return 0;
}

bool 
MergeDriverClusterer::allowed(girl::region reg1Id, girl::region reg2Id, int /*level*/)
{
  assert(m_clusterer);
  assert(m_regSet);
  assert(m_labels.type() == CV_32S && m_labels.rows > 0);

  girl::hashMap<girl::region, int>::iterator it;
  it = m_regToIndexMap.find(reg1Id);
  assert(it != m_regToIndexMap.end());
  int i1 = it->second;
  it = m_regToIndexMap.find(reg2Id);
  assert(it != m_regToIndexMap.end());
  int i2 = it->second;

  return m_labels.at<int>(i1) == m_labels.at<int>(i2);

  //  std::vector<RegionFeature_Data_Type> buff1(m_regSet->getFeatureDim(), 0);
  //  ise::RegionFeature feature1(&buff1[0]);
  //  m_regSet->getRegionFeatures(reg1Id, feature1);
  //
  //  std::vector<RegionFeature_Data_Type> buff2(m_regSet->getFeatureDim(), 0);
  //  ise::RegionFeature feature2(&buff2[0]);
  //  m_regSet->getRegionFeatures(reg2Id, feature2);
  //
  //  ise::Clusterer::LabelType c1 = m_clusterer->getCluster(&buff1[0], m_regSet->getFeatureDim());
  //  ise::Clusterer::LabelType c2 = m_clusterer->getCluster(&buff2[0], m_regSet->getFeatureDim());
  //
  //  return c1 == c2;
}

girl::region 
MergeDriverClusterer::selectCandidate(girl::region regId, int level)
{
  assert(regId > 0);
  assert(m_segContext);
  assert(m_segContext->getToger());

  girl::regionDeque neighbours;
  m_segContext->getToger()->regionNeighbourhood(regId,  m_neighbourhoodMode, m_neighbourhoodMark, neighbours);

  return selectCandidate(regId, neighbours, level);
}

girl::region 
MergeDriverClusterer::selectCandidate(girl::region regId, const girl::regionDeque &regions, int /*level*/)
{
  assert(m_clusterer);
  assert(m_regSet);

  girl::hashMap<girl::region, int>::iterator it;
  it = m_regToIndexMap.find(regId);
  assert(it != m_regToIndexMap.end());
  int c = it->second;

  //  std::vector<RegionFeature_Data_Type> buff(m_regSet->getFeatureDim(), 0);
  //  ise::RegionFeature feature(&buff[0]);
  //  m_regSet->getRegionFeatures(regId, feature);
  //
  //  ise::Clusterer::LabelType c = m_clusterer->getCluster(&buff[0], m_regSet->getFeatureDim());
  //
  size_t sizeRegions = regions.size();

  for (size_t i = 0 ; i < sizeRegions ; ++i) {
      //      std::vector<RegionFeature_Data_Type> buffTmp(m_regSet->getFeatureDim(), 0);
      //      ise::RegionFeature featureTmp(&buffTmp[0]);
      //      m_regSet->getRegionFeatures(regions[i], featureTmp);
      //
      //      ise::Clusterer::LabelType cTmp = m_clusterer->getCluster(&buffTmp[0], m_regSet->getFeatureDim());
      it = m_regToIndexMap.find(regions[i]);
      assert(it != m_regToIndexMap.end());
      int cTmp = it->second;
      if (c == cTmp)
        return regions[i];
  }

  return regions[0];
}

girl::region 
MergeDriverClusterer::execute(girl::region reg1Id, girl::region reg2Id)
{
  assert(reg1Id != 0);
  assert(reg2Id != 0);
  assert(m_segContext);
  assert(m_segContext->getToger());
  return m_segContext->getToger()->mergeRegions(reg1Id, reg2Id); 
}

void 
MergeDriverClusterer::setParameters(const girl::shared_ptr<ise::RegionSetInterface> &regSet, const girl::shared_ptr<ise::Clusterer> &clusterer)
{
  m_clusterer = clusterer;
  m_regSet = regSet;
}
