/*
  Merge algorithm default implementation.
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Achille Braquelaire, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "iseMergeAlgorithmDefault.hpp"

#include <algorithm>

#include <girl/girlUnorderedSet.hpp>
#include <girl/girlHashFunc.hpp>

#include "iseMergeDriver.hpp"
//#include "iseRegionSet.hpp"
#include "iseSegmentationContext.hpp"
#include "iseRegionColorimetricFeatures.hpp"
#include "iseRegionTools.hpp"

using namespace ise;

/**
 * Try to merge at least once each region of @a regionsToMerge.
 *
 * @a return the number of merge done.
 */
size_t
MergeAlgorithmDefault::apply_aux(girl::regionDeque &regionsToMerge,
				 ise::MergeDriver &driver,
				 int level,
				 girl::regionDeque &keptRegions)
{

  driver.filter(regionsToMerge, level);


  // REMI: Unused the region set purpose was to check validity of the
  // regions, we use the MAP instead.
  // std::string regSetUri;
  // driver.getRegSetUri(regSetUri);
  // const ise::RegionSetInterface *regSet = m_segContext->getRegionSet(regSetUri);
  
  typedef girl::unorderedSet<girl::region> RegionsSet;
  RegionsSet keptRegionsSet; //stores final regions after merge 
  // (and also help to know if a region has already been merged)
  RegionsSet erasedRegionsSet; //stores the merged regions


  size_t numMerges = 0;

  //REM: we pop elements from deque because erasing in the middle of a deque invalidates its iterators !

  while ( ! regionsToMerge.empty() ) 
    {
      const girl::region r = regionsToMerge.front();
      assert(r != 0);
      regionsToMerge.pop_front();

      if (keptRegionsSet.find(r) == keptRegionsSet.end()) //r is not the result of a previous merge [we don't want a region to be merged twice]
	{
	  

	  // Original code:
	  //const ise::RegionColorimetricFeatures *currentRegion = regSet->getRegionFeatures(r);
	  // New Interface:
	  // const ise::RegionFeature* currentRegion = (regSet != NULL) ? regSet->getRegionFeatures(r) : NULL;
	  // if (currentRegion != NULL) //check that region is still valid (r may have been merged before [and, see below, removedRegion are not removed from regionsToMerge...])
	  
	  // REMI: Merge drivers that do not rely on any Region Feature (ex: gradient, tangeant) yield
	  // no regSet, hence avoid querying for features which are not used! Hence we query a map of the erased
	  // regions
	  // bool valid = (regSet == NULL) ? keptRegionsSet.find(r) : (regSet->getRegionFeatures(r) != NULL);
	  const bool valid = (erasedRegionsSet.find(r) == erasedRegionsSet.end());
	  if (valid)
	    {
	      
	      const girl::region mergeCandidate = driver.selectCandidate(r, level);
	      assert(mergeCandidate != r);
	      
	      if (mergeCandidate != 0) 
		{
		  const girl::region keptRegion = driver.execute(r, mergeCandidate);

		  if (keptRegion != 0) //merge has been executed 
		    {
		    
		      girl::region removedRegion = mergeCandidate;
		      if (keptRegion == mergeCandidate)
			removedRegion = r;
		    
		      keptRegionsSet.insert(keptRegion);
		    
		      keptRegionsSet.erase(removedRegion);
		      erasedRegionsSet.insert(removedRegion);
		    
		    
		      assert(keptRegionsSet.find(keptRegion) != keptRegionsSet.end());    //keptRegion in map
		      assert(keptRegionsSet.find(removedRegion) == keptRegionsSet.end()); //removedRegion not in map

		      ++numMerges;
		    }
		  else {
		    //a candidate but merge not done
		    //region not merged thus kept
		    keptRegionsSet.insert(r);
		  }

		}
	      else {
		//no candidate
		//region not merged thus kept
		keptRegionsSet.insert(r);
	      }

	    }

	}

    }
  
  RegionsSet::const_iterator itM = keptRegionsSet.begin();
  const RegionsSet::const_iterator itMEnd = keptRegionsSet.end();
  for ( ; itM != itMEnd; ++itM) 
    keptRegions.push_back(*itM);
  //keptRegionsSet.clear();

  return numMerges;
}


void
MergeAlgorithmDefault::apply(girl::regionDeque &regionsToMerge,
			     ise::MergeDriver &driver,
			     int level,
			     girl::regionDeque &keptRegions)
{
  driver.init(regionsToMerge, level);

  apply_aux(regionsToMerge, driver, level, keptRegions);
}
