/*
  Copyright (C) 2005-2010 Vieux Remi, Roigt Julian, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "iseImageFeatureColorSpace.hpp"

//parallel_for
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
#include <opencv2/core/core.hpp>
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
#include <opencv2/core/internal.hpp>
#endif

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION > 261
#include <opencv2/core/utility.hpp> //for ParallelLoopBody
#endif


using namespace ise;


ImageFeatureColorSpace::ImageFeatureColorSpace(ise::SegmentationContext *segC, int colorSpace)
  : ImageFeatureColorimetricComputer(segC)
  , m_colorSpace(colorSpace)
{
}

const std::string
ImageFeatureColorSpace::baseUri() const
{
  return "ColorSpace";
}

const std::string
ImageFeatureColorSpace::uri() const
{
  std::stringstream ss(std::ios::in|std::ios::out);
  ss<<baseUri()<< IMAGE_FEATURE_URI_SEPARATOR <<m_colorSpace;
  return ss.str();
}

void
ImageFeatureColorSpace::compute()
{
  assert(m_segContext);
  assert(m_segContext->getImage());

  girl::tools::Image *img = m_segContext->getImage();
  assert(img);

  compute(*img);
}


//B:TODO:OPTIM: there is a clear performance improvement for complexe color spaces transforms (Lab, Luv, ...) on big images
//   However, il would be probably better to give more work than a single line to each thread !?!

class ImageConverter
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
 : public cv::ParallelLoopBody
#endif
{
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
  typedef cv::Range Range;
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
  typedef cv::BlockedRange Range;
#endif

public:

  ImageConverter(const girl::tools::Image &img,
		 girl::tools::ImageFeature &imgF,
		 int colorSpace)
    : m_img(img), m_imgF(imgF), m_colorSpace(colorSpace)
  {
    assert(m_img.channels() >= 3);
    assert(m_imgF.channels() == 3);
  }


#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240

  virtual void operator()( const Range& range ) const
  {
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    const int start = range.start;
    const int end = range.end;
#else
    const int start = range.begin();
    const int end = range.end();
#endif
    processRows(start, end);
  }

protected:

#endif //OpenCV >= 240


  inline void processRows(int start, int end) const
  {
    const int chanIn = m_img.channels();
    const int chanOut = m_imgF.channels();

    const int width = m_img.width();
    const int colorSpace = m_colorSpace;
      
    for (int j=start; j<end; ++j) {
      const girl::tools::Image::type *in = m_img.scanLine(j);
      girl::tools::ImageFeature::type *out = m_imgF.scanLine(j);
      
      for (int i = 0 ; i < width ; ++i) {
	int x, y, z;
	ise::conversion::changeFromRGB(colorSpace, in[0], in[1], in[2], x, y, z);
	out[0] = static_cast<girl::tools::ImageFeature::type>(x);
	out[1] = static_cast<girl::tools::ImageFeature::type>(y);
	out[2] = static_cast<girl::tools::ImageFeature::type>(z);
	
	in += chanIn;
	out += chanOut;
      }
    }
  }

protected:

  const girl::tools::Image &m_img;
  girl::tools::ImageFeature &m_imgF;
  int m_colorSpace;
};

void
ImageFeatureColorSpace::compute(const girl::tools::Image &img)
{
  int width = img.width(); 
  int height = img.height();

  int dim = ImageFeatureColorSpace::FEATURE_DIMENSION;
  if (m_colorSpace == ISE_RGB_to_GRAY)
    dim = 1;
  
  m_imgF = girl::shared_ptr<girl::tools::ImageFeature>(new girl::tools::ImageFeature(width, height, dim));
  
  girl::tools::ImageFeature &imgF = *(m_imgF.get());

#if 1

  if (m_colorSpace != ISE_RGB_to_GRAY) {
    ImageConverter ic(img, imgF, m_colorSpace);
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    cv::parallel_for_(cv::Range(0, height), ic);
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
    cv::parallel_for(cv::BlockedRange(0, height), ic);
#else
    ic.processRows(0, height);
#endif
  }
  else if (dim == 1) {
      
    if (img.isContinuous() && imgF.isContinuous()) {
      width *= height;
      height = 1;
    }
    const int colorSpace = m_colorSpace;
    
    const int incS = img.formatSize();
    const int incD = imgF.formatSize();

    for ( int j = 0 ; j < height ; ++j) {
      const girl::tools::Image::type *imgBuf = img.scanLine(j);
      girl::tools::ImageFeature::type *imgfBuf = imgF.scanLine(j);
      
      for (int i = 0 ; i < width ; ++i) {
	int x, y, z;
	ise::conversion::changeFromRGB(colorSpace, imgBuf[0], imgBuf[1], imgBuf[2], x, y, z);
	imgfBuf[0] = static_cast<girl::tools::ImageFeature::type>(x);
      }
      imgBuf += incS;
      imgfBuf += incD;
    }

  }



#else



  if (img.isContinuous() && imgF.isContinuous()) {
    width *= height;
    height = 1;
  }

  const int colorSpace = m_colorSpace;

  const int incS = img.formatSize();
  const int incD = imgF.formatSize();


  if (colorSpace == ISE_RGB_to_RGB) {
    assert(dim == 3);

    for ( int j = 0 ; j < height ; ++j) {
      const girl::tools::Image::type *imgBuf = img.scanLine(j);
      girl::tools::ImageFeature::type *imgfBuf = imgF.scanLine(j);
      
      for (int i = 0 ; i < width ; ++i) {
	imgfBuf[0] = static_cast<girl::tools::ImageFeature::type>(imgBuf[0]);
	imgfBuf[1] = static_cast<girl::tools::ImageFeature::type>(imgBuf[1]);
	imgfBuf[2] = static_cast<girl::tools::ImageFeature::type>(imgBuf[2]);
      }
      imgBuf += incS;
      imgfBuf += incD;
    }

  }
  else if (dim == 3) {

    for ( int j = 0 ; j < height ; ++j) {
      const girl::tools::Image::type *imgBuf = img.scanLine(j);
      girl::tools::ImageFeature::type *imgfBuf = imgF.scanLine(j);
      
      for (int i = 0 ; i < width ; ++i) {
	int x, y, z;
	ise::conversion::changeFromRGB(colorSpace, imgBuf[0], imgBuf[1], imgBuf[2], x, y, z);
	imgfBuf[0] = static_cast<girl::tools::ImageFeature::type>(x);
	imgfBuf[1] = static_cast<girl::tools::ImageFeature::type>(y);
	imgfBuf[2] = static_cast<girl::tools::ImageFeature::type>(z);
      }
      imgBuf += incS;
      imgfBuf += incD;
    }

  }
  else if (dim == 1) {
      
    for ( int j = 0 ; j < height ; ++j) {
      const girl::tools::Image::type *imgBuf = img.scanLine(j);
      girl::tools::ImageFeature::type *imgfBuf = imgF.scanLine(j);
      
      for (int i = 0 ; i < width ; ++i) {
	int x, y, z;
	ise::conversion::changeFromRGB(colorSpace, imgBuf[0], imgBuf[1], imgBuf[2], x, y, z);
	imgfBuf[0] = static_cast<girl::tools::ImageFeature::type>(x);
      }
      imgBuf += incS;
      imgfBuf += incD;
    }

  }

#endif


}

void
ImageFeatureColorSpace::setParameters(int colorSpace)
{
  m_colorSpace = colorSpace;
}

void
ImageFeatureColorSpace::getParameters(int &colorSpace) const
{
  colorSpace = m_colorSpace;
}

void 
ImageFeatureColorSpace::getDimDescription(std::vector<std::string> &dimDesc) const
{
  dimDesc.clear();
  for (size_t i = 0 ; i < getFeatureDim() ; ++i) {
    std::stringstream ss;
    ss<<"ColorSpace "<<m_colorSpace<<" "<<i;
    dimDesc.push_back(ss.str());
  }
}

