/*
  Copyright (C) 2013 Boris Mansencal, Jean-Philippe Domenger boris.mansencal@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "iseImageFeatureColorProjection.hpp"

//parallel_for
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
#include <opencv2/core/core.hpp>
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
#include <opencv2/core/internal.hpp>
#endif

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION > 261
#include <opencv2/core/utility.hpp> //for ParallelLoopBody
#endif


using namespace ise;


ImageFeatureColorProjection::ImageFeatureColorProjection(ise::SegmentationContext *segC)
  : ImageFeatureColorimetricComputer(segC),
    m_a00(0), m_a01(0), m_a02(0), m_a10(0), m_a11(0), m_a12(0), 
    m_tx(0), m_ty(0),
    m_colorSpace(0)
{
}

void
ImageFeatureColorProjection::setParameters(int colorSpace, 
					   double a00, double a01, double a02, 
					   double a10, double a11, double a12,
					   double tx, double ty)
{
  if (colorSpace == ISE_RGB_to_GRAY)
    colorSpace = ISE_RGB_to_RGB;

  m_colorSpace = colorSpace;
  m_a00 = a00;
  m_a01 = a01;
  m_a02 = a02;
  m_a10 = a10;
  m_a11 = a11;
  m_a12 = a12;
  m_tx = tx;
  m_ty = ty;
}

void
ImageFeatureColorProjection::getParameters(int &colorSpace, 
					  double &a00, double &a01, double &a02, 
					  double &a10, double &a11, double &a12, 
					  double &tx, double &ty)
{
  colorSpace = m_colorSpace;
  a00 = m_a00;
  a01 = m_a01;
  a02 = m_a02;
  a10 = m_a10;
  a11 = m_a11;
  a12 = m_a12;
  tx = m_tx;
  ty = m_ty;
}


void
ImageFeatureColorProjection::compute()
{
  assert(m_segContext);
  assert(m_segContext->getImage());

  girl::tools::Image *img = m_segContext->getImage();
  assert(img);

  compute(*img);  
}

//B:TODO:OPTIM: there is a clear performance improvement for complexe color spaces transforms (Lab, Luv, ...) on big images
//   However, il would be probably better to give more work than a single line to each thread !?!

//some CODE DUPLICATION with iseImageFeatureColorSpace.cpp

class ImageConverter
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
 : public cv::ParallelLoopBody
#endif
{
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
  typedef cv::Range Range;
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
  typedef cv::BlockedRange Range;
#endif

public:

  ImageConverter(const girl::tools::Image &img,
		 girl::tools::ImageFeature &imgF,
		 int colorSpace, 
		 double a00, double a01, double a02, 
		 double a10, double a11, double a12,
		 double tx, double ty)
    : m_img(img), m_imgF(imgF), 
      m_colorSpace(colorSpace), 
      m_a00(a00), m_a01(a01), m_a02(a02), m_a10(a10), m_a11(a11), m_a12(a12), 
      m_tx(tx), m_ty(ty)
  {
    assert(m_img.channels() >= 3);
    assert(m_imgF.channels() == 2);
  }

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240

  virtual void operator()( const Range& range ) const
  {
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    const int start = range.start;
    const int end = range.end;
#else
    const int start = range.begin();
    const int end = range.end();
#endif
    processRows(start, end);
  }

protected:

#endif //OpenCV >= 240

  inline void processRows(int start, int end) const
  {
    const int chanIn = m_img.channels();
    const int chanOut = m_imgF.channels();
    
    const int width = m_img.width();
    const int colorSpace = m_colorSpace;
    
    for (int j=start; j<end; ++j) {
      const girl::tools::Image::type *in = m_img.scanLine(j);
      girl::tools::ImageFeature::type *out = m_imgF.scanLine(j);
      
      for (int i = 0 ; i < width ; ++i) {
	int x, y, z;
	ise::conversion::changeFromRGB(colorSpace, in[0], in[1], in[2], x, y, z);

	projectColor(x, y, z, out[0], out[1]);

	in += chanIn;
	out += chanOut;
      }
    }
  }

  inline void projectColor(int x, int y, int z, 
			   girl::tools::ImageFeature::type &a, girl::tools::ImageFeature::type &b) const
  {
    a = m_a00*x + m_a01*y + m_a02*z + m_tx;
    b = m_a10*x + m_a11*y + m_a12*z + m_ty;
  } 
  
  
protected:
  
  const girl::tools::Image &m_img;
  girl::tools::ImageFeature &m_imgF;
  int m_colorSpace;
  double m_a00, m_a01, m_a02, m_a10, m_a11, m_a12, m_tx, m_ty;
};



void
ImageFeatureColorProjection::compute(const girl::tools::Image &img)
{
  const int width = img.width(); 
  const int height = img.height();
  
  int dim = ImageFeatureColorProjection::FEATURE_DIMENSION;
  assert(m_colorSpace != ISE_RGB_to_GRAY);
  
  m_imgF = girl::shared_ptr<girl::tools::ImageFeature>(new girl::tools::ImageFeature(width, height, dim));
  
  girl::tools::ImageFeature &imgF = *(m_imgF.get());

  ImageConverter ic(img, imgF, m_colorSpace, m_a00, m_a01, m_a02, m_a10, m_a11, m_a12, m_tx, m_ty);
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
  cv::parallel_for_(cv::Range(0, height), ic);
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
  cv::parallel_for(cv::BlockedRange(0, height), ic);
#else
  ic.processRows(0, height);
#endif
  
}

void
ImageFeatureColorProjection::getDimDescription(std::vector<std::string> &dimDesc) const
{
  dimDesc.clear();
  for (size_t i = 0 ; i < getFeatureDim() ; ++i) {
    std::stringstream ss;
    ss<<"ColorSpace "<<m_colorSpace<<" projected "<<i;
    dimDesc.push_back(ss.str());
  }
}

const std::string 
ImageFeatureColorProjection::uri() const
{
  std::stringstream ss;
  ss << baseUri() << IMAGE_FEATURE_URI_SEPARATOR << m_colorSpace << IMAGE_FEATURE_URI_SEPARATOR << m_a00 << IMAGE_FEATURE_URI_SEPARATOR << m_a01 << IMAGE_FEATURE_URI_SEPARATOR << m_a02 << IMAGE_FEATURE_URI_SEPARATOR << m_a10 << IMAGE_FEATURE_URI_SEPARATOR << m_a11 << IMAGE_FEATURE_URI_SEPARATOR << m_a12 << IMAGE_FEATURE_URI_SEPARATOR << m_tx << IMAGE_FEATURE_URI_SEPARATOR << m_ty << IMAGE_FEATURE_URI_SEPARATOR;
  return ss.str();
}

const std::string
ImageFeatureColorProjection::baseUri() const
{
  return "ColorProjection";
}

