/*
  1D Histogram functions.
  Copyright (C) 2005-2011 Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef ISE_HISTOGRAM_1D_TOOLS_HPP
#define ISE_HISTOGRAM_1D_TOOLS_HPP

#include <ise/iseSparseHistogram1D.hpp>
#include <ise/iseHistogram1D.hpp>


namespace ise {
 
  extern GIRL_API void convertSparseHistogram1D(const ise::Histogram1D &histo, ise::SparseHistogram1DD &h);

  /**
   * @brief Quantize histogram @a h into @a histo, such as @a histo as @a size bins. 
   *
   */
  extern GIRL_API void quantizeSparseHistogram1D(const ise::SparseHistogram1DD &h, ise::Histogram1D &histo, size_t size);

  /**
   * @brief Quantize histogram @a h into @a histo, such a bin in @a histo is nearbyInt(bin of @a h / quantifFactor). 
   *
   */
  extern GIRL_API void quantizeSparseHistogram1D(const ise::SparseHistogram1DD &h, double quantifFactor, ise::Histogram1D &histo);

  
  
  /**
   * Apply filter @a w to histogram @a histo from index @a indexStart for a size @a size
   * and set result in histogram @a h.
   *
   * @param[in] histo  input histogram
   * @param[in] indexStart  index in histo from where to apply @a w.
   * @param[in] size  size to process from @a indexStart.
   * @param[in] w  applied filter (its size must be odd).
   * @param[out] h  output filtered histogram.
   */
  extern GIRL_API void applyFilter(const ise::Histogram1D &histo,
				   size_t indexStart, size_t size,
				   const std::vector<double> &w,	      
				   ise::Histogram1D &h);

  /**
   * Apply filter @a w to histogram @a histo and set result in histogram @a h.
   *
   * Equivalent to applyFilter(histo, 0, histo.size(), w, h).
   * 
   * @param[in] histo  input histogram
   * @param[in] w  applied filter (its size must be odd).
   * @param[out] h  output filtered histogram.
   */
  extern void applyFilter(const ise::Histogram1D &histo,
			  const std::vector<double> &w,	      
			  ise::Histogram1D &h);

  
  /**
   * Make a gaussian of size @a kernelSize in @a vect.
   * 
   * @param[in] kernelSize  size of gaussian (must be odd).
   * @param[in] sigma       value of sigma for gaussian
   * @param[out] vect       produced gaussian (vector will be resized to @a kernelSize).
   */
  extern GIRL_API void gaussian(size_t kernelSize, double sigma, std::vector<double> &vect);
    

  /**
   * Apply gaussian filter on histogram @a histo,
   * taking care of beginning and end of histogram,
   * and set result in histogram @a h.
   * Gaussian filter is defined by @a sigma and original size @a kernelSize
   *
   * @param[in] histo  input histogram
   * @param[in] kernelSize  size of gaussian (must be odd).
   * @param[in] sigma       value of sigma for gaussian
   * @param[out] h  output filtered histogram.
   */
  extern GIRL_API void applyGaussianFilter(const ise::Histogram1D &histo,
					   size_t kernelSize,
					   double sigma,
					   ise::Histogram1D &h);

  /**
   * Get the number of peaks of histogram @a h.
   */
  extern GIRL_API size_t histoNumPeakDetection(const ise::Histogram1D &h);

  /**
   * Get key of maximal frequency of histogram @a h.
   *
   * @warning If there are several bins equal to maximal frequency, first bin is returned. 
   */
  extern GIRL_API double getKeyOfMaximalFrequency(const ise::Histogram1D &h);


}

#endif /* ! ISE_HISTOGRAM_1D_TOOLS_HPP */
