/*
  1D Histogram functions.
  Copyright (C) 2005-2011 Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "iseHistogram1DTools.hpp"


#include <cmath>
#include <limits>

#ifndef M_PI
//Visual C++ does not define these constants in <cmath>
#define M_PI 3.14159265358979323846
#endif 


namespace ise {

  void
  convertSparseHistogram1D(const ise::Histogram1D &histo, ise::SparseHistogram1DD &h)
  {
    h.clear();
    const double absoluteMin = histo.absoluteMin();
    for (size_t i=0; i<histo.size(); ++i) {
      const double k = absoluteMin + static_cast<double>(i);
      const double f = histo[i];
      h.add(k, f);
    }  
  }
  
  void
  quantizeSparseHistogram1D(const ise::SparseHistogram1DD &h, ise::Histogram1D &histo, size_t size)
  {
    histo.clear();
    
    if (h.empty() || size == 0)
      return;

    const double absoluteMin = h.begin()->first;
    const double absoluteMax = h.rbegin()->first;
    
    const double interval = (absoluteMax - absoluteMin)/static_cast<double>(size-1);
    
    histo = ise::Histogram1D(absoluteMin, interval, size);

#ifndef NDEBUG
    double currIntervalMin = absoluteMin;
#endif //NDEBUG
    double currIntervalMax = absoluteMin+interval;
    const ise::SparseHistogram1DD::const_iterator itEnd = h.end();
    ise::SparseHistogram1DD::const_iterator it = h.begin();
    for (size_t i=0 ;  it != itEnd; ++it) {
      const double k = it->first;
      assert(k >= currIntervalMin);
      while (k >= currIntervalMax) {
#ifndef NDEBUG
	currIntervalMin = currIntervalMax;
#endif //NDEBUG
	currIntervalMax += interval;
	++i;
      }
      assert(i < size);
      histo[i] += it->second;
    }
  
  }

  void
  quantizeSparseHistogram1D(const ise::SparseHistogram1DD &h, double quantifFactor, ise::Histogram1D &histo)
  {
    histo.clear();
    
    if (h.empty() || quantifFactor == 0)
      return;

    const double absoluteMin = h.begin()->first;
    //const double absoluteMax = h.rbegin()->first;

    const double inv_quantifFactor = 1.0 / quantifFactor;

    ise::SparseHistogram1DD h2;

    for (ise::SparseHistogram1DD::const_iterator it = h.begin(); it != h.end(); ++it) {
      const double k = it->first;
      const double k2 = static_cast<int>( (k - absoluteMin)*inv_quantifFactor + 0.5) + absoluteMin;
      h2.add(k2, it->second);
    }

    quantizeSparseHistogram1D(h2, histo, h2.size());
  }




  /**
   * @param[in] histo input  histogram
   * @param[in] indexStart  index in histo from where to apply @a w.
   * @param[in] size  size to process from @a indexStart.
   * @param[in] w  applied filter (its size must be odd).
   * @param[out] h  output filtered histogram.
   */
  void
  applyFilter(const ise::Histogram1D &histo,
	      size_t indexStart, size_t size,
	      const std::vector<double> &w,	      
	      ise::Histogram1D &h)
  {
    assert(&histo != &h);
    assert(indexStart+size<=histo.size());
    assert(histo.size() == h.size());
    assert(w.size()&1); //must be an odd number.

    const size_t kernelSize = w.size();
    const size_t halfKernelSize = kernelSize/2;

    const size_t indexEnd = indexStart+size;
    if (size >= kernelSize) {
      const size_t applyEnd = indexEnd-halfKernelSize;
      for (size_t i = indexStart+halfKernelSize ; i < applyEnd ; ++i) {
	double sum = 0;
	const size_t wStartIndex = i - halfKernelSize;
	for (size_t j = 0 ; j < kernelSize ; ++j) {
	  sum += histo[ wStartIndex+j ] * w[j];
	}
	h[i] = sum;
      }
    }
    else {
      //TODO:OPTIM? memcpy ?
      for (size_t i = indexStart; i<indexEnd; ++i) {
	h[i] = histo[i];
      }
    }
  }

  /**
   * Apply filter @a w to histogram @a histo and set result in histogram @a h.
   *
   * Equivalent to applyFilter(histo, 0, histo.size(), w, h).
   * 
   * @param[in] histo  input histogram 
   * @param[in] w  applied filter (its size must be odd).
   * @param[out] h  output filtered histo.
   */
  void
  applyFilter(const ise::Histogram1D &histo,
	      const std::vector<double> &w,	      
	      ise::Histogram1D &h)
  {
    assert(&histo != &h);
    assert(histo.size() == h.size());
    assert(w.size()&1); //must be an odd number.

    applyFilter(histo, 0, histo.size(), w, h);
  }
 
  /**
   * Make a gaussian of size @a kernelSize in @a vect.
   * 
   * @param[in] kernelSize  size of gaussian (must be odd).
   * @param[in] sigma       value of sigma for gaussian
   * @param[out] vect       produced gaussian (vector will be resized to @a kernelSize).
   */
  void
  gaussian(size_t kernelSize, double sigma, std::vector<double> &vect)
  {
    assert(kernelSize&1); //must be an odd number.

    vect.resize(kernelSize, 0);
    const int size = kernelSize / 2;
    const double f1 = 1.0 / (sigma * sqrt( 2 * M_PI ));
    const double f2 = 1.0 / (2.0 * sigma * sigma);
    for ( int i = -size ; i <= size ; ++i) {
      vect[i+size] = ( f1 * exp( - i*i * f2 ) ); 
    }
  }

  /**
   * Apply gaussian filter on histogram @a histo,
   * taking care of beginning and end of histogram,
   * and set result in histogram @a h.
   * Gaussian filter is defined by @a sigma and original size @a kernelSize
   *
   * @param[in] histo  input histogram
   * @param[in] kernelSize  size of gaussian (must be odd).
   * @param[in] sigma       value of sigma for gaussian
   * @param[out] h  output filtered histogram.
   */
  void
  applyGaussianFilter(const ise::Histogram1D &histo,
		      size_t kernelSize,
		      double sigma,
		      ise::Histogram1D &h)
  {
    assert(kernelSize&1); //must be an odd number.
    
    h = histo;
     
    std::vector<double> g;
    gaussian(kernelSize, sigma, g);
    applyFilter(histo, g, h);

    //gaussians on the beginning and end of histogram
    if (histo.size() >= kernelSize) {
      const size_t halfKernelSize = kernelSize/2;
      for (size_t i = 0 ; i < halfKernelSize ; ++i) {
	const size_t kS = 2*i+1; 
	gaussian(kS, sigma, g);
	
	applyFilter(histo, 0, kS, g, h);
      
	applyFilter(histo, histo.size()-kS, kS, g, h);
      }
    }
    
  }


  /**
   * Get the number of peaks of histogram @a h.
   */
  size_t
  histoNumPeakDetection(const ise::Histogram1D &h)
  {
    const size_t size = h.size();
    size_t n = 0;
    for (size_t i = 1 ; i < size-1 ; ++i) //skip first and last element
      if ( h[i-1] < h[i] && h[i] > h[i+1])
	++n;

    return n;
  }

  /**
   * Get key of maximal frequency of histogram @a h.
   *
   * @warning If there are several bins equal to maximal frequency, first bin is returned. 
   */
  double
  getKeyOfMaximalFrequency(const ise::Histogram1D &h)
  {
    size_t maxI = 0;
    double maxValue = -std::numeric_limits<double>::max(); //warning: not std::numeric_limits<double>::min()
    const size_t size = h.size();
    for (size_t i = 0 ; i<size ; ++i) {
      if (h[i] > maxValue) {
	maxValue = h[i];
	maxI = i;
      }	
    }
    return h.indexToKey(maxI);
  }


} //end namespace ise
