/*
  1D Histogram class.
  Copyright (C) 2005-2009 Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef ISEHISTOGRAM1D_HPP
#define ISEHISTOGRAM1D_HPP


  
#include <cassert>
#include <vector>
#include <girl/girlCommon.hpp>
#include <girl/girlToger.hpp>
#include <girlTools/girlToolsImage.hpp>
#include <ise/iseConversion.hpp>
#include <ise/iseHashHistogram.hpp>

namespace ise {


  class GIRL_API Histogram1D
  {
  public:
    
    /**
     * @brief Build an empty histogram.
     *
     * @sa reset.
     */
    Histogram1D();

    /**
     * @brief Build an histogram of channel @a chan of image @a img in 
     * color space @a colorSpaceConversion.
     *
     * @sa reset.
     */    
    Histogram1D(const girl::tools::Image &img,
		int colorSpaceConversion,
		int chan);
      
    /**
     * @brief Build an histogram of channel @a chan of a region of interest of 
     * image @a img in color space @a colorSpaceConversion.
     *
     * @warning We must have 0<=p1.x()<=p2.x()<img.width() 
     * and  0<=p1.y()<=p2.y()<img.height().
     *
     * All points (x; y) such as p1.x()<=x<=p2.x() and p1.y()<=y<=p2.y()
     * will be traversed.
     *
     * @sa reset.
     */    
    Histogram1D(const girl::tools::Image &img,
		const girl::point &p1, const girl::point &p2,
		int colorSpaceConversion,
		int chan);

    /**
     * @brief Build an histogram of channel @a chan of image @a img in 
     * color space @a colorSpaceConversion only for regions @a regions of 
     * toger @a tog.
     *
     * @sa reset.
     *
     */
    Histogram1D(const girl::tools::Image &img,
		const girl::toger &tog,
		int colorSpaceConversion,
		int chan,
		const girl::regionDeque &regions);
    /**
     * @brief Build an histogram of channel @a chan from HashHistogram @a histo.
     *
     * @a min and @a max are minimum and maximum of values of this channel.
     *
     * @sa reset.
     */
    Histogram1D(const ise::HashHistogram &histo,
		int chan, int min, int max);
    
    /**
     * @brief Build an empty histogram where values will be in [@a min; @a max].
     *
     * @sa reset.
     */
    Histogram1D(int min, int max);

    /**
     * @brief Build an empty histogram where values will be in [@a min; @a min + (@size - 1) * @a interval], 
     * with interval @a interval.
     *
     * @sa reset.
     */
    Histogram1D(double min, double interval, size_t size);


    void clear();


    void reset(const girl::tools::Image &img,
	       int colorSpaceConversion,
	       int chan); 
	       
    void reset(const girl::tools::Image &img,
	       const girl::point &p1, const girl::point &p2,
	       int colorSpaceConversion,
	       int chan);

    void reset(const girl::tools::Image &img,
	       const girl::toger &tog,
	       int colorSpaceConversion,
	       int chan,
	       const girl::regionDeque &regions);

    void reset(const ise::HashHistogram &histo,
	       int chan, int min, int max);
   
    void reset(int min, int max);

    void reset(double min, double interval, size_t size);
    

    size_t size() const
    {
      return m_frequencies.size();
    }

    /**
     * @brief Absolute minimum of histogram for 
     * given colorSpaceConversion and channel.
     * 
     */
    double absoluteMin() const
    {
      return m_min;
    }
    
    /**
     * @brief Absolute maximum of histogram for 
     * given colorSpaceConversion and channel.
     * 
     * @warning if size() is 0, return valued is undefined.
     */
    double absoluteMax() const
    {
      //return m_min + m_interval*(m_frequencies.size()-1);
      const size_t s = m_frequencies.size();
      if (s > 0)
	return m_min + m_interval*(s-1);
      return m_min;
    }
    
    double interval() const
    {
      return m_interval;
    }


    /**
     * @warning @a i must be in [0; size()[,
     * and not in [absoluteMin(); absoluteMax()] 
     */
    double operator[](size_t i) const
    {
      assert(m_frequencies.size() > i);
      return m_frequencies[i];
    }

    /**
     * @warning @a i must be in [0; size()[,
     * and not in [absoluteMin(); absoluteMax()] 
     */
    double &operator[](size_t i)
    {
      assert(m_frequencies.size() > i);
      return m_frequencies[i];
    }


    /**
     * translate index from [0; size()[ to [absoluteMin(); absoluteMax()]
     */
    double indexToKey(size_t i) const
    {
      return m_min + i * m_interval;
    }

    /**
     * translate index from [absoluteMin(); absoluteMax()] to [0; size()[ 
     */
    size_t keyToIndex(double v) const
    {
      return static_cast<size_t>( (v - m_min) /  m_interval );
    }


    /**
     * @brief Minimum values for which frequency is not null in histogram.
     *
     * @warning return absluteMax() on empty histogram.
     */
    double getMin() const;
    
    /**
     * @brief Maximum values for which frequency is not null in histogram.
     *
     * @warning return absluteMin() on empty histogram.
     */
    double getMax() const;
    
    /**
     * @brief Median values of histogram, in [getMin(); getMax()]
     */
    double getMedian() const;

    /**
     * @brief Mean value of whole histogram.
     *
     * Returned value is in [getMin(); getMax()]
     */
    double getMean() const;

    /**
     * @brief Mean value of histogram, 
     * for indexes in [t1; t2]
     *
     * @warning We must have t1<=t2<size().
     * Returned value is in [getMin(); getMax()]
     */
    double getMean(size_t t1, size_t t2) const;
    
    /**
     * @brief Standard deviation of histogram.
     */
    double getStandardDeviation() const;
    
    /**
     * @brief Standard deviation of histogram.
     *
     * @param mean mean value of histogram.
     *
     * @sa getMean.
     */
    double getStandardDeviation(int mean) const;


    /**
     *@brief Return value of maximal frequency
     */
    double getMaxFrequency() const;

    /**
     *@brief Return key for which frequency is maximal.
     */
    double getKeyOfMaxFrequency() const;


  protected:
  
    void initHisto(int colorSpaceConversion, int chan);
    void initHistomm(int min, int max);
    void initHistomm(double min, double interval, size_t size);

    void populate(const girl::tools::Image &img, int chan);
    void populate(const girl::tools::Image &img, int colorSpaceConversion, int chan);
    void populate(const girl::tools::Image &img, const girl::point &p1, const girl::point &p2, int chan);
    void populate(const girl::tools::Image &img, const girl::point &p1, const girl::point &p2, int colorSpaceConversion, int chan);
    void populate(const ise::HashHistogram &histo, int chan);

    double standardDeviation(double m) const;

    
    

  protected:
    
    std::vector<double> m_frequencies;
    double m_min;
    double m_interval;
  };


}

#endif /* ! ISEHISTOGRAM1D_HPP */
