/*
  1D Histogram class.
  Copyright (C) 2005-2009 Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "iseHistogram1D.hpp"

#include <cassert>
#include <limits>
#include <cmath>

using namespace ise;

Histogram1D::Histogram1D()
  :
  m_frequencies(0),
  m_min(0),
  m_interval(1)
{

}

Histogram1D::Histogram1D(const girl::tools::Image &img,
			 int colorSpaceConversion,
			 int chan)
  :
  m_frequencies(0),
  m_min(0),
  m_interval(1)
{
  reset(img, colorSpaceConversion, chan);
}

// Histogram1D::Histogram1D(const girl::tools::ImageFeature &img,
// 			 int chan)
//   :
//   m_frequencies(0),
//   m_min(0),
//   m_interval(1)
// { 
//   reset(img, chan);
// }

Histogram1D::Histogram1D(const girl::tools::Image &img,
			 const girl::point &p1, const girl::point &p2, 
			 int colorSpaceConversion,
			 int chan)
  :
  m_frequencies(0),
  m_min(0),
  m_interval(1)
{
  reset(img, p1, p2, colorSpaceConversion, chan);
}

Histogram1D::Histogram1D(const girl::tools::Image &img,
			 const girl::toger &tog,
			 int colorSpaceConversion,
			 int chan,
			 const girl::regionDeque &regions)
  :
  m_frequencies(0),
  m_min(0),
  m_interval(1)
{
  reset(img, tog, colorSpaceConversion, chan, regions);
}

      
Histogram1D::Histogram1D(const ise::HashHistogram &histo,
			 int chan, int min, int max)
  :
  m_frequencies(0),
  m_min(0),
  m_interval(1)
{
  
  reset(histo, chan, min, max);
}

Histogram1D::Histogram1D(int min, int max)
  :
  m_frequencies(0),
  m_min(0),
  m_interval(1)
{
  initHistomm(min, max);
}

Histogram1D::Histogram1D(double min, double interval, size_t size)
  :
  m_frequencies(size),
  m_min(min),
  m_interval(interval)
{
  initHistomm(min, interval, size);
}

void
Histogram1D::clear()
{
  m_frequencies.resize(0);
  m_min = 0;
  m_interval = 1;
}


void
Histogram1D::reset(const girl::tools::Image &img,
		   int colorSpaceConversion,		       
		   int chan)
{
  initHisto(colorSpaceConversion, chan);

  if (ISE_RGB_to_RGB == colorSpaceConversion) {
    populate(img, chan);
  }
  else {
    populate(img, colorSpaceConversion, chan);
  } 
}

void
Histogram1D::reset(const girl::tools::Image &img,
		   const girl::point &p1, const girl::point &p2,
		   int colorSpaceConversion,		       
		   int chan)
{
  initHisto(colorSpaceConversion, chan);

  if (ISE_RGB_to_RGB == colorSpaceConversion) {
    populate(img, p1, p2, chan);
  }
  else {
    populate(img, p1, p2, colorSpaceConversion, chan);
  } 
}




namespace {
  class RegionHistoComputer : public girl::toger::pointIterator
  {
  public:  
    RegionHistoComputer(const girl::tools::Image &img,
			ise::conversion::PFConversion fct,
			double pMin, double interval,
			std::vector<double> &frequencies)
      :
      m_img(img),
      m_fct(fct),
      m_min(pMin), m_invInterval(1.0),
      m_frequencies(frequencies)
    {
      assert(interval > 0);
      m_invInterval = 1.0 / interval;
    }
  
    inline size_t keyToIndex(int v) const
    {
      return static_cast<size_t>( (v - m_min) * m_invInterval );
    }
		      
    virtual void processPoint(const girl::point &pt)
    {
      const girl::tools::Pixel p = m_img.getPixel(pt.x(), pt.y());

      const int v = m_fct(p[0], p[1], p[2]);
      assert(keyToIndex(v)  < m_frequencies.size());
      m_frequencies[keyToIndex(v)]++;
    }

  protected:
    const girl::tools::Image &m_img;
    ise::conversion::PFConversion m_fct;
    double m_min;
    double m_invInterval;
    std::vector<double> &m_frequencies;
  };
}

void 
Histogram1D::reset(const girl::tools::Image &img,
		   const girl::toger &tog,
		   int colorSpaceConversion,
		   int chan,
		   const girl::regionDeque &regions)
{
  if (tog.numRegions() == regions.size()) { 
    //all regions
    // => faster to do init on whole image
    reset(img, colorSpaceConversion, chan);
  }
  else {
    initHisto(colorSpaceConversion, chan);

    RegionHistoComputer rhc(img, 
			    ise::conversion::changeSpace(colorSpaceConversion)[chan],
			    m_min, m_interval,
			    m_frequencies);
    
    girl::regionDeque::const_iterator it = regions.begin();
    const girl::regionDeque::const_iterator itEnd = regions.end();
    for ( ; it != itEnd; ++it)
      tog.iterateOnRegionDomain(*it, rhc);

  }

}
 
void
Histogram1D::reset(const ise::HashHistogram &histo,
		   int chan, int min, int max)
{
  initHistomm(min, max);

  populate(histo, chan);
}

void
Histogram1D::reset(int min, int max)
{
  initHistomm(min, max);
}

void
Histogram1D::reset(double min, double interval, size_t size)
{
  initHistomm(min, interval, size);
}

void
Histogram1D::initHisto(int colorSpaceConversion, int chan)
{
  initHistomm(ise::conversion::absoluteMin(colorSpaceConversion, chan),
	      ise::conversion::absoluteMax(colorSpaceConversion, chan));
}

void
Histogram1D::initHistomm(int pMin, int pMax)
{
  assert(pMin<=pMax);

  m_frequencies.clear();
  m_frequencies.resize(pMax - pMin + 1, 0);
  m_min = pMin;
  m_interval = 1;
}

void
Histogram1D::initHistomm(double pMin, double interval, size_t size)
{
  m_frequencies.clear();
  m_frequencies.resize(size, 0);
  m_min = pMin;
  m_interval = interval;
}




void
Histogram1D::populate(const girl::tools::Image &img, int chan)
{
  const int h = img.height();
  const int w = img.width();
  for (int j=0; j<h; ++j) {
    const girl::tools::Image::type *p = img.scanLine(j);
    for (int i=0; i<w; ++i) {
      assert(keyToIndex(p[chan])  < m_frequencies.size());
      m_frequencies[keyToIndex(p[chan])]++;
      p += img.formatSize();
    }
  }
}

//B: CODE DUPLICATION : to pay conversion only if really needed
//TODO: test performances
void
Histogram1D::populate(const girl::tools::Image &img, int colorSpaceConversion, int chan)
{
  ise::conversion::PFConversion conv = ise::conversion::changeSpace(colorSpaceConversion)[chan];
  
  const int h = img.height();
  const int w = img.width();
  for (int j=0; j<h; ++j) {
    const girl::tools::Image::type *p = img.scanLine(j);
    for (int i=0; i<w; ++i) {
      const int v = conv(p[0], p[1], p[2]);
      assert(keyToIndex(v)  < m_frequencies.size());
      m_frequencies[keyToIndex(v)]++;
      p += img.formatSize();
    }
  }
}


void
Histogram1D::populate(const girl::tools::Image &img, 
		      const girl::point &p1, const girl::point &p2, 
		      int chan)
{
  assert(p1.x() <= p2.x());
  assert(p1.y() <= p2.y());
  assert(p1.x() >= 0);
  assert(p1.y() >= 0);
  assert(p2.x() < img.width());
  assert(p2.y() < img.height());

  const size_t w = p2.x()-p1.x()+1;
  const size_t h = p2.y()-p1.y()+1;
  if (w==0 || h==0) { //no point
    return;
  }
  
  const int formatSize = img.formatSize();
  const int shiftX = p1.x()*formatSize;
  
  for (int j=p1.y(); j<=p2.y(); ++j) {
    const girl::tools::Image::type *p = img.scanLine(j) + shiftX;
    for (int i=p1.x(); i<=p2.x(); ++i) {
      assert(keyToIndex(p[chan])  < m_frequencies.size());
      m_frequencies[keyToIndex(p[chan])]++;

      p += formatSize;
    }
  }
}

//B: CODE DUPLICATION : to pay conversion only if really needed
//TODO: test performances
void
Histogram1D::populate(const girl::tools::Image &img, 
		      const girl::point &p1, const girl::point &p2, 
		      int colorSpaceConversion, int chan)
{
  assert(p1.x() <= p2.x());
  assert(p1.y() <= p2.y());
  assert(p1.x() >= 0);
  assert(p1.y() >= 0);
  assert(p2.x() < img.width());
  assert(p2.y() < img.height());

  ise::conversion::PFConversion conv = ise::conversion::changeSpace(colorSpaceConversion)[chan];

  const size_t w = p2.x()-p1.x()+1;
  const size_t h = p2.y()-p1.y()+1;
  if (w==0 || h==0) { //no point
    return;
  }
  
  const int formatSize = img.formatSize();
  const int shiftX = p1.x()*formatSize;
  
  for (int j=p1.y(); j<=p2.y(); ++j) {
    const girl::tools::Image::type *p = img.scanLine(j) + shiftX;
    for (int i=p1.x(); i<=p2.x(); ++i) {
      const int v = conv(p[0], p[1], p[2]);
      assert(keyToIndex(v)  < m_frequencies.size());
      m_frequencies[keyToIndex(v)]++;

      p += formatSize;
    }
  }
}



void
Histogram1D::populate(const ise::HashHistogram &histo, int chan)
{
  ise::HashHistogram::const_iterator it = histo.begin();
  const ise::HashHistogram::const_iterator itEnd = histo.end();
  for ( ; it!= itEnd; ++it) {
    const int v = it->first[chan];
    const size_t offv = keyToIndex(v);
    assert(offv  < m_frequencies.size());
    m_frequencies[offv] += it->second;
  }

}



double
Histogram1D::getMin() const
{
  size_t imin = 0;
  for (size_t i=0; i<m_frequencies.size(); ++i) {
    imin = i;
    if (m_frequencies[i] != 0)
      break;
  }
  assert(m_frequencies.size() == 0 || m_frequencies[imin] != 0 || imin == m_frequencies.size()-1);
  
  return indexToKey(imin);
  
}
    
double
Histogram1D::getMax() const
{
  size_t imax = 0;
  if (m_frequencies.size() > 0) {
    for (size_t i=m_frequencies.size()-1; i!=static_cast<size_t>(-1); --i) {
      imax = i;
      if (m_frequencies[i] != 0)
	break;
    }
    assert(m_frequencies[imax] != 0 || imax == 0);
  }
  
  return indexToKey(imax); 
}
    

double
Histogram1D::getMedian() const
{
  double sum = 0;
  const size_t size = m_frequencies.size();
  for (size_t i=0; i<size; ++i) {
    sum += m_frequencies[i];
  }

  const double half = sum/2.0;
  double s = 0.0;
  size_t i = 0;
  for ( ; i<size; ++i) {
    s += m_frequencies[i];
    if (s >= half)
      break;
  }
  
  return indexToKey(i); 
}

double
Histogram1D::getMean() const
{
  double m = 0;
  double a = 0;
  const size_t size = m_frequencies.size();
  for (size_t i=0; i<size; ++i) {
    m += (i+1)*m_frequencies[i];
    a += m_frequencies[i];
  }
  const double mean = (a <= std::numeric_limits<double>::epsilon()) ? 0.0 : m/a -1;

  const double smean =  indexToKey(mean); 
  
  assert(smean >= absoluteMin() && (m_frequencies.size() == 0 || smean <= absoluteMax()));

  return smean;
}

double
Histogram1D::getMean(size_t t1, size_t t2) const
{
  assert(t2 < m_frequencies.size());
  assert(t1 <= t2);

  double p = 0;
  double mean = 0;
  for (size_t i = t1; i <= t2 ; ++i) {
    p += m_frequencies[i];
    mean += (i+1) * m_frequencies[i];
  }

  mean /= p;

  return indexToKey(mean);
}

double
Histogram1D::standardDeviation(double mean) const
{
  double stdD = 0.0;
  
  if (m_frequencies.size() > 1) {
    //const float m = mean + m_shift;

    double s = 0.0;
    double n = 0;
    for (size_t i=0; i<m_frequencies.size(); ++i) {
      const float t = i - mean;
      const double freq = m_frequencies[i];
      s += freq * t*t;
      n += freq;
    }
    if (n>0)
      stdD = sqrt( 1.0f/n * s);
  }
  
  return stdD;
}

double
Histogram1D::getStandardDeviation() const
{
  double m = 0;
  double a = 0;
  for (size_t i=0; i<m_frequencies.size(); ++i) {
    m += (i+1)*m_frequencies[i];
    a += m_frequencies[i];
  }
  const double mean = (a <= std::numeric_limits<double>::epsilon()) ? 0 : m/static_cast<float>(a) -1;
  return standardDeviation(mean);
}

double
Histogram1D::getStandardDeviation(int m) const
{
  const double mean = static_cast<double>(m);
  return standardDeviation(mean);
}


double
Histogram1D::getMaxFrequency() const
{
 double maxVal = -std::numeric_limits<double>::max();
 for (size_t i = 0; i < size(); ++i) {
  if (m_frequencies[i] > maxVal) {
   maxVal = m_frequencies[i];
  }
 }
 return maxVal;
}


double
Histogram1D::getKeyOfMaxFrequency() const
{
  size_t maxIndex = 0;
 double maxVal = -std::numeric_limits<double>::max();
 for (size_t i = 0; i < size(); ++i) {
  if (m_frequencies[i] > maxVal) {
   maxVal = m_frequencies[i];
   maxIndex = i;
  }
 }
 return indexToKey(maxIndex);
}
