/*
  3D Histogram stored as hash map of colors.
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef ISEHASHHISTOGRAM_HPP
#define ISEHASHHISTOGRAM_HPP

#include <cassert>
#include <cstdlib>

#include <girl/girlCommon.hpp>
#include <girl/girlHashMap.hpp>
#include <girl/girlToger.hpp>
#include <girlTools/girlToolsImage.hpp>

namespace ise {

  class HashHistoRegionComputer;

  class GIRL_API HashHistogram
  {

  public:
    struct Pixel
    {
      Pixel(int A, int B, int C)
      {
	a[0] = A;
	a[1] = B;
	a[2] = C;
      }
      
      bool operator==(const Pixel &p) const
      {
	return a[0] == p.a[0] && a[1] == p.a[1] && a[2] == p.a[2];
      }
      
      int operator[](int i) const
      {
	assert(i>=0 && i<3);
	return a[i];
      }

      int &operator[](int i)
      {
	assert(i>=0 && i<3);
	return a[i];
      }

      int a[3];
    };


  protected:

    class pixelHasher
      :  public std::unary_function<Pixel, std::size_t>
    {
    public:

      inline
      std::size_t
      operator()(const Pixel &p) const
      {
	//return static_cast<size_t>(p.a[0]<<20 | p.a[1]<<10 | p.a[2]);
	return ( ((static_cast<unsigned int>(abs(p.a[0]))) << 16) | ((static_cast<unsigned int>(abs(p.a[1]))) << 8) | (static_cast<unsigned int>(abs(p.a[2]))) );
      }
      
#if _MSC_VER
      static const size_t bucket_size = 4;
      static const size_t min_buckets = 8;
    
      bool operator( )(const Pixel &a,
		       const Pixel &b) const
      {
	if (a[2] != b[2]) {
	  return a[2] < b[2];
	}
	else {
	  if (a[1] != b[1]) {
	    return a[1] < b[1];
	  }
	  else {
	    return a[0] < b[0];
	  }
	}

      }
 
#endif
    };


    typedef girl::hashMap<Pixel, size_t, pixelHasher> HashMapType;

  public:

    HashHistogram();
    
    explicit HashHistogram(const girl::tools::Image &img);
    
    HashHistogram(const girl::tools::Image &img,
		  const girl::toger &tog,
		  const girl::regionDeque &regions);
    

    void reset(const girl::tools::Image &img);
    
    void reset(const girl::tools::Image &img,
	       const girl::toger &tog,
	       const girl::regionDeque &regions);
    
    void populate(const girl::tools::Image &img);

    void populate(const girl::tools::Image &img,
		  const girl::toger &tog,
		  const girl::regionDeque &regions);
    
    size_t sum() const;

    void convert(int colorSpaceConversion);

    void weighted(std::vector<double> weightedChannelsCoeffs);

    size_t getSize() const { return m_hashMap.size(); }

    void clear();

    void addColor(girl::tools::Image::type r, 
		  girl::tools::Image::type g,
		  girl::tools::Image::type b);


    typedef HashMapType::iterator iterator;
    typedef HashMapType::const_iterator const_iterator;

    iterator begin() { return m_hashMap.begin(); }
    iterator end() { return m_hashMap.end(); }
    const_iterator begin() const { return m_hashMap.begin(); }
    const_iterator end() const { return m_hashMap.end(); }


  protected:

    
  protected:

    friend class HashHistoRegionComputer;

  
    HashMapType m_hashMap;


  };

}



#endif /* ! ISEHASHHISTOGRAM_HPP */
