/*
  3D Histogram stored as hash map of colors.
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "iseHashHistogram.hpp"

#include <ise/iseConversion.hpp>

using namespace ise;

HashHistogram::HashHistogram()
  : m_hashMap()
{
  
}

HashHistogram::HashHistogram(const girl::tools::Image &img)
  : m_hashMap()
{
  populate(img);
}

HashHistogram::HashHistogram(const girl::tools::Image &img,
			     const girl::toger &tog,
			     const girl::regionDeque &regions)
  : m_hashMap()
{
  populate(img, tog, regions);
}
    


void 
HashHistogram::reset(const girl::tools::Image &img)
{
  m_hashMap.clear();
  populate(img);
}

void
HashHistogram::reset(const girl::tools::Image &img,
		     const girl::toger &tog,
		     const girl::regionDeque &regions)
{
  m_hashMap.clear();
  populate(img, tog, regions);
}
 

void
HashHistogram::clear()
{
  m_hashMap.clear();
}

void
HashHistogram::addColor(girl::tools::Image::type r, 
			girl::tools::Image::type g,
			girl::tools::Image::type b)
{
  m_hashMap[Pixel(r, g, b)]++;
}


void 
HashHistogram::populate(const girl::tools::Image &img)
{
  const int h = img.height();
  const int w = img.width();
  for (int j=0; j<h; ++j) {
    const girl::tools::Image::type *p = img.scanLine(j);
    for (int i=0; i<w; ++i) {
      const Pixel pix(p[0], p[1], p[2]);
      m_hashMap[pix]++;
      p += img.formatSize();
    }
  }
}

namespace ise {

  class HashHistoRegionComputer : public girl::toger::pointIterator
  {
  public:
    HashHistoRegionComputer(const girl::tools::Image &img,
			    HashHistogram::HashMapType &hashMap)
      : m_img(img), m_hashMap(hashMap)
    {

    }

    void processPoint(const girl::point &pt)
    {
      const girl::tools::Pixel p = m_img.getPixel(pt.x(), pt.y());
      const HashHistogram::Pixel pix(p[0], p[1], p[2]);
      m_hashMap[pix]++;
    }
    
  protected:
    const girl::tools::Image &m_img;
    HashHistogram::HashMapType &m_hashMap;
    
  };

}

void
HashHistogram::populate(const girl::tools::Image &img,
			const girl::toger &tog,
			const girl::regionDeque &regions)
{
  ise::HashHistoRegionComputer hhrc(img, m_hashMap);
  
  girl::regionDeque::const_iterator it = regions.begin();
  const girl::regionDeque::const_iterator itEnd = regions.end();
  for ( ; it != itEnd; ++it)
    tog.iterateOnRegionDomain(*it, hhrc);
}
 

void HashHistogram::convert(int colorSpaceConversion)
{
  HashMapType hashMap;
  
  ise::conversion::PFConversion *conv = ise::conversion::changeSpace(colorSpaceConversion);
  
  
  HashMapType::const_iterator it = m_hashMap.begin();
  const HashMapType::const_iterator itEnd = m_hashMap.end();
  for ( ; it != itEnd; ++it) {

#if 1
    const Pixel &pix = it->first;
    
    const int a0 = pix.a[0];
    const int a1 = pix.a[1];
    const int a2 = pix.a[2];

    const Pixel pixT(conv[0](a0, a1, a2),
		     conv[1](a0, a1, a2),
		     conv[2](a0, a1, a2));
#else

    const Pixel pix = it->first;
    
    const int a0 = pix.a[0];
    const int a1 = pix.a[1];
    const int a2 = pix.a[2];

    const Pixel pixT = Pixel(conv[0](pix.a[0], pix.a[1], pix.a[2]),
 			     conv[1](pix.a[0], pix.a[1], pix.a[2]),
 			     conv[2](pix.a[0], pix.a[1], pix.a[2]));

#endif    


    hashMap[pixT] = it->second;
  }
  
  std::swap(hashMap, m_hashMap);
}

void HashHistogram::weighted(std::vector<double> weightedChannelsCoeffs)
{
  HashMapType hashMap;

  HashMapType::const_iterator it = m_hashMap.begin();
  const HashMapType::const_iterator itEnd = m_hashMap.end();
  for ( ; it != itEnd; ++it) {

#if 1
    const Pixel &pix = it->first;

    const int a0 = pix.a[0];
    const int a1 = pix.a[1];
    const int a2 = pix.a[2];

    const Pixel pixT(a0*weightedChannelsCoeffs[0],
		     a1*weightedChannelsCoeffs[1],
		     a2*weightedChannelsCoeffs[2]);
#else

    const Pixel pix = it->first;

    const int a0 = pix.a[0];
    const int a1 = pix.a[1];
    const int a2 = pix.a[2];

    const Pixel pixT = Pixel(pix.a[0]*weightedChannelsCoeffs[0],
 			     pix.a[1]*weightedChannelsCoeffs[1],
 			     pix.a[2]*weightedChannelsCoeffs[2]);

#endif


    hashMap[pixT] = it->second;
  }

  std::swap(hashMap, m_hashMap);
}

size_t
HashHistogram::sum() const
{
  HashMapType::const_iterator it = m_hashMap.begin();
  const HashMapType::const_iterator itEnd = m_hashMap.end();
  size_t sum = 0;
  for ( ; it != itEnd; ++it) {
    sum += it->second;
  }
  return sum;
}

