/*
  Copyright (C) 2005-2012 Boris Mansencal, Remi Vieux, Jean-Philippe Domenger  boris.mansencal@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "iseGeometricComputer.hpp"

#include <cmath>
#ifndef M_PI
//Visual C++ does not define these constants in <cmath>
#define M_PI 3.14159265358979323846
#endif 

namespace ise
{
  
  GeometricComputer::GeometricComputer()
    : m_area(0.0),
      m_x(0.0),
      m_x2(0.0),
      m_y(0.0),
      m_y2(0.0),
      m_xy(0.0),
      m_xCoordCenter(0.0),
      m_yCoordCenter(0.0),
      m_majorAxisLength(0.0),
      m_minorAxisLength(0.0),
      m_angleMajorAxisAbscice(0.0)
  {

  }


  void GeometricComputer::clear()
  {
    m_area = 0;
    m_x = 0;
    m_x2 = 0;
    m_y = 0;
    m_y2 = 0;
    m_xy = 0.0;
    m_xCoordCenter = 0;
    m_yCoordCenter = 0;
    m_majorAxisLength = 0;
    m_minorAxisLength = 0;
    m_angleMajorAxisAbscice = 0;
  }

  void GeometricComputer::process(const girl::toger &tog, girl::region r)
  {
    tog.iterateOnRegionDomain(r, *this); //will call processPoint() on each pixel of region
  }
    
  void GeometricComputer::processPoint(const girl::point& pt)
  {
    ++m_area;
    m_x += pt.x();
    m_x2 += pt.x() * pt.x();
    m_y += pt.y();
    m_y2 += pt.y() * pt.y();
    m_xy += pt.x() * pt.y();
  }

  GeometricComputer &
  GeometricComputer::operator+=(const GeometricComputer &o)
  {
    m_area += o.m_area;
    m_x += o.m_x;
    m_x2 += o.m_x2;
    m_y += o.m_y;
    m_y2 += o.m_y2;
    m_xy += o.m_xy;
    return *this;
  }

  void GeometricComputer::finalize()
  {
    if (m_area > 0) {

      const double inv_area = 1.0/ m_area;
      m_xCoordCenter = m_x * inv_area;
      m_yCoordCenter = m_y * inv_area;
      
      // src : http://en.wikipedia.org/wiki/Eigenvalue_algorithm#Eigenvalues_of_2.C3.972_matrices

      //variances
      const double a = ((m_x2 - (m_x*m_x*inv_area))*inv_area);
      const double d = ((m_y2 - (m_y*m_y*inv_area))*inv_area);
      const double c = ((m_xy - (m_x*m_y*inv_area))*inv_area);
      const double b = c;
      
      //eigen values : lambda1 & lambda2
      const double delta = sqrt(4.0*b*c+(a-d)*(a-d))/2.0;
      const double tmp = (a+d)/2.0;
      const double lambda1 = tmp + delta;
      const double lambda2 = tmp - delta;
      
      //std::cerr<<"region="<<id<<" a="<<a<<" b="<<b<<" c="<<c<<" d="<<d<<std::endl;
      //std::cerr<<"   delta="<<delta<<" tmp="<<tmp<<" lambda1="<<lambda1<<" lambda2="<<lambda2<<"\n";

      ////m_majorAxisLength = lambda1;
      ////m_minorAxisLength = lambda2;
      m_majorAxisLength = sqrt(lambda1)*2;
      m_minorAxisLength = sqrt(lambda2)*2;

      //http://www.math.harvard.edu/archive/21b_fall_04/exhibits/2dmatrices/index.html
      /*
	double v1x = 1, v1y = 0;
	double v2x = 0, v2y = 1;
	if (c != 0) {
	v1x = lambda1-d;
	v1y = c;
	
	v2x = lambda2-d;
	v2y = c;
	}
      */
      
      //    double trace = a + b;
      //    double trace2 = trace * trace;
      //    double det = (a * b) - (c * c);
      //    double res1, res2;
      //    double val = ( (trace2 - 4 * det) == 0 ? 0 : std::sqrt( std::abs(trace2 - 4 * det)) );
      //
      //    res1 = 0.5 * ( trace + val);
      //    res2 = 0.5 * ( trace - val);
      //
      //    if (res1 > res2) {
      //      m_majorAxisLength = res1 ;
      //      m_minorAxisLength = res2;
      //    }
      //    else {
      //      m_majorAxisLength = res2 ;
      //      m_minorAxisLength = res1;
      //    }
      
      //m_angleMajorAxisAbscice = atan2(c, a - m_minorAxisLength) * 180 / M_PI;
      m_angleMajorAxisAbscice = 0.5 * atan2(2*c, a - d) * 180 / M_PI;
      
      //B: clockwise !
      
    }

  }


}//end ise namespace
