/*
  Copyright (C) 2005-2013 Remi vieux, Jean-Philippe Domenger, Achille Braquelaire, Jenny Benois-Pineau, Julian Roigt, Boris Mansencal  remi.vieux@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*
  This work was (partly) funded by the X-Media project, sponsored by
  the European Commission as part of the Information Society Technologies
  (IST) program under EC grant number IST-FP6-026978.
*/

#ifndef ISEGENERICKMEANS_HPP
#define ISEGENERICKMEANS_HPP

#include <vector>
#include <cassert>
#include <girl/girlCommon.hpp>

#include <opencv2/core/core.hpp>
#include <opencv2/flann/flann.hpp>
//parallel_for
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 243 && CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
#include <opencv2/core/internal.hpp>
#endif

namespace ise {

  template <typename T, typename Distance> class GenericKMeansParallelizer; //forward declaration.


  template <typename T, typename Distance = cvflann::L2<T> >
  class GenericKMeans
  {
  public:

    explicit GenericKMeans<T, Distance>(unsigned int numberOfClusters = 0)
    : m_numberClasses(numberOfClusters),
      m_numIter(20), 
      m_centers(),
      m_distance(),
      m_weights(NULL),
      m_medoid(false)
    {

      srand(1); //TO AVOID RANDOMNESS !!!!!
    }

    void setParameters(std::vector<T> *weights = NULL, bool medoid = false);

    void setNumberClasses(size_t numberClasses) { m_numberClasses = numberClasses; }


    // returns the dimension of the centroids
    inline int getDimension() const
    {
      assert(! m_centers.empty());
      return m_centers.cols;
    }
    inline bool getMedoid() const
    {
      return m_medoid;
    }

    inline void setMedoid(bool medoid)
    {
      m_medoid = medoid;
    }

    inline unsigned int getNumIter() const
    {
      return m_numIter;
    }

    inline void setNumIter(unsigned int numIter)
    {
      m_numIter = numIter;
    }

    inline const cv::Mat &getCenters() const
    {
      return m_centers;
    }

    //Faire une methode template ???

    //B:design: is it logical to pass a DistanceType here ??? 
    //We may compute the clusters we one distance and here query distance with another ?
    inline double getCentroidsDistance(int i, int j /*, ise::DistanceType dtype = ise::EUCLIDEAN, void *userData = NULL*/)
    {
      assert(! m_centers.empty());
      assert(i < m_centers.rows);
      assert(j < m_centers.rows);
      /*
      ise::DistanceComputer distance;
      distance.setDistanceType(dtype);
      distance.setUserData(userData);
      */
      return m_distance(m_centers.ptr<T>(i), m_centers.ptr<T>(j), m_centers.cols);
    }


    void compute(const cv::Mat &instances, cv::Mat &labels);

    void compute(const cv::Mat &instances, cv::Mat &labels, cv::Mat &centers);


    int getCluster(const T *instance, size_t dim) const;


  private:

    void clusterInstance(const T *instance, size_t size, int &cluster, double &distToCentroid) const;

    //void buildClusterer(const cv::Mat &instances, const ise::DistanceComputer &distance, std::vector<T> *weights = NULL, bool medoid = false);

    void buildClusterer(const cv::Mat &instances);


    //void initClusterCenters(const cv::Mat &data, cv::Mat &centers, unsigned int &numCenters, const ise::DistanceComputer &distance);
    
    void initClusterCenters(const cv::Mat &data);



    //void doClustering(const cv::Mat &data, cv::Mat &centers, unsigned int &numIterations, const ise::DistanceComputer &distance, bool medoid, std::vector<T> *weights = NULL);

    void doClustering(const cv::Mat &data);


    void getClusterWeight(const cv::Mat &data, const cv::Mat &centers, std::vector<size_t> &weights) const;



    //void assignPointsToClusters(const cv::Mat &data, std::vector<unsigned int> &assignment, const ise::DistanceComputer &distance) const;

    void assignPointsToClusters(const cv::Mat &data, cv::Mat &assignment) const;


    //void computeCenters(const cv::Mat &data, std::vector<unsigned int> &assignment, cv::Mat &centers, bool medoid, const ise::DistanceComputer &distance, std::vector<T> *weights = NULL);


    void computeCenters(const cv::Mat &data, cv::Mat &assignment);


    T getRandom() const
    {
      return static_cast<T>(rand())/RAND_MAX;
    }

  private:

    friend class GenericKMeansParallelizer<T, Distance>;

    size_t m_numberClasses;
    unsigned int m_numIter;

    cv::Mat m_centers;

    Distance m_distance;
    std::vector<T> *m_weights;
    bool m_medoid;

    //cache
    mutable cv::Mat m_assignment;
    mutable std::vector<T> m_clusterSize;
    cv::Mat m_clusterPosition;

  };

  /* impl: 
     We use cv::Mat and not cv::Mat_
     because with the latter, we have to use "mat.ptr<T>"
     to be able to use templated method ptr().

     cf book "C++ templates: The complete guide" p132
     
     We have a class with a template method
     and we use it in an other template class (or method)
     => we have to use ".template " syntaxe

   */


  //TODO: make a template method to pass the distance !?
  template <typename T, typename Distance>
  void GenericKMeans<T, Distance>::setParameters(std::vector<T> *weights, bool medoid)
  {
    //m_distance = distance;
    m_weights = weights;
    m_medoid = medoid;
  }
  
  template <typename T, typename Distance>
  void GenericKMeans<T, Distance>::compute(const cv::Mat &instances, cv::Mat &labels)
  { 
    assert(instances.channels() == 1);
    assert((sizeof(T)==sizeof(double) && instances.type() == CV_64F)
	   || (sizeof(T)==sizeof(float) && instances.type() == CV_32F));

    buildClusterer(instances);

    assert(m_assignment.rows == instances.rows);
    assert(m_assignment.cols == 1);

    labels = m_assignment;
  }

  template <typename T, typename Distance>
  void GenericKMeans<T, Distance>::compute(const cv::Mat &instances, cv::Mat &labels, cv::Mat &centers)
  {
    compute(instances, labels);

    computeCenters(instances, m_assignment);

    centers = m_centers;
  }


  template <typename T, typename Distance>
  void GenericKMeans<T, Distance>::buildClusterer(const cv::Mat &instances)
  {
    if (instances.rows <= (int)m_numberClasses) {
      m_centers = instances;
      m_assignment = cv::Mat(instances.rows, 1, CV_32S);
      int *a=m_assignment.ptr<int>(0);
      for (int i=0; i<instances.rows; ++i)
	*a = i;
      return;
    }


#ifdef DEBUG
    std::cout << "Beginning KMeans++ clustering (mat opencv) ..." << std::endl;
    std::cout << "Clustering " << instances.rows*instances.cols << " elements in " << m_numberClasses << " classes." << std::endl;
#endif

#ifdef TIMING
    double tT = (double)cv::getTickCount();
#endif //TIMING

    initClusterCenters(instances);

#ifdef TIMING
    tT = ((double)cv::getTickCount() - tT)/cv::getTickFrequency();
    std::cout<<"initClusterCenters: time="<<tT<<"s"<<std::endl;
#endif //TIMING


#ifdef DEBUG
    std::cout << "Finished center initialisation." << std::endl;
#endif

#ifdef TIMING
     tT = (double)cv::getTickCount();
#endif //TIMING
    
    doClustering(instances);

#ifdef TIMING
    tT = ((double)cv::getTickCount() - tT)/cv::getTickFrequency();
    std::cout<<"doClustering: time="<<tT<<"s"<<std::endl;
#endif //TIMING
    
#ifdef DEBUG
    std::cout << "End KMeans++ clustering. Gathering statistics" << std::endl;
#endif
  }
  
  template <typename T, typename Distance>
  void GenericKMeans<T, Distance>::clusterInstance(const T *instance, size_t size,
						   int &cluster,
						   double &distToCentroid) const
  {
    assert(! m_centers.empty());
    assert(size == (size_t)m_centers.cols);

    assert(m_centers.isContinuous());
    const T *c = m_centers.ptr<T>(0);

    distToCentroid = m_distance(instance, c, size);
    cluster = 0;

    c += size;
    for (unsigned int j = 1; j < (unsigned int)m_centers.rows; ++j) { //start from 1
      const double currentValue = m_distance(instance, c, size);
      if (currentValue < distToCentroid) {
	distToCentroid = currentValue;
	cluster = j;
      }
      c += size;
    }
  }

  template<typename T, typename Distance>
  int GenericKMeans<T, Distance>::getCluster(const T *instance, size_t dim) const
  {
    int cluster;
    double distance;
    clusterInstance(instance, dim, cluster, distance);
    return cluster;
  }

  /**
   * Chooses a number of centers from the data set as follows:
   *  - One center is chosen randomly.
   *  - Now repeat numCenters-1 times:
   *      - Repeat numLocalTries times:
   *          - Add a point x with probability proportional to the
   *            distance squared from x to the closest existing center
   *      - Add the point chosen above that results in the smallest potential.
   *      D. Arthur, S. Vassilvitskii, "k-means++ : The Advantage of
   *      Careful Seeding", SODA07, New Orleans, Louisiana, p1027-1035, 2007.
   *
   **/

  static int s_numLocalTries = 5;


  // template<typename T>
  // void GenericKMeans<T, Distance>::initClusterCenters(const cv::Mat &data,
  // 					    cv::Mat &centers,
  // 					    unsigned int &numCenters,
  // 					    const ise::DistanceComputer &distance)
  // {
  //   assert(numCenters > 0 && numCenters <= static_cast<int>(data.rows));
  //   double currentPot = 0;
  //   std::vector<double> closestDistSq;

  //   const size_t data_size = static_cast<int>(data.rows);
    
  //   // Choose one random center and set the closestDistSq values
  //   size_t index0 = static_cast<int>(getRandom()*data_size);
  //   // dim of data
  //   int dim = data.cols;
    
  //   std::vector<T> vIndex0 = buildVect(data, 0);

  //   centers.push_back(vIndex0);

  //   for (size_t i = 0; i < data_size; ++i) {
  //     closestDistSq.push_back(distance.getDistance(buildVect(data, i), vIndex0));
  //     currentPot += closestDistSq[i];
  //   }

  //   // Choose each center
  //   for (int centerCount = 1; centerCount < numCenters; ++centerCount) {

  //     // Repeat several trials
  //     double bestNewPot = -1;
  //     size_t bestNewIndex = data_size;
  //     for (int localTrial = 0; localTrial < s_numLocalTries; ++localTrial) {
  //       // Choose our center - have to be slightly careful to return a
  //       // valid answer even accounting for possible rounding errors
  //       double randVal = getRandom() * currentPot;
  //       size_t index;
  //       for (index = 0; index < data_size-1; ++index) {
  //         if (randVal <= closestDistSq[index])
  //           break;
  //         else
  //           randVal -= closestDistSq[index];
  //       }
	
  // 	std::vector<T> vIndex = buildVect(data, index);

  //       // Compute the new potential
  //       double newPot = 0;
  //       for (size_t i = 0; i < data_size; ++i) {
  //         newPot += std::min(distance.getDistance(buildVect(data, i), vIndex), closestDistSq[i]);
  //       }
  //       // Store the best result
  //       if (bestNewPot < 0 || newPot < bestNewPot) {
  //         bestNewPot = newPot;
  //         bestNewIndex = index;
  //       }
  //     }

  //     std::vector<T> vBestNewIndex = buildVect(data, bestNewIndex);

  //     // Add the appropriate center
  //     centers.push_back(vBestNewIndex);
  //     currentPot = bestNewPot;
  //     for (size_t i = 0; i < data_size; ++i) {
  //       closestDistSq[i] = std::min(distance.getDistance(buildVect(data, i), vBestNewIndex),
  // 				    closestDistSq[i]);
  //     }
  //   }
  // }


  // __sync_fetch_and_add is only available for integer types
  // found version for double here : http://www.memoryhole.net/kyle/2012/06/a_use_for_volatile_in_multithr.html
  static inline double qthread_dincr(double *operand, double incr)
  {
    typedef unsigned long long uint64_t; //B
    
    union {
      double   d;
      uint64_t i;
    } oldval, newval, retval;
    do {
      oldval.d = *(volatile double *)operand;
      newval.d = oldval.d + incr;
      __asm__ __volatile__ ("lock; cmpxchgq %1, (%2)"
			    : "=a" (retval.i)
			    : "r" (newval.i), "r" (operand),
			      "0" (oldval.i)
			    : "memory");
    } while (retval.i != oldval.i);
    return oldval.d;
  }
  

  template <typename T, typename Distance>
  class NewPotParallelizer
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    : public cv::ParallelLoopBody
#endif
  {
  public:
    
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    typedef cv::Range Range;
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
    typedef cv::BlockedRange Range;
#endif


    NewPotParallelizer(Distance distance, const T *data, const T *dataIndex, 
		       const typename Distance::ResultType *closestDistSq, 
		       const size_t size, 
		       typename Distance::ResultType &newPot)
      : m_distance(distance), m_data(data), m_dataIndex(dataIndex), m_closestDistSq(closestDistSq), 
	m_size(size), 
	m_newPot(newPot)
    {}

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240

  virtual void operator() (const Range &range ) const 
  {
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    const size_t start = range.start;
    const size_t end = range.end;
#else
    const size_t start = range.begin();
    const size_t end = range.end();
#endif
    process(start, end);
  }
    
private:

#endif //OpenCV >= 240

    inline void process(size_t start, size_t end) const
    {
      const T *d = m_dataIndex;
      const T *d_i = m_data + start*m_size;
      const size_t s = m_size;
      const typename Distance::ResultType *cds = m_closestDistSq + start;
      typename Distance::ResultType pot = 0;
      for (size_t i = start; i < end; ++i) {
	pot += std::min(m_distance(d_i, d, s), *cds);

	d_i += s;
	++cds;
      }

      qthread_dincr(&m_newPot, pot); //atomic update of newPot
    }

private:
    Distance m_distance;
    const T *m_data;
    const T *m_dataIndex;
    const typename Distance::ResultType *m_closestDistSq;
    const size_t m_size;
    typename Distance::ResultType &m_newPot;
  };


  template<typename T, typename Distance>
  void GenericKMeans<T, Distance>::initClusterCenters(const cv::Mat &data)
  {
    assert(static_cast<int>(m_numberClasses) > 0 && m_numberClasses <= static_cast<unsigned int>(data.rows)); 
    //B:TODO: handle case where data.rows < m_numberClasses !!!

    double currentPot = 0;
    
    const size_t data_size = data.rows;
    const size_t size = data.cols;
    const unsigned int numberOfClasses = m_numberClasses;

    m_centers = cv::Mat(numberOfClasses, size, CV_MAKETYPE(cv::DataDepth<T>::value, 1)); 
    m_clusterSize.resize(numberOfClasses);
    m_clusterPosition = cv::Mat(numberOfClasses, size, CV_MAKETYPE(cv::DataDepth<T>::value, 1));

#ifdef TIMING
    double ts = (double)cv::getTickCount();
#endif //TIMING

    // Choose one random center (the first one) and set the closestDistSq values    
    const int randIdx = 0;
    T *center0 =  m_centers.ptr<T>(randIdx);
    const T *data0 = data.ptr<T>(randIdx);
    std::copy(data0, data0+size, center0); //TODO:OPTIM?:use memcpy
    size_t nbCenters = 1;
    std::vector<typename Distance::ResultType> closestDistSq(data_size); //TODO:OPTIM: allocation + hidden memset
    assert(data.isContinuous());
    const T *d_i = data.ptr<T>(0);
    for (size_t i = 0; i < data_size; ++i) { //TODO:OPTIM:PARALLEL_FOR ???
      closestDistSq[i] = m_distance(d_i, (const T *)center0, size);
      currentPot += closestDistSq[i];
      d_i += size;
    }

#ifdef TIMING
    ts = ((double)cv::getTickCount() - ts)/cv::getTickFrequency();
    std::cout<<":::: time1 ="<<ts<<"s"<<std::endl;
    double st0=0, st1=0, st0b=0;
    ts = (double)cv::getTickCount();
#endif //TIMING
    

    const int lNumLocalTries = s_numLocalTries;

    // Choose each center
    for (unsigned int centerCount = 1; centerCount < numberOfClasses; ++centerCount) { //start from 1

#ifdef TIMING
      double tt = (double)cv::getTickCount();
#endif //TIMING

      // Repeat several trials
      double bestNewPot = -1;
      size_t bestNewIndex = data_size;
      for (int localTrial = 0; localTrial < lNumLocalTries; ++localTrial) {
        // Choose our center - have to be slightly careful to return a
        // valid answer even accounting for possible rounding errors
        double randVal = getRandom() * currentPot;
        size_t index;
        for (index = 0; index < data_size-1; ++index) {
          if (randVal <= closestDistSq[index])
            break;
          else
            randVal -= closestDistSq[index];
        }
	
	assert(index < data_size);
	const T *dataIndex = data.ptr<T>(index);

#ifdef TIMING
	double tt0 = (double)cv::getTickCount();
#endif //TIMING

        // Compute the new potential
        typename Distance::ResultType newPot = 0;
	assert(data.isContinuous());
	const T *d_i = data.ptr<T>(0);
#if 0
        for (size_t i = 0; i < data_size; ++i) { //TODO:OPTIM:PARALLEL_FOR ???
          newPot += std::min(m_distance(d_i, dataIndex, size), closestDistSq[i]);
	  d_i += size;
        }
#else
	NewPotParallelizer<T, Distance> npp(m_distance, d_i, dataIndex, &closestDistSq[0], size, newPot);
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
	cv::parallel_for_(cv::Range(0, data_size), npp);
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
	cv::parallel_for(cv::BlockedRange(0, data_size), npp);
#else
	npp.process(0, data_size);
#endif
      
#endif //0

#ifdef TIMING
	tt0 = ((double)cv::getTickCount() - tt0)/cv::getTickFrequency(); st0b += tt0;
#endif //TIMING


        // Store the best result
        if (bestNewPot < 0 || newPot < bestNewPot) {
          bestNewPot = newPot;
          bestNewIndex = index;
        }
      }

#ifdef TIMING
      tt = ((double)cv::getTickCount() - tt)/cv::getTickFrequency();
      st0 += tt;
      tt = (double)cv::getTickCount();
#endif //TIMING

      // Add the appropriate center
      assert(bestNewIndex < data_size);
      const T *dataBestNewIndex = data.ptr<T>(bestNewIndex);
      std::copy(dataBestNewIndex, dataBestNewIndex+size, m_centers.ptr<T>(nbCenters)); //TODO:OPTIM?: use memcpy
      ++nbCenters;
      //m_centers.push_back(vBestNewIndex);
      currentPot = bestNewPot;
      assert(data.isContinuous());
      const T *d_i = data.ptr<T>(0);
      for (size_t i = 0; i < data_size; ++i) { //TODO:OPTIM:PARALLEL_FOR ???
        closestDistSq[i] = std::min(m_distance(d_i, dataBestNewIndex, size), closestDistSq[i]);
	d_i += size;
      }

#ifdef TIMING
      tt = ((double)cv::getTickCount() - tt)/cv::getTickFrequency();
      st1 += tt;
#endif //TIMING

    }
    assert(nbCenters == m_numberClasses);

#ifdef TIMING
    ts = ((double)cv::getTickCount() - ts)/cv::getTickFrequency();
    std::cout<<":::: timeN ="<<ts<<"s [numberOfClasses="<<numberOfClasses<<"]"<<std::endl;
    std::cout<<"::::    st0="<<st0<<"s\n";
    std::cout<<"::::    st0b="<<st0b<<"s\n";
    std::cout<<"::::    st1="<<st1<<"s\n";
#endif //TIMING

  }



  // template<typename T>
  // void GenericKMeans<T, Distance>::doClustering(const cv::Mat &data,
  //     cv::Mat &centers,
  //     unsigned int &numIterations,
  //     const ise::DistanceComputer &distance,
  //     bool medoid,
  //     std::vector<T> *weights)
  // {
  //   int data_size = data.rows;
  //   std::vector<unsigned int> assignment(data_size);
  //   assignPointsToClusters(data, assignment, distance);

  //   std::vector<unsigned int> newAssignment(data_size);

  //   while (numIterations > 0) {
  //     computeCenters(data, assignment, centers, medoid, distance, weights);

  //     assert(newAssignment.size() == data_size);
  //     assignPointsToClusters(data, newAssignment, distance);

  //     bool isProgress = false;
  //     for (size_t i = 0; i < assignment.size(); ++i)
  //       if (assignment[i] != newAssignment[i])
  //         isProgress = true;
  //     if (!isProgress) {
  //       std::cout << "Clustering stopped with " << numIterations
  //       << " left" << std::endl;
  //       break;
  //     }
  //     assignment = newAssignment;
  //     --numIterations;
  //   }
  // }

  template<typename T, typename Distance>
  void GenericKMeans<T, Distance>::doClustering(const cv::Mat &data)
  {
    const int data_size = data.rows;

    //TODO: cache at least assignement !  => so no need to compute assignments !

    m_assignment = cv::Mat(data_size, 1, CV_32S);
    cv::Mat &assignment = m_assignment; //no copy, just reference

    assignPointsToClusters(data, assignment);

    cv::Mat newAssignment(data_size, 1, CV_32S); //TODO:OPTIM:: allocation

    assert(assignment.isContinuous());
    assert(newAssignment.isContinuous());

#ifdef TIMING
    double stT = 0; size_t lNumIter = m_numIter;
    double tT0 = (double)cv::getTickCount();
#endif //TIMING


    while (m_numIter > 0) {

#ifdef TIMING
      double tT = (double)cv::getTickCount();
#endif //TIMING

      computeCenters(data, assignment);

#ifdef TIMING
      tT = ((double)cv::getTickCount() - tT)/cv::getTickFrequency();  stT += tT;
#endif //TIMING

      assert(newAssignment.rows == data_size);
      assignPointsToClusters(data, newAssignment);

      bool isProgress = false;
      const int *p1 = assignment.ptr<int>(0);
      const int *p2 = newAssignment.ptr<int>(0);
      for (int i = 0; i < assignment.rows; ++i) {
        if (*p1 != *p2) {
          isProgress = true;
	  break;
	}
	++p1;
	++p2;
      }
      if (!isProgress) {
#ifdef VERBOSE
        std::cout << "Clustering stopped with " << m_numIter << " left" << std::endl;
#endif //VERBOSE
        break;
      }
      std::swap(assignment, newAssignment); //assignement = newAssignment;

      --m_numIter;
    }

#ifdef TIMING
    tT0 = ((double)cv::getTickCount() - tT0)/cv::getTickFrequency();
    std::cout<<"computeCenters acc: time="<<stT<<"s [numIter="<<lNumIter<<" data_size="<<data_size<<"]"<<std::endl;
    std::cout<<" total="<<tT0<<"\n";
#endif //TIMING

  }

  
  // template<typename T>
  // void GenericKMeans<T>::assignPointsToClusters(const cv::Mat &data,
  // 						std::vector<unsigned int> &assignment,
  // 						const ise::DistanceComputer &distance) const
  // {
  //   int data_size = data.rows;
  //   for (size_t i = 0; i < data_size; ++i) {
  //     double dist;
  //     std::vector<T> v = buildVect(data, i);
  //     clusterInstance(v, distance, assignment[i], dist);
  //   }
  // }

  template <typename T, typename Distance>
  class GenericKMeansParallelizer
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    : public cv::ParallelLoopBody
#endif
  {
  public:
    
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    typedef cv::Range Range;
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
    typedef cv::BlockedRange Range;
#endif

    GenericKMeansParallelizer(const GenericKMeans<T, Distance> &gkm, const cv::Mat &data, cv::Mat &assignment)
      : m_gkm(gkm), m_data(data), m_assignment(assignment)
    {}

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240

  virtual void operator() (const Range &range ) const 
  {
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    const size_t start = range.start;
    const size_t end = range.end;
#else
    const size_t start = range.begin();
    const size_t end = range.end();
#endif
    process(start, end);
  }
    
private:

#endif //OpenCV >= 240

    inline void process(size_t start, size_t end) const
    {
      for (size_t i=start; i<end; ++i) {
	double dist;
	m_gkm.clusterInstance(m_data.ptr<T>(i), m_data.cols, *m_assignment.ptr<int>(i), dist);
      }	
    }

private:
    const GenericKMeans<T, Distance> &m_gkm;
    const cv::Mat &m_data; 
    cv::Mat &m_assignment;
};


  template<typename T, typename Distance>
  void GenericKMeans<T, Distance>::assignPointsToClusters(const cv::Mat &data, cv::Mat &assignment) const
  {
    /*
    const int data_size = data.rows;
    for (int i = 0; i < data_size; ++i) { //TODO:OPTIM:PARALLEL
      double dist;
      clusterInstance(data.ptr<T>(i), data.cols, *assignment.ptr<int>(i), dist);
    }
    */

    GenericKMeansParallelizer<T, Distance> p(*this, data, assignment);
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 243
    cv::parallel_for_(cv::Range(0, data.rows), p);
#elif CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 240
    cv::parallel_for(cv::BlockedRange(0, data.rows), p);
#else
    p.process(0, data.rows);
#endif

  }



  // template<typename T>
  // void GenericKMeans<T>::computeCenters(const cv::Mat_<double> &data,
  //     std::vector<unsigned int> &assignment ,
  //     cv::Mat &centers, bool medoid,
  //     const ise::DistanceComputer &distance,
  //     std::vector<double> *weights)
  // {
  //   int data_size = data.rows;
  //   assert(data_size > 0 || data.cols > 0);
  //   assert(centers.size() > 0 || centers[0].size() > 0);
  //   assert(assignment.size() == data_size);
  //   if (weights) {
  //     assert(weights->size() == data_size);
  //   }

  //   const size_t numClusters = centers.size();
  //   std::vector<double> clusterSize(numClusters, 0);
  //   ise::matrix<T> clusterPosition(numClusters, centers[0].size());

  //   for (size_t i = 0; i < assignment.size(); ++i) {
  //     const int cIndex = assignment[i];
  //     const size_t s = clusterPosition[cIndex].size();
  //     double weight = 1;
  //     if (weights)
  //       weight = (*weights)[i];
  //     clusterSize[cIndex]+= weight;

  //     std::vector<T> v = buildVect(data, i);
      
  //     for (size_t j = 0; j < s; ++j) {
  //       clusterPosition[cIndex][j] += (v[j]*weight);
  //     }
  //   }
    
  //   for (size_t i = 0; i < numClusters; ++i)
  //     if (clusterSize[i] > 0) {
  //       const size_t s = clusterPosition[i].size();
  //       for (size_t j = 0; j < s; ++j)
  //         clusterPosition[i][j] /= clusterSize[i];
  //     }

  //   if (medoid) {
  //     for (size_t i = 0; i < clusterPosition.size(); ++i) {
	

  //       double bestValue = distance.getDistance(buildVect(data, 0), clusterPosition[i]);
  //       size_t bestIndex = 0;
  //       for (size_t j = 1; j < data_size; ++j) {
  //         const double currentValue = distance.getDistanceCutOff(buildVect(data, j), clusterPosition[i], bestValue);
  //         if (currentValue < bestValue) {
  //           bestValue = currentValue;
  //           bestIndex = j;
  //         }
  //       }

  //       centers[i] = buildVect(data, bestIndex);
  //     }
  //     return;
  //   }

  //   centers = clusterPosition;
  // }

  template<typename T, typename Distance>
  void GenericKMeans<T, Distance>::computeCenters(const cv::Mat &data, cv::Mat &assignment)
  {
    size_t data_size = data.rows;
    assert(data_size > 0 || data.cols > 0);
    assert(! m_centers.empty());
    assert((size_t)assignment.rows == data_size);
    assert(!m_weights || m_weights->size() == data_size);
    assert(data.cols == m_centers.cols);

    const size_t numClusters = m_centers.rows;
    std::vector<T> &clusterSize = m_clusterSize; //no allocation
    memset(&clusterSize[0], 0, clusterSize.size()*sizeof(T));

    cv::Mat &clusterPosition = m_clusterPosition;
    clusterPosition = cv::Mat::zeros(numClusters, m_centers.cols, CV_MAKETYPE(cv::DataDepth<T>::value, 1)); //no allocation, just memset


    const bool medoid = m_medoid;
    const size_t size = data.cols;
    const std::vector<T> *weights = m_weights;

    assert(assignment.isContinuous());
    const int *p = assignment.ptr<int>(0);
    for (size_t i = 0; i < data_size; ++i) { //REM: not easily parallel, concurrent write to clusterPosition
      const int cIndex = *p;
      ++p;

      const T weight = ( weights == NULL ? 1 : (*weights)[i] );

      clusterSize[cIndex] += weight;
      
      const T *data_i = data.ptr<T>(i);
      T *cp_cIndex = clusterPosition.ptr<T>(cIndex);
      for (size_t j = 0; j < size; ++j) {
        cp_cIndex[j] += data_i[j]*weight;
      }
    }
    
    for (size_t i = 0; i < numClusters; ++i) { //not necessary to parallelize (numClusters rather low)
      if (clusterSize[i] > 0) {
	const T inv_clusterSize = 1.0 / clusterSize[i];
	T *cp_i = clusterPosition.ptr<T>(i);
        for (size_t j = 0; j < size; ++j)
          cp_i[j] *= inv_clusterSize;
      }
    }

    if (medoid) {
      for (int i = 0; i < clusterPosition.rows; ++i) { //TODO:OPTIM:PARALLEL
	const T *cp_i = clusterPosition.ptr<T>(i);
	const T *data0 = data.ptr<T>(0);
        double bestValue = m_distance(data0, cp_i, size);
        size_t bestIndex = 0;
        for (size_t j = 1; j < data_size; ++j) { //start from 1
	  const T *data_j = data.ptr<T>(j);
          //const double currentValue = distance.getDistanceCutOff(data_j, data_j+size, cp_i, bestValue); //B: not implemented (and must be slower)
	  const double currentValue = m_distance(data_j, cp_i, size);
          if (currentValue < bestValue) {
            bestValue = currentValue;
            bestIndex = j;
          }
        }

	assert(bestIndex < (size_t)data.rows);
	const T *dataBestIndex = data.ptr<T>(bestIndex);
	std::copy(dataBestIndex, dataBestIndex+size, m_centers.ptr<T>(i)); //TODO:OPTIM:use memcpy
      }
      return;
    }

    std::swap(m_centers, clusterPosition);
  }



  template <typename T, typename Distance>
  void GenericKMeans<T, Distance>::getClusterWeight(const cv::Mat &data, const cv::Mat &centers, std::vector<size_t> &weights) const
  {
    assert(data.rows > 0 && centers.rows > 0);
    assert(data.cols == centers.cols);

    cv::Mat assignment(data.rows, 1, CV_32S);

    assignPointsToClusters(data, assignment);

    weights.clear();
    weights.assign(centers.rows, 0);

    assert(assignment.isContinuous());
    int *p = assignment.ptr<int>(0);
    for (int i=0; i<data.rows; ++i) {
      weights[*p]++;
      ++p;
    }
  }


} //end namespace ise

#endif /* ! ISEGENERICKMEANS_HPP */
