/*
  Copyright (C) 2005-2010 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef ISEFEATURECOMPUTERHANDLER_HPP_
#define ISEFEATURECOMPUTERHANDLER_HPP_

#include <cassert>
#include <string>
#include <vector>

#include <girl/girlCommon.hpp>
#include <girl/girlHashMap.hpp>
#include <girl/girlSharedPtr.hpp>
#include <girl/girlToger.hpp>
#include <girl/girlUnorderedSet.hpp>

#include "iseFeatureComputer.hpp"


/**
 * @brief Class that register the feature computers and the dependencies between them
 * Hence, it can trigger the computation/update of features in the appropriate order
 */
namespace ise
{

  class GIRL_API FeatureComputerHandler : public girl::toger::observer
  {

    typedef FeatureComputerPtr FeatureComputerPtrType;
    typedef FeatureComputer::FeatureComputerListType DependancyListType;
    //  typedef  FeatureComputerListType;

  public:

    FeatureComputerHandler();
    virtual
    ~FeatureComputerHandler();


    void
    update(const FeatureComputerHandler&);

    /**
     * Adds a computer @param computer to the handler. Required
     * computers for the computer added MUST ALREADY be in the
     * handler, i.e. being already added with the same function.
     * @param doCompute specify whether the computation of features
     * should be triggered when adding the computer to the handler.
     * Otherwise, it is client responsibility to trigger it when required
     *
     */
    bool
    add(const FeatureComputerPtrType &computer, bool doCompute = false);

    /**
     * Remove computer identified by its @param uri from
     * the handler.
     * @return true if computer was actually removed, false otherwise
     *
     */
    bool
      remove(const std::string &title, bool removeDependencies,
	     DependancyListType &removed);


    /**
     * Check if a computer with same uri exists in the handler
     */
    bool
    exists(const std::string &uri) const;


    /**
     * Return the computer identified by it's uri.
     */
    // R: TODO: REMOVE, use only the next function?
    const girl::shared_ptr<FeatureComputer>
    getFeatureComputer(const std::string &uri) const;

    /**
     * Return the computer identified by it's uri @uri
     * and dynamically casted as a <T>. If cast fails, the
     * returned ptr is NULL.
     */
    //    template <typename T>
    //    const girl::shared_ptr<T>
    //    getFeatureComputer(const std::string &uri) const;

    /**
     * Return the list of titles of the computers currently recorded
     * in the handler
     */
    //    void
    //    getComputerList(std::vector<std::string> &titles) const;

    /**
     * Return the list of uris of the computers which can be
     * casted in the type specified in template parameter
     */
    template <typename T>
    void
    getComputerList(FeatureComputer::FeatureComputerListType &list) const;

    /**
     * Return the number of computers stored in the handler
     */
    size_t
    getNbComputer() const;


    /**
     * Change the computer name and update the list of dependencies
     */
    //    void
    //    renameComputer(const std::string &oldName, const std::string &newName);




    /**
     * Remove all computed features from the computers currently in the handler
     */
    void
    reset();

    /**
     * Clear the list of computers from the handler
     */
    void
    clear();


  public:
    // inherited girl::toger::observer methods
    bool doNewRegion()    const { return true; }
    bool doUpdateRegion() const { return true; }
    bool doMergeRegions() const { return true; }
    bool doDeleteRegion() const { return true; }
    bool doRenameRegion() const { return true; }

    void newRegion(const girl::regionDeque &);

    void updateRegion(const girl::regionDeque &);

    void mergeRegions(girl::region keptRegion, girl::region deletedRegion);

    void deleteRegion(girl::region);

    void renameRegion(girl::region oldRegion, girl::region newRegion);

#ifndef NDEBUG

    /**
     * Check that every computer in the handler has dependencies to other
     * computers also present in the handler
     */
    bool
    checkConsistency();
#endif

  protected:

    //  typedef girl::unorderedSet<girl::shared_ptr<FeatureComputer> > DependancyListType;

    typedef struct Node {
      girl::shared_ptr<FeatureComputer> m_computer;
      DependancyListType m_computersNeedingMe;
      //      DependancyListType m_computersINeed;
      Node() : m_computer(), m_computersNeedingMe() {}
      ~Node() {}
    } Node;

    typedef girl::hashMap<std::string, Node> ContainerType;


    ContainerType m_list;

    std::vector<std::string> m_sortedComputers;



  protected:

    /**
     * fill the vector with an ordered list of computers that can be
     * processed sequentially. Use this function before calling the
     * observers function in order to get the dependency hierarchy
     * between the current FeatureComputers.
     * NB: if WithImageFeature=false the non-toger feature computers are not inserted into the list
     */
    void
    getFeatureHierarchy(std::vector<std::string> &sorted, bool withImageFeature = false) const;
    
    void
    getFeatureHierarchy(const FeatureComputerPtrType &,
			  girl::unorderedSet<std::string> &updated,
			  std::vector<std::string> &sorted,
			  bool withImageFeature) const;
    
  };


  //  template <typename T>
  //  const girl::shared_ptr<T>
  //  FeatureComputerHandler::getFeatureComputer(const std::string &uri) const
  //  {
  //    ContainerType::const_iterator it = m_list.find(uri);
  //    if (it != m_list.end())
  //          return it->second.m_computer;
  //    }
  //    return girl::shared_ptr<T>();
  //  }


  template <typename T>
  void
  FeatureComputerHandler::getComputerList(FeatureComputer::FeatureComputerListType &list) const
  {
    ContainerType::const_iterator it = m_list.begin();
    const ContainerType::const_iterator itEnd = m_list.end();
    for (; it != itEnd; ++it) {
        girl::shared_ptr<T> ptr = girl::dynamic_pointer_cast<T>(it->second.m_computer);
        if (ptr != girl::shared_ptr<T>()) {
            list.insert(ptr->uri());
        }
    }
  }
}

#endif /* ISEFEATURECOMPUTERHANDLER_HPP_ */
