/*
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <algorithm>
#include <cassert>
#include <deque>
#include <iostream>

#include "iseFeatureComputerHandler.hpp"

static
bool
P_checkHaveSameDependancies(const girl::shared_ptr<ise::FeatureComputer> &a,
    const girl::shared_ptr<ise::FeatureComputer> &b) {
  bool returnValue = true;
  ise::FeatureComputer::FeatureComputerListType la;
  a->getRequiredComputers(la);
  ise::FeatureComputer::FeatureComputerListType lb;
  b->getRequiredComputers(lb);
  if (la.size() != lb.size())
    returnValue = false;
  if (returnValue) {
      ise::FeatureComputer::FeatureComputerListType::const_iterator it = la.begin();
      const ise::FeatureComputer::FeatureComputerListType::const_iterator itEnd = la.end();
      for (; it != itEnd; ++it) {
          if (lb.find(*it) == lb.end()) {
              returnValue = false;
              break;
          }
      }
  }
  if (!returnValue) {
      std::cerr << "Trying to add duplicate node ("<< a->uri() << ") in FeatureComputerHandler with different dependancies of existing node" << std::endl;
  }
  return returnValue;
}

namespace ise
{

  FeatureComputerHandler::FeatureComputerHandler() :
                  m_list()
  {
  }

  FeatureComputerHandler::~FeatureComputerHandler()
  {
  }

  void
  FeatureComputerHandler::update(const FeatureComputerHandler& h)
  {
    // if (this == &h)
    //   return *this;
    
    std::vector<std::string> order;
    h.getFeatureHierarchy(order, true);
    // remove
    std::deque<std::string> toRemove;
    ContainerType::iterator cIt = m_list.begin();
    const ContainerType::iterator cItEnd = m_list.end();
    for (; cIt != cItEnd; ++cIt) {
      std::vector<std::string>::const_iterator it = std::find(order.begin(), order.end(), cIt->first);
      if (it == order.end()) {
	toRemove.push_back(cIt->first);
      } 
    }
    while (!toRemove.empty()) {
      DependancyListType removed;
      std::string rm = toRemove.front();
#ifndef NDEBUG
      bool ok = 
#endif //NDEBUG
	remove(rm, false, removed);
      assert(ok);
      toRemove.pop_front();
    }
    

    std::vector<std::string>::const_iterator it = order.begin();
    const std::vector<std::string>::const_iterator itEnd = order.end();
    for (; it != itEnd; ++it) {
        ContainerType::iterator myIt = m_list.find(*it);
        FeatureComputerPtr computer = h.getFeatureComputer(*it);
        if (myIt == m_list.end()) {
            // new feature computer
            add(computer, true);
        } else {
            // merge lists
            assert(P_checkHaveSameDependancies(myIt->second.m_computer, computer));
            ContainerType::const_iterator lIt = h.m_list.find(*it);
            assert(lIt != h.m_list.end());
            DependancyListType::const_iterator depIt = lIt->second.m_computersNeedingMe.begin();
            const DependancyListType::const_iterator depItEnd = lIt->second.m_computersNeedingMe.end();
            for (; depIt != depItEnd; ++depIt) {
                myIt->second.m_computersNeedingMe.insert(*depIt);
            }
	    // Called in case of changing cacheState
	    //computer->compute();
	    computer->update();
        }
    }


    //    ContainerType::const_iterator it = h.m_list.begin();
    //    const ContainerType::const_iterator itEnd = h.m_list.end();
    //    for (; it != itEnd; ++it) {
    //        ContainerType::iterator myIt = m_list.find(it->first);
    //        if (myIt == m_list.end()) {
    //            // new feature computer to this
    //            // will compute the features... but must check for multithreadable version, it might
    //            // be better to store the list of new computers and launch them concurrently
    //            add(it->second.m_computer, true);
    //            // m_list[it->first] = it->second;
    //        } else {
    //            // merge lists
    //            assert(P_checkHaveSameDependancies(myIt->second.m_computer, it->second.m_computer));
    //            DependancyListType::const_iterator depIt = it->second.m_computersNeedingMe.begin();
    //            const DependancyListType::const_iterator depItEnd = it->second.m_computersNeedingMe.end();
    //            for (; depIt != depItEnd; ++depIt) {
    //                myIt->second.m_computersNeedingMe.insert(*depIt);
    //            }
    //        }
    //    }

    //    return *this;
}

  bool
  FeatureComputerHandler::add(const FeatureComputerPtrType &computer, bool doCompute)
  {

    std::string title = computer->uri();
    if (m_list.find(title) == m_list.end()) {
        Node n;
        n.m_computer = computer;
        //typedef std::vector<girl::shared_ptr<FeatureComputer> > RequiredFeatureListType;
        DependancyListType required;
        computer->getRequiredComputers(required);
        DependancyListType::const_iterator reqIt = required.begin();
        const DependancyListType::const_iterator reqItEnd = required.end();
      //  std::cout << "Adding " << title << ", ancestors: ";
        for (; reqIt != reqItEnd; ++reqIt) {
        //    std::cout << *reqIt << " ";
            ContainerType::iterator it = m_list.find(*reqIt);
            assert(it != m_list.end());
            it->second.m_computersNeedingMe.insert(title);
            //     n.m_computersINeed.insert(/*it->second.m_computer->getTitle()*/ *reqIt);
        }
       // std::cout << std::endl;
        assert(m_list.find(title) == m_list.end());
        m_list[title] = n;
        if (doCompute)
          computer->compute();
    }
#ifndef NDEBUG
    else {
      //      assert(P_checkHaveSameDependancies(computer, m_list[title].m_computer));
	return false;
    }
#endif
    getFeatureHierarchy(m_sortedComputers);
    return true;
  }

  bool
  FeatureComputerHandler::remove(const std::string &computerUri,
				 bool removeDependencies,
				 DependancyListType &removed)
  {
    //std::string computerUri = computer->uri();
    ContainerType::iterator it = m_list.find(computerUri);
    assert(it != m_list.end());
    bool removeDepOk = true;
    if (removeDependencies) {
        while (it->second.m_computersNeedingMe.size() > 0) {
            //FeatureComputerPtrType dep = *(it->second.m_computersNeedingMe.begin());
            std::string toRemove = *(it->second.m_computersNeedingMe.begin());
            removeDepOk &= remove(toRemove/**(it->second.m_computersNeedingMe.begin())*/, removeDependencies, removed);
        }
    }
    // remove only if I am not needed by anyone else
    if (it->second.m_computersNeedingMe.size() == 0) {
        //     std::string myTitle = it->second.m_computer->getTitle();
        DependancyListType myRequiredExtractors;
        FeatureComputerPtrType computer = it->second.m_computer;
        computer->getRequiredComputers(myRequiredExtractors);
        DependancyListType::const_iterator dep = myRequiredExtractors.begin();
        const DependancyListType::const_iterator depEnd = myRequiredExtractors.end();
        for (; dep != depEnd; ++dep) {
            ContainerType::iterator cIt = m_list.find(*dep);
            assert(cIt != m_list.end());
            cIt->second.m_computersNeedingMe.erase(computerUri);
            // if (removeDependencies)
            //   remove(cIt->second.m_computer->getTitle(), removeDependencies);
        }
	removed.insert(computerUri);
        m_list.erase(it);
	getFeatureHierarchy(m_sortedComputers);
        return removeDepOk && true;
    }

    return false;

  }

  bool
  FeatureComputerHandler::exists(const std::string &uri) const
  {
    return m_list.find(uri) != m_list.end();
  }


  const FeatureComputerHandler::FeatureComputerPtrType
  FeatureComputerHandler::getFeatureComputer(const std::string &title) const
  {
    ContainerType::const_iterator it = m_list.find(title);
    if (it != m_list.end()) {
        return it->second.m_computer;
    } else {
        assert(false);
        return girl::shared_ptr<FeatureComputer>();
    }
  }

  //  void
  //  FeatureComputerHandler::getComputerList(std::vector<std::string> &titles) const
  //  {
  //    titles.clear();
  //    titles.reserve(m_list.size());
  //    ContainerType::const_iterator it = m_list.begin();
  //    const ContainerType::const_iterator itEnd = m_list.end();
  //    for (; it != itEnd; ++it) {
  //        titles.push_back(it->first);
  //    }
  //  }

  size_t
  FeatureComputerHandler::getNbComputer() const
  {
    return m_list.size();
  }

  //  void
  //  FeatureComputerHandler::renameComputer(const std::string &oldName, const std::string &newName)
  //  {
  //    ContainerType::iterator oldIt = m_list.find(oldName);
  //    assert(oldIt != m_list.end());
  //    assert(m_list.find(newName) == m_list.end());
  //    DependancyListType::const_iterator it = oldIt->second.m_computersNeedingMe.begin();
  //    const DependancyListType::const_iterator itEnd = oldIt->second.m_computersNeedingMe.end();
  //    for (; it != itEnd; ++it) {
  //        ContainerType::iterator depIt = m_list.find(*it);
  //        assert(depIt != m_list.end());
  //        depIt->second.m_computersINeed.erase(oldName);
  //        depIt->second.m_computersINeed.insert(newName);
  //    }
  //    oldIt->second.m_computer->setTitle(newName);
  //    m_list[newName] = oldIt->second;
  //    m_list.erase(oldIt);
  //  }

  void
  FeatureComputerHandler::reset()
  {
    ContainerType::iterator it = m_list.begin();
    const ContainerType::const_iterator itEnd = m_list.end();
    for (; it != itEnd; ++it) {
        it->second.m_computer->reset();
    }
  }

  void
  FeatureComputerHandler::clear()
  {
    m_list.clear();
  }




  void
  FeatureComputerHandler::newRegion(const girl::regionDeque &regs)
  {
    // std::vector<std::string> order;
    // getFeatureHierarchy(order);
    assert(m_sortedComputers.size() <= m_list.size());
    std::vector<std::string>::const_iterator it = m_sortedComputers.begin();
    const std::vector<std::string>::const_iterator itEnd = m_sortedComputers.end();
    for (; it != itEnd; ++it) {
        ContainerType::iterator iter = m_list.find(*it);
        assert(iter != m_list.end());
        iter->second.m_computer->newRegion(regs);
    }
  }

  void
  FeatureComputerHandler::updateRegion(const girl::regionDeque &regs)
  {
    // std::vector<std::string> order;
    // getFeatureHierarchy(order);
    assert(m_sortedComputers.size() <= m_list.size());
    std::vector<std::string>::const_iterator it = m_sortedComputers.begin();
    const std::vector<std::string>::const_iterator itEnd = m_sortedComputers.end();
    for (; it != itEnd; ++it) {
        ContainerType::iterator iter = m_list.find(*it);
        assert(iter != m_list.end());
        iter->second.m_computer->updateRegion(regs);
    }
  }

  void
  FeatureComputerHandler::mergeRegions(girl::region keptRegion,
      girl::region deletedRegion)
  {
    // std::vector<std::string> order;
    // getFeatureHierarchy(order);
    assert(m_sortedComputers.size() <= m_list.size());
    std::vector<std::string>::const_iterator it = m_sortedComputers.begin();
    const std::vector<std::string>::const_iterator itEnd = m_sortedComputers.end();
    for (; it != itEnd; ++it) {
        ContainerType::iterator iter = m_list.find(*it);
        assert(iter != m_list.end());
        iter->second.m_computer->mergeRegions(keptRegion, deletedRegion);
    }
  }

  void
  FeatureComputerHandler::deleteRegion(girl::region reg)
  {
    // std::vector<std::string> order;
    // getFeatureHierarchy(order);
    assert(m_sortedComputers.size() <= m_list.size());
    std::vector<std::string>::const_iterator it = m_sortedComputers.begin();
    const std::vector<std::string>::const_iterator itEnd = m_sortedComputers.end();
    for (; it != itEnd; ++it) {
        ContainerType::iterator iter = m_list.find(*it);
        assert(iter != m_list.end());
        iter->second.m_computer->deleteRegion(reg);
    }
  }

  void
  FeatureComputerHandler::renameRegion(girl::region oldRegion, girl::region newRegion)
  {
    // std::vector<std::string> order;
    // getFeatureHierarchy(order);
    assert(m_sortedComputers.size() <= m_list.size());
    std::vector<std::string>::const_iterator it = m_sortedComputers.begin();
    const std::vector<std::string>::const_iterator itEnd = m_sortedComputers.end();
    for (; it != itEnd; ++it) {
        ContainerType::iterator iter = m_list.find(*it);
        assert(iter != m_list.end());
        iter->second.m_computer->renameRegion(oldRegion, newRegion);
    }
  }


  // TODO: improve, return a true hierarchy that can be parallelized
  // i.e. return the computers by their level in the hierarchy
  void
  FeatureComputerHandler::getFeatureHierarchy(std::vector<std::string> &sorted, bool withImageFeature) const
  {
    sorted.clear();
    girl::unorderedSet<std::string> updatedNodes;
    ContainerType::const_iterator it = m_list.begin();
    const ContainerType::const_iterator itEnd = m_list.end();
    for (; it != itEnd; ++it) {
        std::string uri = it->second.m_computer->uri();
        if (updatedNodes.find(uri) == updatedNodes.end())
          getFeatureHierarchy(it->second.m_computer, updatedNodes, sorted, withImageFeature);
    }

//    std::cout << "Computing in this order" << std::endl;
//    for (size_t i = 0; i < sorted.size(); ++i) {
//        std::cout << sorted[i] << std::endl;
//    }
  }

  void
  FeatureComputerHandler::getFeatureHierarchy(const FeatureComputerPtrType &computer,
      girl::unorderedSet<std::string> &updated,
      std::vector<std::string> &sorted,
      bool withImageFeature) const
  {
    std::string uri = computer->uri();
    if (updated.find(uri) != updated.end())
      return;
    DependancyListType dependencies;
    computer->getRequiredComputers(dependencies);
    if (dependencies.size() > 0) {
        DependancyListType::const_iterator it = dependencies.begin();
        const DependancyListType::const_iterator itEnd = dependencies.end();
        for (; it != itEnd; ++it) {
            if (updated.find(*it) == updated.end()) {
                ContainerType::const_iterator depIt = m_list.find(*it);
                assert(depIt != m_list.end());
                getFeatureHierarchy(depIt->second.m_computer, updated, sorted, withImageFeature);
            }
        }
    }

    if (withImageFeature) {
      sorted.push_back(uri);
    } else {
      if (computer->isTogerObserver()) {
        sorted.push_back(uri);
      }
    }
    updated.insert(uri);
  }
  
  

#ifndef NDEBUG
  bool
  FeatureComputerHandler::checkConsistency()
  {
    ContainerType::const_iterator it = m_list.begin();
    const ContainerType::const_iterator itEnd = m_list.end();
    for (; it != itEnd; ++it) {
        //      std::cout << it->second.m_computer->getTitle() << std::endl;
        DependancyListType::const_iterator dep;
        DependancyListType::const_iterator depEnd;
        // std::cout << "\t Is required by: ";
        for (dep = it->second.m_computersNeedingMe.begin(),
            depEnd = it->second.m_computersNeedingMe.end();
            dep != depEnd; ++dep) {
            // std::cout << *dep <<  " ";
            if (m_list.find(*dep) == m_list.end())
              return false;
        }
        // std::cout << std::endl;
        // std::cout << "\t Requires: ";
        DependancyListType required;
        it->second.m_computer->getRequiredComputers(required);
        for (dep = required.begin(),
            depEnd = required.end();
            dep != depEnd; ++dep) {
            // std::cout << *dep << " ";
            if (m_list.find(*dep) == m_list.end())
              return false;
        }
        // std::cout << std::endl;
    }
    return true;
  }
#endif

}
