/*
  Copyright (C) 2005-2011 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*
 * iseFeatureComputer.hpp
 *
 *      Base class for all Feature Computer, i.e. classes that compute
 *      descriptors. Descriptors can be local (at pixel level), regional,
 *      possibly more kind in the future.
 *      The interface provides the necessary functions to handle those
 *      descriptors in a uniform manner, through the SegmentationContext
 */

#ifndef ISEFEATURECOMPUTER_HPP_
#define ISEFEATURECOMPUTER_HPP_

#include <set>
#include <string>
#include <vector>

#include <girl/girlCommon.hpp>
#include <girl/girlSharedPtr.hpp>
#include <girl/girlToger.hpp>
#include <girl/girlUnorderedSet.hpp>

namespace ise
{
  class SegmentationContext;

  class GIRL_API FeatureComputer : public girl::toger::observer
  {

  public:

    typedef std::set<std::string> FeatureComputerListType;

    /**
     * Cache type :
     * - @a NONE: Features are not stored and always recomputed.
     * - @a LAZY: Features are stored, but only computed when asked for.
     * - @a ALWAYS: Features are stored and updated on each operation.
     */
    typedef enum {
      NONE = 0,
      LAZY = 1,
      ALWAYS = 2
    } CacheType;
    


  public:

    /**
     * @brief Constructor
     *
     * @param segContext SegmentationContext for computing the features
     */
    FeatureComputer(SegmentationContext *segContext) :
      m_segContext(segContext),
      m_cacheType(NONE) 
      {}

    virtual ~FeatureComputer() {}


    /**
      * Get the description of the current region set, i.e. what feature is computed and
      * a description of the feature dimension semantics
      * @param title is filled with the name of the feature
      * @param dimDesc is filled with a description of the semantics of each
      * dimension of the feature vector
      */

//     inline void
//     getDescription(std::string &title, std::vector<std::string> &dimDesc) const
//     {
//       title = getTitle();
//       getDimDescription(dimDesc);
//     }

//     const std::string&
//     getTitle() const
//     {
//       return m_title;
//     }
//
//     void
//     setTitle(const std::string &title)
//     {
//       m_title = title;
//     }


     // virtual functions to be overloaded by the computers
  public:

     /**
      * Initiate the computation of the features. This is virtual because it means
      * different things depending on the computer type: ImageFeature needs to compute
      * according to the loaded image (or to another image feature). RegionFeatures
      * needs to call newRegion for all regions in the toger.
      */
     virtual
     void compute() = 0;


     /**
       * This function updates the features. Needed when changing the computer cacheState
       * for example. Update will be different according to the subtype of FeatureComputer:
       * imagefeature needs not to be updated, while regionFeature needs to be updated if the
       * segmentation or cache state has changed.
       */
     virtual
     void update() = 0;

    /**
     * @return The dimensionality of the descriptor computed by this computer
     */
    virtual size_t
    getFeatureDim() const = 0;


    /**
     * @param dimDesc filled with a description of the semantics of each dimension
     * of the descriptor
     */
    virtual void
    getDimDescription(std::vector<std::string> &dimDesc) const = 0;

    /**
     * fill @computers with the titles of the required feature computer for this
     * computer
     */
    virtual void
    getRequiredComputers(FeatureComputerListType &computers) const = 0;


    /**
     * Remove all features stored in the computer
     */
    virtual void
    reset() = 0;

    /**
     * @param helpContent filled with a relevant description of the algorithm, i.e. what
     * it does, what are the parameters, etc.
     */
    virtual const std::string
    // Uncomment to see the computers that did not fill the help function at compile time
    //help() const = 0;
    help() const { return "TODO!"; }



    /**
     * Must return true if the computer features needs to be updated for events occurring
     * in the toger (i.e. new regions, merge, rename, update), false otherwise. Example
     * the ImageFeature needs not to be updated
     */
    virtual bool
    isTogerObserver() const = 0;


    /**
     * This overloaded function should return an encoding of a particular instance
     * of a FeatureComputer as a String, e.g. the type of feature computer with its
     * parameters. As such it is not a URI (todo:rename) but an algorithm instance
     * identifier
     */
    virtual const std::string
    uri() const = 0;


    /**
     * Set cache state
     */
    virtual void
    setCacheState(CacheType cacheType) = 0;

    /**
     * Get SegContext
     */
    SegmentationContext *getSegContext() const { return m_segContext; }

  protected:

    SegmentationContext *m_segContext;

    CacheType            m_cacheType;

  };

  typedef girl::shared_ptr<FeatureComputer> FeatureComputerPtr;

}

#endif /* ISEFEATURECOMPUTER_HPP_ */
