/*
  Copyright (C) 2005-2010 Jean-Philippe Domenger, Boris Mansencal,
  Julian Roigt, Remi Vieux remi.vieux@labri.fr
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef ISEFactoryTemplate_HXX
#define ISEFactoryTemplate_HXX

#include <cassert>

#include <girl/girlCommon.hpp>
#include <girl/girlSharedPtr.hpp>

#include "iseAbstractFactoryTemplate.hxx"

namespace ise {
  /**
   * Template for creating factories: speed up the code
   * Use this class if you want to create a factory with
   * no parameters in the constructor
   * It is achieved in one typedef with the right template
   * parameters
   * @see
   * http://www.yolinux.com/TUTORIALS/C++Singleton.html (Last consult Jan. 2011)
   */
  template <typename Interface, typename ConcreteClass>
  class GIRL_API FactoryTemplate : public AbstractFactoryTemplate<Interface>
  {
  public:

    /**
     * Factory is created as a singleton
     * @return The instance of the factory object
     */
    static FactoryTemplate<Interface, ConcreteClass>& instance();

    /**
     * Object create function
     * @return An object of type Interface
     */
    girl::shared_ptr<Interface>
    create(ise::SegmentationContext *segContext);

  private:

    FactoryTemplate<Interface, ConcreteClass>() {}
    FactoryTemplate<Interface, ConcreteClass>(const FactoryTemplate<Interface, ConcreteClass>&);
    virtual ~FactoryTemplate<Interface, ConcreteClass>() {}

  private:
    static FactoryTemplate<Interface, ConcreteClass>* m_instance;

  };

  template <typename Interface, typename ConcreteClass>
  FactoryTemplate<Interface, ConcreteClass>&
  FactoryTemplate<Interface, ConcreteClass>::instance()
  {
    if (!m_instance)
      m_instance = new FactoryTemplate<Interface, ConcreteClass>();
    assert(m_instance);
    return *m_instance;
  }

  template <typename Interface, typename ConcreteClass>
  girl::shared_ptr<Interface>
  FactoryTemplate<Interface, ConcreteClass>::create(ise::SegmentationContext *segContext)
  {
    girl::shared_ptr<Interface> result(new ConcreteClass(segContext));
    return result;
  }

  template <typename Interface, typename ConcreteClass>
  FactoryTemplate<Interface, ConcreteClass>*
  FactoryTemplate<Interface, ConcreteClass>::m_instance = NULL;

}
#endif /* ISEFactoryTemplate_HXX */
