/*
  Copyright (C) 2005-2011 Remi vieux, Jean-Philippe Domenger, Achille Braquelaire, Jenny Benois-Pineau, Julian Roigt  remi.vieux@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*
  This work was (partly) funded by the X-Media project, sponsored by
  the European Commission as part of the Information Society Technologies
  (IST) program under EC grant number IST-FP6-026978.
*/
#ifndef ISEDISTANCES_HPP_
#define ISEDISTANCES_HPP_

#include <algorithm>
#include <cassert>
#include <cmath>
#include <limits>
#include <map>
#include <vector>


namespace ise {

  struct GIRL_API Distances {


    /**
     * return manhattan distance between @a a and @a b.
     */

    template<typename T>
    static double manhattanDist(const std::vector<T>& a, const std::vector<T> &b);

    template<typename InputIterator>
    static double manhattanDist(InputIterator itA, InputIterator itAEnd, InputIterator itB);


    template<typename T>
    static double manhattanDistCutOff(const std::vector<T>& a, const std::vector<T> &b, double cutOff);

    /**
     * returns Euclidean distance (squared) between @a a and @a b.
     */
    template<typename T>
    static double distSq(const std::vector<T>& a, const std::vector<T>& b);

    template<typename InputIterator>
    static double distSq(InputIterator itA, InputIterator itAEnd, InputIterator itB);

    /**
     * return Euclidean distance (squared) between @a a and @a b.
     * Stop computation if distance > cutOff (hence do not compute full distance)
     */
    template<typename T>
    static double distSqCutOff(const std::vector<T>& a, const std::vector<T>& b, double cutOff);

    /**
     * Minkowski distance
     */
    template<typename T>
    static double minkowskiDist(const std::vector<T>& a, const std::vector<T>& b, int p);

    template<typename InputIterator>
      static double minkowskiDist(InputIterator itA, InputIterator itAEnd, InputIterator itB, int p);



    /*
     * Chebyshev distance (aka L^inf distance)
     *
     */
    template<typename T>
    static double chebyshevDist(const std::vector<T>& a, const std::vector<T>& b);

    template<typename InputIterator>
    static double chebyshevDist(InputIterator itA, InputIterator itAEnd, InputIterator itB);


    /**
     * return Chi^2 distance between @a a and @a b.
     * d = sum_{i} ((a_{i}-b_{i})^2)/(a_{i}+b{i})
     */
    template<typename T>
    static double chiSquareDist(const std::vector<T>& a, const std::vector<T>& b);

    template<typename InputIterator>
    static double chiSquareDist(InputIterator itA, InputIterator itAEnd, InputIterator itB);


    template<typename T>
    static double chiSquareDistCutOff(const std::vector<T>& a, const std::vector<T>& b, double cutOff);

    /**
     * return Cosine distance between @a a and @a b.
     * d = a.b/(||a||*||b||), a.b = dot product, ||a|| L2 norm of vector a
     * note: no cutOff available for this distance function
     */
    template<typename T>
    static double cosineDist(const std::vector<T>& a, const std::vector<T>& b);

    template<typename InputIterator>
    static double cosineDist(InputIterator itA, InputIterator itAEnd, InputIterator itB);


    /**
     * Non-Intersection:
     * D(A, B) = N*(1-S(A, B))
     * S(A, B) = 1/N sum_{i=0}^{N}min(A[i], B[i])
     * see: On measuring the distance between histograms
     * Char and Srihari, Pattern Recognition 35(2002) 1355-1370 for all the following distance measures
     */

    template<typename T>
    static double intersectionDist(const std::vector<T>& a, const std::vector<T>& b);

    template<typename InputIterator>
    static double intersectionDist(InputIterator itA, InputIterator itAEnd, InputIterator itB);


    /**
     * Kullbach-Lieber Distance (not a true metric)
     */

    template<typename T>
      static double KLDist(const std::vector<T>& a, const std::vector<T>& b);

    template<typename InputIterator>
      static double KLDist(InputIterator itA, InputIterator itAEnd, InputIterator itB);

    /*
     * Bhattacharyya distance
     */

    template<typename T>
    static double BhattaDist(const std::vector<T>& a, const std::vector<T>& b);

    template<typename InputIterator>
    static double BhattaDist(InputIterator itA, InputIterator itAEnd, InputIterator itB);


    /*
     * Matusita distance
     */
    template<typename T>
    static double MatusitaDist(const std::vector<T>& a, const std::vector<T>& b);

    template<typename InputIterator>
    static double MatusitaDist(InputIterator itA, InputIterator itAEnd, InputIterator itB);


    /**
     * Return sum_{i=0}^{n} 1+k*alpha(a[i]-b[i])^2, k= i/offset (partie entiere)
     * Modified version of euclidean distance
     */

    template<typename T>
      static double penalizedDistSq(const std::vector<T>& a, const std::vector<T>& b, int offset = 100, double alpha = 0.2 );

    template<typename InputIterator>
      static double penalizedDistSq(InputIterator itA, InputIterator itAEnd, InputIterator itB, int offset = 100, double alpha = 0.2);


    static double coeffVal(int val, int index, double alpha)
    {
      int a = (index/val);
      if (a)
        return 1 + a*alpha;
      else
        return 1 - alpha;
    }


  };

  template<typename T>
  double Distances::manhattanDist(const std::vector<T> &a, const std::vector<T> &b)
  {
    assert(a.size() == b.size());
    return manhattanDist(a.begin(), a.end(), b.begin());
  }


  template<typename InputIterator>
  double Distances::manhattanDist(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double sum = 0.0;
    for ( ; itA != itAEnd; ++itA, ++itB) {
      const double d = (*itA - *itB);
      sum += fabs(d);
    }
    return sum;
  }


  template<typename T>
  double Distances::manhattanDistCutOff(const std::vector<T> &a, const std::vector<T> &b, double cutOff)
   {
     assert(a.size() == b.size());
     double sum = 0.0;
     typename std::vector<T>::const_iterator itA = a.begin();
     typename std::vector<T>::const_iterator itB = b.begin();
     const typename std::vector<T>::const_iterator itAEnd = a.end();
     for ( ; itA != itAEnd; ++itA, ++itB) {
       const double d = (*itA - *itB);
       sum += fabs(d);
       if (sum > cutOff)
         break;
     }
     return sum;
   }

  template<typename T>
  double Distances::distSq(const std::vector<T> &a, const std::vector<T> &b)
  {
    assert(a.size() == b.size());
    return distSq(a.begin(), a.end(), b.begin());
  }

  template<typename InputIterator>
  double Distances::distSq(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double sum = 0.0;
    for ( ; itA != itAEnd; ++itA, ++itB) {
      const double d = (*itA - *itB);
      sum += d*d;
    }
    return sum;    
  }

  template<typename T>
  double Distances::distSqCutOff(const std::vector<T>& a, const std::vector<T>& b,
      double cutOff) {
        assert(a.size() == b.size());
        double sum = 0.0;
        typename std::vector<T>::const_iterator itA = a.begin();
        typename std::vector<T>::const_iterator itB = b.begin();
        const typename std::vector<T>::const_iterator itAEnd = a.end();
        const double cmp = static_cast<double>(cutOff);

        for ( ; itA != itAEnd; ++itA, ++itB) {

          const double d = (*itA - *itB);
          sum += d*d;
          if (sum > cmp) {

            break;
          }
        }
        return sum;
  }

  template<typename T>
  double Distances::minkowskiDist(const std::vector<T> &a, const std::vector<T> &b, int p)
  {
    assert(a.size() == b.size());
    return minkowskiDist(a.begin(), a.end(), b.begin(), p);
  }

  template<typename InputIterator>
  double Distances::minkowskiDist(InputIterator itA, InputIterator itAEnd, InputIterator itB, int p)
  {
    double sum = 0.0;
    for ( ; itA != itAEnd; ++itA, ++itB) {
      const double d = (*itA - *itB);
      sum += pow(d, p);
    }
    return pow(sum, -p);    
  }


  template<typename T>
  double Distances::chebyshevDist(const std::vector<T>& a, const std::vector<T>& b)
  {
    assert(a.size() == b.size());
    return chebyshevDist(a.begin(), a.end(), b.begin());
  }

  template<typename InputIterator>
  double Distances::chebyshevDist(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double max = -std::numeric_limits<double>::max();
    for ( ; itA != itAEnd; ++itA, ++itB) {
      const double d = fabs(*itA - *itB);
      if (d > max)
        max = d;
    }
    return max;
  }



  template<typename T>
  double Distances::chiSquareDist(const std::vector<T>& a, const std::vector<T>& b)
  {
    assert (a.size() == b.size());
    return chiSquareDist(a.begin(), a.end(), b.begin());
  }

  template<typename InputIterator>
  double Distances::chiSquareDist(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double sum = 0.;
    for ( ; itA != itAEnd; ++itA, ++itB) {
      if (*itA > std::numeric_limits<double>::epsilon()
          || *itB > std::numeric_limits<double>::epsilon()) { // avoid division by 0
            const double square = *itA - *itB;
            sum += (square*square)/(*itA + *itB);
      }
    }
    return sum;
  }
  

  template<typename T>
  double Distances::chiSquareDistCutOff(const std::vector<T>& a, const std::vector<T>& b, double cutOff) {
    assert(a.size() == b.size());

    typename std::vector<T>::const_iterator itA = a.begin();
    typename std::vector<T>::const_iterator itB = b.begin();
    const typename std::vector<T>::const_iterator itAEnd = a.end();

    const double cmp = static_cast<double>(cutOff);
    double sum = 0.;
    for ( ; itA != itAEnd; ++itA, ++itB) {
      if (*itA > std::numeric_limits<double>::epsilon()
          || *itB > std::numeric_limits<double>::epsilon()) { // avoid division by 0
            const double square = *itA - *itB;
            sum += (square*square)/(*itA + *itB);
            if (sum > cmp)
              break;
      }
    }
    return sum;
  }

  template<typename T>
  double Distances::cosineDist(const std::vector<T>& a, const std::vector<T>& b)
  {
    assert(a.size() == b.size());
    return cosineDist(a.begin(), a.end(), b.begin());
  }

  template<typename InputIterator>
  double Distances::cosineDist(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double anorm = 0.0f;
    double bnorm = 0.0f;
    double dotProd = 0.0f;

    for ( ; itA != itAEnd; ++itA, ++itB) {
      anorm += ((*itA)*(*itA));
      bnorm += ((*itB)*(*itB));
      dotProd += ((*itA)*(*itB));
    }
    anorm = sqrt(anorm);
    bnorm = sqrt(bnorm);
    double cosinus = dotProd/(anorm*bnorm);
    return acos(cosinus);
  }


  template<typename T>
  double Distances::intersectionDist(const std::vector<T>& a, const std::vector<T>& b)
  {
    assert(a.size() == b.size());
    assert(a.size() > 0);
    return intersectionDist(a.begin(), a.end(), b.begin());
  }

  template<typename InputIterator>
  double Distances::intersectionDist(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double sum = 0.0;
    double s = (itAEnd - itA);
    for ( ; itA != itAEnd; ++itA, ++itB)
      sum += std::min(*itA, *itB);
    return s - sum;
  }



  template<typename T>
  double Distances::KLDist(const std::vector<T>& a, const std::vector<T>& b)
  {
    assert(a.size() == b.size());
    return KLDist(a.begin(), a.end(), b.begin());
  }

  template<typename InputIterator>
  double Distances::KLDist(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double sum = 0.0;
    for ( ; itA != itAEnd; ++itA, ++itB)
      if (*itA != 0 && *itB != 0)
	sum += (*itB)*log((*itB)/(*itA));
    return sum;
  }


  template<typename T>
  double Distances::BhattaDist(const std::vector<T>& a, const std::vector<T>& b)
  {
    assert(a.size() == b.size());
    //assert(a.size() > 0);
    return BhattaDist(a.begin(), a.end(), b.begin());
  }

  template<typename InputIterator>
  double Distances::BhattaDist(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double sum = 0.0;
    for ( ; itA != itAEnd; ++itA, ++itB)
      sum += sqrt((*itA)*(*itB));

    return -log(sum);
  }


  template<typename T>
  double Distances::MatusitaDist(const std::vector<T>& a, const std::vector<T>& b)
  {
    assert(a.size() == b.size());
    //assert(a.size() > 0);
    return MatusitaDist(a.begin(), a.end(), b.begin());
  }

  template<typename InputIterator>
  double Distances::MatusitaDist(InputIterator itA, InputIterator itAEnd, InputIterator itB)
  {
    double sum = 0.0;
    for ( ; itA != itAEnd; ++itA, ++itB) {
      double d = sqrt(*itA) - sqrt(*itB);
      sum += d*d;
    }
    return sqrt(sum);    
  }


  template<typename T>
  double Distances::penalizedDistSq(const std::vector<T> &a, const std::vector<T> &b, int offset, double alpha)
  {
    assert(a.size() == b.size());
    return penalizedDistSq(a.begin(), a.end(), b.begin(), offset, alpha);
  }

  template<typename InputIterator>
  double Distances::penalizedDistSq(InputIterator itA, InputIterator itAEnd, InputIterator itB, int offset, double alpha)
  {
    double sum = 0.0;
    int i = 0;
    for ( ; itA != itAEnd; ++itA, ++itB) {
      const double d = (*itA - *itB);
      const double coeff = coeffVal(offset, i++, alpha);
      sum += d*d*coeff;
    }
    return sum;    
  }


}
#endif /* ISEDISTANCES_HPP_ */
