/*
  Copyright (C) 2005-2009 Remi vieux, Jean-Philippe Domenger, Achille Braquelaire, Jenny Benois-Pineau, Julian Roigt  remi.vieux@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*
  This work was (partly) funded by the X-Media project, sponsored by
  the European Commission as part of the Information Society Technologies
  (IST) program under EC grant number IST-FP6-026978.
*/

#ifndef ISEDISTANCECOMPUTER_HPP_
#define ISEDISTANCECOMPUTER_HPP_

#include <vector>
#include <girl/girlCommon.hpp>
#include "iseDistances.hpp"

namespace ise
{

  typedef enum {
    SQUARED_EUCLIDEAN = 1,
    CHI_SQUARE = 2,
    COSINE = 3,
    EMD = 4,
    EUCLIDEAN = 5,
    MANHATTAN = 6,
    INTERSECTION = 7,
    KL = 8,
    BHATTA = 9,
    MATSUSITA = 10,
    LP = 11
  } DistanceType;

  class GIRL_API DistanceComputer
  {
  public:
      /**
       * Pass any additional requirements for distance computation to userData
       */
      DistanceComputer(void *userData = NULL, DistanceType d_type = ise::EUCLIDEAN);

      virtual ~DistanceComputer();

      template<typename T> double getDistance(const std::vector<T> & a, const std::vector<T> & b) const;

      template<typename InputIterator>
	double getDistance(InputIterator itA, InputIterator itAEnd, InputIterator itB) const;


      template<typename T> double getDistanceCutOff(const std::vector<T> & a, const std::vector<T> & b, double cutOff) const;
      void setDistanceType(DistanceType type);

      inline void *getUserData() const
      {
          return m_userData;
      }

      inline void setUserData(void *userData)
      {
          m_userData = userData;
      }

      DistanceType getDistanceType() { return m_distanceType; }

  private:

    DistanceType m_distanceType;

    void *m_userData;

  };

  template<typename T>
  double DistanceComputer::getDistance(const std::vector<T>& ma, const std::vector<T>& mb) const
  {
    return getDistance(ma.begin(), ma.end(), mb.begin());

  }

  template<typename InputIterator>
  double DistanceComputer::getDistance(InputIterator itA, InputIterator itAEnd, InputIterator itB) const 
  {
    double dist = 0.0;
    // brackets are necessary when you declare variables within a case label
    switch(m_distanceType) {
    case LP: {
      assert(m_userData);
      int *minkowskiFactor = static_cast<int*>(m_userData);
      dist = ise::Distances::minkowskiDist(itA, itAEnd, itB, *minkowskiFactor);
      break;
    }
    case MATSUSITA: {
      dist = ise::Distances::MatusitaDist(itA, itAEnd, itB);
      break;
    }
    case BHATTA: {
      dist = ise::Distances::BhattaDist(itA, itAEnd, itB);
      break;
    }
    case KL: {
      dist = ise::Distances::KLDist(itA, itAEnd, itB);
      break;
    }
    case INTERSECTION: {
      dist = ise::Distances::intersectionDist(itA, itAEnd, itB);
      break;
    }
    case MANHATTAN: {
      dist = ise::Distances::manhattanDist(itA, itAEnd, itB);
      break;
    }
    case EUCLIDEAN: {
      dist = sqrt(ise::Distances::distSq(itA, itAEnd, itB));
      break;
    }
    case EMD : {
      assert(m_userData);
      //ise::matrix<double> *gtDist = static_cast<ise::matrix<double>* >(m_userData);
      //dist = ise::Distances::emdDist(itA, itAEnd, itB, *gtDist);
      
      break;
    }
    case COSINE: {
      dist = ise::Distances::cosineDist(itA, itAEnd, itB);
      break;
    }
    case CHI_SQUARE: {
      dist = ise::Distances::chiSquareDist(itA, itAEnd, itB);
      break;
    }
    case SQUARED_EUCLIDEAN: {
      dist = ise::Distances::distSq(itA, itAEnd, itB);
      break;
    }
    default: {
      dist = ise::Distances::distSq(itA, itAEnd, itB);
      break;
    }
    }
    return dist;
  }



  template<typename T>
  double DistanceComputer::getDistanceCutOff(const std::vector<T>& ma, const std::vector<T>& mb, double cutOff) const
  {
    double dist;
    // brackets are necessary when you declare variables within a case label
    switch(m_distanceType) {
    case LP: {
      //      int *minkowskiFactor = static_cast<int*>(data);
      //      dist = ise::Distances::minkowskiDist(ma, mb, *minkowskiFactor);
      //      break;
      assert(false);
      throw "LP cutoff distance not implemented";
    }
    case MATSUSITA: {
      //      dist = ise::Distances::MatusitaDist(ma, mb);
      //      break;
      assert(false);
      throw "MATSUSITA cutoff distance not implemented";
    }
    case BHATTA: {
      //      dist = ise::Distances::BhattaDist(ma, mb);
      //      break;
      assert(false);
      throw "BHATTACHARY cutoff dist not implemented";
    }
    case KL: {
      //      dist = ise::Distances::KLDist(ma, mb);
      //      break;
      assert(false);
      throw "KL cutoff dist not implemented";
    }
    case INTERSECTION: {
      //      dist = ise::Distances::intersectionDist(ma, mb);
      //      break;
      assert(false);
      throw "INTERSECTION cutoff dist not implemented";
    }
    case MANHATTAN: {
      dist = ise::Distances::manhattanDistCutOff(ma, mb, cutOff);
      break;
      //      throw "Manhattant cutoff dist not implemented";
    }
    case EUCLIDEAN: {
      dist = sqrt(ise::Distances::distSqCutOff(ma, mb, cutOff));
      break;
    }
    case EMD : {
      //      ise::matrix<double> *gtDist = static_cast<ise::matrix<double>* >(data);
      //      dist = ise::Distances::emdDist(ma, mb, *gtDist);
      //      break;
      throw "EMD cutoff dist not implemented";
    }
    case COSINE: {
      // No cutoff available for cosine, return normal distance
      dist = ise::Distances::cosineDist(ma, mb);
      break;
    }
    case CHI_SQUARE: {
      dist = ise::Distances::chiSquareDistCutOff(ma, mb, cutOff);
      break;
    }
    case SQUARED_EUCLIDEAN: {
      dist = ise::Distances::distSqCutOff(ma, mb, cutOff);
      break;
    }
    default: {
      dist = ise::Distances::distSqCutOff(ma, mb, cutOff);
      break;
    }
    }
    return dist;
  }
}

#endif /* ISEDISTANCECOMPUTER_HPP_ */
