/*
  Color spaces conversions functions.
  Copyright (C) 1996, 2004-2009 Luc Brun, Achille Braquelaire, Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <cmath>
#include <cassert>
#include <cstdio>
#include <iostream>

#include "iseConversion.hpp"


/****************************************************************************/
/* fichier de iseConversion (R, G, B)--> Espace C1C2C3 */
/* Pour rajouter un espace Il faut : */
/* 1) Rajouter 3 fonctions de type PFiseConversion RGB_to_C1, RGB_to_C2, RGB_to_C3 */
/* 2) Rajouter 3 fonctions de type PFiseConversion C1C2C3_to_R, C1C2C3_to_G et C1C2C3_to_B */
/* 3) Toute fonction suplementaire est a ecrire en fin de fichier */
/* 4) Rajouter ces fonctions dans changeSpace et restoreSpace*/
/* 5) Ne pas oublier de declarer l'espace dans le header */
/****************************************************************************/
 

#define MAX( a, b ) ((a) > (b) ? (a) : (b))
#define MIN( a, b ) ((a) < (b) ? (a) : (b))
#define FLOAT_TO_INT(a) static_cast<int>(a<0?a-0.5f:a+0.5f)

/*
static
inline
unsigned char
CLAMP_TO_UCHAR(int a)
{
  return (a<0?0:(a>255?255:a));
}
*/

static
inline
unsigned char
CLAMP_TO_UCHAR(float a)
{
  return (a<0.f ? 0 : (a>255.f ? 255 : static_cast<int>(a+0.5f)));
  //B: no need to do a FLOAT_TO_INT !
}



#define ONE_THIRD_F 1.0f / 3.0f
#define TWO_THIRD_F 2.0f / 3.0f


#ifdef _MSC_VER
//visual C++ is not C99 conformant, 
//cbrt is not available

//Other implementation available here :
// http://metamerist.com/cbrt/cbrt.htm

#define ONE_THIRD 1.0 / 3.0

static
double
cbrt(double x)
{
  return pow(x, ONE_THIRD);
}
#endif



/*
//B

[1] 
"Image numérique couleur: de l'acquisition au traitement"
Alain Trémeau, Christine Fernandez-Maloigne, Pierre Bonton
Ed: Dunod

[2] 
"RGB coordinates of the Macbeth ColorChecker" Danny Pascale (2006)


( Xr Xg Xb )   ( R )   ( X )
( Yr Yg Yb ) * ( G ) = ( Y )
( Zr Zg Zb )   ( B )   ( Z )

( Rx Ry Rz ) * ( X )   ( R ) 
( Gx Gy Gz ) * ( Y ) = ( G ) 
( Bx By Bz ) * ( Z )   ( B ) 


Xw, Yw, Zw, Uw, Vw used in XYZ<->Luv/ab
correspond to: Xr, Yr, Zr, u'r, v'r in formulae at 
http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Lab.html
http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Luv.html
correspoding to : Xn, Yn, Zn in [2]
 Xn, Yn and Zn are the XYZ values of the reference white"


According to http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Luv.html
We have :
Uw = 4*Xw / (Xw + 15*Yw + 3*Zw)
Vw = 9*Yw / (Xw + 15*Yw + 3*Zw)

REM: it is not the same formulae in [1] p 51 : -3*Z instaed of +3*Z in denominator !! 
It is +3*Z averywhere else I have looked, for exemple here  http://en.wikipedia.org/wiki/CIELUV also list 
Error in [1] ???

*/

#define GAMMA



//#define ILLUMINANT_C
#define ILLUMINANT_D65 //B: change default ilumminant to D65 

#ifdef ILLUMINANT_C
/*--------------------------------*/
/* Illuminant C */
/*--------------------------------*/
#define Xr 0.166f
#define Yr 0.06f
#define Zr 0.0f

#define Xg 0.174f
#define Yg 0.587f
#define Zg 0.066f

#define Xb 0.093f
#define Yb 0.005f
#define Zb 0.46f

#define Xw 98.04f
#define Yw 100.0f
#define Zw 118.10f
#define Uw 0.2009f
#define Vw 0.4609f

#define Rx 6.686f
#define Ry -1.832f
#define Rz -1.332f

#define Gx -0.684f
#define Gy 1.893f
#define Gz 0.118f

#define Bx 0.098f
#define By -0.272f
#define Bz 2.157f
#endif //ILLUMINANT_C

#ifdef ILLUMINANT_E
/*--------------------------------*/
/* Illuminant E                   */
/*--------------------------------*/
#define Xr 2.769f
#define Yr 1.0f
#define Zr 0.0f

#define Xg 1.7518f
#define Yg 4.5907f
#define Zg 0.0565f

#define Xb 1.133f
#define Yb 0.0601f
#define Zb 5.5943f

#define Xw 100.0f
#define Yw 100.0f
#define Zw 100.0f
#define Uw 0.2105f
#define Vw 0.4737f
#endif //ILLUMINANT_E

#ifdef ILLUMINANT_D65
/*--------------------------------*/
/* Illuminant D65                 */
/*--------------------------------*/
//Where these values come from ?
/*
#define Xr 0.478f
#define Yr 0.263f
#define Zr 0.02f

#define Xg 0.299f 
#define Yg 0.655f
#define Zg 0.16f 

#define Xb 0.175f
#define Yb 0.081f
#define Zb 0.908f 

#define Xw 95.02f
#define Yw 100.0f
#define Zw 108.81f
#define Uw 0.1978f
#define Vw 0.4683f
*/
//B: the values given in OpenCV documentation
// http://docs.opencv.org/modules/imgproc/doc/miscellaneous_transformations.html#cvtcolor
// "CIE XYZ.Rec 709 with D65 white point" are more precise than in [2] ?
// In [2], we have the same values than OpenCV but rounded to 4 digits after comma in [2]
// p9, Table 6 "RGB (R709) to XYZ" cell
// The values at http://www.brucelindbloom.com/index.html?Eqn_RGB_XYZ_Matrix.html
// for sRGB D65 are slightly different
// We keep values of [2] (because we may also want to use the Bradford matrix from D50->D65 in [2])
//
// For the reference white, the values in OpenCV source 
// OpenCV-2.4.3/modules/imgproc/src/color.cpp in D65 array (times 100) are :
// 95.0456, 100.0, 108.8754
// The values in [2], p10, Table 7, are slightly different :
// 95.047, 100.0, 108.883
// http://www.brucelindbloom.com/index.html?Eqn_ChromAdapt.html gives the same values than [2]
// saying "come from ASTM E308-01"

//Uw & Vw are computed from Xw, Yw, Zw with previously given formulae.

#define Xr 0.4125f
#define Yr 0.2127f
#define Zr 0.0193f

#define Xg 0.3576f 
#define Yg 0.7152f
#define Zg 0.1192f 

#define Xb 0.1804f
#define Yb 0.0722f
#define Zb 0.9503f 


#define Rx 3.2405
#define Gx -0.9693
#define Bx 0.0556

#define Ry -1.5371
#define Gy 1.8760
#define By -0.2040

#define Rz -0.4985
#define Gz 0.0416
#define Bz 1.0572


#define Xw 95.047f
#define Yw 100.0f
#define Zw 108.883f

#define Uw 0.19784f
#define Vw 0.468336f


#endif //ILLUMINANT_D65

#ifdef ILLUMINANT_A
/*--------------------------------*/
/* Illuminant A                  */
/*--------------------------------*/
#define Xr 0.892f
#define Yr 0.322f
#define Zr 0.00f

#define Xg 0.33f  
#define Yg 0.863f 
#define Zg 0.011f

#define Xb 0.083f
#define Yb 0.004f  
#define Zb 0.409f

#define Xw 109.83f
#define Yw 100.0f
#define Zw 35.55f 
#define Uw 0.2560f
#define Vw 0.5243f
#endif //ILLUMINANT_A




/* Some auxiliary functions */
static char sortRGB(int R, int G, int B, int  *minrgb, int  *maxrgb);
static int P_HMinMax_to_R(int H, long min, long max);
static int P_HMinMax_to_G(int H, long min, long max);
static int P_HMinMax_to_B(int H, long min, long max);



/**************************************************************************/
/********************************** RGB ***********************************/
/**************************************************************************/

static int RGB_min[] = {0, 0, 0};
static int RGB_max[] = {255, 255, 255};

/* RGB to RGB */
static int P_RGB_to_R(int r, int /* g */, int /* b */) { return r; }
static int P_RGB_to_G(int /* r */, int g, int /* b */) { return g; }
static int P_RGB_to_B(int /* r */, int /* g */, int b) { return b; }

static
void
P_RGB_to_RGB(int r, int g, int b, int &x, int &y, int &z)
{
  x = r;
  y = g;
  z = b;
}




/**************************************************************************/
/********************************* XYZ *********************************/
/**************************************************************************/
//private auxiliary functions

#ifdef GAMMA
//B:
//gamma corrections equations 
//found here : http://www.easyrgb.com/index.php?X=MATH&H=02#text2   in RGB->XYZ & XYZ->RGB
//They are also present in [2].
//In [2] it says for sRGB :
// offset=0.055, gamma=0.42, transition=0.003, slope=12.92
//
// ??? Is it the correct values ??? 1/0.42=2.380952 != 2.4   !!!???
// It is also 2.4 in OpenCV color.cpp

static inline
float
gammaCorrected_to_linear(int R)
{
  float Rf = R / 255.f;
  if (Rf > 0.04045f)
    Rf = pow((( Rf + 0.055 ) / 1.055), 2.4);
  else
    Rf = Rf / 12.92;
  return Rf;
}

static
void
RGBgammaCorrected_to_linear(int R, int G, int B, float &Rf, float &Gf, float &Bf)
{
  //B:REM:Impl: using pow() & double is faster than using powf() & float on X86_64... SSE ?

  Rf = gammaCorrected_to_linear(R);
  Gf = gammaCorrected_to_linear(G);
  Bf = gammaCorrected_to_linear(B);
}

static inline
int
linear_to_gammaCorrected(float Rf)
{
  if (Rf > 0.0031308)
    Rf = 1.055 * pow(Rf, 1/2.4) - 0.055;
  else 
    Rf *= 12.92;
  
  Rf *= 255.f;
    
  return CLAMP_TO_UCHAR(Rf);
}

static
void
RGBlinear_to_gammaCorrected(float Rf, float Gf, float Bf, 
			    int &R, int &G, int &B)
{
  R = linear_to_gammaCorrected(Rf);
  G = linear_to_gammaCorrected(Gf);
  B = linear_to_gammaCorrected(Bf);
}

#endif //GAMMA

static
void
P_RGB_to_XYZ(int R, int G, int B, float *X, float *Y, float *Z)
{
#ifdef GAMMA

  float Rf, Gf, Bf;
  RGBgammaCorrected_to_linear(R, G, B, Rf, Gf, Bf);

  Rf *= 100.f;
  Gf *= 100.f;
  Bf *= 100.f;
  
  *X = Xr*Rf + Xg*Gf + Xb*Bf;
  *Y = Yr*Rf + Yg*Gf + Yb*Bf;
  *Z = Zr*Rf + Zg*Gf + Zb*Bf;

#else

  *X = Xr*R + Xg*G + Xb*B;
  *Y = Yr*R + Yg*G + Yb*B;
  *Z = Zr*R + Zg*G + Zb*B;

#endif
}

static
float
P_RGB_to_XYZ_Y(int R, int G, int B)
{
#ifdef GAMMA

  float Rf, Gf, Bf;
  RGBgammaCorrected_to_linear(R, G, B, Rf, Gf, Bf);

  Rf *= 100.f;
  Gf *= 100.f;
  Bf *= 100.f;
  
  return Yr*Rf + Yg*Gf + Yb*Bf;

#else
  return Yr*R + Yg*G + Yb*B;
#endif

}


void
P_XYZ_to_RGB(float X, float Y, float Z, int &R, int &G, int &B)
{
#ifdef GAMMA

  X *= 0.01f;
  Y *= 0.01f;
  Z *= 0.01f;

  float Rf = Rx*X + Ry*Y + Rz*Z;
  float Gf = Gx*X + Gy*Y + Gz*Z;
  float Bf = Bx*X + By*Y + Bz*Z;
  
  RGBlinear_to_gammaCorrected(Rf, Gf, Bf, R, G, B);

#else
  const float Rf = Rx*X + Ry*Y + Rz*Z + 0.5f;
  const float Gf = Gx*X + Gy*Y + Gz*Z + 0.5f;
  const float Bf = Bx*X + By*Y + Bz*Z + 0.5f;
  R = CLAMP_TO_UCHAR(Rf);
  G = CLAMP_TO_UCHAR(Gf);
  B = CLAMP_TO_UCHAR(Bf);
#endif
}


static
int
P_XYZ_to_R(float X, float Y, float Z)
{
#ifdef GAMMA
  X *= 0.01f;
  Y *= 0.01f;
  Z *= 0.01f;

  const float Rf = Rx*X + Ry*Y + Rz*Z;
  return linear_to_gammaCorrected(Rf);
#else
  const float R = Rx*X + Ry*Y + Rz*Z;
  return CLAMP_TO_UCHAR(R);
#endif
}

static
int
P_XYZ_to_G(float X, float Y, float Z)
{
#ifdef GAMMA
  X *= 0.01f;
  Y *= 0.01f;
  Z *= 0.01f;

  const float Gf = Gx*X + Gy*Y + Gz*Z;
  return linear_to_gammaCorrected(Gf);
#else
  const float G = Gx*X + Gy*Y + Gz*Z;
  return CLAMP_TO_UCHAR(G);
#endif
}

static
int
P_XYZ_to_B(float X, float Y, float Z)
{
#ifdef GAMMA
  X *= 0.01f;
  Y *= 0.01f;
  Z *= 0.01f;

  const float Bf = Bx*X + By*Y + Bz*Z;
  return linear_to_gammaCorrected(Bf);
#else
  const float B = Bx*X + By*Y + Bz*Z;
  return CLAMP_TO_UCHAR(B);
#endif
}




/**************************************************************************/
/********************************* Luv *********************************/
/**************************************************************************/

//static int Luv_min[] = {0, -190, -150};
//static int Luv_max[] = {121, 253, 175};
//B
#ifdef GAMMA
static int Luv_min[] = {0, -83, -134};
static int Luv_max[] = {100, 175, 107};
#else
static int Luv_min[] = {0, -119, -204};
static int Luv_max[] = {142, 258, 154};
#endif

static
void
P_XYZ_to_Luv(float X, float Y, float Z, float &L, float &u, float &v)
{
  //http://www.easyrgb.com/index.php?X=MATH&H=16#text16
  //http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Luv.html & [1] p50/51

  if ((X==0.0f)&&(Y==0.0f)&&(Z==0.0f)) {
    L = u = v = 0.0;
    return;
  }
  
  float Lp = 0.0;
  
  const float YparYw = Y/Yw;
  if (YparYw > 0.008856f)
    Lp = 116.0f*cbrtf(YparYw) -16.0f; 
  else
    Lp = 903.3f*YparYw;
  
  L = Lp;
  
  const float up = 4.f*X/(X + 15.f*Y + 3.f*Z);
  u = 13.f*Lp*(up-Uw);
  
  const float vp = 9.f*Y/(X + 15.f*Y + 3.f*Z);
  v = 13.f*Lp*(vp-Vw);
}

static
float
P_XYZ_Y_to_Luv_L(float Y)
{
  const float YparYw = Y/Yw;
  if (YparYw > 0.008856f)
    return 116.0f*cbrtf(YparYw) -16.0f; 
  else
    return 903.3f*YparYw;
}

static
float
P_RGB_to_Luv_L(int R, int G, int B)
{
  const float Y = P_RGB_to_XYZ_Y(R, G, B);

  return P_XYZ_Y_to_Luv_L(Y);
}


static
void
P_RGB_to_Luv(int R, int G, int B,
	     int &L, int &u, int &v)
{
  float X, Y, Z;
  P_RGB_to_XYZ(R, G, B, &X, &Y, &Z);
  
  float Lf, uf, vf;
  P_XYZ_to_Luv(X, Y, Z, Lf, uf, vf);

  L = FLOAT_TO_INT(Lf);
  u = FLOAT_TO_INT(uf);
  v = FLOAT_TO_INT(vf);
}


static
void
P_Luv_to_XYZ(int L, int u, int v, float &X, float &Y, float &Z)
{
  //http://www.brucelindbloom.com/index.html?Eqn_Luv_to_XYZ.html & p50/51

  // not same formula here :   //http://www.easyrgb.com/index.php?X=MATH&H=17#text17


  //B: wrong si (L==0 && u==0) || (L==0 && v==0) !?!


  const float KEpsilon = 903.3f * 0.008856f; //static ?
  if (L > KEpsilon) {
    const float Lp = (L+16.f)/116.0f; //TODO:OPTIM: use inv_116 ?
    Y = Yw * Lp*Lp*Lp;
  }
  else {
    //it was :
    //static const float inv_K = 1.0/903.327;
    //changed according to http://www.brucelindbloom.com/index.html?Eqn_Luv_to_XYZ.html and value used in P_RGB_to_LUV() !
    static const float inv_K = 1.0f/903.3f; //static ?
    
    Y = Yw*L*inv_K;
  }

  const float one_third = (1.f/3.f); //static?

  const float a = one_third*(52*L/(u+13.f*L*Uw) -1.f);
  
  const float b = -5*Y;

  const float c = -one_third;

  const float d = Y*(39*L/(v + 13*L*Vw) - 5.f);

  X = (d-b)/(a-c);

  Z = X*a + b;

  /*

  const float Q = u/(13.0*L)+Uw;
  const float R = v/(13.0*L)+Vw;

  //B: not same formula than at www.brucelindbloom.com
  //TODO: check equivalence !!
  
  const float A = 3.0*(*Y)*(5.0*R-3.0);
  *Z = ( (Q-4.0)*A - 15.0*Q*R*(*Y) )/(12.0*R);
  
  *X = -( A/R + 3.0*(*Z) );

  */
}

static
int
P_RGB_to_L_of_Luv(int R, int G, int B)
{
  return FLOAT_TO_INT(P_RGB_to_Luv_L(R, G, B));
}

static
int
P_RGB_to_u(int R, int G, int B)
{
  float X, Y, Z;
  P_RGB_to_XYZ(R, G, B, &X, &Y, &Z);

  if ((X==0.0) && (Y==0.0) && (Z==0.0))
    return 0;

  const float L = P_XYZ_Y_to_Luv_L(Y);
 
  const float up = 4*X/(X+15*Y+3*Z);
  const float u = 13*L*(up-Uw);
  return FLOAT_TO_INT(u);
}

static
int
P_RGB_to_v(int R, int G, int B)
{
  float X, Y, Z;
  P_RGB_to_XYZ(R, G, B, &X, &Y, &Z);

  if ((X==0.0) && (Y==0.0) && (Z==0.0))
    return 0;

  const float L = P_XYZ_Y_to_Luv_L(Y);

  const float vp = 9*Y/(X+15*Y+3*Z);
  const float v = 13*L*(vp-Vw);
  return FLOAT_TO_INT(v);
}   



static
int
P_Luv_to_R(int L, int u, int v)
{
  float X, Y, Z;
  P_Luv_to_XYZ(L, u, v, X, Y, Z);
  return P_XYZ_to_R(X, Y, Z);
}

static
int
P_Luv_to_G(int L, int u, int v)
{
  float X, Y, Z;
  P_Luv_to_XYZ(L, u, v, X, Y, Z);
  return P_XYZ_to_G(X, Y, Z);
}

static
int
P_Luv_to_B(int L, int u, int v)
{
  float X, Y, Z;
  P_Luv_to_XYZ(L, u, v, X, Y, Z);
  return P_XYZ_to_B(X, Y, Z);
}

void
P_Luv_to_RGB(int L, int u, int v, int &r, int &g, int &b)
{
  float X, Y, Z;
  P_Luv_to_XYZ(L, u, v, X, Y, Z);

  P_XYZ_to_RGB(X, Y, Z, r, g, b);
}


/**************************************************************************/
/********************************* Lab *********************************/
/**************************************************************************/

#ifdef GAMMA
static int Lab_min[] = {0, -86, -108};
static int Lab_max[] = {100, 98, 94};
#else
static int Lab_min[] = {0, -118, -147};
static int Lab_max[] = {142, 134, 129};
#endif

static
void
P_XYZ_to_Lab(float X, float Y, float Z, float &L, float &a, float &b)
{  
  const float XparXw = X/Xw;
  const float YparYw = Y/Yw;
  const float ZparZw = Z/Zw;
  
  float fx = 0.0f, fy = 0.0f, fz = 0.0f;
  // according to http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Lab.html & [1] p52
  //B: the L of Lab & Luv are the same
  // However according to http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Lab.html formulae
  // we need 'fy' to compute a & b, thus L is not computed exactly the same way here than in P_XYZ_to_LUV
  // (TODO:check: does it produce differences ???)
  
  const float Epsilon = 0.008856f; //static
  const float K = 903.3f; //static
  const float inv_116 = 1.0f / 116.0f; //static

  //B:impl: cbrtf seems slightly faster than cbrt 
  if (XparXw > Epsilon)
    fx = cbrtf(XparXw); 
  else
    fx = (K * XparXw + 16.0f) * inv_116;

  if (YparYw > Epsilon)
    fy = cbrtf(YparYw);
  else
    fy = (K * YparYw + 16.0f) * inv_116;

  if (ZparZw > Epsilon)
    fz = cbrtf(ZparZw);
  else
    fz = (K * ZparZw + 16.0f) * inv_116;
      
  L = 116.0f * fy  - 16.f;
  a = 500.f * (fx - fy);
  b = 200.f * (fy - fz);
}



static
void
P_RGB_to_Lab(int R, int G, int B,
	     int &L, int &a, int &b)
{
  float X, Y, Z;
  P_RGB_to_XYZ(R, G, B, &X, &Y, &Z);
  
  float Lf, af, bf;
  P_XYZ_to_Lab(X, Y, Z, Lf, af, bf);

  L = FLOAT_TO_INT(Lf);
  a = FLOAT_TO_INT(af);
  b = FLOAT_TO_INT(bf);
}


static
void
P_Lab_to_XYZ(int L, int a, int b, float &X, float &Y, float &Z)
{
  //formula from http://www.brucelindbloom.com/index.html?Eqn_Lab_to_XYZ.html

  const float fy = (L + 16.0f) / 116.0f;
  const float fx = a / 500.0f + fy;
  const float fz = fy - b / 200.0f;
  //TODO:OPTIM: replace "divide by constant" by "multiply by constant'=inv(constant)" !?
  
  const float fx_3 = fx*fx*fx;
  const float K = 903.3f;//static ?
  const float Epsilon = 0.008856f;//static ?
  const float inv_K = 1.0f / K;//static ?
  if (fx_3 > Epsilon)
    X = Xw * fx_3;
  else
    X = Xw * (116.0f * fx - 16.0f)*inv_K;
  
  const float KEpsilon = K * Epsilon; //static ?
  if (L > KEpsilon) {
    const float Lp = (L+16)/116.0f; //TODO:OPTIM: use inv_116 ?
    Y = Yw * (Lp*Lp*Lp);
  }
  else {
    Y = Yw * L * inv_K;
  }
  
  const float fz_3 = fz*fz*fz;
  if (fz_3 > Epsilon)
    Z = Zw * fz_3;
  else
    Z = Zw * (116.0f * fz - 16.0f)*inv_K;
}

static
void
P_Lab_to_RGB(int L, int a, int b,
	     int &R, int &G, int &B)
{
  float X, Y, Z;
  P_Lab_to_XYZ(L, a, b, X, Y, Z);
  
  P_XYZ_to_RGB(X, Y, Z, R, G, B);
}


static
int
P_RGB_to_L_of_Lab(int R, int G, int B)
{
  return FLOAT_TO_INT(P_RGB_to_Luv_L(R, G, B)); //L of Lab & Luv is the same !
}

static
int
P_RGB_to_a(int R, int G, int B)
{
  float X, Y, Z;
  P_RGB_to_XYZ(R, G, B, &X, &Y, &Z);
  
  const float XparXw = X/Xw;
  const float YparYw = Y/Yw;
  
  float fx = 0.0f, fy = 0.0f;
  
  const float Epsilon = 0.008856f; //static
  const float K = 903.3f; //static
  const float inv_116 = 1.0f / 116.0f; //static

  //B:impl: cbrtf seems slightly faster than cbrt 
  if (XparXw > Epsilon)
    fx = cbrtf(XparXw); 
  else
    fx = (K * XparXw + 16.0f) * inv_116;

  if (YparYw > Epsilon)
    fy = cbrtf(YparYw);
  else
    fy = (K * YparYw + 16.0f) * inv_116;

      
  const float af = 500.f * (fx - fy);

  return FLOAT_TO_INT(af);
}

static
int
P_RGB_to_b(int R, int G, int B)
{
  float X, Y, Z;
  P_RGB_to_XYZ(R, G, B, &X, &Y, &Z);
  
  const float YparYw = Y/Yw;
  const float ZparZw = Z/Zw;
  
  float fy = 0.0f, fz = 0.0f;
  // according to http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Lab.html & [1] p52
  //B: the L of Lab & Luv are the same
  // However according to http://www.brucelindbloom.com/index.html?Eqn_XYZ_to_Lab.html formulae
  // we need 'fy' to compute a & b, thus L is not computed exactly the same way here than in P_XYZ_to_LUV
  // (TODO:check: does it produce differences ???)
  
  const float Epsilon = 0.008856f; //static
  const float K = 903.3f; //static
  const float inv_116 = 1.0f / 116.0f; //static

  //B:impl: cbrtf seems slightly faster than cbrt 
  if (YparYw > Epsilon)
    fy = cbrtf(YparYw);
  else
    fy = (K * YparYw + 16.0f) * inv_116;

  if (ZparZw > Epsilon)
    fz = cbrtf(ZparZw);
  else
    fz = (K * ZparZw + 16.0f) * inv_116;
      
  const float bf = 200.f * (fy - fz);

  return FLOAT_TO_INT(bf);
}   

static
int
P_Lab_to_R(int L, int u, int v)
{
  float X, Y, Z;
  P_Lab_to_XYZ(L, u, v, X, Y, Z);
  return P_XYZ_to_R(X, Y, Z);
}

static
int
P_Lab_to_G(int L, int u, int v)
{
  float X, Y, Z;
  P_Lab_to_XYZ(L, u, v, X, Y, Z);
  return P_XYZ_to_G(X, Y, Z);
}

static
int
P_Lab_to_B(int L, int u, int v)
{
  float X, Y, Z;
  P_Lab_to_XYZ(L, u, v, X, Y, Z);
  return P_XYZ_to_B(X, Y, Z);
}




/**************************************************************************/
/********************************** HSV ***********************************/
/**************************************************************************/

/* sa */


static int HSV_min[] = {0, 0, 0};
static int HSV_max[] = {359, 255, 255};


/* Additional function to sort r, g, b values.
 * Return a char characterizing the order of r, g, b values :
 *  0=rgb, 1=gbr, 2=grb, 3=rbg, 4=bgr, 5=brg
 * Gives the maximum and minimum values in maxrgb and minrgb parameters
 */
static char 
sortRGB(int R, int G, int B, int *minrgb, int *maxrgb)
{
  static char order;

  if (G > B)
    {
      if (R > G)
	{
	  *maxrgb = R;
	  *minrgb = B;
	  order = 0;
	}
      else 
	{
	  if (B > R)
	    {
	      *maxrgb = G;
	      *minrgb = R;
	      order = 1;
	    }
	  else
	    {
	      *maxrgb = G;
	      *minrgb = B;
	      order = 2;
	    }
	}
    }
  else 
    {
      if (R > B)
	{
	  *maxrgb = R;
	  *minrgb = G;
	  order = 3;
	}
      else
	{
	  if (G > R)
	    {
	      *maxrgb = B;
	      *minrgb = R;
	      order = 4;
	    }
	  else
	    {
	      *maxrgb = B;
	      *minrgb = G;
	      order = 5;
	    }
	}
    }

  return order;
}


static
int 
P_RGB_to_H(int R, int G, int B)
{
  int H;
  
  int min, max;
  //  fprintf(stderr, "(%d,%d,%d) // ", R, G, B);
  char order = sortRGB(R, G, B, &min, &max);
  if (max == min)
    H = 0;
  else
    {
      float floatH;

      switch(order)
	{
	case 0 : 
	  floatH = 1.0 - static_cast<float>(R - G) / static_cast<float>(R - B);
	  break;
	case 1 :
	  floatH = 3.0 - static_cast<float>(G - B) / static_cast<float>(G - R);
	  break;
	case 2 :
	  floatH = 1.0 + static_cast<float>(G - R) / static_cast<float>(G - B);
	  break;
	case 3 :
	  floatH = 5.0 + static_cast<float>(R - B) / static_cast<float>(R - G);
	  break;
	case 4 :
	  floatH = 3.0 + static_cast<float>(B - G) / static_cast<float>(B - R);
	  break;
	case 5 :
	  floatH = 5.0 - static_cast<float>(B - R) / static_cast<float>(B - G);
	  break;
	default :
	  fprintf(stderr, "Error:  Internal error in RGB to H transformation");
	  floatH = 0.0;
	}

      H = static_cast<int>(floatH / 6 * 359);
      //fprintf(stderr, "(%d,%d,%d)  %g --> %ld \n", R, G, B, floatH, H);
    }
  
  return H;
}


static
int 
P_RGB_to_S(int R, int G, int B)
{
  long S;
  
  int max = MAX(R, G);
  max = MAX(max, B);

  int min = MIN(R, G);
  min = MIN(min, B);

  if (!max)
    S = 0;
  else 
    S = ((max - min) * 255) / max;
  
  return static_cast<int>(S);  
}


static
int
P_RGB_to_V_HS(int R, int G, int B)
{
  int V;
  
  V = MAX(R, G);
  V = MAX(V, B);

  return V;
}

static
void
P_RGB_to_HSV(int r, int g, int b, int &h, int &s, int &v)
{
  h = 0;
  s = 0;

  int min, max;
  const char order = sortRGB(r, g, b, &min, &max);

  v = max;

  if (max != min) {
    float floatH;
    
    switch(order) {
    case 0 : 
      floatH = 1.0f - static_cast<float>(r - g) / static_cast<float>(r - b);
      break;
    case 1 :
      floatH = 3.0f - static_cast<float>(g - b) / static_cast<float>(g - r);
      break;
    case 2 :
      floatH = 1.0f + static_cast<float>(g - r) / static_cast<float>(g - b);
      break;
    case 3 :
      floatH = 5.0f + static_cast<float>(r - b) / static_cast<float>(r - g);
      break;
    case 4 :
      floatH = 3.0f + static_cast<float>(b - g) / static_cast<float>(b - r);
      break;
    case 5 :
      floatH = 5.0f - static_cast<float>(b - r) / static_cast<float>(b - g);
      break;
    default :
      std::cerr<<"Error:  Internal error in RGB to H transformation\n";
      floatH = 0.0f;
    }
    
    h = static_cast<int>(floatH / 6.f * 359.f);
    s = ((max - min) * 255  / max); //B: float ???

  }
  
}




/*  HSV to RGB */

/* some auxiliary functions to compute R, G and B from  hue, Min of R, G, B and Max of R, G, B*/
/* useful both for HSV and HMMD conversions */

static
int 
P_HMinMax_to_R(int H, long min, long max)
{
  float floatH = static_cast<float>(H);

  long R = max;

  if (floatH != -1) {
    floatH = floatH / 255.0 * 360.0;
    
    if (floatH == 360)
      floatH = 0.0;
    
    floatH = floatH / 60.0;
    long i = static_cast<long>(floatH);
    float f = floatH - i;
    float q = max * (1 - f) + min * f;
    float t = max * f + min * (1 - f);
    
    switch (i) {
    case 0:
    case 5: 
      R = max;
      break;
    case 2 :
    case 3 :
      R = min;
      break;
    case 1 :
      R = static_cast<long>(q + 0.5);
      break;
    case 4 :
      R = static_cast<long>(t + 0.5);
      break;
    }
  }
  
  return static_cast<int>(R);
}


static
int 
P_HMinMax_to_G(int H, long min, long max)
{
  float floatH = static_cast<float>(H);

  long G = max;

  if (floatH != -1) {
    floatH = floatH / 255.0 * 360.0;
    
    if (floatH == 360)
      floatH = 0.0;
    
    floatH = floatH / 60.0;
    long i = static_cast<long>(floatH);
    float f = floatH -  i;
    float q = static_cast<float>(max) * (1 - f) + static_cast<float>(min) * f;
    float t = static_cast<float>(max) * f + static_cast<float>(min) * (1 - f);
    
    switch (i) {
    case 0:
      G = static_cast<long>(t + 0.5);
      break;
    case 1: 
    case 2 :
      G = max;
      break;
    case 3 :
      G = static_cast<long>(q + 0.5);
      break;
    case 4 :
    case 5 :
      G = min;
      break;
    }
  }
  
  return static_cast<int>(G);
}


static
int 
P_HMinMax_to_B(int H, long min, long max)
{
  float floatH = static_cast<float>(H);

  long B = max;
 
  if (floatH != -1)
    {
      floatH = floatH / 255.0 * 360.0;
    
      if (floatH == 360)
	floatH = 0.0;
    
      floatH = floatH / 60.0;
      long i = static_cast<long>(floatH);
      float f = floatH - i;
      float q = max * (1 - f) + min * f;
      float t = max * f + min * (1 - f);

      switch (i)
	{
	case 0:
	case 1: 
	  B = min;
	  break;
	case 2 :
	  B = static_cast<long>(t + 0.5);
	  break;
	case 3 :
	case 4 :   
	  B = max;
	  break;
	case 5 :
	  B = static_cast<long>(q + 0.5);
	  break;
	}
    }
  
  return static_cast<int>(B);
}


static
int 
P_HSV_to_R(int H, int S, int V)
{
  long max = V;
  long min = static_cast<long>(max * (1.0 - S / 255.0) + 0.5);

  return P_HMinMax_to_R(H, min, max);
}


static
int 
P_HSV_to_G(int H, int S, int V)
{
  long max = V;
  long min = static_cast<long>(max * (1.0 - S / 255.0) + 0.5);

  return P_HMinMax_to_G(H, min, max);
}


static
int 
P_HSV_to_B(int H, int S, int V)
{
  long max = V;
  long min = static_cast<long>(max * (1.0 - S / 255.0) + 0.5);

  return P_HMinMax_to_B(H, min, max);
}

static
void
P_HSV_to_RGB(int h, int s, int v, int &r, int &g, int &b)
{
  //TODO: optimized version
  r = P_HSV_to_R(h, s, v);
  g = P_HSV_to_G(h, s, v);
  b = P_HSV_to_B(h, s, v);
}



/**************************************************************************/
/********************************* YCbCr **********************************/
/**************************************************************************/

/* sa */

static int YCbCr_min[] = {16, 16, 16};
static int YCbCr_max[] = {235, 239, 239};


/* RGB to YCbCr */
/*  Y is constrained in [16, 235] whereas Cb and Cr are constrained in [16, 240] */
/* Conversion matrices obtained from Poynton FAQ : case where R', G', and B' belong to [0, 255] */
/* Note : conversion appearing in MPEG-7 software in ColorLayoutExtraction.cpp      */ 
/*        seems to corresponds to the conversion where R', G', and B' belong to [0, 1]  */
/*        another (and different) conversion exists in HierarchicalKeyFrameExtraction : */
/*        I have no idea where the coefficients used in this conversion come from. Here they are : */
/*   Y  = (int)(4096.0 + (46.5375*dRVal) + (156.6726*dGVal) + (15.7899*dBVal)) >> 8 */
/*   Cb = (int)(32768.0 + (112.0*dBVal) - (25.76*dRVal) - (86.24*dGVal)) >> 8 */
/*   Cr = (int)(32768.0 + (112.0*dRVal) - (101.696*dGVal) - (10.304*dBVal)) >> 8 */
/* Any idea ? */

static
int 
P_RGB_to_Y_CbCr(int R, int G, int B)
{
  const long Y = static_cast<long>(16 + (65.738 * static_cast<float>(R) + 129.057 * static_cast<float>(G) + 25.064 * static_cast<float>(B)) / 256.0);

  int result = static_cast<int>(MAX(Y, 16));
  result = MIN(result, 235);    
  
  return result;
}


static
int 
P_RGB_to_Cb(int R, int G, int B)
{
  const long Cb = static_cast<long>(128 + (-37.945 * static_cast<float>(R) - 74.494 * static_cast<float>(G) + 112.439 * static_cast<float>(B)) / 256.0);

  int result = static_cast<int>(MAX(Cb, 16));
  result = MIN(result, 240);    
  
  return result;
}


static
int 
P_RGB_to_Cr(int R, int G, int B)
{

  const long Cr = static_cast<long>(128 + (112.439 * static_cast<float>(R) - 94.154 * static_cast<float>(G) - 18.285 *  static_cast<float>(B)) / 256.0);

  int result = static_cast<int>(MAX(Cr, 16));
  result = MIN(result, 240);    
  
  return result;
}

static
void
P_RGB_to_YCbCr(int r, int g, int b, int &y, int &cb, int &cr)
{
  y = P_RGB_to_Y_CbCr(r, g, b);
  cb = P_RGB_to_Cb(r, g, b);
  cr = P_RGB_to_Cr(r, g, b);
}



/*  YCbCr to RGB */
static
int 
P_YCbCr_to_R(int Y, int /* Cb */, int Cr)
{
  const float R = ((298.082f * static_cast<float>(Y - 16) + 408.583f * static_cast<float>(Cr - 128)) / 256.0f );
  return CLAMP_TO_UCHAR(R);
}


static
int 
P_YCbCr_to_G(int Y, int Cb, int Cr)
{
  const float G = ((298.082f * static_cast<float>(Y - 16) - 100.291f * static_cast<float>(Cb - 128) - 208.120f * static_cast<float>(Cr - 128)) / 256.0f );
  return CLAMP_TO_UCHAR(G);
}

static
int 
P_YCbCr_to_B(int Y, int Cb, int /* Cr */)
{
  const float B = ((298.082f * static_cast<float>(Y - 16) + 516.411f * static_cast<float>(Cb - 128)) / 256.0f );
  return CLAMP_TO_UCHAR(B);
}

static
void
P_YCbCr_to_RGB(int y, int cb, int cr, int &r, int &g, int &b)
{
  //TODO: optimized version
  r = P_YCbCr_to_R(y, cb, cr);
  g = P_YCbCr_to_G(y, cb, cr);
  b = P_YCbCr_to_B(y, cb, cr);
}


/**************************************************************************/
/********************************** YUV ***********************************/
/**************************************************************************/

/* sa */

static int YUV_min[] = {0, 0, 0};
static int YUV_max[] = {255, 128, 128};


/* RGB to YUV */
/* extract from MPEG-7 software */
/* This is the conversion which seems to be used when referring to YCbCr color space in DominantColor */
/* But according to Poynton Colour FAQ, it corresponds to the color space YPbPr*/

//B: conversion to use float everywhere

static
int 
P_RGB_to_Y_UV(int R, int G, int B)
{
  const float Y = 0.299f * R + 0.587f * G + 0.114f * B;
  return CLAMP_TO_UCHAR(Y);
}


static
int 
P_RGB_to_U(int R, int G, int B)
{
  //const float U = -0.169f * R - 0.331f * G + 0.500f * B;
  const float U = -0.168736f * R - 0.331264f * G + 0.5f * B;
  return FLOAT_TO_INT(U);
}


static
int
P_RGB_to_V_YU(int R, int G, int B)
{
  //const float V = 0.500f * R - 0.419f * G - 0.081f * B;
  const float V = 0.5f * R - 0.418688f * G - 0.081312f * B;
  return FLOAT_TO_INT(V);
}


static
void
P_RGB_to_YUV(int r, int g, int b, int &y, int &u, int &v)
{
  //TODO: optimized version
  y = P_RGB_to_Y_UV(r, g, b);
  u = P_RGB_to_U(r, g, b);
  v = P_RGB_to_V_YU(r, g, b);
}


/*  YUV to RGB */
static
int 
P_YUV_to_R(int Y, int /* U */, int V)
{
  //const float R = Y + (1.4017f * static_cast<float>(V - 128) + 0.5f); 
  const float R = Y + 1.402f * V;

  return CLAMP_TO_UCHAR(R);
}


static
int 
P_YUV_to_G(int Y, int U, int V)
{
  //const float G = Y - (0.3437f * static_cast<float>(U - 128) + 0.7142f * static_cast<float>(V - 128) + 0.5f);
  const float G = Y + -0.344136f*U + -0.714136*V;

  return CLAMP_TO_UCHAR(G);
}


static
int 
P_YUV_to_B(int Y, int U, int /* V */)
{
  //const float B = Y + (1.772f * static_cast<float>(U - 128) + 0.5f); 
  const float B = Y + 1.772f*U;

  return CLAMP_TO_UCHAR(B);
}

static
void
P_YUV_to_RGB(int y, int u, int v, int &r, int &g, int &b)
{
  //TODO: optimized version
  r = P_YUV_to_R(y, u, v);
  g = P_YUV_to_G(y, u, v);
  b = P_YUV_to_B(y, u, v);
}


/**************************************************************************/
/********************************** GRAY ***********************************/
/**************************************************************************/

static int GRAY_min[] = {0, 0, 0};
static int GRAY_max[] = {255, 255, 255};

/* RGB to RGB */
static int P_RGB_to_G1(int R, int  G, int B ) { 
 const long U = static_cast<long>(0.299 * static_cast<float>(R) + 0.587 * static_cast<float>(G) + 0.114 * static_cast<float>(B) + 0.5);

  int result = static_cast<int>(MAX(U, 0));
  result = MIN(result, 255);    
  
  return result; 
}
static int P_RGB_to_G2(int r , int g,  int b) { return P_RGB_to_G1(r, g, b); }
static int P_RGB_to_G3(int r , int g,  int b) { return P_RGB_to_G1(r, g, b); }

static
void
P_RGB_to_GRAY(int r, int g, int b, int &G1, int &G2, int &G3)
{
  //TODO: optimized version
  G1 = P_RGB_to_G1(r, g, b);
  G2 = P_RGB_to_G2(r, g, b);
  G3 = P_RGB_to_G3(r, g, b);
}

static int P_GRAY_to_R(int /* G1 */, int /* G2 */, int /* G3 */) { return 0; }
static int P_GRAY_to_G(int /* G1 */, int /* G2 */, int /* G3 */) { return 0; }
static int P_GRAY_to_B(int /* G1 */, int /* G2 */, int /* G3 */) { return 0; }

static
void
P_GRAY_to_RGB(int /*i1*/, int /*i2*/, int /*i3*/, int &r, int &g, int &b)
{
  //TODO: optimized version
  r = 0;
  g = 0;
  b = 0;
}

/****************************************************************************/

namespace ise {
  namespace conversion {

    PFConversion *
    changeSpace(int choice)
    {
      static PFConversion iseConversion[][3]= 
	{
	  {P_RGB_to_R, P_RGB_to_G, P_RGB_to_B}, // 0
	  {NULL, NULL, NULL}, // 1
	  {NULL, NULL, NULL}, // 2
	  {P_RGB_to_L_of_Lab, P_RGB_to_a, P_RGB_to_b}, // 3
	  {P_RGB_to_L_of_Luv, P_RGB_to_u, P_RGB_to_v}, // 4
	  {NULL, NULL, NULL}, // 5
	  {NULL, NULL, NULL}, // 6
	  {NULL, NULL, NULL}, // 7
	  {NULL, NULL, NULL}, // 8
	  {P_RGB_to_H, P_RGB_to_S, P_RGB_to_V_HS}, // 9
	  {NULL, NULL, NULL}, // 10
	  {P_RGB_to_Y_CbCr, P_RGB_to_Cb, P_RGB_to_Cr}, // 11
	  {P_RGB_to_Y_UV, P_RGB_to_U, P_RGB_to_V_YU}, // 12
	  {P_RGB_to_G1, P_RGB_to_G2, P_RGB_to_G3} // 13
	};
      return(iseConversion[choice]);
    }


    PFConversion *
    restoreSpace(int choice)
    {
      static PFConversion iseConversion_inverse[][3]= 
	{
	  {P_RGB_to_R, P_RGB_to_G, P_RGB_to_B}, // 0
	  {NULL, NULL, NULL}, // 1
	  {NULL, NULL, NULL}, //2 
	  {P_Lab_to_R, P_Lab_to_G, P_Lab_to_B}, //3
	  {P_Luv_to_R, P_Luv_to_G, P_Luv_to_B}, //4
	  {NULL, NULL, NULL}, //5
	  {NULL, NULL, NULL}, //6
	  {NULL, NULL, NULL}, //7
	  {NULL, NULL, NULL}, //8
	  {P_HSV_to_R, P_HSV_to_G, P_HSV_to_B}, //9
	  {NULL, NULL, NULL}, //10
	  {P_YCbCr_to_R, P_YCbCr_to_G, P_YCbCr_to_B}, //11
	  {P_YUV_to_R, P_YUV_to_G, P_YUV_to_B}, //12
	  {P_GRAY_to_R, P_GRAY_to_G, P_GRAY_to_B} //13
	};
      return(iseConversion_inverse[choice]);
    }

    int
    absoluteMin(int choice, int chan)
    {
      static int *iseConversion_min[] =
	{
	  RGB_min, 
	  NULL,
	  NULL,
	  Lab_min,
	  Luv_min,
	  NULL,
	  NULL,
	  NULL,
	  NULL,
	  HSV_min,
	  NULL,
	  YCbCr_min,
	  YUV_min,
	  GRAY_min
	};

      return iseConversion_min[choice][chan];
    }


    int
    absoluteMax(int choice, int chan)
    {
      static int *iseConversion_max[] =
	{
	  RGB_max, 
	  NULL,
	  NULL,
	  Lab_max,
	  Luv_max,
	  NULL,
	  NULL,
	  NULL,
	  NULL,
	  HSV_max,
	  NULL,
	  YCbCr_max,
	  YUV_max,
	  GRAY_max
	};

      return iseConversion_max[choice][chan];
    }

    Change3Components changeSpace3(int colorSpace)
    {
      static Change3Components iseConversion_changeFromRGB[] = {
	P_RGB_to_RGB,
	NULL,
	NULL,
	P_RGB_to_Lab,
	P_RGB_to_Luv,
	NULL,
	NULL,
	NULL,
	NULL,
	P_RGB_to_HSV,
	NULL,
	P_RGB_to_YCbCr,
	P_RGB_to_YUV,
	P_RGB_to_GRAY
      };
      return iseConversion_changeFromRGB[colorSpace];
    }

    Change3Components restoreSpace3(int colorSpace)
    {
      static Change3Components iseConversion_changeToRGB[] = {
	P_RGB_to_RGB,
	NULL,
	NULL,
	P_Lab_to_RGB,
	P_Luv_to_RGB,
	NULL,
	NULL,
	NULL,
	NULL,
	P_HSV_to_RGB,
	NULL,
	P_YCbCr_to_RGB,
	P_YUV_to_RGB,
	P_GRAY_to_RGB
      };
      return iseConversion_changeToRGB[colorSpace];
    }

    void changeFromRGB(int colorSpace, int r, int g, int b, int &x, int &y, int &z)
    {
      (changeSpace3(colorSpace))(r, g, b, x, y, z);
    }
    
    void changeToRGB(int colorSpace, int x, int y, int z, int &r, int &g, int &b)
    {
      (restoreSpace3(colorSpace))(x, y, z, r, g, b);
    }

  }

}







