/*
  Colorimetric moements computer.
  Copyright (C) 2004-2009 Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef ISE_COLORIMETRICMOMENTSCOMPUTER_HPP
#define ISE_COLORIMETRICMOMENTSCOMPUTER_HPP

#include <girl/girlCommon.hpp>
#include <girl/girlToger.hpp>
#include <girlTools/girlToolsImage.hpp>
#include <girlTools/girlToolsImageFeature.hpp>

#include "iseConversion.hpp"
#include <girlTools/girlToolsVector.hpp>

namespace ise {

  class GIRL_API MomentsComputerIF : public girl::toger::pointIterator
  {
  public:
    
    MomentsComputerIF(const girl::tools::ImageFeature *imgFeature,
		      double *m0,
		      double *m1,
		      double *m2)
      : m_imageFeature(imgFeature)
      , m_m2(m2)
      , m_m1(m1)
      , m_m0(m0)
    {
      clear();
    }


    void clear()
    {
      if (m_imageFeature->isNull())
	return;
      for (int i = 0 ; i < m_imageFeature->channels() ; ++i) {
	m_m2[i] = 0.0;
	m_m1[i] = 0.0;
      }
      (*m_m0) = 0.0;
    }

    void processPoint(const girl::point &pt)
    {

      if (m_imageFeature->isNull())
	return;

      const girl::tools::PixelFromImage<girl::tools::ImageFeature::type> p = m_imageFeature->getPixel(pt.x(), pt.y());
      
      (*m_m0)++;// += 1;

      for (int i = 0 ; i < m_imageFeature->channels() ; ++i) {
        double val = static_cast<double> (p[i]);
	m_m1[i] += val;     // c[i];
	m_m2[i] += val*val; //cs[i];
      }
    }
    
  private:

    MomentsComputerIF(const MomentsComputerIF &);
    MomentsComputerIF &operator=(const MomentsComputerIF &);

  public:
    const girl::tools::ImageFeature *m_imageFeature;
    double *m_m2;
    double *m_m1;
    double *m_m0;
  };

  class GIRL_API ColorimetricMomentsComputer : public girl::toger::pointIterator
  {
  public:
  
    /**
     * @brief Constructor.
     */
    explicit ColorimetricMomentsComputer(const girl::tools::Image &img)
      : m_img(img),
	m_m2(0.0, 0.0, 0.0),
	m_m1(0.0, 0.0, 0.0),
	m_m0(0.0)
    {}

    void clear()
    {
      m_m2 = girl::tools::Flt3(0.0, 0.0, 0.0);
      m_m1 = girl::tools::Flt3(0.0, 0.0, 0.0);
      m_m0 = 0.0;
    }

    void processPoint(const girl::point &pt)
    {
      const girl::tools::Pixel p = m_img.getPixel(pt.x(), pt.y());
  
      m_m0 += 1;
      
      const int x = p[0];
      const int y = p[1];
      const int z = p[2];

      const girl::tools::Flt3 c(static_cast<girl::tools::Flt>(x), static_cast<girl::tools::Flt>(y), static_cast<girl::tools::Flt>(z));
      //const girl::tools::Flt3 cs(c[0]*c[0], c[1]*c[1], c[2]*c[2]);
      const girl::tools::Flt3 cs(static_cast<girl::tools::Flt>(x*x), static_cast<girl::tools::Flt>(y*y), static_cast<girl::tools::Flt>(z*z));
      m_m1 += c;
      m_m2 += cs;
      
    }

  private:

    ColorimetricMomentsComputer(const ColorimetricMomentsComputer&);

    ColorimetricMomentsComputer &operator=(const ColorimetricMomentsComputer&);

  public:
    const girl::tools::Image &m_img;
    girl::tools::Flt3 m_m2;
    girl::tools::Flt3 m_m1;
    girl::tools::Flt m_m0;
  };

 


  // class GIRL_API ColorimetricMomentsComputerGI : public girl::toger::pointIterator
  // {
  // public:
  
  //   /**
  //    * @brief Constructor.
  //    */
  //   ColorimetricMomentsComputerGI(const GenericImage<int, 3> &img,
  // 				  double &m0,
  // 				  double *m1,
  // 				  double *m2)
  //     : m_img(img),
  // 	m_m2(m2),
  // 	m_m1(m1),
  // 	m_m0(m0)
  //   {}

  //   void processPoint(const girl::point &pt)
  //   {
  //     const GenericImage<int, 3>::pixel_t p = m_img.getPixel(pt.x(), pt.y());
  
  //     m_m0 += 1;
      
  //     for (int i = 0 ; i < 3 ; ++i) {
  // 	m_m1[i] += static_cast<double>(p[i]);
  // 	m_m2[i] += p[i]*p[i];
  //     }
  //   }

  // private:

  //   ColorimetricMomentsComputerGI(const ColorimetricMomentsComputerGI&);

  //   ColorimetricMomentsComputerGI &operator=(const ColorimetricMomentsComputerGI&);

  // public:
  //   const GenericImage<int, 3> &m_img;
  //   double *m_m2;
  //   double *m_m1;
  //   double &m_m0;
  // };



  class GIRL_API ColorimetricMomentsComputerCS : public girl::toger::pointIterator
  {
  public:

    /**
     * @brief Constructor.
     */
    ColorimetricMomentsComputerCS(const girl::tools::Image &img, int conversion)
      : m_conv(ise::conversion::changeSpace(conversion)),
  	m_img(img),
  	m_m2(0.0, 0.0, 0.0),
  	m_m1(0.0, 0.0, 0.0),
  	m_m0(0.0)
    {}


    void clear()
    {
      m_m2 = girl::tools::Flt3(0.0, 0.0, 0.0);
      m_m1 = girl::tools::Flt3(0.0, 0.0, 0.0);
      m_m0 = 0.0;
    }

    virtual void processPoint(const girl::point &pt)
    {
      const girl::tools::Pixel p = m_img.getPixel(pt.x(), pt.y());
  
      m_m0 += 1;

      const int x = m_conv[0](p[0], p[1], p[2]);
      const int y = m_conv[1](p[0], p[1], p[2]);
      const int z = m_conv[2](p[0], p[1], p[2]);

      const girl::tools::Flt3 c(static_cast<double>(x), static_cast<double>(y), static_cast<double>(z));
      //const girl::tools::Flt3 cs(c[0]*c[0], c[1]*c[1], c[2]*c[2]);
      const girl::tools::Flt3 cs(static_cast<double>(x*x), static_cast<double>(y*y), static_cast<double>(z*z));
      m_m1 += c;
      m_m2 += cs;
      
    }
    
  private:
    ColorimetricMomentsComputerCS(const ColorimetricMomentsComputerCS &);

    ColorimetricMomentsComputerCS &operator=(const ColorimetricMomentsComputerCS &);
    
  public:
    ise::conversion::PFConversion *m_conv;
    const girl::tools::Image &m_img;
    girl::tools::Flt3 m_m2;
    girl::tools::Flt3 m_m1;
    double m_m0;
  };


  // class GIRL_API ColorimetricMomentsComputerCSOneChannel : public girl::toger::pointIterator
  // {
  // public:

  //   /**
  //    * @brief Constructor.
  //    */
  //   ColorimetricMomentsComputerCSOneChannel(const girl::tools::Image &img, int conversion, int channel)
  //     : m_conv(ise::conversion::changeSpace(conversion)),
  // 	m_img(img),
  // 	m_m2(0.0),
  // 	m_m1(0.0),
  //       m_m0(0.0),
  //       m_channel(channel)
  //   {}

  //   virtual void processPoint(const girl::point &pt)
  //   {
  //     const girl::tools::Pixel p = m_img.getPixel(pt.x(), pt.y());
  
  //     m_m0 += 1;

  //     const int v = m_conv[m_channel](p[0], p[1], p[2]);

  //     m_m1 += v;
  //     m_m2 += v*v;
      
  //   }
    
  // private:
  //   ColorimetricMomentsComputerCSOneChannel(const ColorimetricMomentsComputerCSOneChannel &);

  //   ColorimetricMomentsComputerCSOneChannel &operator=(const ColorimetricMomentsComputerCSOneChannel &);
    
  // public:
  //   ise::conversion::PFConversion *m_conv;
  //   const girl::tools::Image &m_img;
  //   double m_m2;
  //   double m_m1;
  //   double m_m0;
  //   int m_channel;
  // };

}




#endif /* ! ISE_COLORIMETRICMOMENTSCOMPUTER_HPP */
