/*
  Color binary tree.
  Copyright (C) 2004-2009 Luc Brun, Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "iseColorBinaryTree.hpp"

#include <cstdio> //fprintf
#ifdef DEBUG
#include <iostream>
#endif

#include "iseColorFrequency.hpp"

#include "iseDefinitions.h"

using namespace ise;

#define CALLMEMBERFUNCTION(object, ptrToMember)  ((object).*(ptrToMember))
#define CALLMEMBERFUNCTIONP(objectPtr, ptrToMember)  ((objectPtr)->*(ptrToMember))


ColorBinaryTree::ColorBinaryTree()
  : m_ecart(0.0)
  , m_cutAxis(0)
  , m_moment0(0.0)
  , m_number(0)
  , m_isLeaf(true)
  , m_axisChoiceFn(&ColorBinaryTree::varChoice)
  , m_valueFn(&ColorBinaryTree::weightedMSE)
  , m_partitionFn(&ColorBinaryTree::cut)
{
  init();
}

ColorBinaryTree::~ColorBinaryTree()
{
  delete m_child[0];
  delete m_child[1];
}

void
ColorBinaryTree::clear()
{
  delete m_child[0];
  delete m_child[1];
  
  init();
}

void
ColorBinaryTree::init()
{
  m_child[0] = NULL;
  m_child[1] = NULL;

  m_range[0].set(0, 0);
  m_range[1].set(0, 0);
  m_range[2].set(0, 0);

  m_var[0] = 0.0;
  m_var[1] = 0.0;
  m_var[2] = 0.0;

  m_meanAxis[0] = 0;
  m_meanAxis[1] = 0;
  m_meanAxis[2] = 0;

  m_ecart = 0.0;
  m_cutAxis = 0;
  
  m_moment0 = 0.0;
  m_moment1[0] = 0.0;
  m_moment1[1] = 0.0;
  m_moment1[2] = 0.0;
  m_moment2[0] = 0.0;
  m_moment2[1] = 0.0;
  m_moment2[2] = 0.0;

  m_number = 0;
  m_isLeaf = true;
}

void
ColorBinaryTree::set(const ColorFrequency &colFreq,
		     int chosenCluster,
		     int chosenAxis,
		     int chosenInit,
		     int chosenCut)
{
  
  static PFaxe choix_de_l_axe[] = 
    {
      &ColorBinaryTree::varChoice, 
      &ColorBinaryTree::lengthChoice, 
      &ColorBinaryTree::varLengthChoice,
      &ColorBinaryTree::varMedianChoice,
      &ColorBinaryTree::mseChoice,
      &ColorBinaryTree::splitVarChoice
    };

  static PFcluster valeur_cluster[] = 
    {
      &ColorBinaryTree::weightedMSE,
      &ColorBinaryTree::nonWeightedMSE,
      &ColorBinaryTree::cardinal
    };
  static PFinit choix_init[] = 
    {
      &ColorBinaryTree::initialisation,
      &ColorBinaryTree::initialisationDangerous
    };
  static PFdecoupe choix_decoupe[] = 
    {
      &ColorBinaryTree::cut,
      &ColorBinaryTree::cutProject,
      &ColorBinaryTree::cutMedian,
      &ColorBinaryTree::cutMedianDangerous
    };

 
  PFinit init = choix_init[chosenInit];

  m_partitionFn = choix_decoupe[chosenCut];
  m_valueFn = valeur_cluster[chosenCluster];
  m_axisChoiceFn = choix_de_l_axe[chosenAxis];
  
  m_ecart = 0.0;
  m_child[0] = NULL;
  m_child[1] = NULL;
  
  // On fixe les intervalles
  for (int i=0; i<3; ++i) {
    m_range[i].set(colFreq.getMin(i), colFreq.getMax(i)+1);
  }
  
  //Calcul des differents moments 
#ifdef DEBUG
  std::cerr<<"RACINE : variances :\n";
#endif
  double mom0 = 0.0;
  double mom1[3] = {0.0, 0.0, 0.0};
  double mom2[3] = {0.0, 0.0, 0.0};
  CALLMEMBERFUNCTIONP(this, init)(colFreq, &mom0, mom1, mom2);
  assert(mom0 != 0.0);
  m_moment0 = mom0;
  for (int i=0; i<3; ++i) {
      m_moment1[i] = mom1[i];
      m_moment2[i] = mom2[i];
      m_meanAxis[i] = mom1[i]/mom0;
      m_var[i] = (mom2[i] - mom1[i]*mom1[i]/mom0)/mom0;
#ifdef DEBUG
      std::cerr<<"sur l'axe "<<i<<" : "<<m_var[i]<<"\n";
#endif
    }
  
  m_ecart =  CALLMEMBERFUNCTIONP(this, m_valueFn)();
  m_cutAxis = CALLMEMBERFUNCTIONP(this, m_axisChoiceFn)(colFreq);

  assert(m_cutAxis>=0 && m_cutAxis<3);

#ifdef DEBUG
  std::cerr<<std::endl<<std::endl;
#endif

  m_isLeaf = true;
}

void
ColorBinaryTree::createTable(ColorFrequency &colFreq,
			     int &nbOfColors,
			     double ecart_max)
{
  int nbColors=1;
  
  double ecart_initial = m_ecart;
  double ecart1 = 100.0;
  double ecart2;
  bool end = false;
  
  do {
      
      createTable_aux(colFreq);
      ++nbColors;
      ecart2 = (100*m_ecart)/ecart_initial;
      /* on s' arrete si on est pres de ecart max et si l' */
      /* ecart intra classe decroit lentement*/
      end = (ecart1-ecart2< 2.0)&&(ecart2 < ecart_max+2.0);
      /* On s' arrete de toute facon si on est en dessous de */
      /* ecart max*/
      end = end || (ecart2 <= ecart_max);
      ecart1 = ecart2;
#ifdef DEBUG
      printf("nb de couleurs = %d\n", nbColors);
      printf("ecart max dans l' arbre :%f\n", ecart2);
#endif
  }
  while (nbColors < nbOfColors);
  /*    while (!end);*/
  nbOfColors = nbColors;
}


//B: do not use @a color !!!???
void
ColorBinaryTree::seekColorInTable(int *color,
				  int &colorIndex)
{
  if (m_isLeaf) {
    colorIndex = m_number;
  }
  else {
    assert(m_child[0] != NULL);
    assert(m_child[1] != NULL);
    if (color[m_cutAxis] < ((m_child[0]->m_range[m_cutAxis].max())*REDUCTION))
      m_child[0]->seekColorInTable(color, colorIndex);
    else
      m_child[1]->seekColorInTable(color, colorIndex);
  }
}

void
ColorBinaryTree::labelLeafes(int *table)
{
  int numero = 0;
  label(table, numero);
}

//-------------------------------------------

/**
 * Compute where chosen axis must be cut,
 * and moments of two resulting clusters.
 */
int
ColorBinaryTree::cut(const ColorFrequency &colFreq,
		     double *moment0,
		     momentsChildren moment1,
		     momentsChildren moment2)
{
  assert(m_cutAxis>=0 && m_cutAxis<3);
  double S0 = m_moment0;  //B: replaced float with double !!!
  int axe1 = (m_cutAxis+1)%3;
  int axe2 = (axe1 + 1)%3;
  int b_inf1 = m_range[m_cutAxis].min();
  int b_sup1 = m_range[m_cutAxis].max();

#ifdef DEBUG
  int b_inf2 = m_range[axe1].min();
  int b_sup2 = m_range[axe1].max();
  int b_inf3 = m_range[axe2].min();
  int b_sup3 = m_range[axe2].max();
#endif

  int i; /* variables de boucle */
  int index_decoupe = b_inf1; //B
  double mom0 = 0.0, mom1[3] = {0.0, 0.0, 0.0}, mom2[3] = {0.0, 0.0, 0.0};//B
  double  En[3], En2[3];
  double sn, terme, ui, freq_axei;
  double max = 0.0;
  double moments[3][2];
  double MSE;
  int demi_intervalle = (REDUCTION>>1);
  bool end = false;
  double X_moins_ya, det2 = 0.0;
  double a[3];
  double b2_par_delta2, delta2 = 0.0;
  /* initialisations */

  /************************************/
  /* sn : cardinal de Ct */
  /* En[], En2[] moments d'ordre 1 et 2*/
  /* a : moyenne du cluster */
  /* delta2 norme au carre du verteur tel que delta_i = longueur de */
  /* l'intervalle i */
  /************************************/
  sn=0.0;
  for (int j=0; j<3; ++j) {
    En[j]=0.0;
    En2[j] =0.0;
    a[j] = m_moment1[j]/S0;
    delta2 += m_range[j].width()*m_range[j].width();
  }
  delta2 *= REDUCTION*REDUCTION;
  assert(delta2>0.0);
  b2_par_delta2 = S0*(S0/delta2);
  i= b_inf1;

#ifdef DEBUG
  printf("sur l'axe I%d : [%d,%d[\n", m_cutAxis+1, b_inf1, b_sup1);
  printf("sur l'axe I%d : [%d,%d[\n", axe1+1, b_inf2, b_sup2);
  printf("sur l'axe I%d : [%d,%d[\n", axe2+1, b_inf3, b_sup3);
#endif

  /* Calculs */
  while ((i<b_sup1)&&(!end)) {

    freq_axei = colFreq.projection(m_cutAxis, i, m_range, moments[axe1], moments[axe2]);
      

    ui = (i<<K_REDUCTION);
    if (i<0)
      ui -= demi_intervalle;
    else
      ui += demi_intervalle;
      
    sn += freq_axei;
    terme = ui*freq_axei;
      
    En[m_cutAxis] += terme;
    En[axe1] += moments[axe1][0];
    En[axe2] += moments[axe2][0];
    En2[m_cutAxis] +=  ui*terme;
    En2[axe1] += moments[axe1][1];
    En2[axe2] += moments[axe2][1];
    MSE=0.0;
    if ((sn != 0.0)&&(sn != S0)) {
      for (int j=0; j<3; ++j) {
	X_moins_ya = En[j]/sn-a[j];
	MSE += X_moins_ya*X_moins_ya;
      }
      MSE = (MSE/(S0-sn))*sn;
    }
      
#ifdef DEBUG
    printf("MSE(%d) = %f, z= %f\n", i, MSE, sn/S0);
#endif
      
    if (max <  MSE) {
      max = MSE;
      index_decoupe = i+1;
      mom0 = sn;
      det2 = b2_par_delta2*max;
      for (int j=0; j<3; ++j) {
	mom1[j] = En[j];
	mom2[j] = En2[j];
      }
    }
    end = ((sn+sn) > S0)&&(sn*(S0-sn) <= det2);
    ++i;
  }

  /* transposition des resultats dans les variables de sorties */
  moment0[0] = mom0;
  moment0[1] = S0-mom0;
  for (int j=0; j<3; ++j) {
    moment1[0][j] = mom1[j];
    moment1[1][j] = m_moment1[j]-mom1[j];
    moment2[0][j] = mom2[j];
    moment2[1][j] = m_moment2[j]-mom2[j];
  }
#ifdef DEBUG
  printf("en sortie de boucle i = %d\n", i);
  printf("gain= %d%%\n", (int)(100*(b_sup1-i)/(b_sup1-b_inf1)));
#endif

  return index_decoupe;
} 
  

int
ColorBinaryTree::cutProject(const ColorFrequency &colFreq,
			    double *moment0,
			    momentsChildren moment1,
			    momentsChildren moment2)
{
  
  double S0 = m_moment0;
  int axe1 = (m_cutAxis+1)%3;
  int axe2 = (axe1 + 1)%3;
  int b_inf1 = m_range[m_cutAxis].min();
  int b_sup1 = m_range[m_cutAxis].max();

#ifdef DEBUG

  int b_inf2 = m_range[axe1].min();
  int b_sup2 = m_range[axe1].max();
  int b_inf3 = m_range[axe2].min();
  int b_sup3 = m_range[axe2].max();

#endif

  int i; /* variables de boucle */
  int index_decoupe = b_inf1; //B
  double mom0 = 0.0, mom1[3] = {0.0, 0.0, 0.0}, mom2[3] = {0.0, 0.0, 0.0};//B
  double  En[3], En2[3];
  double sn, terme, ui, freq_axei;
  double max = 0.0;
  double moments[3][2];
  double MSE;
  int demi_intervalle = (REDUCTION>>1);
  bool end = false;
  double X_moins_ya;
  //double det2;
  double a;
  double b2_par_delta2, delta2=0.0;
  /* initialisations */
  /************************************/
  /* sn : cardinal de Ct */
  /* En[], En2[] moments d'ordre 1 et 2*/
  /* a : moyenne du cluster */
  /* delta2 norme au carre du verteur tel que delta_i = longueur de */
  /* l'intervalle i */
  /************************************/
  sn=0.0;
  for (int j=0; j<3; ++j) {
    En[j]=0.0;
    En2[j] =0.0;
      
  }
  a = m_moment1[m_cutAxis]/S0;
  delta2 = m_range[m_cutAxis].width();
  delta2 = (delta2*REDUCTION)*(delta2*REDUCTION);
  b2_par_delta2 = S0*(S0/delta2);
  i = b_inf1;
  b_sup1 = static_cast<int>(b_sup1+a)/2; /* cf wong */
#ifdef DEBUG
  printf("sur l'axe I%d : [%d,%d[\n", m_cutAxis+1, b_inf1, b_sup1);
  printf("sur l'axe I%d : [%d,%d[\n", axe1+1, b_inf2, b_sup2);
  printf("sur l'axe I%d : [%d,%d[\n", axe2+1, b_inf3, b_sup3);
#endif
  /* Calculs */
  while ((i<b_sup1)&&(!end)) {
    freq_axei= colFreq.projection(m_cutAxis,
				    i,
				    m_range,
				    moments[axe1],
				    moments[axe2]);
		
      
    ui = (i<<K_REDUCTION);
    if (i<0)
      ui -= demi_intervalle;
    else
      ui += demi_intervalle;

    sn += freq_axei;
    terme = ui*freq_axei;

    En[m_cutAxis]   += terme;
    En[axe1] += moments[axe1][0];
    En[axe2] += moments[axe2][0];
    En2[m_cutAxis] +=  ui*terme;
    En2[axe1] += moments[axe1][1];
    En2[axe2] += moments[axe2][1];

    MSE=0.0;
    if ((sn != 0.0)&&(sn != S0)) {
	  
      X_moins_ya = En[m_cutAxis]-sn*a;
      MSE = X_moins_ya*X_moins_ya;
      MSE = MSE/(sn*(S0-sn));
    }
    if (max <=  MSE) {
      max = MSE;
      index_decoupe = i+1;
      mom0 = sn;
      //det2 =b2_par_delta2*max;
      for (int j=0; j<3; ++j) {
	mom1[j] = En[j];
	mom2[j] = En2[j];
      }
    }
    end = (sn == S0);
    /*(sn+sn) > S0)&&(sn*(S0-sn) <= det2);*/
    ++i;
  }
  /* transposition des resultats dans les variables de sorties */
  moment0[0] = mom0;
  moment0[1] = S0-mom0;
  for (int j=0; j<3; ++j) {
    moment1[0][j] = mom1[j];
    moment1[1][j] = m_moment1[j]-mom1[j];
    moment2[0][j] = mom2[j];
    moment2[1][j] = m_moment2[j]-mom2[j];
  }
#ifdef DEBUG
  printf("en sortie de boucle i = %d\n", i);
  printf("gain= %d%%\n", (int)(100*(b_sup1-i)/(b_sup1-b_inf1)));
#endif
  return index_decoupe;
}


int
ColorBinaryTree::cutMedian(const ColorFrequency &colFreq,
			   double *moment0,
			   momentsChildren moment1,
			   momentsChildren moment2)
{
  double S0 = m_moment0; //B: replaced float with double !!!
  int axe1 = (m_cutAxis+1)%3;
  int axe2 = (axe1 + 1)%3;
  int b_inf1 = m_range[m_cutAxis].min();
  int b_sup1 = m_range[m_cutAxis].max();

#ifdef DEBUG

  int b_inf2 = m_range[axe1].min();
  int b_sup2 = m_range[axe1].max();
  int b_inf3 = m_range[axe2].min();
  int b_sup3 = m_range[axe2].max();

#endif

  int i; /* variables de boucle */
  double moments[3][2];
  bool end = false;
  double  En[3], En2[3];
  double sn, terme, ui, freq_axei=0.0;
  int demi_intervalle = (REDUCTION>>1);

  /************************************/  
  /* initialisations */
  /************************************/
  
  i= b_inf1;
  sn =0.0;
  for (int j=0; j<3; ++j) {
    En[j]=0.0;
    En2[j] =0.0;
  }
#ifdef DEBUG

  printf("sur l'axe I%d : [%d,%d[\n", m_cutAxis+1, b_inf1, b_sup1);
  printf("sur l'axe I%d : [%d,%d[\n", axe1+1, b_inf2, b_sup2);
  printf("sur l'axe I%d : [%d,%d[\n", axe2+1, b_inf3, b_sup3);

#endif
  /* Calculs */
  while ((i<b_sup1-1)&&(!end)) {
    freq_axei= colFreq.projection(m_cutAxis, i, m_range, moments[axe1], moments[axe2]);
      
    ui = (i<<K_REDUCTION);
    if (i<0)
      ui -= demi_intervalle;
    else
      ui += demi_intervalle;
      
    sn += freq_axei;
    terme = ui*freq_axei;
    moments[m_cutAxis][0] = terme;
    moments[m_cutAxis][1] = ui*terme;
    for (int j=0; j <3; ++j) {
      En[j]  += moments[j][0];
      En2[j] += moments[j][1];
    }
      
    end = (sn+sn) > S0;
    ++i;
  }
  /* transposition des resultats dans les variables de sorties */
  if (sn != S0) {
    moment0[0] = sn;
    moment0[1] = S0-sn;
    for (int j=0; j<3; ++j) {
      moment1[0][j] = En[j];
      moment1[1][j] = m_moment1[j]-En[j];
      moment2[0][j] = En2[j];
      moment2[1][j] = m_moment2[j]-En2[j];
    }
  }
  else {
    --i;
    moment0[0] = sn-freq_axei;
    moment0[1] = freq_axei;
      
    for (int j=0; j<3; ++j) {
      moment1[0][j] = En[j]-moments[j][0];
      moment1[1][j] = moments[j][0];
      moment2[0][j] = En2[j]-moments[j][1];
      moment2[1][j] = moments[j][1];
    }
  }
#ifdef DEBUG
  printf("en sortie de boucle i = %d\n", i);
  printf("gain= %d%%\n", 100*(b_sup1-i)/(b_sup1-b_inf1));
#endif

  return i;
}


int
ColorBinaryTree::cutMedianDangerous(const ColorFrequency &colFreq,
				    double *moment0,
				    momentsChildren moment1,
				    momentsChildren moment2)
{
  double S0   =  m_moment0; //B: replaced float with double !!!
  int axe1   = (m_cutAxis+1)%3;
  int axe2   = (axe1 + 1)%3;
  int b_inf1 = m_range[m_cutAxis].min();
  int b_sup1 = m_range[m_cutAxis].max();
#ifdef DEBUG
  int b_inf2 = m_range[axe1].min();
  int b_sup2 = m_range[axe1].max();
  int b_inf3 = m_range[axe2].min();
  int b_sup3 = m_range[axe2].max();
#endif
  int i; /* variables de boucle */
  double moments[3][2];
  bool end = false;
  double  En[3];
  double sn, terme, ui, freq_axei = 0.0;
  int demi_intervalle = (REDUCTION>>1);
  
  /* initialisations */
  /************************************/
  
  i= b_inf1;
  sn =0.0;
  for (int j=0; j<3; ++j)
    En[j]=0.0;
  
#ifdef DEBUG
  printf("sur l'axe I%d : [%d,%d[\n", m_cutAxis+1, b_inf1, b_sup1);
  printf("sur l'axe I%d : [%d,%d[\n", axe1+1, b_inf2, b_sup2);
  printf("sur l'axe I%d : [%d,%d[\n", axe2+1, b_inf3, b_sup3);
#endif

  /* Calculs */
  while ((i<b_sup1-1)&&(!end)) {
    freq_axei= colFreq.projection(m_cutAxis, i, m_range, moments[axe1], moments[axe2]);
         
    ui = (i<<K_REDUCTION);
    if (i<0)
      ui -= demi_intervalle;
    else
      ui += demi_intervalle;
      
    sn += freq_axei;
    terme = ui*freq_axei;
    moments[m_cutAxis][0] = terme;
    for (int j=0; j<3; ++j)
      En[j]   += moments[j][0];
      
    end = (sn+sn) > S0;
    ++i;
  }
  /* transposition des resultats dans les variables de sorties */
  if (sn != S0) {
    moment0[0] = sn;
    moment0[1] = S0-sn;
    for (int j=0; j<3; ++j) {
      moment1[0][j] = En[j];
      moment1[1][j] = m_moment1[j]-En[j];
      moment2[0][j] = 0.0;
      moment2[1][j] = 0.0;
	  
    }
  }
  else {
    --i;
    moment0[0] = sn-freq_axei;
    moment0[1] = freq_axei;
      
    for (int j=0; j<3; ++j) {
      moment1[0][j] = En[j]-moments[j][0];
      moment1[1][j] = moments[j][0];
      moment2[0][j] = 0.0;
      moment2[1][j] = 0.0;
    }
  }
#ifdef DEBUG
  printf("en sortie de boucle i = %d\n", i);
  printf("gain= %d%%\n", 100*(b_sup1-i)/(b_sup1-b_inf1));
#endif
  return i;
}

//-------------------------------------------------------

double
ColorBinaryTree::cutChoice(const ColorFrequency &colFreq)
{
  double S0 = m_moment0; //B: replaced float with double !!!
  int axe1 = (m_cutAxis+1)%3;
  int axe2 = (axe1 + 1)%3;
  int b_inf1 = m_range[m_cutAxis].min();
  int b_sup1 = m_range[m_cutAxis].max();

#ifdef DEBUG

  int b_inf2 = m_range[axe1].min();
  int b_sup2 = m_range[axe1].max();
  int b_inf3 = m_range[axe2].min();
  int b_sup3 = m_range[axe2].max();

#endif

  int i; /* variables de boucle */
  double  En[3], En2[3];
  double sn, terme, ui, freq_axei;
  double max = 0.0;
  double moments[3][2];
  double MSE;
  int demi_intervalle = (REDUCTION>>1);
  bool end = false;
  double X_moins_ya, det2 = 0.0;
  double a[3];
  double b2_par_delta2, delta2 = 0.0;
  /* initialisations */
  /************************************/
  /* sn : cardinal de Ct */
  /* En[], En2[] moments d'ordre 1 et 2*/
  /* a : moyenne du cluster */
  /* delta2 norme au carre du verteur tel que delta_i = longueur de */
  /* l'intervalle i */
  /************************************/
  sn=0.0;
  for (int j=0; j<3; ++j) {
    En[j]=0.0;
    En2[j] =0.0;
    a[j] = m_moment1[j]/S0;
    delta2 += m_range[j].width()*m_range[j].width();
  }
  delta2 *= REDUCTION*REDUCTION;
  b2_par_delta2 = S0*(S0/delta2);
  i= b_inf1;
#ifdef DEBUG
  printf("sur l'axe I%d : [%d,%d[\n", m_cutAxis+1, b_inf1, b_sup1);
  printf("sur l'axe I%d : [%d,%d[\n", axe1+1, b_inf2, b_sup2);
  printf("sur l'axe I%d : [%d,%d[\n", axe2+1, b_inf3, b_sup3);
#endif
  /* Calculs */
  while ((i<b_sup1)&&(!end)) {
    freq_axei= colFreq.projection(m_cutAxis, i, m_range, moments[axe1], moments[axe2]);
      
    ui = (i<<K_REDUCTION);
    if (i<0)
      ui -= demi_intervalle;
    else
      ui += demi_intervalle;
      
    sn += freq_axei;
    terme = ui*freq_axei;
      
    En[m_cutAxis] += terme;
    En[axe1] += moments[axe1][0];
    En[axe2] += moments[axe2][0];
    En2[m_cutAxis] += ui*terme;
    En2[axe1] += moments[axe1][1];
    En2[axe2] += moments[axe2][1];
    MSE=0.0;
    if ((sn != 0.0)&&(sn != S0)) {
      for (int j=0; j<3; ++j) {
	X_moins_ya = En[j]/sn-a[j];
	MSE += X_moins_ya*X_moins_ya;
      }
      MSE = (MSE/(S0-sn))*sn;
    }
      
    if (max <  MSE) {
      max = MSE;
      det2 =b2_par_delta2*max;
    }
    end = ((sn+sn) > S0)&&(sn*(S0-sn) <= det2);
    ++i;
  }
  
#ifdef DEBUG
  printf("en sortie de boucle i = %d\n", i);
  printf("gain= %d%%\n", (int)(100*(b_sup1-i)/(b_sup1-b_inf1)));
#endif

  return max;
  
}


//----------------------------------------------------------

/*******************************************************************/
/* Calcul des moments d'ordre 0, 1 et 2 d'un cluster. Utilise pour */
/* l'initialisation                                                */
/*******************************************************************/
void
ColorBinaryTree::initialisation(const ColorFrequency &colFreq,
				double *moment0, 
				double *moment1, 
				double *moment2)
{
  int i;
  int axeI1 = 0;
  int axeI2 = 1;
  int axeI3 = 2;
  int b_inf1 = m_range[axeI1].min();
  int b_sup1 = m_range[axeI1].max();
  double freq_axe_I, terme, sn;
  double En[3], En2[3];
  double mom[3][2];
  int ui, demi_intervalle = (REDUCTION>>1);
  
  
  sn=0.0;
  for (int j=0; j<3; ++j) {
    En[j]=0.0;
    En2[j]=0.0;
  }
  
  /* Calcul de la somme; de la somme des carres; du cardinal de */
  /* l'ensemble */
  for (i=b_inf1; i<b_sup1; ++i) {
    freq_axe_I= colFreq.projection(axeI1, i, m_range, mom[axeI2], mom[axeI3]);

    ui = (i*REDUCTION);
    if (i<0)
      ui -= demi_intervalle;
    else 
      ui += demi_intervalle;

    sn += freq_axe_I;
    terme = ui*freq_axe_I;
      
    En[0] += terme;
    En2[0] = En2[0] + ui*terme;
    for (int j=1; j<3; ++j) {
      En[j] += mom[j][0];
      En2[j] += mom[j][1];
    }
      
  }
  
  /* On transvase dans les variables de sortie */

  *moment0 = sn;
  for (int j=0; j<3; ++j) {
    moment1[j] = En[j];
    moment2[j] = En2[j];
  }
}

void
ColorBinaryTree::initialisationDangerous(const ColorFrequency &colFreq,
					 double *moment0, 
					 double *moment1, 
					 double *moment2)
{
  int i;
  int axeI1 = 0;
  int axeI2 = 1;
  int axeI3 = 2;
  int b_inf1 = m_range[axeI1].min();
  int b_sup1 = m_range[axeI1].max();
  double freq_axe_I, terme, sn;
  double En[3];
  double mom[3][2];
  int ui, demi_intervalle = (REDUCTION>>1);
  

  sn=0.0;
  for (int j=0; j<3; ++j)
    En[j]=0.0;
  
  /* Calcul de la somme; de la somme des carres; du cardinal de */
  /* l'ensemble */
  for (i=b_inf1; i<b_sup1; ++i) {
    freq_axe_I = colFreq.projection(axeI1, i, m_range, mom[axeI2], mom[axeI3]);

    ui = (i*REDUCTION);
    if (i<0)
      ui -= demi_intervalle;
    else 
      ui += demi_intervalle;
      
    sn += freq_axe_I;
    terme = ui*freq_axe_I;
      
    En[0] += terme;
    for (int j=1; j<3; ++j)
      En[j] += mom[j][0];
  }
  
  /* On transvase dans les variables de sortie */
  
  *moment0 = sn;
  for (int j=0; j<3; ++j) {
    moment1[j] = En[j];
    moment2[j] = 0.0;
  }
}


//-----------------------------------------


/******************************************************/
/*choix de l'axe a couper */
/* choix_axe est egal a une de ces fonctions */
/******************************************************/

/* L'axe apportant la plus forte diminution de l'MSE */
int
ColorBinaryTree::mseChoice(const ColorFrequency &colFreq)
{
  double max_mse, max =0.0;
  int cutAxis = 0;

  for (int i=0; i<3; ++i) {
    const int longueur = m_range[i].width();
    m_cutAxis = i;
    max_mse = cutChoice(colFreq);
    if ((max_mse > max) && (longueur>1)) {
      cutAxis = i;
      max = max_mse;
    }
  }

  return cutAxis;
}

/* L'axe apportant la plus forte diminution de a variance sur un axe */
int
ColorBinaryTree::splitVarChoice(const ColorFrequency &colFreq)
{
  double max_mse, max =0.0;
  int cutAxis = 0;
  double m0 = 0.0;
  momentsChildren m1, m2;

  for (int i=0; i<3;++i) {
    const int longueur = m_range[i].width();
    m_cutAxis = i;
    //const int index = 
    cutProject(colFreq, &m0, m1, m2);
    double diff_moy=0.0;
    for (int j=0; j<3; ++j) {
      const double X_moins_ya = m1[0][j]/m0 - m_moment1[j]/m_moment0;
      diff_moy += X_moins_ya*X_moins_ya;
    }
    if (m_moment0 == m0)
      max_mse = 0.0;
    else
      max_mse = (m0/(m_moment0-m0))*diff_moy;

    if ((max_mse > max) && (longueur>1)) {
      cutAxis = i;
      max = max_mse;
    }
  }

  return cutAxis;
}


/* L'axe de plus grande variance */
int
ColorBinaryTree::varChoice(const ColorFrequency & /* colFreq */)
{
  double var_max= 0.0;
  int cutAxis = 0;
  
  for (int i=0; i<3; ++i) {
    const int longueur = m_range[i].width();
    if ((m_var[i] > var_max)&&(longueur>1)) {
      cutAxis = i;
      var_max = m_var[i];
    }
  }
  return cutAxis;
}

/* L'axe le plus long */
int
ColorBinaryTree::lengthChoice(const ColorFrequency & /* colFreq */)
{
  int longueur_max = 0;
  int cutAxis = 0;
  
  for (int i=0; i<3; ++i) {
    const int longueur = m_range[i].width();
    if (longueur > longueur_max) {
      cutAxis = i;
      longueur_max = longueur;
    }
  }
  return cutAxis;
}

/* L'axe tel que var(i)*longueur(i) soit max */
int
ColorBinaryTree::varLengthChoice(const ColorFrequency & /* colFreq */)
{
  double mom2_max = 0.0;
  int cutAxis = 0;
  
  for (int i=0; i<3; ++i) {
    const int longueur = m_range[i].width();
    const double mom2 = m_var[i]*longueur;
    if ((mom2>mom2_max) && (longueur >1)) {
      cutAxis = i;
      mom2_max = mom2;
    }
  }
  return cutAxis;
}

/* L'axe tel que Var(i)*ecart_median(i) est max */
/* ecart_max(i) est la distance entre la moyenne sur l'axe et le point */
/* median sur cet axe. Avec pt_median(i) = ( Max(i)+Min(i) )/2 */
int
ColorBinaryTree::varMedianChoice(const ColorFrequency &colFreq)
{

  int min[3];
  int moy[3];  
  for (int i=0; i<DIM; ++i) {
      moy[i] = m_meanAxis[i]/REDUCTION;
      min[i] = m_range[i].min();
    } 
  int *median = colFreq.getMedian(min, moy);

  int cutAxis = 0;
  double mom2_max =0.0;
  for (int i=0; i<DIM; ++i) {
    const int longueur = m_range[i].width();
    double mom2 = m_var[i]*(moy[i]-median[i]);
    if ((mom2 > mom2_max) && (longueur > 1)) {
      cutAxis = i;
      mom2_max = mom2;
    }
  }

  return cutAxis;
}

//-----------------------------------------

/*******************************************/
/* Calcul de l'homogneite du cluster       */
/* valeur est egale a une de ces fonctions */
/*******************************************/

/* retourne l' MSE pondere ie :*/
/* La somme des distances au barycentre divise par le cardinal */
double 
ColorBinaryTree::weightedMSE()
{
  double error =0.0;  
  for (int i=0; i<3; ++i)
    error += m_var[i];

  return error;
}

/* Retourne la somme des distances au barycentre */
double 
ColorBinaryTree::nonWeightedMSE()
{
  double error =0.0;  
  for (int i=0; i<3; ++i)
    error += m_var[i];
  
  error = error*m_moment0;

  return error;
}

/* Retourne le cardinal */
double 
ColorBinaryTree::cardinal()
{
  int longueur_max = 0;
  
  for (int i=0; i<3; ++i) {
    const int longueur = m_range[i].width();
    if (longueur > longueur_max)
      longueur_max = longueur;
  }

  if (longueur_max <= 1)
    return 0;
  
  return m_moment0;
}


//-----------------------------------------

ColorBinaryTree *
ColorBinaryTree::createChild()
{
  ColorBinaryTree *child = new ColorBinaryTree();
  
  for (int i=0; i<3; ++i)    
    child->m_range[i] = m_range[i];
    
  child->m_isLeaf = true;
  
  child->m_axisChoiceFn = m_axisChoiceFn;
  child->m_valueFn = m_valueFn;
  child->m_partitionFn = m_partitionFn;
    
  return child;
}

void
ColorBinaryTree::createTable_aux(const ColorFrequency &colFreq)
{
  //ColorBinaryTree *child;
  double moment0[2] = {0.0, 0.0};
  momentsChildren moment1, moment2;
  //double m0, m1, m2;
  
  if (m_isLeaf) {

    const int index_decoupe = CALLMEMBERFUNCTIONP(this, m_partitionFn)(colFreq, moment0, moment1, moment2);
    
#ifdef DEBUG		
    printf("index_decoupe suivant I%d: %d\n", m_cutAxis+1, index_decoupe);
#endif

    ColorBinaryTree *child;

    /* On corrige les intervalles */
      
    assert(m_child[0] == NULL);
    child = m_child[0] = createChild();
    assert(child->m_range[m_cutAxis].min()<=index_decoupe);
    child->m_range[m_cutAxis].setMax(index_decoupe);
      
    assert(m_child[1] == NULL);
    child = m_child[1] = createChild();
    assert(child->m_range[m_cutAxis].max()>=index_decoupe);
    child->m_range[m_cutAxis].setMin(index_decoupe);
      
    /* loop on two children (right & left) */
    for (int i=0; i<2; ++i) {
      child = m_child[i];
      assert(child);
      const double m0 = child->m_moment0 = moment0[i]; 
	  
      /* boucle sur les 3 coordonnees */
      for (int j=0; j<3; ++j) {
	const double m1 = child->m_moment1[j]= moment1[i][j]; 
	const double m2 = child->m_moment2[j]= moment2[i][j]; 
	child->m_meanAxis[j]= m1/m0;
	child->m_var[j] = (m2-m1*m1/m0)/m0;
	      
#ifdef DEBUG
	printf("var et moyenne sur l'axe I%d : var =%f moyenne = %d\n", (j+1), child->m_var[j], child->m_meanAxis[j]);
#endif	
					
	if (child->m_var[j] < -1.0) {
	  fprintf(stderr, "ERREUR DANS La quantification var negative\n");
	  assert(false);
	  exit(0);
	}

      }

      child->m_cutAxis = CALLMEMBERFUNCTIONP(child, m_axisChoiceFn)(colFreq);
      assert(m_cutAxis>=0 && m_cutAxis<3);
      child->m_ecart = CALLMEMBERFUNCTIONP(child, m_valueFn)();
    }		    
            
#ifdef DEBUG
    printf("\n\n");
#endif

    /* On corrige le pere */
    m_isLeaf = false;

  }
  else {
    assert(m_child[0] != NULL);
    assert(m_child[1] != NULL);

    if ( m_child[0]->m_ecart > m_child[1]->m_ecart )
      m_child[0]->createTable_aux(colFreq);
    else
      m_child[1]->createTable_aux(colFreq);
  }

  m_ecart = std::max(m_child[0]->m_ecart, m_child[1]->m_ecart);
}

/****************************************************************/
/********** Remplie la table de couleurs  en prenant la couleur */
/********** moyenne de chaque cluster dans l'arbre              */
/****************************************************************/
void 
ColorBinaryTree::label(int *table,
		       int &numero)
{
  if (m_isLeaf) {
    int index = 3*numero;
    for (int i=0; i<3; ++i) {
      //*(table+index)= m_meanAxis[i]; //B: currently we don't use colortabel !!!
      ++index;
    }
      
    m_number = numero;
    ++numero;
  }
  else {
    assert(m_child[0] != NULL);
    assert(m_child[1] != NULL);
    m_child[0]->label(table, numero);
    m_child[1]->label(table, numero);
  }
}




